/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite;

import java.sql.SQLException;
import java.util.logging.Logger;
import org.sqlite.jdbc.JdbcSQLException;
import org.sqlite.swig.SWIGTYPE_p_sqlite3_stmt;
import org.sqlite.swig.SWIGTYPE_p_void;
import org.sqlite.types.SQLite3StmtPtr;
import org.sqlite.types.SQLite3StmtPtrPtr;
import org.sqlite.types.SQLiteTransient;
import static org.sqlite.swig.SQLite3.*;

/**
 * sqlite3_stmt wrapper class.
 * @author calico
 */
public class Statement {
    private final Database db;
    private final SWIGTYPE_p_sqlite3_stmt stmt;
    /** MANAGED Statement is NULL */
    private final SQLite3StmtPtrPtr ppStmt;
    private boolean isClosed;
    
    Statement(Database db, SWIGTYPE_p_sqlite3_stmt stmt) throws SQLException {
        this(db, stmt, null);
    }
    
    Statement(Database db, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        this(db, ppStmt.getSQLite3StmtPtr(), ppStmt);
    }
    
    private Statement(Database db, SWIGTYPE_p_sqlite3_stmt stmt, SQLite3StmtPtrPtr ppStmt) throws SQLException {
        this.db = db;
        this.stmt = stmt;
        this.ppStmt = ppStmt;
        db.addStatement(this);
    }
    
    public int getParameterCount() {
        return sqlite3_bind_parameter_count(stmt);
    }
    
    public int getParameterIndex(String parameterName) {
        return sqlite3_bind_parameter_index(stmt, parameterName);
    }
    
    public String getParameterName(int parameterIndex) {
        return sqlite3_bind_parameter_name(stmt, parameterIndex);
    }
    
    public int getColumnCount() {
        return sqlite3_column_count(stmt);
    }
    
    public int getDataCount() {
        return sqlite3_data_count(stmt);
    }
    
    public String getColumnLabel(int columnIndex) {
        return sqlite3_column_name(stmt, columnIndex - 1);
    }
    
    public String getColumnName(int columnIndex) {
        return sqlite3_column_origin_name(stmt, columnIndex - 1);
    }
    
    public int getColumnType(int columnIndex) {
        return sqlite3_column_type(stmt, columnIndex - 1);
    }
    
    public String getColumnTableName(int columnIndex) {
        return sqlite3_column_table_name(stmt, columnIndex - 1);
    }
    
    public String getColumnTypeName(int columnIndex) {
        return sqlite3_column_decltype(stmt, columnIndex - 1);
    }
    
    public String getString(int columnIndex) {
        return sqlite3_column_text(stmt, columnIndex - 1);
    }
    
    public int getInt(int columnIndex) {
        return sqlite3_column_int(stmt, columnIndex - 1);
    }
    
    public long getLong(int columnIndex) {
        return sqlite3_column_int64(stmt, columnIndex - 1);
    }
    
    public double getDouble(int columnIndex) {
        return sqlite3_column_double(stmt, columnIndex - 1);
    }
    
    public byte[] getBytes(int columnIndex) {
        return sqlite3_column_blob_by_bytes(stmt, columnIndex - 1);
    }
    
    public SWIGTYPE_p_void getBlob(int columnIndex) {
        return sqlite3_column_blob(stmt, columnIndex - 1);
    }
    
    public long getByteLength(int columnIndex) {
        return sqlite3_column_bytes(stmt, columnIndex - 1);
    }
    
    public void bindNull(int parameterIndex) throws SQLException {
        int ret = sqlite3_bind_null(stmt, parameterIndex);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    public void bindInt(int parameterIndex, int val) throws SQLException {
        int ret = sqlite3_bind_int(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    public void bindLong(int parameterIndex, long val) throws SQLException {
        int ret = sqlite3_bind_int64(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    public void bindDouble(int parameterIndex, double val) throws SQLException {
        int ret = sqlite3_bind_double(stmt, parameterIndex, val);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    private static final SQLiteTransient SQLITE_TRANSIENT = new SQLiteTransient();
    
    public void bindText(int parameterIndex, String val) throws SQLException {
        int ret = sqlite3_bind_text(stmt, parameterIndex, val, -1, SQLITE_TRANSIENT);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    public void bindBytes(int parameterIndex, byte[] val) throws SQLException {
        bindBytes(parameterIndex, val, val.length);
    }
    
    public void bindBytes(int parameterIndex, byte[] val, int bytes) throws SQLException {
        int ret = sqlite3_bind_blob_by_bytes(stmt, parameterIndex, val, bytes);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
    }
    
    public void clearBinding() throws SQLException {
        final int max = getParameterCount() + 1;
        for (int i = 1; i < max; ++i) {
            bindNull(i);
        }
    }

    public int reset() throws SQLException {
        int ret = sqlite3_reset(stmt);
        if (ret != SQLITE_OK) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
        return ret;
    }

    public int step() throws SQLException {
        int ret = 0;
        if (db.getBusyTimeout() == 0) {
            // no limit
            while ((ret = sqlite3_step(stmt)) == SQLITE_BUSY) {
                // waiting...
            }
        } else {
            ret = sqlite3_step(stmt);
            if (ret == SQLITE_BUSY) {
                // timeout
                throw new SQLException("Timeout expired.");
            }
        }
        if (ret != SQLITE_DONE && ret != SQLITE_ROW) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
        return ret;
    }

    public int execute() throws SQLException {
        int ret = step();
        if (ret != SQLITE_DONE) {
            throw new JdbcSQLException(sqlite3_db_handle(stmt));
        }
        return ret;        
    }
    
    public boolean isExpired() {
        return (sqlite3_expired(stmt) != 0);
    }
    
    public boolean producedResultSet() {
        return (getColumnCount() != 0);
    }
    
    /*public*/ long getHandle() {
        return SQLite3StmtPtr.addressOf(stmt);
    }
    
    public boolean isClosed() {
        return (db.isClosed() || isClosed);
    }
    
    public void close() throws SQLException {
        close(false);
    }

    /**
     * invoke from Database#closeStatements() only.
     */
    void closeForced() throws SQLException {
        close(true);
    }
    
    private void close(boolean isForced) throws SQLException {
        if (!isClosed) {
            int ret = sqlite3_finalize(stmt);
            if (ret == SQLITE_ABORT) {
                Logger.getLogger(Statement.class.getName()).warning("Transaction aborted.");
            } else if (ret != SQLITE_OK) {
                throw new JdbcSQLException(sqlite3_db_handle(stmt));
            }

            isClosed = true;
            if (!isForced) {
                db.removeStatement(this);
            }
            if (ppStmt != null) {
                // UNMANAGED Statement
                ppStmt.delete();
            }
        }
    }
    
    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }
    
    
}
