/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.sqlite.jdbc;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import org.sqlite.Database;
import org.sqlite.Statement;
import org.sqlite.types.SQLite3StmtPtrPtr;

/**
 *
 * @author calico
 */
public class JdbcPreparedStatement extends JdbcStatement implements PreparedStatement {

    private final String sql;
    private SQLite3StmtPtrPtr pStmt = new SQLite3StmtPtrPtr();
    private Statement stmt;
    
    public JdbcPreparedStatement(Database db, JdbcConnection conn, String sql) throws SQLException {
        super(db, conn);
        this.stmt = db.prepare(sql, pStmt);
        this.sql = sql;
        this.stmt.reset();
    }
    
    // START implements
    public ResultSet executeQuery() throws SQLException {
        validateStatementOpen();

        if (!stmt.producedResultSet()) {
            throw new SQLException("No ResultSet was produced.");
        }
        
        rs = new JdbcResultSet(this, stmt);
        cntUpdate = -1;
        return rs;
    }

    public int executeUpdate() throws SQLException {
        validateStatementOpen();

        if (stmt.producedResultSet()) {
            throw new SQLException("ResultSet was produced.");
        }
        
        stmt.execute();
        cntUpdate = db.changes();
//        cntUpdate = db.totalChanges();
        rs = null;
        stmt.reset();
        return cntUpdate;
    }

    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        validateStatementOpen();
  
        stmt.bindNull(parameterIndex);
    }

    public void setBoolean(int parameterIndex, boolean x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindInt(parameterIndex, (x ? 1 : 0));
    }

    public void setByte(int parameterIndex, byte x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindInt(parameterIndex, x);
    }

    public void setShort(int parameterIndex, short x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindInt(parameterIndex, x);
    }

    public void setInt(int parameterIndex, int x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindInt(parameterIndex, x);
    }

    public void setLong(int parameterIndex, long x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindLong(parameterIndex, x);
    }

    public void setFloat(int parameterIndex, float x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindDouble(parameterIndex, x);
    }

    public void setDouble(int parameterIndex, double x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindDouble(parameterIndex, x);
    }

    public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
        validateStatementOpen();
  
        stmt.bindText(parameterIndex, x.toString());
    }

    public void setString(int parameterIndex, String x) throws SQLException {
        validateStatementOpen();
  
        if (x != null) {
            stmt.bindText(parameterIndex, x);
        } else {
            stmt.bindNull(parameterIndex);
        }
    }

    public void setBytes(int parameterIndex, byte[] x) throws SQLException {
        validateStatementOpen();
  
        if (x != null) {
            stmt.bindBytes(parameterIndex, x);
        } else {
            stmt.bindNull(parameterIndex);
        }
    }

    public void setDate(int parameterIndex, Date x) throws SQLException {
        validateStatementOpen();
  
        if (x != null) {
            stmt.bindText(parameterIndex, new SimpleDateFormat("yyyy-MM-dd").format(x));
        } else {
            stmt.bindNull(parameterIndex);
        }
    }

    public void setTime(int parameterIndex, Time x) throws SQLException {
        validateStatementOpen();
  
        if (x != null) {
            stmt.bindText(parameterIndex, new SimpleDateFormat("HH:mm:ss").format(x));
        } else {
            stmt.bindNull(parameterIndex);
        }
    }

    public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
        validateStatementOpen();
  
        if (x != null) {
            stmt.bindText(parameterIndex, new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(x));
        } else {
            stmt.bindNull(parameterIndex);
        }
    }

    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void clearParameters() throws SQLException {
        validateStatementOpen();

        stmt.clearBinding();
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType, int scaleOrLength) throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public void setObject(int parameterIndex, Object x) throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public boolean execute() throws SQLException {
        validateStatementOpen();

        return (executeQuery() != null);
    }

    public void addBatch() throws SQLException {
        // TODO do implements!
        throw new UnsupportedOperationException("Not implemented yet.");
    }

    public void setCharacterStream(int parameterIndex, Reader x, int length) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setRef(int parameterIndex, Ref x) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setBlob(int parameterIndex, Blob x) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setClob(int parameterIndex, Clob x) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void setArray(int parameterIndex, Array x) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public ResultSetMetaData getMetaData() throws SQLException {
        return new JdbcResultSetMetaData(stmt);
    }

    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
        final Calendar calendar = (Calendar) cal.clone();
        calendar.setTime(x);
        setDate(parameterIndex, new Date(calendar.getTime().getTime()));
    }

    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
        final Calendar calendar = (Calendar) cal.clone();
        calendar.setTime(x);
        setTime(parameterIndex, new Time(calendar.getTime().getTime()));
    }

    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
        final Calendar calendar = (Calendar) cal.clone();
        calendar.setTime(x);
        setTimestamp(parameterIndex, new Timestamp(calendar.getTime().getTime()));
    }

    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
        setNull(parameterIndex, sqlType);
    }

    public void setURL(int parameterIndex, URL x) throws SQLException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public ParameterMetaData getParameterMetaData() throws SQLException {
        return new JdbcParameterMetaData(stmt);
    }
    // END implements

    @Override
    public boolean isClosed() throws SQLException {
        return (pStmt == null || super.isClosed());
    }
    
    @Override
    public void close() throws SQLException {
        if (pStmt != null) {
            // TODO Statement#close() の必要要件を満たしていない
            // 注: Statement オブジェクトがクローズされるとき、「その現在の ResultSet オブジェクトが存在すれば」、それもクローズされます。
            if (stmt != null) {
                stmt.close();
                stmt = null;
            }
            pStmt.delete();
            pStmt = null;
        }
        super.close();
    }

    /**
     * PreparedStatement#close()時にResultSetが一緒にcloseされないようにResultSetを切り離す。
     * WARNING! All parameters are unbinded. 
     * @param drs
     * @throws java.sql.SQLException
     */
    @Override
    public void detach(ResultSet drs) throws SQLException {
        validateStatementOpen();
        
        if (rs != null && drs.getStatement() == this) {
            // detach ResultSet
            rs = null;

            // free bind memory
            stmt.clearBinding();

            // renew Statement
            stmt = db.prepare(sql, pStmt);
            stmt.reset();
        }
    }

    /**
     * ResultSetを切り離してPrepaeredStatementをcloseする。
     * @param drs
     * @throws java.sql.SQLException
     */
    public void detachAndClose(ResultSet drs) throws SQLException {
        validateStatementOpen();
        
        if (rs != null && drs.getStatement() == this) {
            // detach ResultSet
            rs = null;

//            // free bind memory
//            stmt.clearBinding();  // clearするとstep()を呼び出しても次の値が取れなくなる
            
            // detach Statement
            stmt = null;
            
            close();
        }
    }
    
    @Override
    protected void finalize() throws Throwable {
        close();
        super.finalize();
    }

    public int findParameter(String parameterName) throws SQLException {
        validateStatementOpen();

        final int parameterIndex = stmt.getParameterIndex(parameterName);
        if (parameterIndex == 0) {
            throw new SQLException("Parameter name is not found.");
        }
        return parameterIndex;
    }
    
    public void setNull(String parameterName, int sqlType) throws SQLException {
        setNull(findParameter(parameterName), sqlType);
    }

    public void setBoolean(String parameterName, boolean x) throws SQLException {
        setBoolean(findParameter(parameterName), x);
    }

    public void setByte(String parameterName, byte x) throws SQLException {
        setByte(findParameter(parameterName), x);
    }

    public void setShort(String parameterName, short x) throws SQLException {
        setShort(findParameter(parameterName), x);
    }

    public void setInt(String parameterName, int x) throws SQLException {
        setInt(findParameter(parameterName), x);
    }

    public void setLong(String parameterName, long x) throws SQLException {
        setLong(findParameter(parameterName), x);
    }

    public void setFloat(String parameterName, float x) throws SQLException {
        setFloat(findParameter(parameterName), x);
    }

    public void setDouble(String parameterName, double x) throws SQLException {
        setDouble(findParameter(parameterName), x);
    }

    public void setBigDecimal(String parameterName, BigDecimal x) throws SQLException {
        setBigDecimal(findParameter(parameterName), x);
    }

    public void setString(String parameterName, String x) throws SQLException {
        setString(findParameter(parameterName), x);
    }

    public void setBytes(String parameterName, byte[] x) throws SQLException {
        setBytes(findParameter(parameterName), x);
    }

    public void setDate(String parameterName, Date x) throws SQLException {
        setDate(findParameter(parameterName), x);
    }

    public void setTime(String parameterName, Time x) throws SQLException {
        setTime(findParameter(parameterName), x);
    }

    public void setTimestamp(String parameterName, Timestamp x) throws SQLException {
        setTimestamp(findParameter(parameterName), x);
    }
    
    public void setBlob(String parameterName, Blob x) throws SQLException {
        setBlob(findParameter(parameterName), x);
    }

    public void setDate(String parameterName, Date x, Calendar cal) throws SQLException {
        setDate(findParameter(parameterName), x, cal);
    }

    public void setTime(String parameterName, Time x, Calendar cal) throws SQLException {
        setTime(findParameter(parameterName), x, cal);
    }

    public void setTimestamp(String parameterName, Timestamp x, Calendar cal) throws SQLException {
        setTimestamp(findParameter(parameterName), x, cal);
    }

    public String findParameterName(int parameterIndex) throws SQLException {
        validateStatementOpen();
        
        return stmt.getParameterName(parameterIndex);
    }
}
