package org.sqlite.udf;

import java.sql.SQLException;
import java.util.logging.Logger;
import org.sqlite.callback.NamedCallback;

/**
 * User-Defined function class.
 * @author calico
 * @see <a href="http://sqlite.org/c3ref/create_function.html">Create Or Redefine SQL Functions</a>
 * @see org.sqlite.jdbc.JdbcConnection#createFunction(Function)
 * @see org.sqlite.jdbc.JdbcConnection#dropFunction(Function)
 * @see org.sqlite.jdbc.JdbcConnection#dropFunction(java.lang.String)
 */
public abstract class Function extends NamedCallback {
    
    /** the number of arguments that the SQL function or aggregate takes */
    protected final int argc;
    
    /**
     * invoke Function(String, -1) constructor.
     * @param name the function name
     * @see Function#Function(java.lang.String, int)
     */
    protected Function(String name) {
        this(name, -1);
    }
    
    /**
     * create Function object with SQLITE_UTF8.
     * @param name the function name
     * @param argc the number of arguments that the SQL function or aggregate takes.
     * @see NamedCallback#NamedCallback(java.lang.String)
     */
    protected Function(String name, int argc) {
        super(name);
        this.argc = argc;
    }
    
    /**
     * create Function object.
     * @param name the function name
     * @param argc the number of arguments that the SQL function or aggregate takes.
     * @param enc the specifies what text encoding this function prefers for its parameters.
     * @see <a href="http://sqlite.org/c3ref/c_any.html">Text Encodings</a>
     * @see NamedCallback#NamedCallback(java.lang.String, int)
     */
    protected Function(String name, int argc, int enc) {
        super(name, enc);
        this.argc = argc;
    }
    
    /**
     * Returns the number of arguments that the SQL function or aggregate takes.
     * @return the number of arguments that the SQL function or aggregate takes
     */
    public int getArgumentCount() {
        return argc;
    }
    
    /**
     * Called from the sqlite3_step() function.
     * @param context the sqlite3_context* value
     * @param argc the number of arguments
     * @param value the sqlite3_value** value
     * @see #xFunc(org.sqlite.udf.Context)
     */
    protected final void xFunc(long context, int argc, long value) {
        // TODO Mysaifu JVMのBug#11980が解決したらアクセス修飾子をprivateに戻すこと！
        // @see http://sourceforge.jp/tracker/index.php?func=detail&aid=11980&group_id=1890&atid=7027
        Context ctx = null;
        try {
            ctx = new Context(context, argc, value);
            xFunc(ctx);
            
        } catch (SQLException ex) {
            ctx.resultErrorCode(ex.getErrorCode());
            ctx.resultError(ex.getMessage());
            
        } catch (Throwable th) {
            if (ctx != null) {
                final String msg = th. toString();
                ctx.resultError((msg != null ? msg : "Unknown error."));
                
            } else {
                Logger.getLogger(Function.class.getName()).fine("Exception occurred: " + th.toString());
            }
        }
    }
    
    /**
     * Called from the sqlite3_step() function.
     * @param ctx sqlite3_context wrapper object
     * @throws java.sql.SQLException
     */
    protected abstract void xFunc(Context ctx) throws SQLException;
}
