/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.ActionEvent.ACTION_PERFORMED;
import static java.awt.event.InputEvent.*;
import static java.awt.event.KeyEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static net.argius.stew.ui.window.Menu.Item.*;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

/**
 * EBhE\̃j[o[B
 */
final class Menu extends JMenuBar implements PropertyChangeListener {

    /**
     * j[ځB
     */
    enum Item {

        NEW,
        CLOSE,
        QUIT,
        CUT,
        COPY,
        PASTE,
        SELECT_ALL,
        FIND,
        TOGGLE_FOCUS,
        CLEAR_MESSAGE,
        STATUS_BAR,
        INFO_TREE,
        COLUMN_NUMBER,
        ALWAYS_ON_TOP,
        REFRESH,
        WIDEN,
        NARROW,
        ADJUST_COLUMN_WIDTH,
        AUTO_ADJUST_MODE,
        AUTO_ADJUST_MODE_NONE,
        AUTO_ADJUST_MODE_HEADER,
        AUTO_ADJUST_MODE_VALUE,
        AUTO_ADJUST_MODE_HEADERANDVALUE,
        EXECUTE,
        BREAK,
        HISTORY_BACK,
        HISTORY_NEXT,
        RETRIEVE_HISTORY,
        ROLLBACK,
        COMMIT,
        CONNECT,
        DISCONNECT,
        ENCRYPTION_KEY,
        EDIT_CONNECTORS,
        SORT,
        IMPORT,
        EXPORT,
        HELP,
        ABOUT;

    }

    private List<JMenuItem> lockingTargets;
    private List<JMenuItem> unlockingTargets;
    private Map<Item, JMenuItem> menuItems;

    Menu() {
        this.lockingTargets = new ArrayList<JMenuItem>();
        this.unlockingTargets = new ArrayList<JMenuItem>();
        this.menuItems = new LinkedHashMap<Item, JMenuItem>();
        final int shortcutKey = getMenuShortcutKeyMask();
        add(createJMenu("file",
                        'F',
                        createJMenuItem(NEW, 'N', getKeyStroke(VK_N, shortcutKey)),
                        createJMenuItem(CLOSE, 'C', getKeyStroke(VK_W, shortcutKey)),
                        new JSeparator(),
                        createJMenuItem(QUIT, 'Q', getKeyStroke(VK_Q, shortcutKey))));
        add(createJMenu("edit",
                        'E',
                        createJMenuItem(CUT, 'T', getKeyStroke(VK_X, shortcutKey)),
                        createJMenuItem(COPY, 'C', getKeyStroke(VK_C, shortcutKey)),
                        createJMenuItem(PASTE, 'P', getKeyStroke(VK_V, shortcutKey)),
                        createJMenuItem(SELECT_ALL, 'A', getKeyStroke(VK_A, shortcutKey)),
                        new JSeparator(),
                        createJMenuItem(FIND, 'F', getKeyStroke(VK_F, shortcutKey)),
                        new JSeparator(),
                        createJMenuItem(TOGGLE_FOCUS, 'G', getKeyStroke(VK_G, shortcutKey)),
                        createJMenuItem(CLEAR_MESSAGE, 'M')));
        add(createJMenu("view",
                        'V',
                        createJCheckBoxMenuItem(STATUS_BAR, 'S', null),
                        createJCheckBoxMenuItem(COLUMN_NUMBER, 'C', null),
                        createJCheckBoxMenuItem(INFO_TREE, 'I', null),
                        createJCheckBoxMenuItem(ALWAYS_ON_TOP, 'T', null),
                        new JSeparator(),
                        createJMenuItem(REFRESH, 'R', getKeyStroke(VK_F5, 0)),
                        new JSeparator(),
                        createJMenuItem(WIDEN, 'W', getKeyStroke(VK_PERIOD, shortcutKey)),
                        createJMenuItem(NARROW, 'N', getKeyStroke(VK_COMMA, shortcutKey)),
                        createJMenuItem(ADJUST_COLUMN_WIDTH, 'A', getKeyStroke(VK_SLASH,
                                                                               shortcutKey)),
                        createAutoAdjustModeItems()));
        add(createJMenu("command",
                        'C',
                        createJMenuItem(EXECUTE, 'X', getKeyStroke(VK_M, shortcutKey)),
                        createJMenuItem(BREAK, 'B', getKeyStroke(VK_PAUSE, shortcutKey)),
                        new JSeparator(),
                        createJMenuItem(HISTORY_BACK, 'P', getKeyStroke(VK_UP, shortcutKey)),
                        createJMenuItem(HISTORY_NEXT, 'N', getKeyStroke(VK_DOWN, shortcutKey)),
                        createJMenuItem(ROLLBACK, 'R'),
                        createJMenuItem(COMMIT, 'M'),
                        new JSeparator(),
                        createJMenuItem(CONNECT, 'C', getKeyStroke(VK_E, shortcutKey)),
                        createJMenuItem(DISCONNECT, 'D', getKeyStroke(VK_D, shortcutKey)),
                        new JSeparator(),
                        createJMenuItem(ENCRYPTION_KEY, 'K'),
                        createJMenuItem(EDIT_CONNECTORS, 'E')));
        add(createJMenu("data",
                        'D',
                        createJMenuItem(SORT, 'S', getKeyStroke(VK_S, ALT_DOWN_MASK)),
                        createJMenuItem(IMPORT, 'I'),
                        createJMenuItem(EXPORT, 'X', getKeyStroke(VK_S, shortcutKey
                                                                        | SHIFT_DOWN_MASK))));
        add(createJMenu("help",
                        'H',
                        createJMenuItem(HELP, 'H'),
                        new JSeparator(),
                        createJMenuItem(ABOUT, 'A')));
        setEnabledStates(false);
    }

    /* @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent) */
    public void propertyChange(PropertyChangeEvent e) {
        final String propertyName = e.getPropertyName();
        final Object source = e.getSource();
        if (source instanceof JLabel && propertyName.equals("ancestor")) {
            menuItems.get(STATUS_BAR).setSelected(((JLabel)source).isVisible());
        } else if (source instanceof ResultSetTable && propertyName.equals("showNumber")) {
            menuItems.get(COLUMN_NUMBER).setSelected((Boolean)e.getNewValue());
        } else if (source instanceof DatabaseInfoTree) {
            menuItems.get(INFO_TREE).setSelected(((Component)source).isEnabled());
        } else if (source instanceof JFrame && propertyName.equals("alwaysOnTop")) {
            menuItems.get(ALWAYS_ON_TOP).setSelected((Boolean)e.getNewValue());
        } else if (source instanceof WindowOutputProcessor && propertyName.equals("autoAdjustMode")) {
            ((JCheckBoxMenuItem)menuItems.get(e.getNewValue())).setSelected(true);
        }
    }

    /**
     * [h̃Tuj[𐶐B
     * @return JMenu
     */
    private JMenu createAutoAdjustModeItems() {
        List<JComponent> a = new ArrayList<JComponent>();
        ButtonGroup buttonGroup = new ButtonGroup();
        Map<Item, Character> mnemonicInfo = new HashMap<Item, Character>();
        mnemonicInfo.put(AUTO_ADJUST_MODE_NONE, 'N');
        mnemonicInfo.put(AUTO_ADJUST_MODE_HEADER, 'H');
        mnemonicInfo.put(AUTO_ADJUST_MODE_VALUE, 'V');
        mnemonicInfo.put(AUTO_ADJUST_MODE_HEADERANDVALUE, 'A');
        for (final Item item : new Item[]{AUTO_ADJUST_MODE_NONE, AUTO_ADJUST_MODE_HEADER,
                                          AUTO_ADJUST_MODE_VALUE, AUTO_ADJUST_MODE_HEADERANDVALUE}) {
            final char mnemonic = mnemonicInfo.get(item);
            final JCheckBoxMenuItem o = new JCheckBoxMenuItem(getString(item.name(), mnemonic));
            o.setMnemonic(mnemonic);
            if (item == AUTO_ADJUST_MODE_NONE) {
                o.setSelected(true);
            }
            menuItems.put(item, o);
            a.add(o);
            buttonGroup.add(o);
            o.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {
                    fireActionPerformed(o, item);
                }

            });
        }
        return createJMenu(AUTO_ADJUST_MODE.name(), 'M', a.toArray(new JComponent[a.size()]));
    }

    /**
     * V[gJbgL[̐ݒB
     * @param item
     * @param keyStroke
     */
    void setAccelerator(Item item, KeyStroke keyStroke) {
        menuItems.get(item).setAccelerator(keyStroke);
    }

    /**
     * ActionListener̒ǉB
     * @param listener ActionListener
     */
    void addActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.add(ActionListener.class, listener);
    }

    /**
     * ActionListener̍폜B
     * @param listener ActionListener
     */
    void removeActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.remove(ActionListener.class, listener);
    }

    /**
     * j[IꂽƂʒmB
     * @param source \[XIuWFNg
     * @param id ID(j[)
     */
    void fireActionPerformed(Object source, Item id) {
        try {
            ActionEvent event = new ActionEvent(source, ACTION_PERFORMED, id.name());
            for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
                listener.actionPerformed(event);
            }
        } catch (Exception ex) {
            WindowOutputProcessor.showErrorDialog(getParent(), ex);
        }
    }

    /**
     * eڂ̗LԂݒ肷B
     * @param commandStarted R}hJnǂ
     */
    void setEnabledStates(boolean commandStarted) {
        final boolean lockingTargetsState = !commandStarted;
        for (JMenuItem item : lockingTargets) {
            item.setEnabled(lockingTargetsState);
        }
        final boolean unlockingTargetsState = commandStarted;
        for (JMenuItem item : unlockingTargets) {
            item.setEnabled(unlockingTargetsState);
        }
    }

    /**
     * JMenu̐B
     * @param key \̃L[
     * @param mnemonic 蓖ĕ
     * @param components R|[lg̔z
     * @return JMenu    
     */
    private static JMenu createJMenu(String key, char mnemonic, JComponent... components) {
        JMenu menu = new JMenu(getString(key, mnemonic));
        menu.setMnemonic(mnemonic);
        for (JComponent component : components) {
            menu.add(component);
        }
        return menu;
    }

    /**
     * JMenuItem̐B
     * @param key j[L[
     * @param mnemonic 蓖ĕ
     * @return JMenuItem
     */
    private JMenuItem createJMenuItem(Item key, char mnemonic) {
        return createJMenuItem(key, mnemonic, null);
    }

    /**
     * JMenuItem̐B
     * @param key j[L[
     * @param mnemonic 蓖ĕ
     * @param shortcut V[gJbgL[
     * @return JMenuItem
     */
    private JMenuItem createJMenuItem(Item key, char mnemonic, KeyStroke shortcut) {
        return bind(new JMenuItem(getString(key.name(), mnemonic), mnemonic), key, shortcut);
    }

    /**
     * JCheckBoxMenuItem̐B
     * @param key j[L[
     * @param mnemonic 蓖ĕ
     * @param shortcut V[gJbgL[
     * @return JCheckBoxMenuItem
     */
    private JCheckBoxMenuItem createJCheckBoxMenuItem(Item key, char mnemonic, KeyStroke shortcut) {
        JCheckBoxMenuItem o = new JCheckBoxMenuItem(getString(key.name(), mnemonic));
        o.setMnemonic(mnemonic);
        bind(o, key, shortcut);
        return o;
    }

    /**
     * JMenuItem(CxgݒȂǋʂ)ĂsB
     * @param o JMenuItem
     * @param key j[L[
     * @param shortcut V[gJbgL[
     * @return čς݂JMenuItem
     */
    private JMenuItem bind(JMenuItem o, final Item key, KeyStroke shortcut) {
        menuItems.put(key, o);
        o.setName(key.name());
        o.setIcon(getImageIcon(String.format("menu-%s.png", key)));
        o.setDisabledIcon(getImageIcon(String.format("menu-disabled-%s.png", key)));
        if (shortcut != null) {
            o.setAccelerator(shortcut);
        }
        o.addActionListener(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                fireActionPerformed(e.getSource(), key);
            }

        });
        switch (key) {
            case CLOSE:
            case QUIT:
            case CUT:
            case COPY:
            case PASTE:
            case SELECT_ALL:
            case FIND:
            case CLEAR_MESSAGE:
            case REFRESH:
            case WIDEN:
            case NARROW:
            case ADJUST_COLUMN_WIDTH:
            case AUTO_ADJUST_MODE:
            case EXECUTE:
            case HISTORY_BACK:
            case HISTORY_NEXT:
            case CONNECT:
            case DISCONNECT:
            case SORT:
            case EXPORT:
                lockingTargets.add(o);
                break;
            case BREAK:
                unlockingTargets.add(o);
                break;
            default:
        }
        return o;
    }

    /**
     * O̎擾
     * @param key bZ[WL[
     * @return 
     */
    private static String getString(String key, Object... args) {
        return Resource.getString("Menu." + key, args);
    }

}
