
--
-- Copyright (C) 2016  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


----------------------------------------------------------------
with interfaces;
with interfaces.c;


with matutils;  use matutils;
with gametypes;  use gametypes;

with ada.numerics.generic_elementary_functions;
with text_io; use text_io;
with utex;

with gl;  use gl;

with sdl;  use sdl;


with cubemapobj;
with rectobj;
with pictobj;

with gl, gl.binding, gl.pointers;
with glu, glu.binding, glu.pointers;
with glext, glext.binding, glext.pointers;

with system;
with Interfaces.C;
use  type interfaces.c.unsigned;
with Interfaces.C.Pointers;
with interfaces.c.strings;

with pngloader;
with shader;

with snd4ada_hpp; --use snd4ada_hpp;



package body gameutils is

	use pngloader;
	use shader;








function odd( i: integer ) return boolean is
begin
	return ( i mod 2 = 1 );
end odd;



function min( x,y: float ) return float is
begin
	if x<y then return x;
	else return y; end if;
end min;

	function mini(i,j: integer) return integer is
	begin
		if i<j then return i;
		else return j; end if;
	end mini;


	procedure myassert( 
		condition : boolean;  
		flag: integer:=0;
		msg: string := ""
		) is
	begin
	  if condition=false then
			put("ASSERTION Failed!  ");
			if flag /= 0 then
				put( "@ " & integer'image(flag) &" : " );
			end if;
			put_line(msg);
			new_line;
			raise program_error;
	  end if;
	end myassert;




function max( x,y : float ) return float is
begin
	if y>x then return y;
	else return x; end if;
end;

function sqr( x:float ) return float is
begin
	return x*x;
end;

function dotprod( x,y,z, a,b,c : float ) return float is
begin
	return x*a + y*b + z*c;
end;









function hordistance( x1,y1, x2,y2 : float ) return float is
begin
	return fmath.sqrt( sqr(x2-x1) + sqr(y2-y1) );
end hordistance;

function signum( x : float ) return float is
begin
	if x>0.0 then
		return +1.0;
	elsif x<0.0 then
		return -1.0;
	else
		return 0.0;
	end if;
end signum;






	use interfaces.c;
	use interfaces.c.strings;
	use glext;
	use glext.pointers;
	use glext.binding;
	use gl;
	use gl.binding;
	use gl.pointers;




procedure InitSDL( width, height : glint;  flags:Uint32;  name: string ) is

use system;

  ires, jerror, error, cver : interfaces.c.int;
  bresult : SDL_bool;

  compiled, linked : aliased SDL_version;

begin

	-- Careful!  Only initialize what we use (otherwise exe won't run):
	error := SDL_Init(
		SDL_INIT_TIMER or
		SDL_INIT_EVENTS or 
		SDL_INIT_VIDEO);
	myassert( error = 0, 1000 );

	jerror := SDL_Init(
		SDL_INIT_GAMECONTROLLER or
		SDL_INIT_JOYSTICK );

	joystik:=false;
	gamepad:=false;
	if  jerror = 0  and then SDL_NumJoysticks>=1  then
		jsa := SDL_JoystickOpen(0);
		gamepad := (sdl_joysticknumaxes(jsa) >= 4);
		joystik := not gamepad;
		put_line("#axes="& glint'image(sdl_joysticknumaxes(jsa)) );
		put_line("#btns="& glint'image(sdl_joysticknumbuttons(jsa)) );
		ires := SDL_JoystickEventState(SDL_QUERY); -- ignore ires (?might this stop spinning?)
		SDL_JoystickUpdate;
		axis_lx := SDL_JoystickGetAxis(jsa, 0);
		axis_ly := SDL_JoystickGetAxis(jsa, 1);
	end if;

	if gamepad then
		put_line("...#axes>=4 so I'm guessing controller is a gamepad...initialized");
	elsif joystik then
		put_line("...#axes<=3 so I'm guessing controller is a joystick...initialized");
	else
		put_line("...no game controller detected...");
	end if;

---------- begin 14feb15 insert ------------------------------------------------
	SDL_SOURCEVERSION( compiled'access );
	put_line("We compiled against SDL version "
		&Uint8'image(compiled.major)&"."
		&Uint8'image(compiled.minor)&"."
		&Uint8'image(compiled.patch) );
	cver := SDL_COMPILEDVERSION;  
	put_line("SDL_compiledversion="&glint'image(cver));
	SDL_GetVersion( linked'access );
	put_line("We linked against SDL version "
		&Uint8'image(linked.major)&"."
		&Uint8'image(linked.minor)&"."
		&Uint8'image(linked.patch) );
---------- end 14feb15 insert --------------------------------------------------

	bresult := SDL_SetHint( SDL_HINT_RENDER_VSYNC, "1" );
	myassert( bresult = SDL_TRUE, 1001 );
	bresult := SDL_SetHint( SDL_HINT_RENDER_SCALE_QUALITY, "1" );
	myassert( bresult = SDL_TRUE, 1002 );




	--// Turn on double buffering.
	error := SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);
	myassert( error = 0, 1003 );
	error := SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
	myassert( error = 0, 1004 );
	error := SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 8);
	myassert( error = 0, 1005 );





	error := SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 3);
	myassert( error = 0, 1006 );
	error := SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 3);
	myassert( error = 0, 1007 );




	error := SDL_GL_SetAttribute( SDL_GL_CONTEXT_PROFILE_MASK, 
											SDL_GL_CONTEXT_PROFILE_CORE );
	myassert( error = 0, 1008 );

	-- Note that OSX currently requires the forward_compatible flag!
	error := SDL_GL_SetAttribute( SDL_GL_CONTEXT_FLAGS, 
											SDL_GL_CONTEXT_FORWARD_COMPATIBLE_FLAG );
	myassert( error = 0, 1009 );



	error := SDL_GL_SetAttribute(SDL_GL_MULTISAMPLESAMPLES, 4);
	myassert( error = 0, 1010 );



	mainWindow := SDL_CreateWindow( To_C(name,true) , 
			SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, 
			width, height, flags);


	mainGLContext := SDL_GL_CreateContext(mainWindow);

	error := SDL_GL_MakeCurrent( mainWindow, mainGLContext );
	myassert( error = 0, 1011 );


end InitSDL;













package myfloat_io is new text_io.float_io(float);

package mygint_io is new text_io.integer_io(glint);


--we skip blank lines and comments beginning with "#":
function is_blank2( line : string; len:integer ) return boolean is
begin

	if( len < 1 ) then return true; end if;

	if line( line'first )='#' then return true;
	elsif line( line'first ) = ' ' then	return true; end if;

	return false;

end is_blank2;






procedure GetInt( Rcd:string;
						Bgn:in out natural;
						Int: in out glint ) is
begin
mygint_io.get( From => Rcd(Bgn..Rcd'last),
					Item => Int,
					Last => Bgn);
Bgn:=Bgn+1;
end GetInt;



procedure getNbInt(tfile:file_type; rcd: in out string; k: in out glint) is
	len,bgn: natural:=1;
begin
	while not end_of_file(tfile) loop
		get_line(tfile, rcd, len);
		if not is_blank2(rcd,len) then exit; end if;
	end loop;
	bgn:=rcd'first;

	GetInt(rcd,bgn,k);

end getNbInt;


procedure GetFlt( Rcd:string;
					  Bgn:in out natural;
					  Flt: in out float ) is
nd: positive;
begin
myfloat_io.get( From => Rcd(Bgn..Rcd'last),
					 Item => Flt,
					 Last => nd);
Bgn := nd+1;
end GetFlt;


procedure getNbFlt(tfile:file_type; rcd: in out string; t: in out float) is
	len,bgn: natural:=1;
begin
	while not end_of_file(tfile) loop
		get_line(tfile, rcd, len);
		if not is_blank2(rcd,len) then exit; end if;
	end loop;
	bgn:=rcd'first;
	GetFlt(rcd,bgn,t);
end getNbFlt;












procedure first_prep is -- main program setup
	rcd : string(1..80);
	ret : glint; --interfaces.c.int;
begin

	snd4ada_hpp.initSnds;

	water := snd4ada_hpp.initLoop(
		Interfaces.C.Strings.New_String("data/lake-water.ogg"),50);

	misr := snd4ada_hpp.initLoop(
		Interfaces.C.Strings.New_String("data/misrlu.ogg"),90);

	turk := snd4ada_hpp.initLoop(
		Interfaces.C.Strings.New_String("data/turkish.ogg"),90);

------- now for transient sounds -----------------------------------

	fanfare := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/fanfare.wav"),50);

	stone := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/concrete.ogg"),99);

	down := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/putdown.wav"),50);

	up := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/pickup.wav"),50);

	die := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/dragondie.wav"),99);

	eat := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/eaten.wav"),99);

	roar := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/roar.wav"),80);

	won := snd4ada_hpp.initSnd(
		Interfaces.C.Strings.New_String("data/won.wav"),80);


	if 
		water<0 or misr<0 or turk<0 or 
		fanfare<0 or stone<0 or down<0 or up<0 or
		die<0 or eat<0 or roar<0 or won<0
	then
		put_line("snd4ada_hpp.initSnds ERROR");
		raise program_error;
	end if;



------- begin SDL prep ---------------------------------------------------------

	ret := SDL_Init(SDL_INIT_VIDEO);
	should_be_zero := SDL_GetCurrentDisplayMode(0, current'access);
	myassert( should_be_zero = 0, 1015 );

	-- MacOSX:  HDPI is normally controlled in the "properties"
	-- box, but does not work for this app.  If there is jerkiness
	-- on your Retina display, use low dpi, which still looks 
	-- pretty good, albeit with some noticable aliasing.
	contextFlags := 
		SDL_WINDOW_SHOWN or SDL_WINDOW_OPENGL or 
		SDL_WINDOW_FULLSCREEN_DESKTOP or SDL_WINDOW_ALLOW_HIGHDPI;

	InitSDL(current.w, current.h, contextFlags, "AdaVenture");
	winwidth:=current.w;
	winheight:=current.h;


	-- this too runs fine...used for testing purposes...
	--contextFlags := SDL_WINDOW_SHOWN or SDL_WINDOW_OPENGL;
	--winwidth:=1000; --1400;
	--winheight:=600; --800;
	--InitSDL(winwidth,winheight, contextFlags, "AdaGate");


-- if joystik or gamepad, read settings here:
	if FileExists(cfgfile) then -- takes precedence over defaults
		put_line("game-controller settings file found");
		text_io.open(tfile, in_file, cfgfile);

		getNbInt(tfile,rcd,gshtl); 
		getNbInt(tfile,rcd,gshtr); 
		getNbInt(tfile,rcd,gjmp); 
		getNbFlt(tfile,rcd,Lsens); 
		getNbFlt(tfile,rcd,Rsens); 

------- end gamepad;  begin joystik --------------------

		getNbInt(tfile,rcd,jbak); 
		getNbInt(tfile,rcd,jfor); 
		getNbInt(tfile,rcd,jshtl); 
		getNbInt(tfile,rcd,jshtr); 
		getNbInt(tfile,rcd,jjmp); 
		getNbFlt(tfile,rcd,Jsens); 

		text_io.close(tfile);

	else
		put(cfgfile & " file not found...");
		put_line("Using default game controller settings");
		gshtl:=4; --shoot left
		gshtr:=5; --shoot right
		gjmp :=8; --jump

		jbak :=0; --moveback
		jfor :=1; --forward
		jshtl:=2; --shoot left
		jshtr:=3; --shoot right
		jjmp :=7; --jump
	end if;

	if joystik then --we actually use only Lsens or Rsens
		Lsens:=Jsens;
		Rsens:=Jsens; -- Jsens is not used
	end if;










	utex.inittext2d("data/rods3whk.png", integer(winwidth),integer(winheight));
	put_line( "Window: wid-X-hit :" 
		& interfaces.c.int'image(winwidth)&" X "
		& interfaces.c.int'image(winheight) );

	cursor := SDL_CreateSystemCursor(SDL_SYSTEM_CURSOR_CROSSHAIR);
	SDL_SetCursor(cursor);
	error := SDL_SetRelativeMouseMode(SDL_TRUE);
	myassert( error = 0, 1016 );

	SDL_GL_GetDrawableSize( mainWindow, Fwid'access, Fhit'access );
	glViewport(0,0,Fwid,Fhit);

	put_line( "Drawable: Fwid-X-Fhit : "
		&interfaces.c.int'image(Fwid)&" X "
		& interfaces.c.int'image(Fhit) );

	key_map := sdl_getkeyboardstate(numkeys'access);
	--put_line("...numkeys=" & interfaces.c.int'image(numkeys) ); -- 512
	--myassert( sdl.keyrange'last <= numkeys, 1017 );




	glgenvertexarrays(1, vertexarrayid'address );
	glbindvertexarray(vertexarrayid);

	glactivetexture(gl_texture0); -- moved here 5nov14 (outside main loop)

	glgenbuffers(1, vertbuff'address);
	glgenbuffers(1, rgbbuff'address);
	glgenbuffers(1, uvbuff'address);
	glgenbuffers(1, elembuff'address);




	glenable(gl_depth_test);
	gldepthfunc( gl_lequal );
	glenable( gl_cull_face );

	--gldepthrange(0.0, 99.0); --does not help 14sep16

	glShadeModel(gl_smooth);



	-- 6nov14  theoretically reduces aliasing (can't tell for sure):
	glEnable(GL_MULTISAMPLE);
	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
	glHint(GL_POLYGON_SMOOTH_HINT, GL_NICEST);


	level:=0;


end first_prep;









--calculate the radian measure of the solid angle between two vectors
function angl( x1,y1,z1, x2,y2,z2 : float ) return float is
	len1 : constant float := fmath.sqrt(x1*x1+y1*y1+z1*z1);
	len2 : constant float := fmath.sqrt(x2*x2+y2*y2+z2*z2);
	dot12, cosang : float;
begin
myassert(len1>0.01, 191, "angl: len1=0.01");
myassert(len2>0.01, 192, "angl: len2=0.01");
-- note:  if these ever fail, simply define offending length to be 0.01

	dot12:=x1*x2+y1*y2+z1*z2;
	cosang := dot12/len1/len2;
	return fmath.arccos(cosang);
end angl;






-- adjust ang1 by multiples of 2pi
-- to minimize ang1-ang0:
function branchnear( ang0, ang1 : float ) return float is
	dd, ang: float := ang1;
begin 
	dd:=ang-ang0;
	while abs(dd)>onepi loop
		ang := ang - signum(dd)*twopi;
		dd:=ang-ang0;
	end loop;
	myassert( abs(ang-ang0)<onepi );
	return ang;
end branchnear;




procedure setCamAng is
	ghoriang : float;
begin
--put_line("gu:614:enter setCamAng");
--put(", xme="&float'image(xme));
--put(", xcam="&float'image(xcam));
--put(", zme="&float'image(zme));
--put(", zcam="&float'image(zcam));
--put(", horiang="&float'image(horiang));
--new_line;

	if abs(xme-xcam)+abs(zme-zcam) < 0.001 then
		ahoriang:=horiang;
	else
		ahoriang := branchnear(horiang,fmath.arctan(xme-xcam,zme-zcam));
	end if;

	ghoriang := 0.75*ahoriang+0.25*horiang;
	choriang := 0.95*choriang + 0.05*ghoriang;

	cxlook := fmath.cos(vertang)*fmath.sin(choriang);
	cylook := fmath.sin(vertang);
	czlook := fmath.cos(vertang)*fmath.cos(choriang);
--put_line("gu:622:exit setCamAng");
end setCamAng;





procedure updateCamera( init: boolean := false )  is
	ixmx, kxl,kxh,kyl,kyh,kzl,kzh, tt: float;
	hoffset: constant float := 0.8; --2.0*margin; --0.4
	voffset: constant float := 0.0; --1.0*margin; --0.2
	ixgoal,iygoal,izgoal,xc,yc,zc,ff: float;

	--note:  we must prevent KO in between *cam, *me
	okcam: boolean := true;
	nc : constant integer := 5;
	c: integer;
	-- # cuts of segment between "me" and "camera"
begin

	-- 1st, check camera versus scene bounds:

	-- set interior X-bounds
	if scene=4 then
		ixmx:=ixmax/3.0;
	else
		ixmx:=ixmax;
	end if;

	-- initialize ideal camera position:
	tt:=1.0;
	ixcam:=xme - tt*hoffset*xlook;
	izcam:=zme - tt*hoffset*zlook;
	iycam:=yme + tt*voffset;


	--if interior then adjust ideal until within room bounds:
	if interior then
	loop
		ixcam:=xme - tt*hoffset*xlook;
		izcam:=zme - tt*hoffset*zlook;
		iycam:=yme + tt*voffset;
		exit when (iycam<iymax) and (abs(ixcam)<ixmx) and (abs(izcam)<izmax);
		tt:=tt*0.9;
		exit when tt<0.01; 
		--this happens when going thru maze and we
		--transition between 5 & 6.
	end loop;
	end if;

	ixgoal:=ixcam;
	iygoal:=iycam;
	izgoal:=izcam;

	-- adjust ideal camera position versus ko zones:
	c:=0;
	while okcam loop

		ff:=float(c)/float(nc); --move from avatar towards ideal
		xc:= xme*(1.0-ff) + ixgoal*ff;
		yc:= yme*(1.0-ff) + iygoal*ff;
		zc:= zme*(1.0-ff) + izgoal*ff;

		okcam:=true;
		for i in 1..nko loop -- check all KOs that apply
		if scene=koscene(i) then --this KO applies here
			kxl:=xc-(koxlo(i)-margin);
			kxh:=(koxhi(i)+margin)-xc;
			kyl:=yc-(koylo(i)-margin);
			kyh:=(koyhi(i)+margin)-yc;
			kzl:=zc-(kozlo(i)-margin);
			kzh:=(kozhi(i)+margin)-zc;
			if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then 
				okcam:=false;
			end if; --intrusion into ko
		end if;
		end loop; --for i

		if okcam then -- this standoff is valid
			ixcam:=xc; iycam:=yc; izcam:=zc;
		end if;

		c:=c+1;
		exit when c>nc;

	end loop; --while not okcam




	if init or not thirdPerson then
		xcam:=ixcam; ycam:=iycam; zcam:=izcam;
		-- need these initialized in case user 
		-- switches to 3rd person before moving,
		-- but also when entering new room...
	end if;

	cylook := fmath.sin(vertang);
	cxlook := fmath.cos(vertang)*fmath.sin(choriang);
	czlook := fmath.cos(vertang)*fmath.cos(choriang);


	--we only adjust camera if moving
	if direction/=0 then --moving

		-- move actual camera position 15% toward ideal
		xcam := 0.85*xcam + 0.15*ixcam;
		ycam := 0.85*ycam + 0.15*iycam;
		zcam := 0.85*zcam + 0.15*izcam;
		setCamAng;

	end if;


end updateCamera;








	-- this assumes mm=ID, (xme,yme,zme)=virtual pos within skybox
	-- [ actual pos versus skybox is always (0,0,0) ]
	procedure updateMVPs( wid,hit : float ) is
		xlk,ylk,zlk, xrt,yrt,zrt,
		xpos,ypos,zpos, xup,yup,zup : float;
	begin

		if not thirdPerson then
			choriang:=horiang;
			cxlook := fmath.cos(vertang)*fmath.sin(choriang);
			cylook := fmath.sin(vertang);
			czlook := fmath.cos(vertang)*fmath.cos(choriang);
		end if;

		if thirdPerson then
			xpos:=xcam; ypos:=ycam; zpos:=zcam;
		else -- 1st person
			xpos:=xme; ypos:=yme; zpos:=zme;
			cxlook:=xlook; cylook:=ylook; czlook:=zlook;
		end if;

		-- Look Vector
		xlk:=xpos+cxlook;
		ylk:=ypos+cylook;
		zlk:=zpos+czlook;

		-- Right unit-Direction
		xrt:= fmath.sin(choriang-halfpi);
		yrt:= 0.0;
		zrt:= fmath.cos(choriang-halfpi);

		-- calculate UP unit-Direction
		cross( xrt,yrt,zrt, cxlook,cylook,czlook, xup,yup,zup );

		perspective(pm, 45.0, wid/hit,  0.1, 100.0);

		lookat(mv, 0.0,0.0,0.0, cxlook,cylook,czlook, xup,yup,zup );

		lookat(imv, xpos,ypos,zpos, xlk,ylk,zlk, xup,yup,zup );

		mmvp:=mv;
		imvp:=imv;
		myMatMult(mmvp,pm);
		myMatMult(imvp,pm);


	end updateMVPs;












procedure updategamestate is
	cupang, swordang, wkeyang,bkeyang,gkeyang: float := 1.0;
	minonear: boolean;
begin

	if scene=8 then
		minonear:=(abs(xme)<5.0) and (abs(zme+1.0)<4.0);
		if minonear and not minosent then
			minofly:=true;
			dragonstart:=float(sdl_getticks)/1000.0;
			minosent:=true;
		end if;

	elsif scene=1 then

		gatenear:=(abs(xme-xgzm)<neargate*1.5) and (abs(zme-zgzm)<neargate*1.5);

		if wkeyheld and gatenear and not opengate then liftgate(gzmk); end if;

		opengate := opengate or (wkeyheld and gatenear);

------- now deal with opening maze

		mazenear :=
			(abs(xme-xmaze)<neargate*1.8) and (abs(zme-zmaze)<neargate*1.8);
		if gkeyheld and mazenear and not openmaze then liftmaze; end if;
		openmaze := openmaze or (gkeyheld and mazenear);



	elsif scene=6 then

		lionnear :=
			(abs(xme-xlion)<neargate*1.8) and (abs(zme-zlion)<neargate*1.8);

		if bkeyheld and lionnear and not openlion then liftlion; end if;

		openlion := openlion or (bkeyheld and lionnear);

	end if;





if not chaliceheld then
	cupang := angl(xlook,ylook,zlook, 
		float(xchalice)-xme,
		float(ychalice)-yme,
		float(zchalice)-zme);
end if;

if not wkeyheld then
	wkeyang := angl(xlook,ylook,zlook, xwkey-xme,  ywkey-yme, zwkey-zme);
end if;

if not bkeyheld then
	bkeyang := angl(xlook,ylook,zlook, xbkey-xme,  ybkey-yme, zbkey-zme);
end if;

if not gkeyheld then
	gkeyang := angl(xlook,ylook,zlook, xgkey-xme,  ygkey-yme, zgkey-zme);
end if;

if not swordheld then
	swordang :=angl(xlook,ylook,zlook, xsword-xme,ysword-yme,zsword-zme);
end if;

	wkeyseen := ( -- used in bat test
		not wkeyheld
		and (abs(xme-xwkey)<nearkey*4.0)
		and (abs(zme-zwkey)<nearkey*4.0) 
		and (wkeyang<fourthpi) );

	bkeyseen := ( -- used in bat test
		not bkeyheld
		and (abs(xme-xbkey)<nearkey*4.0)
		and (abs(zme-zbkey)<nearkey*4.0) 
		and (bkeyang<fourthpi) );



	--swordseen := (
	--	not swordheld
	--	and (abs(xme-xsword)<nearsword*4.0)
	--	and (abs(zme-zsword)<nearsword*4.0) 
	--	and (swordang<fourthpi) );



	gkeynear := (
		not gkeyheld
		and (sgkey=scene)
		and (abs(xme-xgkey)<nearkey)
		and (abs(zme-zgkey)<nearkey) 
		and (gkeyang<thirdpi) );

	bkeynear := (
		not bkeyheld
		and (sbkey=scene)
		and (not bathasbkey)
		and (abs(xme-xbkey)<nearkey)
		and (abs(zme-zbkey)<nearkey) 
		and (bkeyang<thirdpi) );

	wkeynear := (
		not wkeyheld
		and (swkey=scene)
		and (not bathaswkey)
		and (abs(xme-xwkey)<nearkey)
		and (abs(zme-zwkey)<nearkey) 
		and (wkeyang<thirdpi) );

	swordnear := (
		not swordheld
		and (ssword=scene)
		and (abs(xme-xsword)<nearsword)
		and (abs(zme-zsword)<nearsword) 
		and (swordang<thirdpi) );



	chalicenear := (
		not chaliceheld
		and (schalice=scene)
		and (abs(xme-float(xchalice))<nearchalice)
		and (abs(zme-float(zchalice))<nearchalice) 
		and (cupang<fourthpi) );

	pedestalnear := -- pedestal is @ (x,z)=(0,0)
		(   (abs(xme-float(xped))<nearpedestal) 
		and (abs(zme-float(zped))<nearpedestal) 
		and (scene=2)
		and (abs(ylook)<0.5) ); -- looking near horizontal



end updategamestate;




procedure liftgate( n: integer ) is
  -- n is to be removed from sequential list
  -- so we simply make it non-effectual
  -- then redefine gate, KO
begin
	myassert( n <= nko );
	myassert( (n=gxpk) or (n=gxmk) or (n=gzpk) or (n=gzmk) );

	koxlo(n):=0.0;
	koxhi(n):=0.0;
	koylo(n):=0.0;
	koyhi(n):=0.0;
	kozlo(n):=0.0;
	kozhi(n):=0.0;

	lifttime := float(sdl_getticks)/1000.0;

	gatewait:=true;
	snd4ada_hpp.playSnd(stone); --concrete

	if n=gxpk then
		xpup:=true;
	elsif n=gxmk then
		xmup:=true;
	elsif n=gzpk then
		zpup:=true;
	elsif n=gzmk then
		zmup:=true;
	end if;

end liftgate;










procedure liftlion is
begin

	liontime := float(sdl_getticks)/1000.0;

	lionwait:=true;
	snd4ada_hpp.playSnd(stone); --concrete

	liongoingup:=true;

end liftlion;






procedure liftmaze is
begin

	mazetime := float(sdl_getticks)/1000.0;

	mazewait:=true;
	snd4ada_hpp.playSnd(stone); --concrete

	mazegoingup:=true;

end liftmaze;




















--NOTICE:  typically, dt<<0.1 second  (between iterations of main loop)
--         but when it is that big, it's probably because foldtime is stale
procedure moveforward( currenttime: float ) is
	dt : float := currenttime-foldtime;
	lagfac : constant float := 1.0;

	yyme, yhalf,
	kxl,kxh,kyl,kyh,kzl,kzh,
	okxl,okxh, okzl, okzh: float;
	ixmx: float;
begin

if scene=4 then
	ixmx:=ixmax/3.0;
else
	ixmx:=ixmax;
end if;

if 
	not gatewait
	and not lionwait
	and not mazewait
	--and not batwait
then

	foldtime:=currenttime;
	oxme:=xme;
	oyme:=yme;
	ozme:=zme;

	xme:=xme+dt*speed*xlook;
	zme:=zme+dt*speed*zlook;
	direction:=1;

	if scene=2 then
		yme := aheight;
	elsif not interior then
		yme := aheight + land_alt(xme,zme);
	else
		yme := -iymax+aheight;
	end if;




if scene=2 then -- (x,z) in [-10..0, -10..0]

	if (xme>-margin) then xme:=-margin; end if;
	if (xme<-10.0+margin) then xme:=-10.0+margin; end if;
	if (zme>-margin) then zme:=-margin; end if;
	if (zme<-10.0+margin) then zme:=-10.0+margin; end if;

elsif interior then

	-- limit pos to be within walls:
	if (xme>+ixmx-margin) then xme:=+ixmx-margin; end if;
	if (xme<-ixmx+margin) then xme:=-ixmx+margin; end if;
	if (zme>+izmax-margin) then zme:=+izmax-margin; end if;
	if (zme<-izmax+margin) then zme:=-izmax+margin; end if;

else --exterior scene

	-- need a circular limit to reduce the
	-- disappearance of mipmapped objects

	--rad:=( xme*xme+zme*zme );
	--if rad>=256.0 then
	--	xme:=oxme;
	--	yme:=oyme;
	--	zme:=ozme;
	--end if;

	-- limit pos to be within user bounds:
	if (xme>+xmaxu-margin) then xme:=+xmaxu-margin; end if;
	if (xme<-xmaxu+margin) then xme:=-xmaxu+margin; end if;
	if (zme>+zmaxu-margin) then zme:=+zmaxu-margin; end if;
	if (zme<-zmaxu+margin) then zme:=-zmaxu+margin; end if;

end if;

	if scene=2 then
		yyme:=aheight;
		yhalf:=yyme-aheight/2.0;
	elsif scene=1 then
		yyme:=aheight;
		yhalf:=yyme-aheight/2.0;
	else
		yyme:=-iymax+aheight;
		yhalf:=yyme-aheight/2.0;
	end if;

	-- further, limit pos to avoid ko zones:
	for i in 1..nko loop
	if scene=koscene(i) then --this KO applies here


		kxl:=xme-koxlo(i)+margin;
		kxh:=koxhi(i)+margin-xme;

		kyl:=yhalf-koylo(i);
		kyh:=koyhi(i)-yhalf;

		kzl:=zme-kozlo(i)+margin;
		kzh:=kozhi(i)+margin-zme;





		if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then 
		--intrusion into ko

			okxl:=oxme-koxlo(i)+margin;
			okxh:=koxhi(i)+margin-oxme;
			okzl:=ozme-kozlo(i)+margin;
			okzh:=kozhi(i)+margin-ozme;

			if      ( (okxl*okxh>0.0) and (okzl*okzh<=0.0) ) then
				zme:=ozme;
				null;
			elsif ( (okzl*okzh>0.0) and (okxl*okxh<=0.0) ) then
				xme:=oxme;
				null;
			end if;

		end if; --intrusion into ko

	end if;
	end loop; --for i

	updategamestate;

end if;

end moveforward;




-- not yet able to back thru a stargate
procedure movebackward( currenttime: float ) is
	dt : float := currenttime-boldtime;
	--lagfac : constant float := 1.0;
	yyme, yhalf, kxl,kxh,kyl,kyh,kzl,kzh,
	ixmx, okxl,okxh, okzl, okzh: float;
begin

if scene=4 then
	ixmx:=ixmax/3.0;
else
	ixmx:=ixmax;
end if;

if 
	not gatewait
	and not lionwait
	and not mazewait
	--and not batwait
then


	boldtime:=currenttime;

	forwardok:=true;

	oxme:=xme;
	oyme:=yme;
	ozme:=zme;


	xme:=xme-dt*speed*xlook;
	zme:=zme-dt*speed*zlook;
	direction:=-1;

	if scene=2 then
		yyme:=aheight;
		yhalf:=yyme-aheight/2.0;
	elsif scene=2 then
		yme := aheight;
	elsif not interior then
		yme := aheight + land_alt(xme,zme);
	else
		yme := -iymax+aheight;
	end if;




if scene=2 then -- (x,z) in [-10..0, -10..0]

	if (xme>-margin) then xme:=-margin; end if;
	if (xme<-10.0+margin) then xme:=-10.0+margin; end if;
	if (zme>-margin) then zme:=-margin; end if;
	if (zme<-10.0+margin) then zme:=-10.0+margin; end if;

elsif interior then

	-- limit pos to be within walls:
	if (xme>+ixmx-margin) then xme:=+ixmx-margin; end if;
	if (xme<-ixmx+margin) then xme:=-ixmx+margin; end if;
	if (zme>+izmax-margin) then zme:=+izmax-margin; end if;
	if (zme<-izmax+margin) then zme:=-izmax+margin; end if;

else --exterior scene

	-- need a circular limit to reduce the
	-- disappearance of mipmapped objects
	--rad:=( xme*xme+zme*zme );
	--if rad>=256.0 then
	--	xme:=oxme;
	--	yme:=oyme;
	--	zme:=ozme;
	--end if;

	-- limit pos to be within user bounds:
	if (xme>+xmaxu-margin) then xme:=+xmaxu-margin; end if;
	if (xme<-xmaxu+margin) then xme:=-xmaxu+margin; end if;
	if (zme>+zmaxu-margin) then zme:=+zmaxu-margin; end if;
	if (zme<-zmaxu+margin) then zme:=-zmaxu+margin; end if;

end if;


	if scene=1 then
		yyme:=aheight;
		yhalf:=yyme-aheight/2.0;
	else
		yyme:=-iymax+aheight;
		yhalf:=yyme-aheight/2.0;
	end if;


	-- further, limit pos to avoid ko zones:
	for i in 1..nko loop
	if scene=koscene(i) then -- this KO applies here


		kxl:=xme-koxlo(i)+margin;
		kxh:=koxhi(i)+margin-xme;

		kyl:=yhalf-koylo(i);
		kyh:=koyhi(i)-yhalf;

		kzl:=zme-kozlo(i)+margin;
		kzh:=kozhi(i)+margin-zme;


		if (kxl*kxh>0.0) and (kyl*kyh>0.0) and (kzl*kzh>0.0) then 
		--intrusion into ko

			okxl:=oxme-koxlo(i)+margin;
			okxh:=koxhi(i)+margin-oxme;
			okzl:=ozme-kozlo(i)+margin;
			okzh:=kozhi(i)+margin-ozme;

			if      ( (okxl*okxh>=0.0) and (okzl*okzh<0.0) ) then
				zme:=ozme;
			elsif ( (okzl*okzh>=0.0) and (okxl*okxh<0.0) ) then
				xme:=oxme;
			end if;

		end if; -- KO intrusion
	end if;
	end loop; --for i

	updategamestate;

end if;

end movebackward;



doortime: float := 0.0;
-- ensure look direction is facing door
-- otherwise we are trapped in door-hell:
function atdoor(currenttime: float) return boolean is -- scene 1,2
	dx,dz,idx,idz,hrad, ihrad : float;
	dt: float := currenttime - doortime;
begin

if dt>repassage then -- second minimum repassage time

	dx :=  xdoor - xme;
	dz :=  zdoor - zme;
	hrad := fmath.sqrt( dx*dx + dz*dz );

	idx :=  ixdoor - xme;
	idz :=  izdoor - zme;
	ihrad := fmath.sqrt( idx*idx + idz*idz );


	if 
		( (hrad<ddoor) and (scene=1) )
		or 
		( (ihrad<ddoor) and (scene=2) )
	then
		doortime:=currenttime;
		return true;
	else 
		return false;
	end if;

else
	return false;
end if;

end atdoor;












mazepassagetime: float := 0.0;
-- ensure look direction is facing maze
function atmaze(currenttime: float) return boolean is --scene 1,5
	dx,dz,hrad, ihrad, idx,idz : float;
	dt: float := currenttime - mazepassagetime;
begin

if dt>repassage then -- second minimum repassage time

	dx :=  xmaze - xme;
	dz :=  zmaze - zme;
	hrad := fmath.sqrt( dx*dx + dz*dz );

	idx :=  ixmaze - xme;
	idz :=  izmaze - zme;
	ihrad := fmath.sqrt( idx*idx + idz*idz );



	if 
		((hrad<dmaze) and (scene=1))
		or 
		((ihrad<dmaze) and (scene=5 or scene=7))
	then 
		mazepassagetime:=currenttime;
		return true;
	else 
		return false;
	end if;

else
	return false;
end if;

end atmaze;



dungtime: float := 0.0;
-- ensure look direction is facing temple

function attemple(currenttime: float) return boolean is --scene 6,4
	dx,dz,hrad, ihrad : float;
	ok: boolean := false;
	dt: float := currenttime - dungtime;
begin

if dt>repassage then -- second minimum repassage time

	dx :=  xtmpl - xme;
	dz :=  ztmpl - zme;
	hrad := fmath.sqrt( dx*dx + dz*dz );

	dx :=  ixtmpl - xme;
	dz :=  iztmpl - zme;
	ihrad := fmath.sqrt( dx*dx + dz*dz );



	if 
		((scene=6) and (hrad<dtmpl)) or
		((scene=4) and (ihrad<dtmpl))
	then 
		dungtime:=currenttime;
		return true;
	else 
		return false;
	end if;

else
	return false;
end if;

end attemple;





















procedure handle_gc_left( gcx,gcy:sdl.sint16 ) is
-- to update look direction using left game controller stick
	ux : float := float(gcx)/float(32768);
	uy : float := float(gcy)/float(32768);
begin


if abs(ux)<0.15 then 
	ux:=0.0;
else
	ux:=ux-0.15*signum(ux);
end if;

if abs(uy)<0.15 then 
	uy:=0.0; 
else
	uy:=uy-0.15*signum(uy);
end if;


	horiAng := horiAng - 0.04 * ux * Lsens;
	vertAng := vertAng + 0.02 * uy * Lsens;
	setCamAng;

	xlook := fmath.cos(vertAng)*fmath.sin(horiAng);
	ylook := fmath.sin(vertAng);
	zlook := fmath.cos(vertAng)*fmath.cos(horiAng);


	if
	( 
	not forwardOk 
	and (abs(badHoriAng-horiAng)>fourthpi) 
	and not pauseAtLevelChange 
	)  then
		forwardOk:=true;
		badHoriAng:=-10.0*twopi;
	end if;

	updategamestate;

end handle_gc_left;






procedure handle_gc_right( nowtime: float; gcx,gcy:sdl.sint16 ) is
-- to update move direction using right game controller stick
	ux : float := Rsens*float(gcx)/float(32768);
	uy : float := Rsens*float(gcy)/float(32768);
begin

	if    uy < -0.05 then
		moveforward(nowTime);

	elsif uy > +0.05 then
		movebackward(nowTime);

	end if;

	handle_gc_left(gcx,0); -- turns left/right

end handle_gc_right;










































   function FileExists (File : String) return Boolean is
      FileId : text_io.File_Type;
   begin -- TextFileExists
      -----------------------------------------------------
      -- Open and close the file to see if the file exists.
      -----------------------------------------------------
      text_io.Open
         (File => FileId,
          Mode => text_io.In_File,
          Name => File);

      text_io.Close
         (File => FileId);
      -------------------------------------------------
      -- If no exception occurred, the file must exist.
      -------------------------------------------------
      return True;
   exception
      when text_io.Name_Error =>
         return False;
		when others => return false;
   end FileExists;







oldMdTime : float := 0.0;

procedure handle_mouse_drag( nowTime : float ) is

-- to update look direction
-- also handles touch pad...

	mousedx, mousedy : aliased interfaces.c.int;
	mouseBtnState : Uint32;
	dt : float := nowTime-oldMdTime;
begin

	mouseBtnState := SDL_GetRelativeMouseState(mousedx'access, mousedy'access);

	oldMdTime:=nowTime;

	if( dt<0.1 ) then
		horiAng := horiAng - speed*0.0005 * float(mousedx);
		vertAng := vertAng - speed*0.0005 * float(mousedy);
		setCamAng;
	end if;

	xlook := fmath.cos(vertAng)*fmath.sin(horiAng);
	ylook := fmath.sin(vertAng);
	zlook := fmath.cos(vertAng)*fmath.cos(horiAng);


	if
	( 
	not forwardOk 
	and (abs(badHoriAng-horiAng)>fourthpi) 
	and not pauseAtLevelChange 
	)  then
		forwardOk:=true;
		badHoriAng:=-10.0*twopi;
	end if;

	updategamestate;

end handle_mouse_drag;














------------------ end game specific code -----------------------------


function bitmatch( x, y : integer ) return boolean is
	result : boolean := false;
	a : integer := x;
	b : integer := y;
begin
	for i in 1..32 loop
		if ( odd(a) and odd(b) ) then result:=true; end if;
		a:=a/2;
		b:=b/2;
	end loop;
	return result;
end;



procedure output( a : mat44 ) is
begin
	for row in 1..4 loop
	for col in 1..4 loop
		put( float'image( a(row,col) ) &" ");
	end loop;
	new_line;
	end loop;
	new_line;
end;



-------------- 9dec14 begin additions ------------------------------------



-- Warning:  this logic is not fully generalized...
--           Sokoban input files must be screened carefully.
--
--           On the other hand, Ada length function omits
--           confusing control characters at EOL, so we 
--           don't need to distinguish DOS from Unix files.
--
function is_blank( line : string; len:integer ) return boolean is
begin

	if( len < 1 ) then return true; end if;

	for i in 1..len loop
	  if( line(i) = '#' ) then return false; end if;
	end loop;
	return true;
end is_blank;




-------------- 9dec14 end additions ------------------------------------


------- begin addendum 5jan15 -------------------------------------------------

-------------- begin myst setup ---------------------------------------





procedure zeroBtns is
begin
	btn_0:=0;
	btn_1:=0;
	btn_2:=0;
	btn_3:=0;
	btn_4:=0;
	btn_5:=0;
	btn_6:=0;
	btn_7:=0;
	btn_8:=0;
end zeroBtns;






-- note that xmax,ymax,zmax = (20,20,20):
function land_alt( x,z : float ) return float is
	--doorx: constant float := ( -5.0)*twopi/xmax;
	--doorz: constant float := (-10.0)*twopi/zmax;
	cycx: constant float := x*twopi/xmax;
	cycz: constant float := z*twopi/zmax;
	amp : constant float := ymax/50.0; --0.4
	alt, hdist : float;
begin

	-- this is difference from desired height (0)
	--altd := -0.1 + amp*( fmath.sin(doorx)+fmath.sin(doorz) )/2.0;

	alt := -0.1 + amp*( fmath.sin(cycx)+fmath.sin(cycz) )/2.0;

	-- special provision @ castle gate (x,y,z)=(-5,0,-10)
	hdist:=fmath.sqrt( sqr(-5.0-xme) + sqr(-10.0-zme) );

	-- want floor-level=zero @ doorway
	if hdist<5.0 then
		alt := alt*(hdist/5.0);
	end if;

	return alt;
end land_alt;


procedure sendBat is
begin
	batfly:=true;
	batstart:=float(sdl_getticks)/1000.0;
	-- ensure margin<1.0:
	if interior then
		xbat:=0.0;        -- try drop near doorway (or southroom maze6)
		ybat:=iymax-0.1; 
		zbat:=-izmax+1.0; -- try drop near doorway (or southroom maze6)
		if scene=7 then zbat:=0.0; end if;

	else -- far reaches of exterior:
		xbat:=-xmaxu+1.0; ybat:=ymax-1.0; zbat:=-zmaxu+1.0;
	end if;
end sendBat;

-- fly bat in a linear trajectory starting and ending
-- at a randomly chosen, but pickable (x,z) location
-- make the key-pickup sound, then disappear...
-- but only if lying on ground, not if being held.
-- In case of chalice (scene=7), bat grabs it anyway.
procedure drawbat( et: float ) is
	v4, vcc : vec4;
	xt,yt,zt, rtgt,dot, yt0,yt1 : float;
	mindot : constant float := fmath.cos(fourthpi);
	s1: string := "{"; --bat1
	s2: string := "}"; --bat2
	s : string(1..1);
	nx: constant float := 2.0; -- #transitions per second
	i: integer;
	ets, xkey,ykey,zkey: float;
	rng : float;
begin

	if scene=1 then
		xkey:=xwkey; ykey:=ywkey; zkey:=zwkey;
		ets:=et;
	elsif scene=5 then
		xkey:=xbkey; ykey:=ybkey; zkey:=zbkey;
		ets:=et*8.0;
	end if;

	myassert( et>=-1.01 );
	myassert( et<=+1.01 );
	-- et in (-1..1)

	i := integer( batduration * et * nx );
	if odd(i) then
		s:=s1;
	else
		s:=s2;
	end if;

if scene=7 then 
-- bat flys more like dragon...right at you
-- and grabs chalice out of your hand...

	rng := abs(et)*10.0;
	if rng<0.5 then rng:=0.5; end if;
	yt0 := yme + abs(et)*(iymax-yme);

	-- this trajectory approaches from exact [dynamic] look direction:
	xt :=xme+rng*xlook;
	yt1:=yme+rng*ylook;
	zt :=zme+rng*zlook;

	-- this trajectory mixes the two:
	yt:= abs(et)*yt0 + (1.0-abs(et))*yt1;

	rtgt := fmath.sqrt( sqr(xt-xme) + sqr(yt-yme) + sqr(zt-zme) );
	if rtgt<0.5 then rtgt:=0.5; end if;

	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	utex.print3d(s,
		vcc(1),vcc(2),
		vcc(3),vcc(4), 600, rtgt);

else -- more typical bat flight:
	xt:=xkey+abs(ets)*(xbat-xkey);
	yt:=ykey+abs(ets)*(ybat-ykey);
	zt:=zkey+abs(ets)*(zbat-zkey);

	rtgt := fmath.sqrt( sqr(xt-xme) + sqr(yt-yme) + sqr(zt-zme) );
	dot := (xt-xme)*xlook + (yt-yme)*ylook + (zt-zme)*zlook;
	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	if (dot/rtgt > mindot) then
		utex.print3d(s,
			vcc(1),vcc(2),
			vcc(3),vcc(4), 600, rtgt);
	end if;

end if;


end drawbat;







procedure drawspider( et: float ) is
	v4, vcc : vec4;
	xt,yt,zt, rtgt,dot : float;
	mindot : constant float := fmath.cos(fourthpi);
	s: string := ":"; --funny-spider
	ym: constant float := htobj;
	yx: constant float := iymax-htobj;
	yc: constant float := 0.5*(yx+ym);
	yr: constant float := 0.5*(yx-ym);
begin

if scene=2 then

	tt := fmath.sin(et*0.1); -- in [-1..1]

	xt:=-9.95;
	zt:=-0.05;
	yt:=yc+tt*yr;

	rtgt := fmath.sqrt( sqr(xt-xme) + sqr(yt-yme) + sqr(zt-zme) );
	dot := (xt-xme)*xlook + (yt-yme)*ylook + (zt-zme)*zlook;
	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	if (dot/rtgt > mindot) then
		utex.print3d(s,
			vcc(1),vcc(2),
			vcc(3),vcc(4), 100, rtgt);
	end if;

end if;

end drawspider;























ydra : float; --only used for next 2 procs

-- 7sep16 revision:  Once the dragon is seen,
-- you cannot turn away...he approaces from
-- whatever your look direction may be !!!
procedure drawBdragon( et: float ) is -- 0<et<1
	v4, vcc : vec4;
	yt0,
	xt1,yt1,zt1,
	xt,yt,zt : float;

	b2: string := "<"; --Bkill
	b1: string := ">"; --Bchase

	s : string(1..1);

	rng : float := (1.0-et)*20.0;
		--(1.0-et)*fmath.sqrt(sqr(xdra-xme)+sqr(ydra-yme)+sqr(zdra-zme));
begin

	if interior then ydra:=iymax*0.5;
	else ydra:=ymax; end if;

	myassert( et>=-0.01 );
	myassert( et<=+1.01 );
	-- et in (0..1)

	if rng<0.3 then rng:=0.3; end if;
	if et>0.9 then
		s:=b2;
	else
		s:=b1;
	end if;



	-- this trajectory approaches from a fixed point (xdra,ydra,zdra) 
	-- above the horizon chosen at the initial look direction:
	--xt0:=xdra+et*(xme-xdra);
	yt0:=ydra+et*(yme-ydra);
	--zt0:=zdra+et*(zme-zdra);

	-- this trajectory approaches from exact [dynamic] look direction:
	xt1:=xme+rng*xlook;
	yt1:=yme+rng*ylook;
	zt1:=zme+rng*zlook;

	-- this trajectory mixes the two:
	yt:= et*yt1 + (1.0-et)*yt0;

	xt:=xt1;
	zt:=zt1;

	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	utex.print3d(s,
		vcc(1),vcc(2),
			vcc(3),vcc(4), 400, rng);

end drawBdragon;










-- 7sep16 revision:  Once the dragon is seen,
-- you cannot turn away...he approaces from
-- whatever your look direction may be !!!
procedure drawRdragon( et: float ) is -- 0<et<1
	v4, vcc : vec4;
	yt0,
	xt1,yt1,zt1,
	xt,yt,zt : float;

	r2: string := "("; --Rkill
	r1: string := ")"; --Rchase

	s : string(1..1);

	rng : float := (1.0-et)*5.0;
		--(1.0-et)*fmath.sqrt(sqr(xdra-xme)+sqr(ydra-yme)+sqr(zdra-zme));
begin

	if interior then ydra:=iymax*0.5;
	else ydra:=ymax; end if;

	myassert( et>=-0.01 );
	myassert( et<=+1.01 );
	-- et in (0..1)

	if rng<0.3 then rng:=0.3; end if;
	if et>0.9 then
		s:=r2;
	else
		s:=r1;
	end if;


	-- this trajectory approaches from a fixed point (xdra,ydra,zdra) 
	-- above the horizon chosen at the initial look direction:
	--xt0:=xdra+et*(xme-xdra);
	yt0:=ydra+et*(yme-ydra);
	--zt0:=zdra+et*(zme-zdra);

	-- this trajectory approaches from exact [dynamic] look direction:
	xt1:=xme+rng*xlook;
	yt1:=yme+rng*ylook;
	zt1:=zme+rng*zlook;

	-- this trajectory mixes the two:
	yt:= et*yt1 + (1.0-et)*yt0;

	xt:=xt1;
	zt:=zt1;

	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	utex.print3d(s,
		vcc(1),vcc(2),
			vcc(3),vcc(4), 400, rng);

end drawRdragon;













procedure drawMinotaur( et: float ) is -- 0<et<1
	v4, vcc : vec4;
	yt0,
	xt1,yt1,zt1,
	xt,yt,zt : float;
	s: string := "!";
	rng : float := (1.0-et)*20.0;
begin

	ydra:=iymax*0.5;

	myassert( et>=-0.01 );
	myassert( et<=+1.01 );
	-- et in (0..1)

	if rng<0.3 then rng:=0.3; end if;

	-- this trajectory approaches from a fixed point (xdra,ydra,zdra) 
	-- above the horizon chosen at the initial look direction:
	--xt0:=xdra+et*(xme-xdra);
	yt0:=ydra+et*(yme-ydra);

	-- this trajectory approaches from exact [dynamic] look direction:
	xt1:=xme+rng*xlook;
	yt1:=yme+rng*ylook;
	zt1:=zme+rng*zlook;

	-- this trajectory mixes the two:
	yt:= et*yt1 + (1.0-et)*yt0;

	xt:=xt1;
	zt:=zt1;

	v4 := (xt,yt,zt,1.0);
	mymatvec( imvp, v4, vcc );

	utex.print3d(s,
		vcc(1),vcc(2),
			vcc(3),vcc(4), 500, rng);

end drawMinotaur;


















procedure initializeNewMazes is
begin

	for im in scenerng loop
	for row in -mrows..mrows loop
	for col in -mcols..mcols loop
		passthru(im,row,col):=false;
		iswall(im,row,col):=false;
	end loop; --col
	end loop; --row
	end loop; --im

	-- topexit -------------------------------------

	--iswall(6,10,10):=true;
	--iswall(6,9,10):=true;
	--iswall(6,8,10):=true;
	--iswall(6,7,10):=true;
	--iswall(6,6,10):=true;
	--iswall(6,5,10):=true;
	--iswall(6,4,10):=true;
	--iswall(6,3,10):=true;

	iswall(6,5,10):=true;
	iswall(6,3,10):=true;

	iswall(6,5,9):=true;
	iswall(6,3,9):=true;

	iswall(6,10,8):=true;
	iswall(6,9,8):=true;
	iswall(6,8,8):=true;
	iswall(6,7,8):=true;
	iswall(6,5,8):=true;
	iswall(6,3,8):=true;

	iswall(6,10,7):=true;
	iswall(6,7,7):=true;
	iswall(6,5,7):=true;
	iswall(6,3,7):=true;
	iswall(6,2,7):=true;

	iswall(6,10,6):=true;
	iswall(6,7,6):=true;
	iswall(6,5,6):=true;

	iswall(6,10,5):=true;
	iswall(6,9,5):=true;
	iswall(6,8,5):=true;
	iswall(6,7,5):=true;
	iswall(6,6,5):=true;
	iswall(6,5,5):=true;
	iswall(6,4,5):=true;
	iswall(6,3,5):=true; --passthru(6,3,5):=true;
	iswall(6,2,5):=true; passthru(6,2,5):=true;
	iswall(6,1,5):=true;

	iswall(6,5,4):=true;
	iswall(6,1,4):=true;

	iswall(6,10,3):=true;
	iswall(6,9,3):=true;
	iswall(6,8,3):=true;
	iswall(6,7,3):=true;
	iswall(6,5,3):=true;
	iswall(6,3,3):=true;
	iswall(6,2,3):=true;
	iswall(6,1,3):=true;
-------------------------- repeat negating 1st coord:

	--iswall(6,-10,10):=true;
	--iswall(6,-9,10):=true;
	--iswall(6,-8,10):=true;
	--iswall(6,-7,10):=true;
	--iswall(6,-6,10):=true;
	--iswall(6,-5,10):=true;
	--iswall(6,-4,10):=true;
	--iswall(6,-3,10):=true;

	iswall(6,-5,10):=true;
	iswall(6,-3,10):=true;

	iswall(6,-5,9):=true;
	iswall(6,-3,9):=true;

	iswall(6,-10,8):=true;
	iswall(6,-9,8):=true;
	iswall(6,-8,8):=true;
	iswall(6,-7,8):=true;
	iswall(6,-5,8):=true;
	iswall(6,-3,8):=true;

	iswall(6,-10,7):=true;
	iswall(6,-7,7):=true;
	iswall(6,-5,7):=true;
	iswall(6,-3,7):=true;
	iswall(6,-2,7):=true;

	iswall(6,-10,6):=true;
	iswall(6,-7,6):=true;
	iswall(6,-5,6):=true;

	iswall(6,-10,5):=true;
	iswall(6,-9,5):=true;
	iswall(6,-8,5):=true;
	iswall(6,-7,5):=true;
	iswall(6,-6,5):=true;
	iswall(6,-5,5):=true;
	iswall(6,-4,5):=true;
	iswall(6,-3,5):=true; --passthru(6,-3,5):=true;
	iswall(6,-2,5):=true; passthru(6,-2,5):=true;
	iswall(6,-1,5):=true;

	iswall(6,-5,4):=true;
	iswall(6,-1,4):=true;

	iswall(6,-10,3):=true;
	iswall(6,-9,3):=true;
	iswall(6,-8,3):=true;
	iswall(6,-7,3):=true;
	iswall(6,-5,3):=true;
	iswall(6,-3,3):=true;
	iswall(6,-2,3):=true;
	iswall(6,-1,3):=true;

--==========================================

	-- mid -------------------------------------

	iswall(6,10,3):=true;
	iswall(6,9,3):=true;
	iswall(6,8,3):=true;
	iswall(6,7,3):=true;
	iswall(6,5,3):=true;
	iswall(6,3,3):=true;
	iswall(6,2,3):=true;
	iswall(6,1,3):=true;

	iswall(6,5,2):=true;
	iswall(6,2,2):=true;
	iswall(6,1,2):=true;

	iswall(6,10,1):=true;
	iswall(6,9,1):=true;
	iswall(6,8,1):=true;
	iswall(6,7,1):=true;
	iswall(6,6,1):=true;
	iswall(6,5,1):=true;
	iswall(6,4,1):=true;
	iswall(6,2,1):=true;
	iswall(6,1,1):=true;

	iswall(6,4,0):=true;
	iswall(6,2,0):=true;
	iswall(6,1,0):=true;

	iswall(6,10,-1):=true;
	iswall(6,9,-1):=true;
	iswall(6,8,-1):=true;
	iswall(6,6,-1):=true;
	iswall(6,4,-1):=true;
	iswall(6,2,-1):=true;
	iswall(6,1,-1):=true;

	iswall(6,6,-2):=true;
	iswall(6,4,-2):=true;
	iswall(6,2,-2):=true;

	iswall(6,10,-3):=true;
	iswall(6,9,-3):=true;
	iswall(6,8,-3):=true;
	iswall(6,7,-3):=true;
	iswall(6,6,-3):=true;
	iswall(6,4,-3):=true;
	iswall(6,2,-3):=true;

-------------------------- repeat negating 1st coord:

	iswall(6,-10,3):=true;
	iswall(6,-9,3):=true;
	iswall(6,-8,3):=true;
	iswall(6,-7,3):=true;
	iswall(6,-5,3):=true;
	iswall(6,-3,3):=true;
	iswall(6,-2,3):=true;
	iswall(6,-1,3):=true;

	iswall(6,-5,2):=true;
	iswall(6,-2,2):=true;
	iswall(6,-1,2):=true;

	iswall(6,-10,1):=true;
	iswall(6,-9,1):=true;
	iswall(6,-8,1):=true;
	iswall(6,-7,1):=true;
	iswall(6,-6,1):=true;
	iswall(6,-5,1):=true;
	iswall(6,-4,1):=true;
	iswall(6,-2,1):=true;
	iswall(6,-1,1):=true;

	iswall(6,-4,0):=true;
	iswall(6,-2,0):=true;
	iswall(6,-1,0):=true;

	iswall(6,-10,-1):=true;
	iswall(6,-9,-1):=true;
	iswall(6,-8,-1):=true;
	iswall(6,-6,-1):=true;
	iswall(6,-4,-1):=true;
	iswall(6,-2,-1):=true;
	iswall(6,-1,-1):=true;

	iswall(6,-6,-2):=true;
	iswall(6,-4,-2):=true;
	iswall(6,-2,-2):=true;

	iswall(6,-10,-3):=true;
	iswall(6,-9,-3):=true;
	iswall(6,-8,-3):=true;
	iswall(6,-7,-3):=true;
	iswall(6,-6,-3):=true;
	iswall(6,-4,-3):=true;
	iswall(6,-2,-3):=true;



--==========================================

	-- bot -------------------------------------

	iswall(6,10,-3):=true;
	iswall(6,9,-3):=true;
	iswall(6,8,-3):=true;
	iswall(6,7,-3):=true;
	iswall(6,6,-3):=true;
	iswall(6,4,-3):=true;
	iswall(6,2,-3):=true;

	iswall(6,6,-4):=true;
	iswall(6,2,-4):=true;

	iswall(6,10,-5):=true;
	iswall(6,9,-5):=true;
	iswall(6,8,-5):=true;
	iswall(6,6,-5):=true;
	iswall(6,5,-5):=true;
	iswall(6,4,-5):=true;
	iswall(6,3,-5):=true;
	iswall(6,2,-5):=true;

	iswall(6,8,-6):=true;

	iswall(6,8,-7):=true;
	iswall(6,7,-7):=true;
	iswall(6,6,-7):=true;

	iswall(6,10,-8):=true;
	iswall(6,9,-8):=true;
	iswall(6,8,-8):=true;
	iswall(6,6,-8):=true;

	iswall(6,6,-9):=true;

	iswall(6,6,-10):=true;

	--iswall(6,10,-10):=true;
	--iswall(6,9,-10):=true;
	--iswall(6,8,-10):=true;
	--iswall(6,7,-10):=true;
	--iswall(6,6,-10):=true;
	--iswall(6,5,-10):=true;
	--iswall(6,4,-10):=true;
	--iswall(6,3,-10):=true;
	--iswall(6,2,-10):=true;
	--iswall(6,1,-10):=true;
	--iswall(6,0,-10):=true;

-------------------------- repeat negating 1st coord:

	iswall(6,-10,-3):=true;
	iswall(6,-9,-3):=true;
	iswall(6,-8,-3):=true;
	iswall(6,-7,-3):=true;
	iswall(6,-6,-3):=true;
	iswall(6,-4,-3):=true;
	iswall(6,-2,-3):=true;

	iswall(6,-6,-4):=true;
	iswall(6,-2,-4):=true;

	iswall(6,-10,-5):=true;
	iswall(6,-9,-5):=true;
	iswall(6,-8,-5):=true;
	iswall(6,-6,-5):=true;
	iswall(6,-5,-5):=true;
	iswall(6,-4,-5):=true;
	iswall(6,-3,-5):=true;
	iswall(6,-2,-5):=true;

	iswall(6,-8,-6):=true;

	iswall(6,-8,-7):=true;
	iswall(6,-7,-7):=true;
	iswall(6,-6,-7):=true;

	iswall(6,-10,-8):=true;
	iswall(6,-9,-8):=true;
	iswall(6,-8,-8):=true;
	iswall(6,-6,-8):=true;

	iswall(6,-6,-9):=true;

	iswall(6,-6,-10):=true;

--=========== done with scene #6;  begin scene #5 ================

-- midLR = maze entry @ bottom = Zmin


	iswall(5,10,-1):=true;
	iswall(5,9,-1):=true;
	iswall(5,8,-1):=true;
	iswall(5,7,-1):=true;
	iswall(5,6,-1):=true;

	iswall(5,4,-1):=true;
	iswall(5,3,-1):=true;
	iswall(5,2,-1):=true;
	iswall(5,1,-1):=true;
	iswall(5,0,-1):=true;

	iswall(5,6,0):=true;
	iswall(5,4,0):=true;

	iswall(5,10,1):=true;
	iswall(5,9,1):=true;
	iswall(5,8,1):=true;
	iswall(5,6,1):=true;
	iswall(5,4,1):=true;
	iswall(5,2,1):=true;
	iswall(5,0,1):=true;

	iswall(5,6,2):=true;
	iswall(5,4,2):=true;
	iswall(5,2,2):=true;

	iswall(5,10,3):=true;
	iswall(5,9,3):=true;
	iswall(5,8,3):=true;
	iswall(5,7,3):=true;
	iswall(5,6,3):=true;
	iswall(5,4,3):=true;
	iswall(5,2,3):=true;
	iswall(5,0,3):=true;

-------------------------- repeat negating 1st coord:

	iswall(5,-10,-1):=true;
	iswall(5,-9,-1):=true;
	iswall(5,-8,-1):=true;
	iswall(5,-7,-1):=true;
	iswall(5,-6,-1):=true;

	iswall(5,-4,-1):=true;
	iswall(5,-3,-1):=true;
	iswall(5,-2,-1):=true;
	iswall(5,-1,-1):=true;

	iswall(5,-6,0):=true;
	iswall(5,-4,0):=true;

	iswall(5,-10,1):=true;
	iswall(5,-9,1):=true;
	iswall(5,-8,1):=true;
	iswall(5,-6,1):=true;
	iswall(5,-4,1):=true;
	iswall(5,-2,1):=true;

	iswall(5,-6,2):=true;
	iswall(5,-4,2):=true;
	iswall(5,-2,2):=true;

	iswall(5,-10,3):=true;
	iswall(5,-9,3):=true;
	iswall(5,-8,3):=true;
	iswall(5,-7,3):=true;
	iswall(5,-6,3):=true;
	iswall(5,-4,3):=true;
	iswall(5,-2,3):=true;



--==========================================

	-- botL-exitR -------------------------------------

	iswall(5,10,8):=true;
	iswall(5,9,8):=true;
	iswall(5,8,8):=true;
	iswall(5,7,8):=true;
	iswall(5,6,8):=true;
	iswall(5,5,8):=true;
	iswall(5,3,8):=true;
	iswall(5,2,8):=true;
	iswall(5,1,8):=true;
	iswall(5,0,8):=true;

	iswall(5,10,7):=true;
	iswall(5,8,7):=true;
	iswall(5,0,7):=true;

	iswall(5,10,6):=true;
	iswall(5,8,6):=true;
	iswall(5,6,6):=true;
	iswall(5,5,6):=true;
	iswall(5,4,6):=true;
	iswall(5,3,6):=true;
	iswall(5,2,6):=true;
	iswall(5,0,6):=true;

	iswall(5,10,5):=true;
	iswall(5,9,5):=true;
	iswall(5,8,5):=true;
	iswall(5,6,5):=true;
	iswall(5,2,5):=true;
	iswall(5,0,5):=true;

	iswall(5,6,4):=true;
	iswall(5,4,4):=true;
	iswall(5,2,4):=true;
	iswall(5,0,4):=true;


-------------------------- repeat negating 1st coord:


	iswall(5,-10,8):=true;
	iswall(5,-9,8):=true;
	iswall(5,-8,8):=true;
	iswall(5,-7,8):=true;
	iswall(5,-6,8):=true;
	iswall(5,-5,8):=true;
	iswall(5,-3,8):=true;
	iswall(5,-2,8):=true;
	iswall(5,-1,8):=true;

	iswall(5,-10,7):=true;
	iswall(5,-8,7):=true;

	iswall(5,-10,6):=true;
	iswall(5,-8,6):=true;
	iswall(5,-6,6):=true;
	iswall(5,-5,6):=true;
	iswall(5,-4,6):=true;
	iswall(5,-3,6):=true;
	iswall(5,-2,6):=true;

	iswall(5,-10,5):=true;
	iswall(5,-9,5):=true;
	iswall(5,-8,5):=true;
	iswall(5,-6,5):=true;
	iswall(5,-2,5):=true;

	iswall(5,-6,4):=true;
	iswall(5,-4,4):=true;
	iswall(5,-2,4):=true;








------------- begin maze #7 = orange maze to white castle -------------

----- warning:  x,z coords here are interchanged ------------

	iswall(7,-10,-7):=true;

	iswall(7,-9,-7):=true;

	iswall(7,-8,-0):=true;
	iswall(7,-8,-1):=true;
	iswall(7,-8,-2):=true;
	iswall(7,-8,-3):=true;
	iswall(7,-8,-4):=true;
	iswall(7,-8,-7):=true;
	iswall(7,-8,-9):=true;
	iswall(7,-8,-10):=true;

	iswall(7,-7,0):=true;
	iswall(7,-7,-7):=true;

	iswall(7,-6,-0):=true;
	iswall(7,-6,-4):=true;
	iswall(7,-6,-5):=true;
	iswall(7,-6,-7):=true;
	iswall(7,-6,-8):=true;
	iswall(7,-6,-9):=true;
	iswall(7,-6,-10):=true;

	iswall(7,-5,-0):=true;
	iswall(7,-5,-4):=true;
	iswall(7,-5,-5):=true;
	iswall(7,-5,-7):=true;
	iswall(7,-5,-8):=true;
	iswall(7,-5,-9):=true;
	iswall(7,-5,-10):=true;

	iswall(7,-4,-0):=true;
	iswall(7,-4,-4):=true;
	iswall(7,-4,-5):=true;


	iswall(7,-3,-0):=true;
	iswall(7,-3,-2):=true;
	iswall(7,-3,-4):=true;
	iswall(7,-3,-5):=true;
	iswall(7,-3,-6):=true;
	iswall(7,-3,-7):=true;
	iswall(7,-3,-8):=true;
	iswall(7,-3,-9):=true;
	iswall(7,-3,-10):=true;

	iswall(7,-2,0):=true;
	iswall(7,-2,-2):=true;

	iswall(7,-1,-0):=true;
	iswall(7,-1,-2):=true;
	iswall(7,-1,-3):=true;
	iswall(7,-1,-4):=true;
	iswall(7,-1,-8):=true;
	iswall(7,-1,-9):=true;
	iswall(7,-1,-10):=true;

	iswall(7,0,-4):=true; --last of upper half

	iswall(7,1,0):=true;
	iswall(7,1,-1):=true;
	iswall(7,1,-2):=true;
	iswall(7,1,-4):=true;
	iswall(7,1,-6):=true;
	iswall(7,1,-7):=true;
	iswall(7,1,-8):=true;
	iswall(7,1,-9):=true;
	iswall(7,1,-10):=true;

	iswall(7,2,-2):=true;
	iswall(7,2,-4):=true;
	iswall(7,2,-6):=true;
	iswall(7,2,-7):=true;
	iswall(7,2,-8):=true;
	iswall(7,2,-9):=true;
	iswall(7,2,-10):=true;

	iswall(7,3,0):=true;
	iswall(7,3,-2):=true;
	iswall(7,3,-4):=true;
	iswall(7,3,-6):=true;

	iswall(7,4,0):=true;
	iswall(7,4,-2):=true;
	iswall(7,4,-4):=true;
	iswall(7,4,-6):=true;
	iswall(7,4,-8):=true;
	iswall(7,4,-9):=true;
	iswall(7,4,-10):=true;

	iswall(7,5,0):=true;
	iswall(7,5,-2):=true;
	iswall(7,5,-6):=true;
	iswall(7,5,-8):=true;
	iswall(7,5,-9):=true;
	iswall(7,5,-10):=true;


	iswall(7,6,0):=true;
	iswall(7,6,-2):=true;
	iswall(7,6,-3):=true;
	iswall(7,6,-4):=true;
	iswall(7,6,-6):=true;

	iswall(7,7,0):=true;

	iswall(7,8,0):=true;
	iswall(7,8,-2):=true;
	iswall(7,8,-3):=true;
	iswall(7,8,-4):=true;
	iswall(7,8,-6):=true;


	iswall(7,9,0):=true;
	iswall(7,9,-6):=true;

	iswall(7,10, 0):=true;
	iswall(7,10,-6):=true;


------ now, repeat maze#7 but negating 3rd coord ----------------------


	iswall(7,-10,7):=true;

	iswall(7,-9,7):=true;

	iswall(7,-8,1):=true;
	iswall(7,-8,2):=true;
	iswall(7,-8,3):=true;
	iswall(7,-8,4):=true;
	iswall(7,-8,7):=true;
	iswall(7,-8,9):=true;
	iswall(7,-8,10):=true;

	iswall(7,-7,7):=true;

	iswall(7,-6,4):=true;
	iswall(7,-6,5):=true;
	iswall(7,-6,7):=true;
	iswall(7,-6,8):=true;
	iswall(7,-6,9):=true;
	iswall(7,-6,10):=true;

	iswall(7,-5,4):=true;
	iswall(7,-5,5):=true;
	iswall(7,-5,7):=true;
	iswall(7,-5,8):=true;
	iswall(7,-5,9):=true;
	iswall(7,-5,10):=true;

	iswall(7,-4,4):=true;
	iswall(7,-4,5):=true;

	iswall(7,-3,2):=true;
	iswall(7,-3,4):=true;
	iswall(7,-3,5):=true;
	iswall(7,-3,6):=true;
	iswall(7,-3,7):=true;
	iswall(7,-3,8):=true;
	iswall(7,-3,9):=true;
	iswall(7,-3,10):=true;

	iswall(7,-2,2):=true;

	iswall(7,-1,2):=true;
	iswall(7,-1,3):=true;
	iswall(7,-1,4):=true;
	iswall(7,-1,8):=true;
	iswall(7,-1,9):=true;
	iswall(7,-1,10):=true;

	iswall(7,0,4):=true; --last of upper half

	iswall(7,1,1):=true;
	iswall(7,1,2):=true;
	iswall(7,1,4):=true;
	iswall(7,1,6):=true;
	iswall(7,1,7):=true;
	iswall(7,1,8):=true;
	iswall(7,1,9):=true;
	iswall(7,1,10):=true;

	iswall(7,2,2):=true;
	iswall(7,2,4):=true;
	iswall(7,2,6):=true;
	iswall(7,2,7):=true;
	iswall(7,2,8):=true;
	iswall(7,2,9):=true;
	iswall(7,2,10):=true;

	iswall(7,3,2):=true;
	iswall(7,3,4):=true;
	iswall(7,3,6):=true;

	iswall(7,4,2):=true;
	iswall(7,4,4):=true;
	iswall(7,4,6):=true;
	iswall(7,4,8):=true;
	iswall(7,4,9):=true;
	iswall(7,4,10):=true;

	iswall(7,5,2):=true;
	iswall(7,5,6):=true;
	iswall(7,5,8):=true;
	iswall(7,5,9):=true;
	iswall(7,5,10):=true;


	iswall(7,6,2):=true;
	iswall(7,6,3):=true;
	iswall(7,6,4):=true;
	iswall(7,6,6):=true;


	iswall(7,8,2):=true;
	iswall(7,8,3):=true;
	iswall(7,8,4):=true;
	iswall(7,8,6):=true;


	iswall(7,9,6):=true;

	iswall(7,10,6):=true;


-------- scene#7 transitions (letter,z,x) ------------------------
-- (-9,-10)  7A  (-2,+10)		dxme:=+(20+step)/-(20+step);
-- (-7,-10)  7B  ( 0,+10)
-- (-4,-10)  7C  (+3,+10)
-- (-2,-10)  7D  (-9,+10)
-- ( 0,-10)  7E  (-7,+10)
-- (+3,-10)  7F  (-4,+10)

-- entry @ (-10,0)
-- exit  @ (+7,-10) to room8 @ (7,+10)
-- exit  @ (+7,+10) to room8 @ (7,-10)
-------------------------------------------



------------ begin scene #8 labyrinth ---------------------------
--
-- entry @ (x,z)=(-10,7) & (+10,7)  minotaur@(0,-1)=center
-----------------------------------------------------------

-- note:  these x,z coords are NOT reversed

	-- -X wall:
	iswall(8,-8,7):=true;

	iswall(8,-8,-8):=true;
	iswall(8,-8,-7):=true;
	iswall(8,-8,-6):=true;
	iswall(8,-8,-5):=true;
	iswall(8,-8,-4):=true;
	iswall(8,-8,-3):=true;
	iswall(8,-8,-2):=true;
	iswall(8,-8,-1):=true;
	iswall(8,-8,0):=true;
	iswall(8,-8,1):=true;
	iswall(8,-8,2):=true;
	iswall(8,-8,3):=true;
	iswall(8,-8,4):=true;

	iswall(8,-6,-6):=true;
	iswall(8,-6,-5):=true;
	iswall(8,-6,-4):=true;
	iswall(8,-6,-3):=true;
	iswall(8,-6,-2):=true;
	iswall(8,-6,-1):=true;
	iswall(8,-6,0):=true;
	iswall(8,-6,1):=true;
	iswall(8,-6,2):=true;


	-- +X wall:
	iswall(8,6,-6):=true;
	iswall(8,6,-5):=true;
	iswall(8,6,-4):=true;
	iswall(8,6,-3):=true;
	iswall(8,6,-2):=true;
	iswall(8,6,-1):=true;
	iswall(8,6,0):=true;
	iswall(8,6,1):=true;
	iswall(8,6,2):=true;
	iswall(8,6,3):=true;
	iswall(8,6,4):=true;
	iswall(8,6,5):=true;
	iswall(8,6,6):=true;
	iswall(8,6,7):=true;
	iswall(8,6,8):=true;


	iswall(8,8,-8):=true;
	iswall(8,8,-7):=true;
	iswall(8,8,-6):=true;
	iswall(8,8,-5):=true;
	iswall(8,8,-4):=true;
	iswall(8,8,-3):=true;
	iswall(8,8,-2):=true;
	iswall(8,8,-1):=true;
	iswall(8,8,0):=true;
	iswall(8,8,1):=true;
	iswall(8,8,2):=true;
	iswall(8,8,3):=true;
	iswall(8,8,4):=true;
	iswall(8,8,5):=true;
	iswall(8,8,6):=true;
	iswall(8,8,7):=true;
	iswall(8,8,8):=true;
	iswall(8,8,9):=true;
	iswall(8,8,10):=true;

----- end vertical walls;  begin horizontal -------------

	iswall(8,-8,-8):=true;
	iswall(8,-7,-8):=true;
	iswall(8,-6,-8):=true;
	iswall(8,-5,-8):=true;
	iswall(8,-4,-8):=true;
	iswall(8,-3,-8):=true;
	iswall(8,-2,-8):=true;
	iswall(8,-1,-8):=true;
	iswall(8,0,-8):=true;
	iswall(8,1,-8):=true;
	iswall(8,2,-8):=true;
	iswall(8,3,-8):=true;
	iswall(8,4,-8):=true;
	iswall(8,5,-8):=true;
	iswall(8,6,-8):=true;
	iswall(8,7,-8):=true;
	iswall(8,8,-8):=true;

	iswall(8,-6,-6):=true;
	iswall(8,-5,-6):=true;
	iswall(8,-4,-6):=true;
	iswall(8,-3,-6):=true;
	iswall(8,-2,-6):=true;
	iswall(8,-1,-6):=true;
	iswall(8,0,-6):=true;
	iswall(8,1,-6):=true;
	iswall(8,2,-6):=true;
	iswall(8,3,-6):=true;
	iswall(8,4,-6):=true;
	iswall(8,5,-6):=true;
	iswall(8,6,-6):=true;


	iswall(8,-8,4):=true;
	iswall(8,-7,4):=true;
	iswall(8,-6,4):=true;
	iswall(8,-5,4):=true;
	iswall(8,-4,4):=true;
	iswall(8,-3,4):=true;
	iswall(8,-2,4):=true;
	iswall(8,-1,4):=true;
	iswall(8, 0,4):=true;
	iswall(8, 1,4):=true;
	iswall(8, 2,4):=true;
	iswall(8, 3,4):=true;
	iswall(8, 4,4):=true;


	iswall(8,-10,6):=true;
	iswall(8,-9,6):=true;
	iswall(8,-8,6):=true;
	iswall(8,-7,6):=true;
	iswall(8,-6,6):=true;
	iswall(8,-5,6):=true;
	iswall(8,-4,6):=true;
	iswall(8,-3,6):=true;
	iswall(8,-2,6):=true;
	iswall(8,-1,6):=true;
	iswall(8, 0,6):=true;
	iswall(8, 1,6):=true;
	iswall(8, 2,6):=true;
	iswall(8, 3,6):=true;
	iswall(8, 4,6):=true;
	iswall(8, 5,6):=true;
	iswall(8, 6,6):=true;


	iswall(8,-8,8):=true;
	iswall(8,-7,8):=true;
	iswall(8,-6,8):=true;
	iswall(8,-5,8):=true;
	iswall(8,-4,8):=true;
	iswall(8,-3,8):=true;
	iswall(8,-2,8):=true;
	iswall(8,-1,8):=true;
	iswall(8, 0,8):=true;
	iswall(8, 1,8):=true;
	iswall(8, 2,8):=true;
	iswall(8, 3,8):=true;
	iswall(8, 4,8):=true;
	iswall(8, 5,8):=true;
	iswall(8, 6,8):=true;

	iswall(8, 8,8):=true;
	iswall(8, 9,8):=true;
	iswall(8,10,8):=true;




end initializeNewMazes;

-- note transition table below

-- scene		no		so		ea									we

-- 	5		x		in		6Y(5,-10,9)->(6,+10,-4)		6C(5,10,9)->(6,-10,9)
-- 	5		x		in		6Z(5,-10,4)->(6,+10,-9)		6D(5,10,4)->(6,-10,4)

-- 	5		x		in		6V(5,-10,2)->(6,+10,2)		6E(5,+10,2)->(6,-10,2)
-- 	5		x		in		6W(5,-10,0)->(6,+10,0)		6F(5,+10,0)->(6,-10,0)
-- 	5		x		in		6X(5,-10,-2)->(6,+10,-2)	6G(5,+10,-2)->(6,-10,-2)



--		6		out	x		5C(6,-10,9)->(5,+10,9)		6A(6,+10,+9)->(6,-10,-4)
--		6		out	x		5D(6,-10,4)->(5,+10,4)		6B(6,+10,+4)->(6,-10,-9)

--		6		out	x		5E(6,-10,2)->(5,+10,2)		5V(6,+10,+2)->(5,-10,+2)
--		6		out	x		5F(6,-10,0)->(5,+10,0)		5W(6,+10,0)->(5,-10,0)
--		6		out	x		5G(6,-10,-2)->(5,+10,-2)	5X(6,+10,-2)->(5,-10,-2)

--		6		out	x		6A(6,-10,-4)->(6,+10,+9)	5Y(6,+10,-4)->(5,-10,+9)
--		6		out	x		6B(6,-10,-9)->(6,+10,+4)	5Z(6,+10,-9)->(5,-10,+4)

-- denote transition ID by (ea/we, initial-scene, letter):
--
-- {ea,we} + {5,6} + {a,b,c,d,e,f,g, v,w,x,y,z}
-- so ...
-----------------------------------------
-- ea5y: x+20, z-13
-- ea5z: x+20, z-13
-- ea5v: x+20
-- ea5w: x+20
-- ea5x: x+20
--
-- we5c: x-20
-- we5d: x-20
-- we5e: x-20
-- we5f: x-20
-- we5g: x-20
----------------------------------
-- ea6c: x+20
-- ea6d: x+20
-- ea6e: x+20
-- ea6f: x+20
-- ea6g: x+20
-- ea6a: x+20, z+13
-- ea6b: x+20, z+13
--
-- we6a: x-20, z-13
-- we6b: x-20, z-13
-- we6v: x-20
-- we6w: x-20
-- we6x: x-20
-- we6y: x-20, z+13
-- we6z: x-20, z+13

crossa: float := 0.0;
function ata1(now: float) return boolean is
	near: constant float := here;
	dist : float;
	dt : float := now - crossa;
begin
if dt>repassage then
	if (scene/=6) then return false;
	else
		dist := fmath.sqrt( sqr(+10.0-xme) + sqr(+9.0-zme) );
		if dist<near then
			crossa:=now;
--put_line("at A1");
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end ata1;


function ata2(now: float) return boolean is
	near: constant float := here;
	dist : float;
	dt : float := now - crossa;
begin
if dt>repassage then
	if (scene/=6) then return false;
	else
		dist := fmath.sqrt( sqr(-10.0-xme) + sqr(-4.0-zme) );
		if dist<near then
			crossa:=now;
--put_line("at A2");
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end ata2;







crossb: float := 0.0;
function atb1(now: float) return boolean is
	near: constant float := here;
	dist : float;
	dt : float := now - crossb;
begin
if dt>repassage then
	if (scene/=6) then return false;
	else
		dist := fmath.sqrt( sqr(+10.0-xme) + sqr(+4.0-zme) );
		if dist<near then
			crossb:=now;
--put_line("at B1");
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end atb1;


function atb2(now: float) return boolean is
	near: constant float := here;
	dist : float;
	dt : float := now - crossb;
begin
if dt>repassage then
	if (scene/=6) then return false;
	else
		dist := fmath.sqrt( sqr(-10.0-xme) + sqr(-9.0-zme) );
		if dist<near then
			crossb:=now;
--put_line("at B2");
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end atb2;









crossc: float := 0.0;
function atc(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossc;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(+10.0-xme) + sqr(+9.0-zme) );
	dist6 := fmath.sqrt( sqr(-10.0-xme) + sqr(+9.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossc:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atc;



crossd: float := 0.0;
function atd(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossd;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(+10.0-xme) + sqr(+4.0-zme) );
	dist6 := fmath.sqrt( sqr(-10.0-xme) + sqr(+4.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossd:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atd;


crosse: float := 0.0;
function ate(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crosse;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(+10.0-xme) + sqr(+2.0-zme) );
	dist6 := fmath.sqrt( sqr(-10.0-xme) + sqr(+2.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crosse:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end ate;


crossf: float := 0.0;
function atf(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossf;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(+10.0-xme) + sqr(+0.0-zme) );
	dist6 := fmath.sqrt( sqr(-10.0-xme) + sqr(+0.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossf:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atf;


crossg: float := 0.0;
function atg(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossg;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(+10.0-xme) + sqr(-2.0-zme) );
	dist6 := fmath.sqrt( sqr(-10.0-xme) + sqr(-2.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossg:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atg;


crossv: float := 0.0;
function atv(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossv;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(-10.0-xme) + sqr(+2.0-zme) );
	dist6 := fmath.sqrt( sqr(+10.0-xme) + sqr(+2.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossv:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atv;


crossw: float := 0.0;
function atw(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossw;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(-10.0-xme) + sqr(+0.0-zme) );
	dist6 := fmath.sqrt( sqr(+10.0-xme) + sqr(+0.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossw:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atw;


crossx: float := 0.0;
function atx(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossx;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(-10.0-xme) + sqr(-2.0-zme) );
	dist6 := fmath.sqrt( sqr(+10.0-xme) + sqr(-2.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossx:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atx;


crossy: float := 0.0;
function aty(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossy;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(-10.0-xme) + sqr(+9.0-zme) );
	dist6 := fmath.sqrt( sqr(+10.0-xme) + sqr(-4.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossy:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end aty;


crossz: float := 0.0;
function atz(now: float) return boolean is
	near: constant float := here;
	dist5, dist6 : float;
	dt : float := now - crossz;
begin
if dt>repassage then
	dist5 := fmath.sqrt( sqr(-10.0-xme) + sqr(+4.0-zme) );
	dist6 := fmath.sqrt( sqr(+10.0-xme) + sqr(-9.0-zme) );
	if 
		((dist5<near) and (scene=5))
		or 
		((dist6<near) and (scene=6))
	then
		crossz:=now;
		return true;
	else
		return false;
	end if;
else
	return false;
end if;
end atz;



function nearsnake return boolean is
	tooclose : constant float := 1.0;
	dist : float;
begin

	if scene=7 then
	  dist:=sqr(x7snake-xme)+sqr(z7snake-zme);
	elsif scene=8 then
	  dist:=sqr(x8snake-xme)+sqr(z8snake-zme);
	elsif scene=6 then
	  dist:=sqr(x6snake-xme)+sqr(z6snake-zme);
	elsif scene=5 then
	  dist:=sqr(x5snake-xme)+sqr(z5snake-zme);
	else
		dist:=999.0;
	end if;

	if dist<tooclose*tooclose then
		return true;
	else
		return false;
	end if;

end nearsnake;



-- for when we have to draw objects in order from far to near
-- note that bubble sort is actually very efficient for short lists.
procedure sort( 
	f2n: in out sortarray;  -- output permutations
	ox,oz : limarray;       -- pos of each object
	lo, hi : integer;       -- bounds of sort
	eyex,eyez : float       -- xme, zme
	) is

	isave : integer;
	di, dj : float;
begin

	myassert( lo>=nsortrng'first );
	myassert( hi<=nsortrng'last  );

	for i in lo..hi loop
		f2n(i):=i; -- initialize permutation
	end loop;

	for i in reverse lo..hi loop
		for j in 1..i loop

			di := sqr( ox( f2n(i) ) - eyex ) + sqr( oz( f2n(i) ) - eyez );
			dj := sqr( ox( f2n(j) ) - eyex ) + sqr( oz( f2n(j) ) - eyez );
			if( di > dj ) then -- swap i,j
				isave:=f2n(i);
				f2n(i):=f2n(j);
				f2n(j):=isave;
			end if;

		end loop; -- for j
	end loop; -- for i

end sort;















cross7f: float:=0.0;
function at7f(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7f;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(-4.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(+3.0-zme) );
		if dist1<near then
			cross7f:=now;
			xme:=-10.0+step;
			zme:=+3.0;
			return true;
		elsif dist2<near then
			cross7f:=now;
			xme:=+10.0-step;
			zme:=-4.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7f;










cross7e: float:=0.0;
function at7e(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7e;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(-7.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(-0.0-zme) );
		if dist1<near then
			cross7e:=now;
			xme:=-10.0+step;
			zme:=0.0;
			return true;
		elsif dist2<near then
			cross7e:=now;
			xme:=+10.0-step;
			zme:=-7.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7e;









cross7d: float:=0.0;
function at7d(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7d;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(-9.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(-2.0-zme) );
		if dist1<near then
			cross7d:=now;
			xme:=-10.0+step;
			zme:=-2.0;
			return true;
		elsif dist2<near then
			cross7d:=now;
			xme:=+10.0-step;
			zme:=-9.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7d;








cross7c: float:=0.0;
function at7c(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7c;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(+3.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(-4.0-zme) );
		if dist1<near then
			cross7c:=now;
			xme:=-10.0+step;
			zme:=-4.0;
			return true;
		elsif dist2<near then
			cross7c:=now;
			xme:=+10.0-step;
			zme:=+3.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7c;







cross7b: float:=0.0;
function at7b(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7b;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(-0.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(-7.0-zme) );
		if dist1<near then
			cross7b:=now;
			xme:=-10.0+step;
			zme:=-7.0;
			return true;
		elsif dist2<near then
			cross7b:=now;
			xme:=+10.0-step;
			zme:=0.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7b;






cross7a: float:=0.0;
function at7a(now: float) return boolean is
	near: constant float := here;
	dist1, dist2 : float;
	dt : float := now - cross7a;
begin
if dt>repassage then
	if (scene/=7) then return false;
	else
		dist1 := fmath.sqrt( sqr(+10.0-xme) + sqr(-2.0-zme) );
		dist2 := fmath.sqrt( sqr(-10.0-xme) + sqr(-9.0-zme) );
		if dist1<near then
			cross7a:=now;
			xme:=-10.0+step;
			zme:=-9.0;
			return true;
		elsif dist2<near then
			cross7a:=now;
			xme:=+10.0-step;
			zme:=-2.0;
			return true;
		else
			return false;
		end if;
	end if;
else
	return false;
end if;
end at7a;







cross78g: float:=0.0;
function at78g(now: float) return boolean is
	near: constant float := here;
	dist7, dist8 : float;
	dt : float := now - cross78g;
begin
if dt>repassage then
	if (scene=7) then

		dist7 := fmath.sqrt( sqr(-10.0-xme) + sqr(+7.0-zme) );
		if dist7<near then
			cross78g:=now;
			xme:=+10.0-step;
			return true;
		else
			return false;
		end if;

	elsif scene=8 then

		dist8 := fmath.sqrt( sqr(+10.0-xme) + sqr(+7.0-zme) );
		if dist8<near then
			cross78g:=now;
			xme:=-10.0+step;
			return true;
		else
			return false;
		end if;

	else
		return false;
	end if;
else
	return false;
end if;
end at78g;








cross78h: float:=0.0;
function at78h(now: float) return boolean is
	near: constant float := here;
	dist7, dist8 : float;
	dt : float := now - cross78h;
begin
if dt>repassage then
	if (scene=8) then
		dist8 := fmath.sqrt( sqr(-10.0-xme) + sqr(+7.0-zme) );
		if dist8<near then
			cross78h:=now;
			xme:=+10.0-step;
			return true;
		else
			return false;
		end if;
	elsif scene=7 then
		dist7 := fmath.sqrt( sqr(+10.0-xme) + sqr(+7.0-zme) );
		if dist7<near then
			cross78h:=now;
			xme:=-10.0+step;
			return true;
		else
			return false;
		end if;
	else
		return false;
	end if;
else
	return false;
end if;
end at78h;








end gameutils;



