/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.lock;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Properties;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.storage.BrokerPool;
import org.exist.storage.lock.FileLockHeartBeat;
import org.exist.util.FileUtils;
import org.exist.util.ReadOnlyException;

public class FileLock {
    private static final Logger LOG = LogManager.getLogger(FileLock.class);
    private final long HEARTBEAT = 10100L;
    private static final byte[] MAGIC = new byte[]{101, 88, 105, 115, 116, 45, 100, 98};
    private BrokerPool pool;
    private Path lockFile;
    private SeekableByteChannel channel = null;
    private final ByteBuffer buf = ByteBuffer.allocate(MAGIC.length + 8);
    private long lastHeartbeat = -1L;

    public FileLock(BrokerPool pool, Path path) {
        this.pool = pool;
        this.lockFile = path;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean tryLock() throws ReadOnlyException {
        int attempt = 0;
        while (Files.exists(this.lockFile, new LinkOption[0])) {
            if (++attempt > 2) {
                return false;
            }
            try {
                this.read();
            }
            catch (IOException e22) {
                this.message("Failed to read lock file", null);
                e22.printStackTrace();
            }
            if (!this.checkHeartbeat()) continue;
            FileLock e22 = this;
            synchronized (e22) {
                try {
                    this.message("Waiting a short time for the lock to be released...", null);
                    this.wait(10200L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            try {
                if (this.channel.isOpen()) {
                    this.channel.close();
                }
                this.channel = null;
            }
            catch (IOException e22) {}
        }
        try {
            this.lockFile = Files.createFile(this.lockFile, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new ReadOnlyException(this.message("Could not create lock file", e));
        }
        try {
            this.save();
        }
        catch (IOException e) {
            throw new ReadOnlyException(this.message("Caught exception while trying to write lock file", e));
        }
        Properties params = new Properties();
        params.put(FileLock.class.getName(), this);
        this.pool.getScheduler().createPeriodicJob(10100L, new FileLockHeartBeat(this.lockFile.toAbsolutePath().toString()), -1L, params);
        return true;
    }

    public void release() {
        try {
            if (this.channel.isOpen()) {
                this.channel.close();
            }
            this.channel = null;
        }
        catch (Exception e) {
            this.message("Failed to close lock file", e);
        }
        if (Files.exists(this.lockFile, new LinkOption[0])) {
            LOG.info("Deleting lock file: " + this.lockFile.toAbsolutePath().toString());
            FileUtils.deleteQuietly(this.lockFile);
        }
    }

    public Date getLastHeartbeat() {
        return new Date(this.lastHeartbeat);
    }

    public Path getFile() {
        return this.lockFile;
    }

    private boolean checkHeartbeat() {
        long now = System.currentTimeMillis();
        if (this.lastHeartbeat < 0L || now - this.lastHeartbeat > 10100L) {
            this.message("Found a stale lockfile. Trying to remove it: ", null);
            this.release();
            return false;
        }
        return true;
    }

    private void open() throws IOException {
        this.channel = Files.newByteChannel(this.lockFile, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SYNC);
    }

    protected void save() throws IOException {
        try {
            if (this.channel == null) {
                this.open();
            }
            long now = System.currentTimeMillis();
            this.buf.clear();
            this.buf.put(MAGIC);
            this.buf.putLong(now);
            this.buf.flip();
            this.channel.position(0L);
            this.channel.write(this.buf);
            this.lastHeartbeat = now;
        }
        catch (NullPointerException npe) {
            if (this.pool.isShuttingDown()) {
                LOG.info("No need to save FileLock, database is shutting down");
            }
            throw npe;
        }
    }

    private void read() throws IOException {
        if (this.channel == null) {
            this.open();
        }
        this.channel.read(this.buf);
        this.buf.flip();
        if (this.buf.limit() < 16) {
            this.buf.clear();
            throw new IOException(this.message("Could not read file lock.", null));
        }
        byte[] magic = new byte[8];
        this.buf.get(magic);
        if (!Arrays.equals(magic, MAGIC)) {
            throw new IOException(this.message("Bad signature in lock file. It does not seem to be an eXist lock file", null));
        }
        this.lastHeartbeat = this.buf.getLong();
        this.buf.clear();
        DateFormat df = DateFormat.getDateInstance();
        this.message("File lock last access timestamp: " + df.format(this.getLastHeartbeat()), null);
    }

    protected String message(String message, Exception e) {
        StringBuilder str = new StringBuilder(message);
        str.append(' ').append(this.lockFile.toAbsolutePath().toString());
        if (e != null) {
            str.append(": ").append(e.getMessage());
        }
        message = str.toString();
        if (LOG.isInfoEnabled()) {
            LOG.info(message);
        }
        return message;
    }
}

