/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.txn;

import java.io.IOException;
import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Function;
import net.jcip.annotations.GuardedBy;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.EXistException;
import org.exist.security.PermissionDeniedException;
import org.exist.storage.BrokerPool;
import org.exist.storage.BrokerPoolService;
import org.exist.storage.DBBroker;
import org.exist.storage.SystemTask;
import org.exist.storage.SystemTaskManager;
import org.exist.storage.journal.JournalException;
import org.exist.storage.journal.JournalManager;
import org.exist.storage.txn.TransactionException;
import org.exist.storage.txn.Txn;
import org.exist.storage.txn.TxnAbort;
import org.exist.storage.txn.TxnCommit;
import org.exist.storage.txn.TxnStart;
import org.exist.xmldb.XmldbURI;

public class TransactionManager
implements BrokerPoolService {
    private static final Logger LOG = LogManager.getLogger(TransactionManager.class);
    private long nextTxnId = 0L;
    private final BrokerPool pool;
    private final Optional<JournalManager> journalManager;
    private final SystemTaskManager systemTaskManager;
    private final Map<Long, TxnCounter> transactions = new HashMap<Long, TxnCounter>();
    private final Lock lock = new ReentrantLock();

    public TransactionManager(BrokerPool pool, Optional<JournalManager> journalManager, SystemTaskManager systemTaskManager) {
        this.pool = pool;
        this.journalManager = journalManager;
        this.systemTaskManager = systemTaskManager;
    }

    public Txn beginTransaction() {
        return this.withLock((DBBroker broker) -> {
            long txnId = this.nextTxnId++;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Starting new transaction: " + txnId);
            }
            if (this.journalManager.isPresent()) {
                try {
                    this.journalManager.get().journal(new TxnStart(txnId));
                }
                catch (JournalException e) {
                    LOG.error("Failed to create transaction. Error writing to log file.", (Throwable)e);
                }
            }
            Txn txn = new Txn(this, txnId);
            this.transactions.put(txn.getId(), new TxnCounter());
            return txn;
        });
    }

    public void commit(Txn txn) throws TransactionException {
        if (txn.getState() != Txn.State.STARTED) {
            return;
        }
        this.withLock((DBBroker broker) -> {
            if (this.journalManager.isPresent()) {
                try {
                    this.journalManager.get().journalGroup(new TxnCommit(txn.getId()));
                }
                catch (JournalException e) {
                    LOG.error("Failed to write commit record to journal: " + e.getMessage());
                }
            }
            txn.signalCommit();
            txn.releaseAll();
            this.transactions.remove(txn.getId());
            this.processSystemTasks();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Committed transaction: " + txn.getId());
            }
        });
    }

    public void abort(Txn txn) {
        Objects.requireNonNull(txn);
        if (txn.getState() != Txn.State.STARTED) {
            return;
        }
        this.withLock((DBBroker broker) -> {
            this.transactions.remove(txn.getId());
            if (this.journalManager.isPresent()) {
                try {
                    this.journalManager.get().journalGroup(new TxnAbort(txn.getId()));
                }
                catch (JournalException e) {
                    LOG.error("Failed to write abort record to journal: " + e.getMessage());
                }
            }
            txn.signalAbort();
            txn.releaseAll();
            this.processSystemTasks();
        });
    }

    public void close(Txn txn) {
        Objects.requireNonNull(txn);
        if (txn.getState() == Txn.State.CLOSED) {
            return;
        }
        try {
            if (txn.getState() == Txn.State.STARTED) {
                LOG.warn("Transaction was not committed or aborted, auto aborting!");
                this.abort(txn);
            }
        }
        finally {
            txn.setState(Txn.State.CLOSED);
        }
    }

    public void trackOperation(long txnId) {
        TxnCounter count = this.transactions.get(txnId);
        if (count != null) {
            count.increment();
        }
    }

    public Lock getLock() {
        return this.lock;
    }

    public void checkpoint(boolean switchFiles) throws TransactionException {
        long txnId = this.nextTxnId++;
        if (this.journalManager.isPresent()) {
            try {
                this.journalManager.get().checkpoint(txnId, switchFiles);
            }
            catch (JournalException e) {
                throw new TransactionException(e.getMessage(), e);
            }
        }
    }

    @Deprecated
    public void reindex(DBBroker broker) throws IOException {
        broker.pushSubject(broker.getBrokerPool().getSecurityManager().getSystemSubject());
        try {
            broker.reindexCollection(XmldbURI.ROOT_COLLECTION_URI);
        }
        catch (PermissionDeniedException e) {
            LOG.error("Exception during reindex: " + e.getMessage(), (Throwable)e);
        }
        finally {
            broker.popSubject();
        }
    }

    @Override
    public void shutdown() {
        int uncommitted;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Shutting down transaction manager. Uncommitted transactions: " + this.transactions.size());
        }
        this.shutdown((uncommitted = this.uncommittedTransaction()) == 0);
    }

    public void shutdown(boolean checkpoint) {
        long txnId = this.nextTxnId++;
        if (this.journalManager.isPresent()) {
            this.journalManager.get().shutdown(txnId, checkpoint);
        }
        this.transactions.clear();
    }

    private int uncommittedTransaction() {
        int count = 0;
        if (this.transactions.isEmpty()) {
            return count;
        }
        for (Map.Entry<Long, TxnCounter> entry : this.transactions.entrySet()) {
            if (entry.getValue().counter <= 0) continue;
            LOG.warn("Found an uncommitted transaction with id " + entry.getKey() + ". Pending operations: " + entry.getValue().counter);
            ++count;
        }
        if (count > 0) {
            LOG.warn("There are uncommitted transactions. A recovery run may be triggered upon restart.");
        }
        return count;
    }

    public void triggerSystemTask(SystemTask task) {
        this.withLock((DBBroker broker) -> this.systemTaskManager.triggerSystemTask(task));
    }

    public void processSystemTasks() {
        this.withLock((DBBroker broker) -> {
            if (this.transactions.isEmpty()) {
                this.systemTaskManager.processTasks();
            }
        });
    }

    public void debug(PrintStream out) {
        out.println("Active transactions: " + this.transactions.size());
    }

    @GuardedBy(value="lock")
    private void withLock(Consumer<DBBroker> lockedCn) {
        this.withLock((DBBroker broker) -> {
            lockedCn.accept((DBBroker)broker);
            return null;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive exception aggregation
     */
    @GuardedBy(value="lock")
    private <T> T withLock(Function<DBBroker, T> lockedFn) {
        try {
            Throwable throwable = null;
            try (DBBroker broker = this.pool.getBroker();){
                DBBroker dBBroker;
                try {
                    this.lock.lock();
                    dBBroker = lockedFn.apply(broker);
                    this.lock.unlock();
                }
                catch (Throwable throwable2) {
                    try {
                        this.lock.unlock();
                        throw throwable2;
                    }
                    catch (Throwable throwable3) {
                        throwable = throwable3;
                        throw throwable3;
                    }
                }
                return (T)dBBroker;
            }
        }
        catch (EXistException e) {
            LOG.error("Transaction manager failed to acquire broker for running system tasks");
            return null;
        }
    }

    protected static final class TxnCounter {
        int counter = 0;

        protected TxnCounter() {
        }

        public void increment() {
            ++this.counter;
        }
    }
}

