/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

/**
 *	Represents a comparison on a numeric or string value.
 *	Static factory methods return Bounds of different kinds (implemented
 *	by nested subclasses).
 */
public abstract class Bound
{
    boolean strict = false;
    public boolean isStrict() { return strict; }

    /**
     *	Tests a numeric value against this Bound.
     *	@throws IllegalArgumentException if not a numeric Bound.
     */
    public abstract boolean test(double value);
    /**
     *	Tests a String value against this Bound.
     *	@throws IllegalArgumentException if not a String Bound.
     */
    public abstract boolean test(String value);

    public abstract boolean isNumeric();

    public String  stringValue() { throw new IllegalArgumentException();  }
    public double  numericValue() { throw new IllegalArgumentException();  }

    /** returns a new Bound matching numbers less-than the specified value. */
    public static Bound LT(double value) {
	return new NumberLT(value);
    }
    /** returns a Bound matching numbers less-than-or-equal-to the specified value. */
    public static Bound LE(double value) {
	return new NumberLE(value);
    }
    /** returns a Bound matching numbers greater-than-or-equal-to the specified value. */
    public static Bound GE(double value) {
	return new NumberGE(value);
    }
    /** returns a Bound matching numbers greater-than the specified value. */
    public static Bound GT(double value) {
	return new NumberGT(value);
    }

    /** returns a new Bound matching strings less-than the specified value. */
    public static Bound LT(String value) {
	return new StringLT(value);
    }
    /** returns a Bound matching strings less-than-or-equal-to the specified value. */
    public static Bound LE(String value) {
	return new StringLE(value);
    }
    /** returns a Bound matching strings greater-than-or-equal-to the specified value. */
    public static Bound GE(String value) {
	return new StringGE(value);
    }
    /** returns a Bound matching strings greater-than the specified value. */
    public static Bound GT(String value) {
	return new StringGT(value);
    }

    public static abstract class Numeric extends Bound {	
	double bound;

	Numeric(double bound) {
	    this.bound = bound;
	}
	public boolean test(String value) { throw new IllegalArgumentException(); }
	public boolean isNumeric() { return true; }
	public double  numericValue() { return bound; }
    }

    public static class NumberLT extends Numeric {
	NumberLT(double value) { super(value); strict = true; }
	public boolean test(double value) { return value < bound; }
    }

    public static class NumberLE extends Numeric {
	NumberLE(double value) { super(value); }
	public boolean test(double value) { return value <= bound; }
    }

    public static class NumberGE extends Numeric {
	NumberGE(double value) { super(value); }
	public boolean test(double value) { return value >= bound; }
    }

    public static class NumberGT extends Numeric {
	NumberGT(double value) { super(value); strict = true; }
	public boolean test(double value) { return value > bound; }
    }

    public static abstract class StringBound extends Bound {	
	String bound;

	StringBound(String bound) {
	    this.bound = bound;
	}
	public boolean test(double value) { throw new IllegalArgumentException(); }
	public boolean isNumeric() { return false; }
	public String  stringValue() { return bound; }
    }

    public static class StringLT extends StringBound {
	StringLT(String value) { super(value); strict = true; }
	public boolean test(String value) { return value.compareTo(bound) < 0; }
    }

    public static class StringLE extends StringBound {
	StringLE(String value) { super(value); }
	public boolean test(String value) { return value.compareTo(bound) <= 0; }
    }

    public static class StringGE extends StringBound {
	StringGE(String value) { super(value); }
	public boolean test(String value) { return value.compareTo(bound) >= 0; }
    }

    public static class StringGT extends StringBound {
	StringGT(String value) { super(value); strict = true; }
	public boolean test(String value) { return value.compareTo(bound) > 0; }
    }
} // end of class Bound


