/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.dt;

import net.xfra.qizxopen.xquery.*;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Locale;

/**
 *	Conversion utility class.
 */
public class Conversion
{

    public static boolean toBoolean( double d ) {
	return d != 0 && d == d /* inlining !isNaN() */;
    }

    public static boolean toBoolean( String s ) {
	return s.length() > 0;
    }


    public static long toInteger( boolean b ) {
	return b ? 1 : 0;
    }

    public static long toInteger( String s ) throws TypeException {
	try {
	    return Long.parseLong( s );
        }
        catch ( NumberFormatException e ) {
            throw new TypeException("invalid integer value "+s);
        }
    }
    /**
     *	Checks if the value is out of bounds. Can be used to detect arithmetic overflow.
     *	Slightly conservative (returns false above 0x7ffffffffffff000).
     */
    public static boolean isIntegerRange( double value ) {
	return (value >= MIN_INTVALUE && value < MAX_INTVALUE);	// the simpler the better
    }
    // cannot be represented with full precision on double: 
    static double MAX_INTVALUE = (double) (1L << 62), MIN_INTVALUE = - MAX_INTVALUE;

    public static float toFloat( boolean b ) {
	return b ? 1.0f : 0.0f;
    }

    public static float toFloat( String s ) throws TypeException {
	try {
	    return Float.parseFloat( s );
        }
        catch ( NumberFormatException e ) {
            throw new TypeException("invalid float value "+s);
        }
    }

    public static double toDouble( boolean b ) {
	return b ? 1.0 : 0.0;
    }

    public static double toDouble( String s ) throws TypeException {
	try {
	    return Double.parseDouble( s );
        }
        catch ( NumberFormatException e ) {
            throw new TypeException("invalid double value "+s);
        }
    }

    public static BigDecimal toDecimal( boolean b ) {
	return new BigDecimal(b ? 1.0 : 0.0);
    }

    public static BigDecimal toDecimal( long i ) {
	return new BigDecimal( Long.toString(i) ); // nothing more efficient ???
    }

    public static BigDecimal toDecimal( String s ) throws TypeException {
	try {
	    return new BigDecimal( s );
        }
        catch ( NumberFormatException e ) {
            throw new TypeException("invalid decimal value "+s);
        }
    }


    public static String toString( boolean b ) {
	return b ? "true" : "false";
    }

    public static String toString( long i ) throws TypeException  {
	return Long.toString(i);
    }

    static DecimalFormatSymbols syms = new DecimalFormatSymbols(Locale.US);
    static DecimalFormat fmtSciD = new DecimalFormat("0.0################E0##", syms);
    static DecimalFormat fmtDecD = new DecimalFormat("#####0.0################", syms);
    static DecimalFormat fmtSciF = new DecimalFormat("0.0########E0##", syms);
    static DecimalFormat fmtDecF = new DecimalFormat("#####0.0########", syms);

    public static String toString( float f ) throws TypeException  {
	if(f == 0)
	    return "0";	// what about -0 ?
	if(Float.isInfinite(f))
	    return f < 0 ? "-INF" : "INF";
	if(f != f)
	    return "NaN";

	float av = Math.abs(f);
	String mant, exp = null;
	if(av >= 1e-6f && av < 1e6f) {	    // must be decimal:
	    mant = fmtDecF.format(f);
	}
	else {	    // must be in scientific notation:
	    mant = fmtSciF.format(f);
	    int pex = mant.indexOf('E');
	    if(pex > 0) {
		exp = mant.substring(pex + 1, mant.length());	
		mant = mant.substring(0, pex);
	    }
	}
	// remove trailing ".0" or "0" from mantissa:
	if(mant.indexOf('.') >= 0) {
	    int L = mant.length();
	    while( L > 0 && mant.charAt(L-1) == '0' )
		-- L;
	    if(L > 0 && mant.charAt(L-1) == '.')
		--L;
	    mant = mant.substring(0, L);
	}

	return (exp == null)? mant : mant.concat("E").concat(exp);
    }

    public static String toString( double d ) throws TypeException  {
	if(d == 0)
	    return "0";
	if(Double.isInfinite(d))
	    return d < 0 ? "-INF" : "INF";
	if(d != d)
	    return "NaN";

	double av = Math.abs(d);
	String mant, exp = null;
	if(av >= 1e-6 && av < 1e6) {	    // must be decimal:
	    mant = fmtDecD.format(d);
	}
	else {	    // must be in scientific notation:
	    mant = fmtSciD.format(d);
	    int pex = mant.indexOf('E');
	    if(pex > 0) {
		exp = mant.substring(pex + 1, mant.length());	
		mant = mant.substring(0, pex);
	    }
	}
	// remove trailing ".0" from mantissa:
	int L = mant.length();
	if(L > 2 && mant.charAt(L-1) == '0' && mant.charAt(L-2) == '.')
	    mant = mant.substring(0, L-2);

	return (exp == null)? mant : mant.concat("E").concat(exp);
    }

} // end of class Conversion
