/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.impl.*;

/**
 *	Superclass of all compiled expressions.
 *	
 */
public abstract class Expression
{
    /**
     *	Module where the expression is defined.
     */
    public Module module;
    /**
     *	Character offset in module source code.
     */
    public int    location;
    /**
     *	Statically inferred type.
     */
    protected Type type = Type.ANY;

    public void dump( ExprDump d ) {
	d.println( getClass().getName() );
    }

    public Expression atSamePlaceAs( Expression origin ) {
	module = origin.module;
	location = origin.location;
	return this;
    }

    public static Expression[] addExpr( Expression[] exprs, Expression e ) {
	Expression[] nexprs = new Expression[exprs.length + 1];
	System.arraycopy(exprs, 0, nexprs, 0, exprs.length);
        nexprs[exprs.length] = e;
	return nexprs;
    }

    /**
     *  Static analysis. 
     *  Checks sub-expressions and can replace this expression by an other construct.
     */
    public Expression staticCheck( StaticContext context ) {
	return this;
    }

    /**
     *	Returns the static type of the expression. Valid only after Static Analysis.
     */
    public Type getType() {
	return type;
    }

    public void setType(Type type) {
	this.type = type;
    }
    
    /**
     *	Static optimization of Path: indicates whether all the nodes
     *	generated by a path step are within the subtree of the origin.
     */
    public final static int WITHIN_SUBTREE = 1;
    /**
     *	Static optimization of Paths: indicates whether all the nodes generated
     *  by a path step belong to the origin (true for attributes or namespace nodes).
     */
    public final static int WITHIN_NODE = 2;
    /**
     *	Static optimization of Paths: indicates whether all the nodes
     *	generated by a path step are at same tree depth.
     */
    public final static int SAME_DEPTH = 4;
    /**
     *	Static optimization of Paths: indicates whether all the nodes
     *	generated by a path step are in document order and not duplicated.
     */
    public final static int DOCUMENT_ORDER = 8;
    /**
     *	Static optimization: indicates that this sequence needs not be ordered.
     *	Used in static context.
     */
    public final static int UNORDERED = 0x10;
    /**
     *  Static optimization: indicates a constant or literal expression.
     */
    public final static int CONSTANT = 0x20;

    public final static int NUMERIC = 0x40;

    public int getFlags() {
	return 0;
    }

    protected void checkFocus(Focus focus, EvalContext context) throws EvalException {
	if(focus == null) {
	    //new Exception("where are we?").printStackTrace();
	    context.error(this, "no context item");
	}
    }

    /**
     *	Abstract Expression visitor.
     */
    public static abstract class Visitor {

	public abstract boolean examine(Expression focus);

	public boolean visit(Expression[] args) {
	    for(int i = 0; i < args.length; i++)
		if(args[i] != null && !args[i].visit(this))
		    return false;
	    return true;
	}
    }
    /**
     *	Performs a visit by a Visitor. Assumed to stop as soon as false is returned by
     *  the examine method of the Visitor, or by visit on sub-expressions.
     */
    public abstract boolean visit( Visitor visitor );

    /**
     *	Finds the first subexpression matching the given class.
     */
    public Expression findSubExpression( Class classe ) {
	Finder f = new Finder(classe);
	visit(f);
	return f.found;
    }

    private static class Finder extends Visitor {
	Class searched;
	Expression found = null;

	Finder( Class searched ) {
	    this.searched = searched;
	}

	public boolean examine(Expression focus) {
	    if(focus.getClass().isAssignableFrom(searched)) {
		found = focus;
		return false;
	    }
	    return true;
	}
    }

    /**
     *	Computes the result in the specified evaluation context.
     *  The current context item (denoted by '.') is specified by 'dot'.
     */
    public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	throw new XQueryException(getClass() + " evaluation not implemented");
    }

    /**
     *	Evaluates directly as one single boolean.
     *	Can be optimised by concrete boolean expressions.
     */
    public boolean evalAsBoolean( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    boolean value = v.asBoolean();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
        }
        catch (EmptyException e) {
            context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    return false;
        }
        catch (TypeException e) {
            context.error(this, e); return false;
        }
    }

    /**
     *	Evaluates the effective boolean value.
     *	Can be optimised by concrete expressions.
     */
    public boolean evalEffectiveBooleanValue(Focus focus, EvalContext context)
	throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		return false;
	    boolean ebv = v.asBoolean();
	    // a list of 2 or more items yields true, whatever the item values
	    return v.next() ? true : ebv;
        }
        catch (TypeException e) {
            context.error(this, e); return false;
        }
    }

    /**
     *	Evaluates directly as one mandatory integer.
     *	Can be optimised by concrete Int expressions.
     */
    public long evalAsInteger( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    long value = v.asInteger();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
        }
        catch (EmptyException e) {
            context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    return 0;
        }
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as an optional integer. If empty, raises an EmptyException that
     *  can be caught by enclosing expressions.
     *  An enclosing expression that returns () if this argument is () juste doesn't need to
     *  care about that.
     *  An enclosing expression that needs to care about an empty argument has to catch
     *  the expression.
     *  Note: the runtime cost of raising en exception is quite acceptable because the 
     *	      exception is statically created.
     */
    public long evalAsOptInteger( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		throw EmptyException.allowed();
	    long value = v.asInteger();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as one float. Can be optimised by actual Float expressions.
     */
    public float evalAsFloat( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    float value = v.asFloat();
	    if( v.next() )
		context.error(this, Type.ERR_TOO_MANY);
	    return value;
        }
        catch (EmptyException e) {
            context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED)); 
	    return 0;
        }
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as an optional float. If empty, raises an EmptyException that
     *  can be caught by enclosing expressions. (See comments of {@link #evalAsOptInteger})
     */
     public float evalAsOptFloat( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		throw EmptyException.allowed();
	    float value = v.asFloat();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as one double. Can be optimised by actual Double expressions.
     */
    public double evalAsDouble( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, Type.ERR_EMPTY_UNEXPECTED);
	    double value = v.asDouble();
	    if( v.next() )
		context.error(this, Type.ERR_TOO_MANY);
	    return value;
        }
        catch (EmptyException e) {
            context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    return 0;
        }
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as an optional double. If empty, raises an EmptyException that
     *  can be caught by enclosing expressions. (See comments of {@link #evalAsOptInteger})
     */
    public double evalAsOptDouble( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		throw EmptyException.allowed();
	    double value = v.asDouble();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return 0;
        }
    }

    /**
     *	Evaluates directly as one string. Can be optimised by actual String expressions.
     */
    public String evalAsString( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    String value = v.asString();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluates directly as one string, or null if empty (mechanism simpler than for numerics).
     *  Can be optimised by actual String expressions.
     */
    public String evalAsOptString( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		return null;
	    String value = v.asString();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluates directly as one Node. Can be optimised by actual Node expressions.
     */
    public Node evalAsNode( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    Node value = v.asNode();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluates directly as one Node, or null if empty.
     */
    public Node evalAsOptNode(Focus focus, EvalContext context) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		return null;
	    Node value = v.asNode();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluates directly as one single Item.
     */
    public Item evalAsItem( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		context.error(this, new TypeException(Type.ERR_EMPTY_UNEXPECTED));
	    Item value = v.asItem();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluates directly as one Item, or null if empty.
     */
    public Item evalAsOptItem( Focus focus, EvalContext context ) throws XQueryException {
        try {
	    Value v = eval(focus, context);
	    if( !v.next() )
		return null;
	    Item value = v.asItem();
	    if( v.next() )
		context.error(this, new TypeException(Type.ERR_TOO_MANY));
	    return value;
	}
        catch (TypeException e) {
            context.error(this, e); return null;
        }
    }

    /**
     *	Evaluation of trees directly into a serial output, without actual
     *	construction of Nodes.
     *	This working mode allows the processing of huge documents with a low
     *	memory footprint (assuming that the source document is a FONIDocument).
     */
    public void evalAsEvents( XMLEventReceiver output, Focus focus, EvalContext context )
	throws XQueryException, DataModelException {
	Value seq = eval(focus, context);
	try {
	    for(int it = 0 ; seq.next(); )
		if(seq.isNode())	// necessarily to be copied
		    output.traverse( seq.asNode(), true/*all in-scope NS*/ );
		else {
		    output.atom( seq.asString() );
		}
	}
	catch(DataModelException e) {
	    context.error(this, "error in constructor: " + e.getMessage());
	}
    }
} // end of class Expression
