/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.app;

import net.xfra.qizxopen.util.CLOptions;
import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.dm.DocumentManager;
import net.xfra.qizxopen.dm.XMLSerializer;
import net.xfra.qizxopen.dm.DataModelException;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.fn.JavaFunction;
import net.xfra.qizxopen.xquery.SAXXQueryProcessor;	// hack

import java.io.*;
import java.net.URL;

/**
 *  A simple command line application that reads queries from a file or from std input.
 *  If no input specified, works in interactive mode from stdin (repeatedly reads a query).
 */
public class Qizx extends XQueryProcessor
{
    // options:
    public String  queryFile;
    public boolean verbose = !false;
    public boolean push = false;
    public boolean traceJava = false;
    public boolean exprDump = false;
    public boolean display = true;
    public boolean xmlDisplay = !false;		// 
    public boolean wrapDisplay = !true;		// wrap results in 'item' elements
    public boolean clearDocuments = false;	// before each query
    public String  baseURI;
    public String  modules = ".";
    public String  inputURI = null;
    public String  xmlInput = null;
    public String  output = null;
    public String[] globals = new String[0];
    public String[] xmloptions = new String[0];
    public String[] appargs = new String[0];
    public String[] class_name = new String[0];
    public String  timezone;
    public String  collation;

    public int     indent = 2;
    public int     repeats = 1;
    public boolean traceExceptions = false;
    public boolean traceLex = false;

    // 
    Log log = new Log();	// writes to System.err
    PrintWriter stderr = new PrintWriter(System.err, true);
    XMLSerializer  nodeDisplay;

    static CLOptions options = new CLOptions("XQuery");
    static {
	options.declare("-base", CLOptions.NEXT_ARG,
			"default base URI for documents and modules");
	options.declare("-modules", CLOptions.NEXT_ARG,
			"base URI for resolving module locations");
	options.declare("-D", "globals", CLOptions.STICKY_ARG,
	    "variable_name=value!initialize a global variable defined in the query.");
	options.declare("--", "appargs", CLOptions.ALL_ARG,
	    "pass all following arguments to XQuery processor in variable 'arguments'");
	options.declare("-X", "xmloptions", CLOptions.STICKY_ARG,
			"option=value!set a XML serialization option.");
	options.declare("-serial", "push", CLOptions.SET_ARG,
		     "serial output: the query must evaluate as a well-formed document.");
	options.declare("-out", "output", CLOptions.NEXT_ARG,
			"output file (defaults to standard output)");
	options.declare("-input", "inputURI", CLOptions.NEXT_ARG,
		     "URI of a document used as input (XQuery function input())");
	options.declare("-xinput", "xmlInput", CLOptions.NEXT_ARG, 
			"a XML fragment used as input (XQuery function input())");
	options.declare("-timezone", CLOptions.NEXT_ARG,
			"implicit timezone in duration format");
	options.declare("-collation", CLOptions.NEXT_ARG, "default collation");
	//options.declare("-v", "verbose", CLOptions.SET_ARG, "verbose mode");
	options.declare("-q", "verbose", CLOptions.RESET_ARG, "quiet mode");
	options.declare("-mute", "display", CLOptions.RESET_ARG, "no display of results");
	options.declare("-jt", "traceJava", CLOptions.SET_ARG,
			"trace Java extension functions");
	options.declare("-xd", "xmlDisplay", CLOptions.RESET_ARG, null);
	options.declare("-wrap", "wrapDisplay", CLOptions.SET_ARG,
			"do not wrap results in descriptor tags");
	options.declare("-ext", "class_name", CLOptions.NEXT_ARG,
			"allow this class as Java extension");
	options.declare("-help", null, CLOptions.HELP_ARG, "print this help");
	options.argument("<query file>", "queryFile", 0,
			 "a file containing a query to execute.\n"+
			 "               If equal to '-', use the standard input.\n"+
			 "               If absent, enter interactive mode.");
	// undocumented options:
	options.declare("-r", "repeats", CLOptions.NEXT_ARG, null);
	options.declare("-d", "exprDump", CLOptions.SET_ARG, null);
	options.declare("-tex", "traceExceptions", CLOptions.SET_ARG, null);
	options.declare("-tlex", "traceLex", CLOptions.SET_ARG, null);
	options.declare("-cld", "clearDocuments", CLOptions.SET_ARG, null);
    }

    Qizx( String args[] ) throws IOException, CLOptions.Exception {
	nodeDisplay = new XMLSerializer();
	nodeDisplay.setIndent(indent);
	//nodeDisplay.setDepth(4);
	options.parse(args, this);
    }

    static public void main( String args[] )
    {
        try {
	    Qizx app = new Qizx(args);
	    app.run();
	    app = null;
	    System.gc();
        } catch (CLOptions.Exception e) {
	    System.exit(1);
	} catch (DataModelException e) {
	    System.err.println("*** error in XML option: "+e.getMessage());
	    System.exit(1);
        } catch (Exception e) {
	    
            e.printStackTrace();
	    System.exit(1);
        }
    }

    void run() throws Exception {
	nodeDisplay = new XMLSerializer();

	// XML options:
	for(int g = 0; g < xmloptions.length; g++) {
	    int eq = xmloptions[g].indexOf('=');
	    if(eq < 0) {
		System.err.println("*** illegal XML option: "+xmloptions[g]); return;
	    }
	    nodeDisplay.setOption( xmloptions[g].substring(0, eq),
				   xmloptions[g].substring(eq + 1));
	}
	for(int c = 0; c < class_name.length; c++)
	    authorizeClass(class_name[c]);

	FileOutputStream outs = null;
	if( output != null )
	    nodeDisplay.setOutput( new FileOutputStream(output),
				   nodeDisplay.getEncoding());
	JavaFunction.trace = traceJava;
	Lexer.debug = traceLex;
	setModuleManager( new ModuleManager(modules) );
	setDocumentManager( new DocumentManager(baseURI) );
	if(inputURI != null)
	    setDocumentInput(inputURI);
	if(xmlInput != null)
	    setInput(xmlInput);
	if(timezone != null)
	    setImplicitTimezone(timezone);
	if(collation != null)
	    setDefaultCollation(collation);
	initGlobal(QName.get("arguments"), appargs);

	if(queryFile == null)
	    interactive();
	else {
	    boolean stdin = queryFile.equals("-");
	    if(queryFile.startsWith("-") && !stdin) {
		options.printHelp(System.err);
		System.exit(1);
	    }
	    Reader input = stdin ?
		(Reader) (new InputStreamReader(System.in)) : new FileReader(queryFile);
	    execute(input, stdin ? "stdin" : queryFile);
	    System.gc();
	}
    }

    void interactive( ) throws IOException {
	if(verbose) { 
	    String APP = "Qizx/open ";
	    System.err.println(APP+getVersion()+" - copyright 2003-2004 Xavier Franc");
	    System.err.println("[interactive mode]");
	}
	BufferedReader rd = new BufferedReader(new InputStreamReader(System.in));
	for(;;) {
	    // Read a query on one line.
	    // if the first line begins with '\' read on several lines until a single '.'
	    System.out.print("XQuery> ");
	    String L = rd.readLine(), input;
	    if(L == null)
		break;
	    if(L.length() == 0) continue;
	    if(L.charAt(0) != '\\')
		input = L;
	    else {
		input = L.substring(1);
		for( ; (L = rd.readLine()) != null && !L.equals("."); )
		    input += '\n' + L;
	    }
	    try {
		// compile and execute:
		log.reset();
		XQuery query = compileQuery( input, "<input>", log );
 		// errors now trigger an exception
		execAndPrint( query );
	    }
	    catch (XQueryException e) {
		stderr.println("*** "+ e.getMessage());
		if(traceExceptions)
		    e.printStackTrace();
	    }
	    catch (Exception e) {
		e.printStackTrace();
	    }
	}
    }

    void execute( Reader input, String uri ) throws IOException, XQueryException  {
	try {
	    log.reset();
	    XQuery query = compileQuery( input, uri, log );
	    log.flush();
	    if(log.getErrorCount() != 0)
		stderr.println(log.getErrorCount() + " parsing/static error(s)");
	    else
		execAndPrint( query );
	}
	catch (Exception e) {
	    System.err.println("*** " + e.getMessage());
	    if(traceExceptions)
		e.printStackTrace();
	}
    }

    void execAndPrint( XQuery query )
	throws IOException, XQueryException, DataModelException {

	for(int g = 0; g < globals.length; g++) {
	    int eq = globals[g].indexOf('=');
	    if(eq < 0) {
		System.err.println("illegal variable initializer: "+globals[g]); return;
	    }
	    initGlobal( toLocalNS( globals[g].substring(0, eq)),
			globals[g].substring(eq + 1));
	}

	if(exprDump && query != null)
	    query.dump(new ExprDump());

	QName RESULTS = QName.get("query-results");
	QName ITEM = QName.get("item"), ITEMTYPE = QName.get("type");

	for(int rep = 0; rep < repeats; rep++)
	  try {
	    if(clearDocuments) {
		String uri = getDocumentManager().getBaseURI();
		setDocumentManager( new DocumentManager(uri) );
	    }
	    long T1, T0 = System.currentTimeMillis();
	    if(!push) {
		// low level execution: 
		Value v = executeQuery( query );
		T1 = System.currentTimeMillis();
		int it = 0;

		nodeDisplay.reset();
		nodeDisplay.definePrefixHints( query.getInScopeNS() );
		if(display && xmlDisplay) {
		    if(wrapDisplay) {
			nodeDisplay.startDocument();
			nodeDisplay.startElement(RESULTS);
		    }
		}

		for( ; v.next(); it++) {
		    Item item = v.asItem();
		    if(display) 
			if(xmlDisplay) {
			    if(wrapDisplay) {
				nodeDisplay.startElement(ITEM);
				nodeDisplay.attribute(ITEMTYPE, v.getType().toString());
			    }
			    if(Type.NODE.accepts(item.getType()))
				nodeDisplay.traverse(v.asNode(), true);
			    else {
				nodeDisplay.atom(v.asString());
				if(!wrapDisplay)
				    nodeDisplay.text(" ");	// some space
			    }
			    if(wrapDisplay)
				nodeDisplay.endElement(ITEM);
			}
			else {	// old style display
			    if(wrapDisplay) 
				System.out.print(it+" ["+v.getType()+"] ");
			    if(Type.NODE.accepts(item.getType())) {
				nodeDisplay.output(v.asNode());
				nodeDisplay.terminate();
			    }
			    else System.out.println(v.asString());
			}
		}

		if(display && xmlDisplay) {
		    if(wrapDisplay) {
			nodeDisplay.endElement(RESULTS);
			nodeDisplay.endDocument();
		    }
		    nodeDisplay.terminate(); 
		}
		if(verbose) System.err.println("-> "+it+" item(s)");
	    }
	    else {
		executeQuery( query, nodeDisplay );
		T1 = System.currentTimeMillis();
	    }
	    long T2 = System.currentTimeMillis();
	    if(verbose)
	      System.err.println("evaluation time: "+ (T1-T0) + " ms, display time: "
				 +(T2-T1) + " ms");
	  }
	 catch (EvalException e) {
	     if(traceExceptions)
		 e.printStackTrace();
	     e.printStack(log, 20);
	     if(e.getCause() != null && !traceExceptions) {
		 System.err.println("  caused by: " + e.getCause());
	     }
 	 }
    }

} // end of class XQuery

