/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.impl;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.GlobalVariable;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.fn.UserFunction;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.dt.SingleItem;
import net.xfra.qizxopen.xquery.dt.SingleNode;
import net.xfra.qizxopen.xquery.dt.SingleInteger;
import net.xfra.qizxopen.xquery.dt.SingleDouble;
import net.xfra.qizxopen.xquery.dt.SingleString;
import net.xfra.qizxopen.xquery.dt.Conversion;
import net.xfra.qizxopen.dm.DocumentManager;
import net.xfra.qizxopen.dm.FONIDocument;
import net.xfra.qizxopen.dm.DataModelException;
import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.time.Duration;
import net.xfra.qizxopen.util.time.DateTimeException;

import java.util.HashMap;
import java.util.Date;
import java.util.TimeZone;
import java.util.Calendar;
import java.text.Collator;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;

/**
 *	Evaluation context for main query or functions. Holds the local variables.
 *	Caution: due to differed evaluation of some expressions (Flowers etc.), the
 *	context can persist even after exit of the related function.
 */
public class DefaultEvalContext implements EvalContext
{
    protected EvalContext upContext;
    protected UserFunction.Call called;
    protected StaticPart staticPart;
    protected Expression point;	  // currently evaluated operator or builtin function
    protected int depth;

    boolean traceExec;

    protected Value[] locals;
    // fast local registers : dont use arrays
    public long   registerInt0, registerInt1, registerInt2, registerInt3;
    public double registerDouble0, registerDouble1, registerDouble2, registerDouble3;
    public String registerString0, registerString1, registerString2, registerString3;
    public Item   registerItem0, registerItem1, registerItem2, registerItem3;
    public Item   registerItem4, registerItem5, registerItem6, registerItem7;


    // staticPart data:
    protected class StaticPart {
	StaticContext context;
	DocumentManager documentManager;
	HashMap globals; 
	HashMap documents;		// already loaded documents
	HashMap properties; 		// miscellaneous objects set by applications
	Value   input;
	Date    currentDate;
	int     implicitTimezone;	// minutes from GMT
	PrintWriter defaultOutput;
	Log log;
	boolean timeout;
	boolean stopped;
	// non null: pause by waiting on this object
	XQueryProcessor.PauseHandler pauseHandler;
    }

    public DefaultEvalContext( StaticContext context, int localSize ) {
	upContext = null;
	locals = localSize > 0 ? new Value[localSize] : null;
	depth = 200;	// max call depth
	staticPart = new StaticPart();
	staticPart.input = null;
	staticPart.context = context;
	staticPart.globals = new HashMap();
	staticPart.documents = new HashMap();
	Calendar cal = Calendar.getInstance();
	staticPart.currentDate = cal.getTime();
	// we are interested in the current APPARENT TZ, the DST is treated by DateTimeBase
	staticPart.implicitTimezone = // ms to minutes
	    cal.get(Calendar.ZONE_OFFSET) / 60000;
    }

    public DefaultEvalContext( int localSize ) {
	locals = localSize > 0 ? new Value[localSize] : null;
    }

    public StaticContext getStaticContext() {
	return staticPart.context;
    }

    public EvalContext getCallerContext() {
	return upContext;
    }

    public EvalContext subContext( UserFunction.Call called )
	throws EvalException {
	if(depth < 1)
	    throw new EvalException("stack overflow");
	DefaultEvalContext ctx = new DefaultEvalContext(called.getLocalSize());
	ctx.upContext = this;
	ctx.called = called;
	ctx.staticPart = staticPart;
	ctx.depth = depth - 1;
	ctx.traceExec = traceExec;
	return ctx;
    }

    public Value error( Expression origin, String error )
	throws EvalException {
	return error(origin, new EvalException(error));
    }

    public Value error( Expression origin, EvalException error )
	throws EvalException {
	point = origin;
	error.setContext(this);
	throw error;
    }

    /**
     *	Displays the context location as a string
     */
    public String displayFrame() {
	if(point != null && point.module == null) {
	    System.err.println("NULL MODULE"); return point.toString()+" at ?";
	}
	if(called != null && called.prototype==null) {
	    System.err.println("NULL prototype"); return point.toString()+" ?";
	}
	return "in " 
	    + (called== null? "main query"
	       : called.prototype.toString(staticPart.context))
	    + " in "
	    + (point == null? "?nowhere?"
	       : point.module.printLocation( point.location ));
    }

    public void  printStack(Log log, int maxDepth) {
	log.info( point.module, point.location, displayFrame() );
	if(upContext != null)
	    if(maxDepth > 0)
		upContext.printStack(log, maxDepth - 1);
	    else log.info("...");
    }

    public final void at( Expression pt ) throws XQueryException {
	if(pt != null) {
	    point = pt;
	    
	    if(traceExec) {
		String descr = null;
		if(pt instanceof Function.Call) 
		    descr = ((Function.Call) pt).prototype.toString(this);
		else
		    descr = pt.toString();
		System.err.println(" at "+pt.location+" "+descr);
	    }
	}
	if( staticPart.stopped ) {
	    if(staticPart.pauseHandler == null)
		error(pt, staticPart.timeout? "time limit reached" : "stopped");
	    else {
		XQueryProcessor.PauseHandler hdlr = staticPart.pauseHandler;
		setPauseHandler(null);	// dont stop again
		hdlr.pauseAt(this);
		synchronized(hdlr) {
		    try {
			hdlr.wait();
		    }
		    catch (InterruptedException e) { }
		}
	    }
	}
    }

    public void setTimeOut(boolean on) {
	staticPart.stopped = on;
	staticPart.timeout = on;
    }

    public void setStopped(boolean on) {
	staticPart.stopped = on;
	staticPart.timeout = false;
    }

    public void setPauseHandler( XQueryProcessor.PauseHandler pause ) {
	staticPart.pauseHandler = pause;
	staticPart.stopped = pause != null;
    }

    public void setExecTrace(boolean on) {
	traceExec = on;
    }

    public Expression getCurrentLocation() {
	return point;
    }

    public Value loadGlobal( GlobalVariable var ) throws XQueryException {
	Value v = (Value) staticPart.globals.get(var);
	if(v == null)
	    error(var, "variable $"+var.name+" has no specified value");
	return v.bornAgain();	// OPTIM?
    }

    public void setGlobal( GlobalVariable var, Value value ) throws XQueryException {
	value = Type.ITEM.star.check(value);	// expand always
	staticPart.globals.put(var, value);
    }

    public Value loadLocal( int address ) throws XQueryException {
	
	switch(address) {
	case INT_REGISTER + 0:
	    return new SingleInteger(registerInt0);
	case INT_REGISTER + 1:
	    return new SingleInteger(registerInt1);
	case INT_REGISTER + 2:
	    return new SingleInteger(registerInt2);
	case INT_REGISTER + 3:
	    return new SingleInteger(registerInt3);
	case DOUBLE_REGISTER + 0:
	    return new SingleDouble(registerDouble0);
	case DOUBLE_REGISTER + 1:
	    return new SingleDouble(registerDouble1);
	case DOUBLE_REGISTER + 2:
	    return new SingleDouble(registerDouble2);
	case DOUBLE_REGISTER + 3:
	    return new SingleDouble(registerDouble3);
	case STRING_REGISTER + 0:
	    return new SingleString(registerString0);
	case STRING_REGISTER + 1:
	    return new SingleString(registerString1);
	case STRING_REGISTER + 2:
	    return new SingleString(registerString2);
	case STRING_REGISTER + 3:
	    return new SingleString(registerString3);
	case ITEM_REGISTER + 0:
	    return new SingleItem(registerItem0);
	case ITEM_REGISTER + 1:
	    return new SingleItem(registerItem1);
	case ITEM_REGISTER + 2:
	    return new SingleItem(registerItem2);
	case ITEM_REGISTER + 3:
	    return new SingleItem(registerItem3);
	case ITEM_REGISTER + 4:
	    return new SingleItem(registerItem4);
	case ITEM_REGISTER + 5:
	    return new SingleItem(registerItem5);
	case ITEM_REGISTER + 6:
	    return new SingleItem(registerItem6);
	case ITEM_REGISTER + 7:
	    return new SingleItem(registerItem7);
	default:
	    // TODO : OPTIM if a var is used only once, dont clone it by bornAgain()
	    
	    return locals[ address - LAST_REGISTER ].bornAgain();
	}
    }

    public long    loadLocalInteger( int address ) throws XQueryException {
	switch(address) {
	case INT_REGISTER + 0:
	    
	    return registerInt0;
	case INT_REGISTER + 1:	    return registerInt1;
	case INT_REGISTER + 2:	    return registerInt2;
	case INT_REGISTER + 3:	    return registerInt3;
	case DOUBLE_REGISTER + 0:  return (long) registerDouble0;
	case DOUBLE_REGISTER + 1:  return (long) registerDouble1;
	case DOUBLE_REGISTER + 2:  return (long) registerDouble2;
	case DOUBLE_REGISTER + 3:  return (long) registerDouble3;
	case STRING_REGISTER + 0:  return Conversion.toInteger(registerString0);
	case STRING_REGISTER + 1:  return Conversion.toInteger(registerString1);
	case STRING_REGISTER + 2:  return Conversion.toInteger(registerString2);
	case STRING_REGISTER + 3:  return Conversion.toInteger(registerString3);
	case ITEM_REGISTER + 0:  return registerItem0.asInteger();
	case ITEM_REGISTER + 1:  return registerItem1.asInteger();
	case ITEM_REGISTER + 2:  return registerItem2.asInteger();
	case ITEM_REGISTER + 3:  return registerItem3.asInteger();
	case ITEM_REGISTER + 4:  return registerItem4.asInteger();
	case ITEM_REGISTER + 5:  return registerItem5.asInteger();
	case ITEM_REGISTER + 6:  return registerItem6.asInteger();
	case ITEM_REGISTER + 7:  return registerItem7.asInteger();
	default:
	    Value v = locals[ address - LAST_REGISTER ].bornAgain();
	    if( !v.next() )
		throw new TypeException(Type.ERR_EMPTY_UNEXPECTED);
	    long item = v.asInteger();
	    if( v.next() )
		throw new TypeException(Type.ERR_TOO_MANY);
	    return item;
	}
    }

    public double  loadLocalDouble( int address ) throws XQueryException {
	switch(address) {
	case INT_REGISTER + 0:  return registerInt0;
	case INT_REGISTER + 1:  return registerInt1;
	case INT_REGISTER + 2:  return registerInt2;
	case INT_REGISTER + 3:  return registerInt3;

	case DOUBLE_REGISTER + 0:  return registerDouble0;
	case DOUBLE_REGISTER + 1:  return registerDouble1;
	case DOUBLE_REGISTER + 2:  return registerDouble2;
	case DOUBLE_REGISTER + 3:  return registerDouble3;

	case STRING_REGISTER + 0:  return Conversion.toDouble(registerString0);
	case STRING_REGISTER + 1:  return Conversion.toDouble(registerString1);
	case STRING_REGISTER + 2:  return Conversion.toDouble(registerString2);
	case STRING_REGISTER + 3:  return Conversion.toDouble(registerString3);

	case ITEM_REGISTER + 0:  return registerItem0.asDouble();
	case ITEM_REGISTER + 1:  return registerItem1.asDouble();
	case ITEM_REGISTER + 2:  return registerItem2.asDouble();
	case ITEM_REGISTER + 3:  return registerItem3.asDouble();
	case ITEM_REGISTER + 4:  return registerItem4.asDouble();
	case ITEM_REGISTER + 5:  return registerItem5.asDouble();
	case ITEM_REGISTER + 6:  return registerItem6.asDouble();
	case ITEM_REGISTER + 7:  return registerItem7.asDouble();
	default:
	    Value v = locals[ address - LAST_REGISTER ].bornAgain();
	    if( !v.next() )
		//throw new TypeException(Type.ERR_EMPTY_UNEXPECTED);
		throw EmptyException.allowed();
	    double item = v.asDouble();
	    if( v.next() )
		throw new TypeException(Type.ERR_TOO_MANY);
	    return item;
	}
    }

    public String  loadLocalString( int address ) throws XQueryException {
	switch(address) {
	case INT_REGISTER + 0:  return Conversion.toString(registerInt0);
	case INT_REGISTER + 1:  return Conversion.toString(registerInt1);
	case INT_REGISTER + 2:  return Conversion.toString(registerInt2);
	case INT_REGISTER + 3:  return Conversion.toString(registerInt3);

	case DOUBLE_REGISTER + 0:  return Conversion.toString(registerDouble0);
	case DOUBLE_REGISTER + 1:  return Conversion.toString(registerDouble1);
	case DOUBLE_REGISTER + 2:  return Conversion.toString(registerDouble2);
	case DOUBLE_REGISTER + 3:  return Conversion.toString(registerDouble3);

	case STRING_REGISTER + 0:  return registerString0;
	case STRING_REGISTER + 1:  return registerString1;
	case STRING_REGISTER + 2:  return registerString2;
	case STRING_REGISTER + 3:  return registerString3;

	case ITEM_REGISTER + 0:  return registerItem0.asString();
	case ITEM_REGISTER + 1:  return registerItem1.asString();
	case ITEM_REGISTER + 2:  return registerItem2.asString();
	case ITEM_REGISTER + 3:  return registerItem3.asString();
	case ITEM_REGISTER + 4:  return registerItem4.asString();
	case ITEM_REGISTER + 5:  return registerItem5.asString();
	case ITEM_REGISTER + 6:  return registerItem6.asString();
	case ITEM_REGISTER + 7:  return registerItem7.asString();
	default:
	    Value v = locals[ address - LAST_REGISTER ].bornAgain();
	    if( !v.next() )
		throw new TypeException(Type.ERR_EMPTY_UNEXPECTED);
	    String item = v.asString();
	    if( v.next() )
		throw new TypeException(Type.ERR_TOO_MANY);
	    return item;
	}
    }

    public Item    loadLocalItem( int address ) throws XQueryException {
	switch(address) {
	case INT_REGISTER + 0:  return new SingleInteger(registerInt0);
	case INT_REGISTER + 1:  return new SingleInteger(registerInt1);
	case INT_REGISTER + 2:  return new SingleInteger(registerInt2);
	case INT_REGISTER + 3:  return new SingleInteger(registerInt3);

	case DOUBLE_REGISTER + 0:  return new SingleDouble(registerDouble0);
	case DOUBLE_REGISTER + 1:  return new SingleDouble(registerDouble1);
	case DOUBLE_REGISTER + 2:  return new SingleDouble(registerDouble2);
	case DOUBLE_REGISTER + 3:  return new SingleDouble(registerDouble3);

	case STRING_REGISTER + 0:  return new SingleString(registerString0);
	case STRING_REGISTER + 1:  return new SingleString(registerString1);
	case STRING_REGISTER + 2:  return new SingleString(registerString2);
	case STRING_REGISTER + 3:  return new SingleString(registerString3);

	case ITEM_REGISTER + 0:  return registerItem0;
	case ITEM_REGISTER + 1:  return registerItem1;
	case ITEM_REGISTER + 2:  return registerItem2;
	case ITEM_REGISTER + 3:  return registerItem3;
	case ITEM_REGISTER + 4:  return registerItem4;
	case ITEM_REGISTER + 5:  return registerItem5;
	case ITEM_REGISTER + 6:  return registerItem6;
	case ITEM_REGISTER + 7:  return registerItem7;
	default:
	    Value v = locals[ address - LAST_REGISTER ].bornAgain();
	    if( !v.next() )
		throw new TypeException(Type.ERR_EMPTY_UNEXPECTED);
	    Item item = v.asItem();
	    if( v.next() )
		throw new TypeException(Type.ERR_TOO_MANY);
	    return item;
	}
    }

    public void storeLocal( int address, Expression expr, Type dynamicType,
			    Focus focus, EvalContext calling )
	throws XQueryException {
	
	switch(address) {
	case INT_REGISTER + 0:
	    registerInt0 = expr.evalAsInteger(focus, calling); break;
	case INT_REGISTER + 1:
	    registerInt1 = expr.evalAsInteger(focus, calling); break;
	case INT_REGISTER + 2:
	    registerInt2 = expr.evalAsInteger(focus, calling); break;
	case INT_REGISTER + 3:
	    registerInt3 = expr.evalAsInteger(focus, calling); break;
	case DOUBLE_REGISTER + 0:
	    registerDouble0 = expr.evalAsDouble(focus, calling); break;
	case DOUBLE_REGISTER + 1:
	    registerDouble1 = expr.evalAsDouble(focus, calling); break;
	case DOUBLE_REGISTER + 2:
	    registerDouble2 = expr.evalAsDouble(focus, calling); break;
	case DOUBLE_REGISTER + 3:
	    registerDouble3 = expr.evalAsDouble(focus, calling); break;
	case STRING_REGISTER + 0:
	    registerString0 = expr.evalAsString(focus, calling); break;
	case STRING_REGISTER + 1:
	    registerString1 = expr.evalAsString(focus, calling); break;
	case STRING_REGISTER + 2:
	    registerString2 = expr.evalAsString(focus, calling); break;
	case STRING_REGISTER + 3:
	    registerString3 = expr.evalAsString(focus, calling); break;
	case ITEM_REGISTER + 0:
	    registerItem0 = expr.evalAsItem(focus, calling);
	    
	    break;
	case ITEM_REGISTER + 1:
	    registerItem1 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 2:
	    registerItem2 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 3:
	    registerItem3 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 4:
	    registerItem4 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 5:
	    registerItem5 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 6:
	    registerItem6 = expr.evalAsItem(focus, calling); break;
	case ITEM_REGISTER + 7:
	    registerItem7 = expr.evalAsItem(focus, calling); break;
	default:
	    try {
		Value value = expr.eval(focus, calling);
		

		// force checking and expansion:
// 		if(dynamicType == null)
// 		    dynamicType = Type.ITEM.star;
// 		value = dynamicType.check( value );

		if(dynamicType != null)
		    // might expand a sequence into an ArraySequence:
		    value = dynamicType.check( value );

		locals[ address - LAST_REGISTER ] = value;
	    }
	    catch(TypeException err) {
		error(expr, err);
	    }
	    break;
	}
    }

    public void  storeLocal( int address, Value value, boolean current )
	throws XQueryException {
	
	// check non empty
	if(address < LAST_REGISTER && !current && !value.next())
	    throw new TypeException(Type.ERR_EMPTY_UNEXPECTED);
	switch(address) {
	case INT_REGISTER + 0:
	    registerInt0 = value.asInteger(); break;
	case INT_REGISTER + 1:
	    registerInt1 = value.asInteger(); break;
	case INT_REGISTER + 2:
	    registerInt2 = value.asInteger(); break;
	case INT_REGISTER + 3:
	    registerInt3 = value.asInteger(); break;
	case DOUBLE_REGISTER + 0:
	    registerDouble0 = value.asDouble(); break;
	case DOUBLE_REGISTER + 1:
	    registerDouble1 = value.asDouble(); break;
	case DOUBLE_REGISTER + 2:
	    registerDouble2 = value.asDouble(); break;
	case DOUBLE_REGISTER + 3:
	    registerDouble3 = value.asDouble(); break;
	case STRING_REGISTER + 0:
	    registerString0 = value.asString(); break;
	case STRING_REGISTER + 1:
	    registerString1 = value.asString(); break;
	case STRING_REGISTER + 2:
	    registerString2 = value.asString(); break;
	case STRING_REGISTER + 3:
	    registerString3 = value.asString(); break;
	case ITEM_REGISTER + 0:
	    registerItem0 = value.asItem();
	    
	    break;
	case ITEM_REGISTER + 1:
	    registerItem1 = value.asItem(); break;
	case ITEM_REGISTER + 2:
	    registerItem2 = value.asItem(); break;
	case ITEM_REGISTER + 3:
	    registerItem3 = value.asItem(); break;
	case ITEM_REGISTER + 4:
	    registerItem4 = value.asItem(); break;
	case ITEM_REGISTER + 5:
	    registerItem5 = value.asItem(); break;
	case ITEM_REGISTER + 6:
	    registerItem6 = value.asItem(); break;
	case ITEM_REGISTER + 7:
	    registerItem7 = value.asItem(); break;
	default:
	    //if(address == LAST_REGISTER) System.err.println(" store Local'' : "+value);
	    locals[ address - LAST_REGISTER ] = current ? new SingleItem(value) : value;
	    break;
	}
    }

    public void  storeLocalInteger( int address, long value ) {
	switch(address) {
	case INT_REGISTER + 0:
	    registerInt0 = value; break;
	case INT_REGISTER + 1:
	    registerInt1 = value; break;
	case INT_REGISTER + 2:
	    registerInt2 = value; break;
	case INT_REGISTER + 3:
	    registerInt3 = value; break;
	default:	// if too many int variables
	    locals[ address - LAST_REGISTER ] = new SingleInteger(value);
	    break;
	}
    }

    public void setInput( Value input ) {
	staticPart.input = input;
    }

    public Value getInput() {
	return staticPart.input == null ? null : staticPart.input.bornAgain();
    }

    public Object getProperty(String name) {
	return staticPart.properties == null? null : staticPart.properties.get(name);
    }

    public void setProperties(HashMap properties) {
	staticPart.properties = properties;
    }

    public void setDocumentManager( DocumentManager documentManager ) {
	staticPart.documentManager = documentManager;
    }

    public DocumentManager getDocumentManager() {
	return staticPart.documentManager;
    }

    public Value getDocument( String uri ) throws XQueryException {
	Node docRoot = (Node) staticPart.documents.get(uri);
	if(docRoot == null) {
	    try {
		FONIDocument doc = staticPart.documentManager.findDocument(uri);
		docRoot = new FONIDataModel(doc).getDocumentNode();
		staticPart.documents.put(uri, docRoot);
	    }	
	    catch(DataModelException dme) {
		// try to get the cause:
		Throwable t = dme.getCause();
		if(t instanceof Exception)
		    throw new EvalException(dme.getMessage(), (Exception) t);
		else throw new EvalException(dme.getMessage(), dme);
	    }
	}
	return new SingleNode(docRoot);
    }

    public Collator getCollator( String uri ) throws XQueryException {
	return staticPart.context.getCollator(uri);
    }

    public Date     getCurrentDate() {
	return staticPart.currentDate;
    }

    public int      getImplicitTimezone() {
	return staticPart.implicitTimezone;
    }

    public void  setImplicitTimezone( int minutes ) {
	staticPart.implicitTimezone = minutes;
    }

    public void  setImplicitTimezone( String duration ) throws DateTimeException {
	staticPart.implicitTimezone = (int)(Duration.parseDuration(duration).getSeconds() / 60);
    }

    public PrintWriter getDefaultOutput() {
	return staticPart.defaultOutput;
    }
    public void setDefaultOutput(PrintWriter output) {
	staticPart.defaultOutput = output;
    }

    public Log getLog() {
	return staticPart.log != null ? staticPart.log : staticPart.context.getLog();
    }

    public void setLog(Log log) {
        staticPart.log = log;
    }
} // end of class DefaultEvalContext

