/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.util.time.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.dt.MomentValue;
import net.xfra.qizxopen.xquery.dt.SingleMoment;
import net.xfra.qizxopen.xquery.dt.SingleDecimal;
import net.xfra.qizxopen.xquery.dt.Conversion;
import net.xfra.qizxopen.xquery.fn.Prototype;

/**
 *  Implementation of operator '+'. 
 */
public class PlusOp extends NumericOp {

    public PlusOp( Expression expr1, Expression expr2 ) {
        super( expr1, expr2 );
    }

    static Prototype[] protos = { 
        Prototype.op("+", Type.DOUBLE.opt, ExecD.class).hidden()
	  .arg("op1", Type.DOUBLE.opt).arg("op2", Type.UNTYPED_ATOMIC.opt),
        Prototype.op("+", Type.DOUBLE.opt, ExecD.class).hidden()
	  .arg("op1", Type.UNTYPED_ATOMIC.opt).arg("op2", Type.DOUBLE.opt),
        Prototype.op("+", Type.INTEGER.opt, ExecI.class)
	  .arg("op1", Type.INTEGER.opt).arg("op2", Type.INTEGER.opt),
        Prototype.op("+", Type.DECIMAL.opt, ExecDec.class)
	  .arg("op1", Type.DECIMAL.opt).arg("op2", Type.DECIMAL.opt),
        Prototype.op("+", Type.FLOAT.opt, ExecF.class)
	  .arg("op1", Type.FLOAT.opt).arg("op2", Type.FLOAT.opt),
        Prototype.op("+", Type.DOUBLE.opt, ExecD.class)
	  .arg("op1", Type.DOUBLE.opt).arg("op2", Type.DOUBLE.opt),

        Prototype.op("+", Type.DATE_TIME, ExecTime.class)
	  .arg("d", Type.DATE_TIME).arg("seconds", Type.DOUBLE),
        Prototype.op("+", Type.TIME, ExecTime.class)
	  .arg("t", Type.TIME).arg("seconds", Type.DOUBLE),
        Prototype.op("+", Type.DATE, ExecDate.class)
	  .arg("d", Type.DATE).arg("days", Type.INTEGER)
    };
    public Prototype[] getProtos() { return protos; }

    public void dump( ExprDump d ) {
	d.header( this, "Op +" );
        d.display("expr1", operands[0]);
        d.display("expr2", operands[1]);
    }

    public static class ExecI extends Function.OptIntegerCall {
	public long evalAsOptInteger( Focus focus, EvalContext context )
	    throws XQueryException {
	    long e1 = args[0].evalAsOptInteger(focus, context);
	    long e2 = args[1].evalAsOptInteger(focus, context);
	    context.at(this);
	    if(!Conversion.isIntegerRange( e1 + (double) e2))
		context.error(this, "integer overflow");
	    
	    return e1 + e2;
	}
    }

    public static class ExecF extends Function.OptFloatCall {
	public float evalAsOptFloat( Focus focus, EvalContext context )
	    throws XQueryException {
	    float e1 = args[0].evalAsOptFloat(focus, context);
	    float e2 = args[1].evalAsOptFloat(focus, context);
	    context.at(this);
	    return e1 + e2;
	}
    }

    public static class ExecD extends Function.OptDoubleCall {
	public double evalAsOptDouble( Focus focus, EvalContext context )
	    throws XQueryException {
	    double e1 = args[0].evalAsOptDouble(focus, context);
	    double e2 = args[1].evalAsOptDouble(focus, context);
	    context.at(this);
	    return e1 + e2;
	}
    }

    public static class ExecDec extends Function.Call {
	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    Value e1 = args[0].eval(focus, context);
	    Value e2 = args[1].eval(focus, context);
	    if(!e1.next() || !e2.next())
		return Value.empty;
	    context.at(this);
	    return new SingleDecimal( e1.asDecimal().add( e2.asDecimal() ));
	}
    }

    public static class ExecTime extends Function.Call {

        public Value eval( Focus focus, EvalContext context) throws XQueryException {
            context.at(this);
	    Item v = args[0].evalAsOptItem(focus, context);
	    if(v == null)
		return Value.empty;
	    double delta = args[1].evalAsDouble(focus, context);
	    DateTimeBase time = ((MomentValue) v).getValue();
	    DateTimeBase res = time.addSeconds(delta);
	    try {
		if(v.getType() == Type.TIME)
		    res = new Time(res);	// cast
		return new SingleMoment( res, v.getType() );
	    }
	    catch (DateTimeException e) {
		context.error(this, e.getMessage()); return null;
	    }
        }
    }

    public static class ExecDate extends Function.Call {

        public Value eval( Focus focus, EvalContext context) throws XQueryException {
            context.at(this);
	    Item v = args[0].evalAsOptItem(focus, context);
	    if(v == null)
		return Value.empty;
	    long days = args[1].evalAsInteger(focus, context);
	    DateTimeBase date = ((MomentValue) v).getValue();
	    try {
		return new SingleMoment( new Date(date.addSeconds(86400 * days)), Type.DATE );
	    }
	    catch (DateTimeException e) {
		context.error(this, e.getMessage()); return null;
	    }
        }
    }
}
