package hass

import (
	"encoding/hex"
	"strings"
)

const (
	vendorApple   = "Apple"
	vendorGoogle  = "Google"
	vendorSamsung = "Samsung"
)

// VendorByMAC performs a best-effort match of the given MAC address
// to known vendors/manufacturers.
func VendorByMAC(mac string) string {
	h, err := hex.DecodeString(strings.Replace(mac, ":", "", 5))
	if err != nil || len(h) < 3 {
		return ""
	}
	return vendorMACs[[3]byte{h[0], h[1], h[2]}]
}

var vendorMACs = map[[3]byte]string{
	// Apple
	{0x00, 0x03, 0x93}: vendorApple,
	{0x00, 0x05, 0x02}: vendorApple,
	{0x00, 0x0A, 0x27}: vendorApple,
	{0x00, 0x0A, 0x95}: vendorApple,
	{0x00, 0x0D, 0x93}: vendorApple,
	{0x00, 0x10, 0xFA}: vendorApple,
	{0x00, 0x11, 0x24}: vendorApple,
	{0x00, 0x14, 0x51}: vendorApple,
	{0x00, 0x16, 0xCB}: vendorApple,
	{0x00, 0x17, 0xF2}: vendorApple,
	{0x00, 0x19, 0xE3}: vendorApple,
	{0x00, 0x1B, 0x63}: vendorApple,
	{0x00, 0x1C, 0xB3}: vendorApple,
	{0x00, 0x1D, 0x4F}: vendorApple,
	{0x00, 0x1E, 0x52}: vendorApple,
	{0x00, 0x1E, 0xC2}: vendorApple,
	{0x00, 0x1F, 0x5B}: vendorApple,
	{0x00, 0x1F, 0xF3}: vendorApple,
	{0x00, 0x21, 0xE9}: vendorApple,
	{0x00, 0x22, 0x41}: vendorApple,
	{0x00, 0x23, 0x12}: vendorApple,
	{0x00, 0x23, 0x32}: vendorApple,
	{0x00, 0x23, 0x6C}: vendorApple,
	{0x00, 0x23, 0xDF}: vendorApple,
	{0x00, 0x24, 0x36}: vendorApple,
	{0x00, 0x25, 0x00}: vendorApple,
	{0x00, 0x25, 0x4B}: vendorApple,
	{0x00, 0x25, 0xBC}: vendorApple,
	{0x00, 0x26, 0x08}: vendorApple,
	{0x00, 0x26, 0x4A}: vendorApple,
	{0x00, 0x26, 0xB0}: vendorApple,
	{0x00, 0x26, 0xBB}: vendorApple,
	{0x00, 0x30, 0x65}: vendorApple,
	{0x00, 0x3E, 0xE1}: vendorApple,
	{0x00, 0x50, 0xE4}: vendorApple,
	{0x00, 0x56, 0xCD}: vendorApple,
	{0x00, 0x5B, 0x94}: vendorApple,
	{0x00, 0x61, 0x71}: vendorApple,
	{0x00, 0x6D, 0x52}: vendorApple,
	{0x00, 0x7D, 0x60}: vendorApple,
	{0x00, 0x88, 0x65}: vendorApple,
	{0x00, 0x8A, 0x76}: vendorApple,
	{0x00, 0xA0, 0x40}: vendorApple,
	{0x00, 0xB3, 0x62}: vendorApple,
	{0x00, 0xC6, 0x10}: vendorApple,
	{0x00, 0xCD, 0xFE}: vendorApple,
	{0x00, 0xDB, 0x70}: vendorApple,
	{0x00, 0xF3, 0x9F}: vendorApple,
	{0x00, 0xF4, 0xB9}: vendorApple,
	{0x00, 0xF7, 0x6F}: vendorApple,
	{0x04, 0x0C, 0xCE}: vendorApple,
	{0x04, 0x15, 0x52}: vendorApple,
	{0x04, 0x1E, 0x64}: vendorApple,
	{0x04, 0x26, 0x65}: vendorApple,
	{0x04, 0x48, 0x9A}: vendorApple,
	{0x04, 0x4B, 0xED}: vendorApple,
	{0x04, 0x52, 0xF3}: vendorApple,
	{0x04, 0x54, 0x53}: vendorApple,
	{0x04, 0x68, 0x65}: vendorApple,
	{0x04, 0x69, 0xF8}: vendorApple,
	{0x04, 0x72, 0x95}: vendorApple,
	{0x04, 0x99, 0xB9}: vendorApple,
	{0x04, 0xD3, 0xCF}: vendorApple,
	{0x04, 0xDB, 0x56}: vendorApple,
	{0x04, 0xE5, 0x36}: vendorApple,
	{0x04, 0xF1, 0x3E}: vendorApple,
	{0x04, 0xF7, 0xE4}: vendorApple,
	{0x08, 0x00, 0x07}: vendorApple,
	{0x08, 0x2C, 0xB6}: vendorApple,
	{0x08, 0x66, 0x98}: vendorApple,
	{0x08, 0x6D, 0x41}: vendorApple,
	{0x08, 0x70, 0x45}: vendorApple,
	{0x08, 0x74, 0x02}: vendorApple,
	{0x08, 0x87, 0xC7}: vendorApple,
	{0x08, 0x8E, 0xDC}: vendorApple,
	{0x08, 0xC7, 0x29}: vendorApple,
	{0x08, 0xE6, 0x89}: vendorApple,
	{0x08, 0xF4, 0xAB}: vendorApple,
	{0x08, 0xF6, 0x9C}: vendorApple,
	{0x08, 0xF8, 0xBC}: vendorApple,
	{0x08, 0xFF, 0x44}: vendorApple,
	{0x0C, 0x15, 0x39}: vendorApple,
	{0x0C, 0x19, 0xF8}: vendorApple,
	{0x0C, 0x30, 0x21}: vendorApple,
	{0x0C, 0x3B, 0x50}: vendorApple,
	{0x0C, 0x3E, 0x9F}: vendorApple,
	{0x0C, 0x4D, 0xE9}: vendorApple,
	{0x0C, 0x51, 0x01}: vendorApple,
	{0x0C, 0x74, 0xC2}: vendorApple,
	{0x0C, 0x77, 0x1A}: vendorApple,
	{0x0C, 0xBC, 0x9F}: vendorApple,
	{0x0C, 0xD7, 0x46}: vendorApple,
	{0x0C, 0xE4, 0x41}: vendorApple,
	{0x10, 0x00, 0x20}: vendorApple,
	{0x10, 0x1C, 0x0C}: vendorApple,
	{0x10, 0x29, 0x59}: vendorApple,
	{0x10, 0x30, 0x25}: vendorApple,
	{0x10, 0x40, 0xF3}: vendorApple,
	{0x10, 0x41, 0x7F}: vendorApple,
	{0x10, 0x93, 0xE9}: vendorApple,
	{0x10, 0x94, 0xBB}: vendorApple,
	{0x10, 0x9A, 0xDD}: vendorApple,
	{0x10, 0xCE, 0xE9}: vendorApple,
	{0x10, 0xDD, 0xB1}: vendorApple,
	{0x14, 0x10, 0x9F}: vendorApple,
	{0x14, 0x20, 0x5E}: vendorApple,
	{0x14, 0x5A, 0x05}: vendorApple,
	{0x14, 0x60, 0xCB}: vendorApple,
	{0x14, 0x7D, 0xDA}: vendorApple,
	{0x14, 0x87, 0x6A}: vendorApple,
	{0x14, 0x88, 0xE6}: vendorApple,
	{0x14, 0x8F, 0xC6}: vendorApple,
	{0x14, 0x95, 0xCE}: vendorApple,
	{0x14, 0x98, 0x77}: vendorApple,
	{0x14, 0x99, 0xE2}: vendorApple,
	{0x14, 0x9D, 0x99}: vendorApple,
	{0x14, 0xBD, 0x61}: vendorApple,
	{0x14, 0xC2, 0x13}: vendorApple,
	{0x14, 0xC8, 0x8B}: vendorApple,
	{0x14, 0xD0, 0x0D}: vendorApple,
	{0x14, 0xD1, 0x9E}: vendorApple,
	{0x18, 0x20, 0x32}: vendorApple,
	{0x18, 0x34, 0x51}: vendorApple,
	{0x18, 0x3E, 0xEF}: vendorApple,
	{0x18, 0x55, 0xE3}: vendorApple,
	{0x18, 0x56, 0xC3}: vendorApple,
	{0x18, 0x65, 0x90}: vendorApple,
	{0x18, 0x7E, 0xB9}: vendorApple,
	{0x18, 0x81, 0x0E}: vendorApple,
	{0x18, 0x9E, 0xFC}: vendorApple,
	{0x18, 0xAF, 0x61}: vendorApple,
	{0x18, 0xAF, 0x8F}: vendorApple,
	{0x18, 0xE7, 0xF4}: vendorApple,
	{0x18, 0xEE, 0x69}: vendorApple,
	{0x18, 0xF1, 0xD8}: vendorApple,
	{0x18, 0xF6, 0x43}: vendorApple,
	{0x1C, 0x1A, 0xC0}: vendorApple,
	{0x1C, 0x36, 0xBB}: vendorApple,
	{0x1C, 0x5C, 0xF2}: vendorApple,
	{0x1C, 0x91, 0x48}: vendorApple,
	{0x1C, 0x91, 0x80}: vendorApple,
	{0x1C, 0x9E, 0x46}: vendorApple,
	{0x1C, 0xAB, 0xA7}: vendorApple,
	{0x1C, 0xB3, 0xC9}: vendorApple,
	{0x1C, 0xE6, 0x2B}: vendorApple,
	{0x20, 0x32, 0xC6}: vendorApple,
	{0x20, 0x37, 0xA5}: vendorApple,
	{0x20, 0x3C, 0xAE}: vendorApple,
	{0x20, 0x69, 0x80}: vendorApple,
	{0x20, 0x76, 0x8F}: vendorApple,
	{0x20, 0x78, 0xF0}: vendorApple,
	{0x20, 0x7D, 0x74}: vendorApple,
	{0x20, 0x9B, 0xCD}: vendorApple,
	{0x20, 0xA2, 0xE4}: vendorApple,
	{0x20, 0xAB, 0x37}: vendorApple,
	{0x20, 0xC9, 0xD0}: vendorApple,
	{0x20, 0xE2, 0xA8}: vendorApple,
	{0x20, 0xE8, 0x74}: vendorApple,
	{0x20, 0xEE, 0x28}: vendorApple,
	{0x24, 0x1B, 0x7A}: vendorApple,
	{0x24, 0x1E, 0xEB}: vendorApple,
	{0x24, 0x24, 0x0E}: vendorApple,
	{0x24, 0x5B, 0xA7}: vendorApple,
	{0x24, 0x5E, 0x48}: vendorApple,
	{0x24, 0xA0, 0x74}: vendorApple,
	{0x24, 0xA2, 0xE1}: vendorApple,
	{0x24, 0xAB, 0x81}: vendorApple,
	{0x24, 0xD0, 0xDF}: vendorApple,
	{0x24, 0xE3, 0x14}: vendorApple,
	{0x24, 0xF0, 0x94}: vendorApple,
	{0x24, 0xF6, 0x77}: vendorApple,
	{0x28, 0x0B, 0x5C}: vendorApple,
	{0x28, 0x37, 0x37}: vendorApple,
	{0x28, 0x5A, 0xEB}: vendorApple,
	{0x28, 0x6A, 0xB8}: vendorApple,
	{0x28, 0x6A, 0xBA}: vendorApple,
	{0x28, 0x77, 0xF1}: vendorApple,
	{0x28, 0xA0, 0x2B}: vendorApple,
	{0x28, 0xC5, 0x38}: vendorApple,
	{0x28, 0xC7, 0x09}: vendorApple,
	{0x28, 0xCF, 0xDA}: vendorApple,
	{0x28, 0xCF, 0xE9}: vendorApple,
	{0x28, 0xE0, 0x2C}: vendorApple,
	{0x28, 0xE1, 0x4C}: vendorApple,
	{0x28, 0xE7, 0xCF}: vendorApple,
	{0x28, 0xEC, 0x95}: vendorApple,
	{0x28, 0xED, 0x6A}: vendorApple,
	{0x28, 0xF0, 0x33}: vendorApple,
	{0x28, 0xF0, 0x76}: vendorApple,
	{0x28, 0xFF, 0x3C}: vendorApple,
	{0x2C, 0x1F, 0x23}: vendorApple,
	{0x2C, 0x20, 0x0B}: vendorApple,
	{0x2C, 0x33, 0x61}: vendorApple,
	{0x2C, 0x61, 0xF6}: vendorApple,
	{0x2C, 0xB4, 0x3A}: vendorApple,
	{0x2C, 0xBC, 0x87}: vendorApple,
	{0x2C, 0xBE, 0x08}: vendorApple,
	{0x2C, 0xF0, 0xA2}: vendorApple,
	{0x2C, 0xF0, 0xEE}: vendorApple,
	{0x30, 0x10, 0xE4}: vendorApple,
	{0x30, 0x35, 0xAD}: vendorApple,
	{0x30, 0x57, 0x14}: vendorApple,
	{0x30, 0x63, 0x6B}: vendorApple,
	{0x30, 0x90, 0x48}: vendorApple,
	{0x30, 0x90, 0xAB}: vendorApple,
	{0x30, 0xD9, 0xD9}: vendorApple,
	{0x30, 0xF7, 0xC5}: vendorApple,
	{0x34, 0x08, 0xBC}: vendorApple,
	{0x34, 0x12, 0x98}: vendorApple,
	{0x34, 0x15, 0x9E}: vendorApple,
	{0x34, 0x31, 0x8F}: vendorApple,
	{0x34, 0x36, 0x3B}: vendorApple,
	{0x34, 0x42, 0x62}: vendorApple,
	{0x34, 0x51, 0xC9}: vendorApple,
	{0x34, 0x7C, 0x25}: vendorApple,
	{0x34, 0xA3, 0x95}: vendorApple,
	{0x34, 0xA8, 0xEB}: vendorApple,
	{0x34, 0xAB, 0x37}: vendorApple,
	{0x34, 0xC0, 0x59}: vendorApple,
	{0x34, 0xE2, 0xFD}: vendorApple,
	{0x34, 0xFD, 0x6A}: vendorApple,
	{0x38, 0x0F, 0x4A}: vendorApple,
	{0x38, 0x48, 0x4C}: vendorApple,
	{0x38, 0x53, 0x9C}: vendorApple,
	{0x38, 0x65, 0xB2}: vendorApple,
	{0x38, 0x66, 0xF0}: vendorApple,
	{0x38, 0x71, 0xDE}: vendorApple,
	{0x38, 0x89, 0x2C}: vendorApple,
	{0x38, 0xB5, 0x4D}: vendorApple,
	{0x38, 0xC9, 0x86}: vendorApple,
	{0x38, 0xCA, 0xDA}: vendorApple,
	{0x38, 0xEC, 0x0D}: vendorApple,
	{0x38, 0xF9, 0xD3}: vendorApple,
	{0x3C, 0x06, 0x30}: vendorApple,
	{0x3C, 0x07, 0x54}: vendorApple,
	{0x3C, 0x15, 0xC2}: vendorApple,
	{0x3C, 0x22, 0xFB}: vendorApple,
	{0x3C, 0x2E, 0xF9}: vendorApple,
	{0x3C, 0x2E, 0xFF}: vendorApple,
	{0x3C, 0x4D, 0xBE}: vendorApple,
	{0x3C, 0x7D, 0x0A}: vendorApple,
	{0x3C, 0xA6, 0xF6}: vendorApple,
	{0x3C, 0xAB, 0x8E}: vendorApple,
	{0x3C, 0xBF, 0x60}: vendorApple,
	{0x3C, 0xCD, 0x36}: vendorApple,
	{0x3C, 0xD0, 0xF8}: vendorApple,
	{0x3C, 0xE0, 0x72}: vendorApple,
	{0x40, 0x26, 0x19}: vendorApple,
	{0x40, 0x30, 0x04}: vendorApple,
	{0x40, 0x33, 0x1A}: vendorApple,
	{0x40, 0x3C, 0xFC}: vendorApple,
	{0x40, 0x4D, 0x7F}: vendorApple,
	{0x40, 0x6C, 0x8F}: vendorApple,
	{0x40, 0x70, 0xF5}: vendorApple,
	{0x40, 0x83, 0x1D}: vendorApple,
	{0x40, 0x98, 0xAD}: vendorApple,
	{0x40, 0x9C, 0x28}: vendorApple,
	{0x40, 0xA6, 0xD9}: vendorApple,
	{0x40, 0xB3, 0x95}: vendorApple,
	{0x40, 0xBC, 0x60}: vendorApple,
	{0x40, 0xC7, 0x11}: vendorApple,
	{0x40, 0xCB, 0xC0}: vendorApple,
	{0x40, 0xD3, 0x2D}: vendorApple,
	{0x40, 0xE6, 0x4B}: vendorApple,
	{0x40, 0xF9, 0x46}: vendorApple,
	{0x44, 0x00, 0x10}: vendorApple,
	{0x44, 0x18, 0xFD}: vendorApple,
	{0x44, 0x2A, 0x60}: vendorApple,
	{0x44, 0x35, 0x83}: vendorApple,
	{0x44, 0x4A, 0xDB}: vendorApple,
	{0x44, 0x4C, 0x0C}: vendorApple,
	{0x44, 0x90, 0xBB}: vendorApple,
	{0x44, 0xA8, 0xFC}: vendorApple,
	{0x44, 0xC6, 0x5D}: vendorApple,
	{0x44, 0xD8, 0x84}: vendorApple,
	{0x44, 0xE6, 0x6E}: vendorApple,
	{0x44, 0xF0, 0x9E}: vendorApple,
	{0x44, 0xF2, 0x1B}: vendorApple,
	{0x44, 0xFB, 0x42}: vendorApple,
	{0x48, 0x26, 0x2C}: vendorApple,
	{0x48, 0x3B, 0x38}: vendorApple,
	{0x48, 0x43, 0x7C}: vendorApple,
	{0x48, 0x4B, 0xAA}: vendorApple,
	{0x48, 0x60, 0xBC}: vendorApple,
	{0x48, 0x74, 0x6E}: vendorApple,
	{0x48, 0xA1, 0x95}: vendorApple,
	{0x48, 0xA9, 0x1C}: vendorApple,
	{0x48, 0xB8, 0xA3}: vendorApple,
	{0x48, 0xBF, 0x6B}: vendorApple,
	{0x48, 0xD7, 0x05}: vendorApple,
	{0x48, 0xE9, 0xF1}: vendorApple,
	{0x4C, 0x20, 0xB8}: vendorApple,
	{0x4C, 0x32, 0x75}: vendorApple,
	{0x4C, 0x56, 0x9D}: vendorApple,
	{0x4C, 0x57, 0xCA}: vendorApple,
	{0x4C, 0x6B, 0xE8}: vendorApple,
	{0x4C, 0x74, 0xBF}: vendorApple,
	{0x4C, 0x7C, 0x5F}: vendorApple,
	{0x4C, 0x7C, 0xD9}: vendorApple,
	{0x4C, 0x8D, 0x79}: vendorApple,
	{0x4C, 0xAB, 0x4F}: vendorApple,
	{0x4C, 0xB1, 0x99}: vendorApple,
	{0x4C, 0xB9, 0x10}: vendorApple,
	{0x50, 0x1F, 0xC6}: vendorApple,
	{0x50, 0x32, 0x37}: vendorApple,
	{0x50, 0x7A, 0x55}: vendorApple,
	{0x50, 0x7A, 0xC5}: vendorApple,
	{0x50, 0x82, 0xD5}: vendorApple,
	{0x50, 0xA6, 0x7F}: vendorApple,
	{0x50, 0xBC, 0x96}: vendorApple,
	{0x50, 0xDE, 0x06}: vendorApple,
	{0x50, 0xEA, 0xD6}: vendorApple,
	{0x50, 0xED, 0x3C}: vendorApple,
	{0x50, 0xF4, 0xEB}: vendorApple,
	{0x54, 0x09, 0x10}: vendorApple,
	{0x54, 0x26, 0x96}: vendorApple,
	{0x54, 0x2B, 0x8D}: vendorApple,
	{0x54, 0x33, 0xCB}: vendorApple,
	{0x54, 0x4E, 0x90}: vendorApple,
	{0x54, 0x62, 0xE2}: vendorApple,
	{0x54, 0x72, 0x4F}: vendorApple,
	{0x54, 0x99, 0x63}: vendorApple,
	{0x54, 0x9F, 0x13}: vendorApple,
	{0x54, 0xAE, 0x27}: vendorApple,
	{0x54, 0xE4, 0x3A}: vendorApple,
	{0x54, 0xE6, 0x1B}: vendorApple,
	{0x54, 0xEA, 0xA8}: vendorApple,
	{0x58, 0x0A, 0xD4}: vendorApple,
	{0x58, 0x1F, 0xAA}: vendorApple,
	{0x58, 0x40, 0x4E}: vendorApple,
	{0x58, 0x55, 0xCA}: vendorApple,
	{0x58, 0x6B, 0x14}: vendorApple,
	{0x58, 0x7F, 0x57}: vendorApple,
	{0x58, 0xB0, 0x35}: vendorApple,
	{0x58, 0xD3, 0x49}: vendorApple,
	{0x58, 0xE2, 0x8F}: vendorApple,
	{0x58, 0xE6, 0xBA}: vendorApple,
	{0x5C, 0x09, 0x47}: vendorApple,
	{0x5C, 0x1D, 0xD9}: vendorApple,
	{0x5C, 0x50, 0xD9}: vendorApple,
	{0x5C, 0x52, 0x30}: vendorApple,
	{0x5C, 0x59, 0x48}: vendorApple,
	{0x5C, 0x70, 0x17}: vendorApple,
	{0x5C, 0x87, 0x30}: vendorApple,
	{0x5C, 0x8D, 0x4E}: vendorApple,
	{0x5C, 0x95, 0xAE}: vendorApple,
	{0x5C, 0x96, 0x9D}: vendorApple,
	{0x5C, 0x97, 0xF3}: vendorApple,
	{0x5C, 0xAD, 0xCF}: vendorApple,
	{0x5C, 0xF5, 0xDA}: vendorApple,
	{0x5C, 0xF7, 0xE6}: vendorApple,
	{0x5C, 0xF9, 0x38}: vendorApple,
	{0x60, 0x03, 0x08}: vendorApple,
	{0x60, 0x06, 0xE3}: vendorApple,
	{0x60, 0x30, 0xD4}: vendorApple,
	{0x60, 0x33, 0x4B}: vendorApple,
	{0x60, 0x69, 0x44}: vendorApple,
	{0x60, 0x70, 0xC0}: vendorApple,
	{0x60, 0x7E, 0xC9}: vendorApple,
	{0x60, 0x83, 0x73}: vendorApple,
	{0x60, 0x8B, 0x0E}: vendorApple,
	{0x60, 0x8C, 0x4A}: vendorApple,
	{0x60, 0x92, 0x17}: vendorApple,
	{0x60, 0x93, 0x16}: vendorApple,
	{0x60, 0x9A, 0xC1}: vendorApple,
	{0x60, 0xA3, 0x7D}: vendorApple,
	{0x60, 0xBE, 0xC4}: vendorApple,
	{0x60, 0xC5, 0x47}: vendorApple,
	{0x60, 0xD9, 0xC7}: vendorApple,
	{0x60, 0xF4, 0x45}: vendorApple,
	{0x60, 0xF8, 0x1D}: vendorApple,
	{0x60, 0xFA, 0xCD}: vendorApple,
	{0x60, 0xFB, 0x42}: vendorApple,
	{0x60, 0xFE, 0xC5}: vendorApple,
	{0x64, 0x0B, 0xD7}: vendorApple,
	{0x64, 0x20, 0x0C}: vendorApple,
	{0x64, 0x5A, 0x36}: vendorApple,
	{0x64, 0x5A, 0xED}: vendorApple,
	{0x64, 0x6D, 0x2F}: vendorApple,
	{0x64, 0x70, 0x33}: vendorApple,
	{0x64, 0x76, 0xBA}: vendorApple,
	{0x64, 0x9A, 0xBE}: vendorApple,
	{0x64, 0xA3, 0xCB}: vendorApple,
	{0x64, 0xA5, 0xC3}: vendorApple,
	{0x64, 0xB0, 0xA6}: vendorApple,
	{0x64, 0xB9, 0xE8}: vendorApple,
	{0x64, 0xC7, 0x53}: vendorApple,
	{0x64, 0xD2, 0xC4}: vendorApple,
	{0x64, 0xE6, 0x82}: vendorApple,
	{0x68, 0x09, 0x27}: vendorApple,
	{0x68, 0x2F, 0x67}: vendorApple,
	{0x68, 0x5B, 0x35}: vendorApple,
	{0x68, 0x64, 0x4B}: vendorApple,
	{0x68, 0x96, 0x7B}: vendorApple,
	{0x68, 0x9C, 0x70}: vendorApple,
	{0x68, 0xA8, 0x6D}: vendorApple,
	{0x68, 0xAB, 0x1E}: vendorApple,
	{0x68, 0xAE, 0x20}: vendorApple,
	{0x68, 0xD9, 0x3C}: vendorApple,
	{0x68, 0xDB, 0xCA}: vendorApple,
	{0x68, 0xEF, 0x43}: vendorApple,
	{0x68, 0xFB, 0x7E}: vendorApple,
	{0x68, 0xFE, 0xF7}: vendorApple,
	{0x6C, 0x19, 0xC0}: vendorApple,
	{0x6C, 0x3E, 0x6D}: vendorApple,
	{0x6C, 0x40, 0x08}: vendorApple,
	{0x6C, 0x4A, 0x85}: vendorApple,
	{0x6C, 0x4D, 0x73}: vendorApple,
	{0x6C, 0x70, 0x9F}: vendorApple,
	{0x6C, 0x72, 0xE7}: vendorApple,
	{0x6C, 0x8D, 0xC1}: vendorApple,
	{0x6C, 0x94, 0xF8}: vendorApple,
	{0x6C, 0x96, 0xCF}: vendorApple,
	{0x6C, 0xAB, 0x31}: vendorApple,
	{0x6C, 0xC2, 0x6B}: vendorApple,
	{0x6C, 0xE8, 0x5C}: vendorApple,
	{0x70, 0x11, 0x24}: vendorApple,
	{0x70, 0x14, 0xA6}: vendorApple,
	{0x70, 0x3C, 0x69}: vendorApple,
	{0x70, 0x3E, 0xAC}: vendorApple,
	{0x70, 0x48, 0x0F}: vendorApple,
	{0x70, 0x56, 0x81}: vendorApple,
	{0x70, 0x70, 0x0D}: vendorApple,
	{0x70, 0x73, 0xCB}: vendorApple,
	{0x70, 0x81, 0xEB}: vendorApple,
	{0x70, 0xA2, 0xB3}: vendorApple,
	{0x70, 0xCD, 0x60}: vendorApple,
	{0x70, 0xDE, 0xE2}: vendorApple,
	{0x70, 0xE7, 0x2C}: vendorApple,
	{0x70, 0xEA, 0x5A}: vendorApple,
	{0x70, 0xEC, 0xE4}: vendorApple,
	{0x70, 0xEF, 0x00}: vendorApple,
	{0x70, 0xF0, 0x87}: vendorApple,
	{0x74, 0x1B, 0xB2}: vendorApple,
	{0x74, 0x42, 0x8B}: vendorApple,
	{0x74, 0x65, 0x0C}: vendorApple,
	{0x74, 0x81, 0x14}: vendorApple,
	{0x74, 0x8D, 0x08}: vendorApple,
	{0x74, 0x8F, 0x3C}: vendorApple,
	{0x74, 0x9E, 0xAF}: vendorApple,
	{0x74, 0xB5, 0x87}: vendorApple,
	{0x74, 0xE1, 0xB6}: vendorApple,
	{0x74, 0xE2, 0xF5}: vendorApple,
	{0x78, 0x02, 0x8B}: vendorApple,
	{0x78, 0x31, 0xC1}: vendorApple,
	{0x78, 0x3A, 0x84}: vendorApple,
	{0x78, 0x4F, 0x43}: vendorApple,
	{0x78, 0x64, 0xC0}: vendorApple,
	{0x78, 0x67, 0xD7}: vendorApple,
	{0x78, 0x6C, 0x1C}: vendorApple,
	{0x78, 0x7B, 0x8A}: vendorApple,
	{0x78, 0x7E, 0x61}: vendorApple,
	{0x78, 0x88, 0x6D}: vendorApple,
	{0x78, 0x9F, 0x70}: vendorApple,
	{0x78, 0xA3, 0xE4}: vendorApple,
	{0x78, 0xCA, 0x39}: vendorApple,
	{0x78, 0xD1, 0x62}: vendorApple,
	{0x78, 0xD7, 0x5F}: vendorApple,
	{0x78, 0xE3, 0xDE}: vendorApple,
	{0x78, 0xFD, 0x94}: vendorApple,
	{0x7C, 0x01, 0x91}: vendorApple,
	{0x7C, 0x04, 0xD0}: vendorApple,
	{0x7C, 0x11, 0xBE}: vendorApple,
	{0x7C, 0x24, 0x99}: vendorApple,
	{0x7C, 0x50, 0x49}: vendorApple,
	{0x7C, 0x6D, 0x62}: vendorApple,
	{0x7C, 0x6D, 0xF8}: vendorApple,
	{0x7C, 0x9A, 0x1D}: vendorApple,
	{0x7C, 0xA1, 0xAE}: vendorApple,
	{0x7C, 0xAB, 0x60}: vendorApple,
	{0x7C, 0xC3, 0xA1}: vendorApple,
	{0x7C, 0xC5, 0x37}: vendorApple,
	{0x7C, 0xD1, 0xC3}: vendorApple,
	{0x7C, 0xF0, 0x5F}: vendorApple,
	{0x7C, 0xFA, 0xDF}: vendorApple,
	{0x7C, 0xFC, 0x16}: vendorApple,
	{0x80, 0x00, 0x6E}: vendorApple,
	{0x80, 0x0C, 0x67}: vendorApple,
	{0x80, 0x49, 0x71}: vendorApple,
	{0x80, 0x4A, 0x14}: vendorApple,
	{0x80, 0x5F, 0xC5}: vendorApple,
	{0x80, 0x82, 0x23}: vendorApple,
	{0x80, 0x92, 0x9F}: vendorApple,
	{0x80, 0xB0, 0x3D}: vendorApple,
	{0x80, 0xBE, 0x05}: vendorApple,
	{0x80, 0xD6, 0x05}: vendorApple,
	{0x80, 0xE6, 0x50}: vendorApple,
	{0x80, 0xEA, 0x96}: vendorApple,
	{0x80, 0xED, 0x2C}: vendorApple,
	{0x84, 0x29, 0x99}: vendorApple,
	{0x84, 0x38, 0x35}: vendorApple,
	{0x84, 0x41, 0x67}: vendorApple,
	{0x84, 0x68, 0x78}: vendorApple,
	{0x84, 0x78, 0x8B}: vendorApple,
	{0x84, 0x85, 0x06}: vendorApple,
	{0x84, 0x89, 0xAD}: vendorApple,
	{0x84, 0x8C, 0x8D}: vendorApple,
	{0x84, 0x8E, 0x0C}: vendorApple,
	{0x84, 0xA1, 0x34}: vendorApple,
	{0x84, 0xAB, 0x1A}: vendorApple,
	{0x84, 0xAC, 0x16}: vendorApple,
	{0x84, 0xAD, 0x8D}: vendorApple,
	{0x84, 0xB1, 0x53}: vendorApple,
	{0x84, 0xFC, 0xAC}: vendorApple,
	{0x84, 0xFC, 0xFE}: vendorApple,
	{0x88, 0x19, 0x08}: vendorApple,
	{0x88, 0x1F, 0xA1}: vendorApple,
	{0x88, 0x4D, 0x7C}: vendorApple,
	{0x88, 0x53, 0x95}: vendorApple,
	{0x88, 0x63, 0xDF}: vendorApple,
	{0x88, 0x64, 0x40}: vendorApple,
	{0x88, 0x66, 0x5A}: vendorApple,
	{0x88, 0x66, 0xA5}: vendorApple,
	{0x88, 0x6B, 0x6E}: vendorApple,
	{0x88, 0xA4, 0x79}: vendorApple,
	{0x88, 0xA9, 0xB7}: vendorApple,
	{0x88, 0xAE, 0x07}: vendorApple,
	{0x88, 0xB2, 0x91}: vendorApple,
	{0x88, 0xB9, 0x45}: vendorApple,
	{0x88, 0xC0, 0x8B}: vendorApple,
	{0x88, 0xC6, 0x63}: vendorApple,
	{0x88, 0xCB, 0x87}: vendorApple,
	{0x88, 0xE8, 0x7F}: vendorApple,
	{0x88, 0xE9, 0xFE}: vendorApple,
	{0x8C, 0x00, 0x6D}: vendorApple,
	{0x8C, 0x29, 0x37}: vendorApple,
	{0x8C, 0x2D, 0xAA}: vendorApple,
	{0x8C, 0x58, 0x77}: vendorApple,
	{0x8C, 0x7A, 0xAA}: vendorApple,
	{0x8C, 0x7B, 0x9D}: vendorApple,
	{0x8C, 0x7C, 0x92}: vendorApple,
	{0x8C, 0x85, 0x90}: vendorApple,
	{0x8C, 0x86, 0x1E}: vendorApple,
	{0x8C, 0x8E, 0xF2}: vendorApple,
	{0x8C, 0x8F, 0xE9}: vendorApple,
	{0x8C, 0xEC, 0x7B}: vendorApple,
	{0x8C, 0xFA, 0xBA}: vendorApple,
	{0x8C, 0xFE, 0x57}: vendorApple,
	{0x90, 0x27, 0xE4}: vendorApple,
	{0x90, 0x3C, 0x92}: vendorApple,
	{0x90, 0x60, 0xF1}: vendorApple,
	{0x90, 0x72, 0x40}: vendorApple,
	{0x90, 0x81, 0x2A}: vendorApple,
	{0x90, 0x81, 0x58}: vendorApple,
	{0x90, 0x84, 0x0D}: vendorApple,
	{0x90, 0x8C, 0x43}: vendorApple,
	{0x90, 0x8D, 0x6C}: vendorApple,
	{0x90, 0x9C, 0x4A}: vendorApple,
	{0x90, 0xA2, 0x5B}: vendorApple,
	{0x90, 0xB0, 0xED}: vendorApple,
	{0x90, 0xB2, 0x1F}: vendorApple,
	{0x90, 0xB9, 0x31}: vendorApple,
	{0x90, 0xC1, 0xC6}: vendorApple,
	{0x90, 0xDD, 0x5D}: vendorApple,
	{0x90, 0xE1, 0x7B}: vendorApple,
	{0x90, 0xFD, 0x61}: vendorApple,
	{0x94, 0x0C, 0x98}: vendorApple,
	{0x94, 0x16, 0x25}: vendorApple,
	{0x94, 0x5C, 0x9A}: vendorApple,
	{0x94, 0x94, 0x26}: vendorApple,
	{0x94, 0xB0, 0x1F}: vendorApple,
	{0x94, 0xBF, 0x2D}: vendorApple,
	{0x94, 0xE9, 0x6A}: vendorApple,
	{0x94, 0xEA, 0x32}: vendorApple,
	{0x94, 0xF6, 0xA3}: vendorApple,
	{0x94, 0xF6, 0xD6}: vendorApple,
	{0x98, 0x00, 0xC6}: vendorApple,
	{0x98, 0x01, 0xA7}: vendorApple,
	{0x98, 0x03, 0xD8}: vendorApple,
	{0x98, 0x10, 0xE8}: vendorApple,
	{0x98, 0x46, 0x0A}: vendorApple,
	{0x98, 0x50, 0x2E}: vendorApple,
	{0x98, 0x5A, 0xEB}: vendorApple,
	{0x98, 0x60, 0xCA}: vendorApple,
	{0x98, 0x9E, 0x63}: vendorApple,
	{0x98, 0xB8, 0xE3}: vendorApple,
	{0x98, 0xCA, 0x33}: vendorApple,
	{0x98, 0xD6, 0xBB}: vendorApple,
	{0x98, 0xDD, 0x60}: vendorApple,
	{0x98, 0xE0, 0xD9}: vendorApple,
	{0x98, 0xF0, 0xAB}: vendorApple,
	{0x98, 0xFE, 0x94}: vendorApple,
	{0x9C, 0x04, 0xEB}: vendorApple,
	{0x9C, 0x20, 0x7B}: vendorApple,
	{0x9C, 0x28, 0xB3}: vendorApple,
	{0x9C, 0x29, 0x3F}: vendorApple,
	{0x9C, 0x35, 0xEB}: vendorApple,
	{0x9C, 0x4F, 0xDA}: vendorApple,
	{0x9C, 0x58, 0x3C}: vendorApple,
	{0x9C, 0x64, 0x8B}: vendorApple,
	{0x9C, 0x76, 0x0E}: vendorApple,
	{0x9C, 0x84, 0xBF}: vendorApple,
	{0x9C, 0x8B, 0xA0}: vendorApple,
	{0x9C, 0xE3, 0x3F}: vendorApple,
	{0x9C, 0xE6, 0x5E}: vendorApple,
	{0x9C, 0xF3, 0x87}: vendorApple,
	{0x9C, 0xF4, 0x8E}: vendorApple,
	{0x9C, 0xFC, 0x01}: vendorApple,
	{0x9C, 0xFC, 0x28}: vendorApple,
	{0xA0, 0x18, 0x28}: vendorApple,
	{0xA0, 0x3B, 0xE3}: vendorApple,
	{0xA0, 0x4E, 0xA7}: vendorApple,
	{0xA0, 0x4E, 0xCF}: vendorApple,
	{0xA0, 0x56, 0xF3}: vendorApple,
	{0xA0, 0x78, 0x17}: vendorApple,
	{0xA0, 0x99, 0x9B}: vendorApple,
	{0xA0, 0xA3, 0x09}: vendorApple,
	{0xA0, 0xD7, 0x95}: vendorApple,
	{0xA0, 0xED, 0xCD}: vendorApple,
	{0xA0, 0xFB, 0xC5}: vendorApple,
	{0xA4, 0x31, 0x35}: vendorApple,
	{0xA4, 0x5E, 0x60}: vendorApple,
	{0xA4, 0x67, 0x06}: vendorApple,
	{0xA4, 0x77, 0xF3}: vendorApple,
	{0xA4, 0x83, 0xE7}: vendorApple,
	{0xA4, 0xB1, 0x97}: vendorApple,
	{0xA4, 0xB8, 0x05}: vendorApple,
	{0xA4, 0xC3, 0x61}: vendorApple,
	{0xA4, 0xD1, 0x8C}: vendorApple,
	{0xA4, 0xD1, 0xD2}: vendorApple,
	{0xA4, 0xD9, 0x31}: vendorApple,
	{0xA4, 0xE9, 0x75}: vendorApple,
	{0xA4, 0xF1, 0xE8}: vendorApple,
	{0xA8, 0x20, 0x66}: vendorApple,
	{0xA8, 0x4A, 0x28}: vendorApple,
	{0xA8, 0x5B, 0x78}: vendorApple,
	{0xA8, 0x5C, 0x2C}: vendorApple,
	{0xA8, 0x60, 0xB6}: vendorApple,
	{0xA8, 0x66, 0x7F}: vendorApple,
	{0xA8, 0x81, 0x7E}: vendorApple,
	{0xA8, 0x86, 0xDD}: vendorApple,
	{0xA8, 0x88, 0x08}: vendorApple,
	{0xA8, 0x8E, 0x24}: vendorApple,
	{0xA8, 0x91, 0x3D}: vendorApple,
	{0xA8, 0x96, 0x8A}: vendorApple,
	{0xA8, 0xBB, 0xCF}: vendorApple,
	{0xA8, 0xBE, 0x27}: vendorApple,
	{0xA8, 0xFA, 0xD8}: vendorApple,
	{0xA8, 0xFE, 0x9D}: vendorApple,
	{0xAC, 0x15, 0xF4}: vendorApple,
	{0xAC, 0x1D, 0x06}: vendorApple,
	{0xAC, 0x1F, 0x74}: vendorApple,
	{0xAC, 0x29, 0x3A}: vendorApple,
	{0xAC, 0x3C, 0x0B}: vendorApple,
	{0xAC, 0x49, 0xDB}: vendorApple,
	{0xAC, 0x61, 0xEA}: vendorApple,
	{0xAC, 0x7F, 0x3E}: vendorApple,
	{0xAC, 0x87, 0xA3}: vendorApple,
	{0xAC, 0x88, 0xFD}: vendorApple,
	{0xAC, 0x90, 0x85}: vendorApple,
	{0xAC, 0xBC, 0x32}: vendorApple,
	{0xAC, 0xCF, 0x5C}: vendorApple,
	{0xAC, 0xE4, 0xB5}: vendorApple,
	{0xAC, 0xFD, 0xEC}: vendorApple,
	{0xB0, 0x19, 0xC6}: vendorApple,
	{0xB0, 0x34, 0x95}: vendorApple,
	{0xB0, 0x35, 0xB5}: vendorApple,
	{0xB0, 0x48, 0x1A}: vendorApple,
	{0xB0, 0x65, 0xBD}: vendorApple,
	{0xB0, 0x70, 0x2D}: vendorApple,
	{0xB0, 0x8C, 0x75}: vendorApple,
	{0xB0, 0x9F, 0xBA}: vendorApple,
	{0xB0, 0xBE, 0x83}: vendorApple,
	{0xB0, 0xCA, 0x68}: vendorApple,
	{0xB0, 0xE5, 0xF9}: vendorApple,
	{0xB4, 0x18, 0xD1}: vendorApple,
	{0xB4, 0x1B, 0xB0}: vendorApple,
	{0xB4, 0x40, 0xA4}: vendorApple,
	{0xB4, 0x4B, 0xD2}: vendorApple,
	{0xB4, 0x56, 0xE3}: vendorApple,
	{0xB4, 0x85, 0xE1}: vendorApple,
	{0xB4, 0x8B, 0x19}: vendorApple,
	{0xB4, 0x9C, 0xDF}: vendorApple,
	{0xB4, 0xF0, 0xAB}: vendorApple,
	{0xB4, 0xF6, 0x1C}: vendorApple,
	{0xB4, 0xFA, 0x48}: vendorApple,
	{0xB8, 0x09, 0x8A}: vendorApple,
	{0xB8, 0x17, 0xC2}: vendorApple,
	{0xB8, 0x2A, 0xA9}: vendorApple,
	{0xB8, 0x37, 0x4A}: vendorApple,
	{0xB8, 0x41, 0xA4}: vendorApple,
	{0xB8, 0x44, 0xD9}: vendorApple,
	{0xB8, 0x53, 0xAC}: vendorApple,
	{0xB8, 0x5D, 0x0A}: vendorApple,
	{0xB8, 0x63, 0x4D}: vendorApple,
	{0xB8, 0x78, 0x2E}: vendorApple,
	{0xB8, 0x7B, 0xC5}: vendorApple,
	{0xB8, 0x81, 0xFA}: vendorApple,
	{0xB8, 0x8D, 0x12}: vendorApple,
	{0xB8, 0x90, 0x47}: vendorApple,
	{0xB8, 0xB2, 0xF8}: vendorApple,
	{0xB8, 0xC1, 0x11}: vendorApple,
	{0xB8, 0xC7, 0x5D}: vendorApple,
	{0xB8, 0xE8, 0x56}: vendorApple,
	{0xB8, 0xF1, 0x2A}: vendorApple,
	{0xB8, 0xF6, 0xB1}: vendorApple,
	{0xB8, 0xFF, 0x61}: vendorApple,
	{0xBC, 0x09, 0x63}: vendorApple,
	{0xBC, 0x3B, 0xAF}: vendorApple,
	{0xBC, 0x4C, 0xC4}: vendorApple,
	{0xBC, 0x52, 0xB7}: vendorApple,
	{0xBC, 0x54, 0x36}: vendorApple,
	{0xBC, 0x67, 0x78}: vendorApple,
	{0xBC, 0x6C, 0x21}: vendorApple,
	{0xBC, 0x92, 0x6B}: vendorApple,
	{0xBC, 0x9F, 0xEF}: vendorApple,
	{0xBC, 0xA5, 0xA9}: vendorApple,
	{0xBC, 0xA9, 0x20}: vendorApple,
	{0xBC, 0xB8, 0x63}: vendorApple,
	{0xBC, 0xE1, 0x43}: vendorApple,
	{0xBC, 0xEC, 0x5D}: vendorApple,
	{0xBC, 0xFE, 0xD9}: vendorApple,
	{0xC0, 0x1A, 0xDA}: vendorApple,
	{0xC0, 0x2C, 0x5C}: vendorApple,
	{0xC0, 0x44, 0x42}: vendorApple,
	{0xC0, 0x63, 0x94}: vendorApple,
	{0xC0, 0x84, 0x7A}: vendorApple,
	{0xC0, 0x9A, 0xD0}: vendorApple,
	{0xC0, 0x9F, 0x42}: vendorApple,
	{0xC0, 0xA5, 0x3E}: vendorApple,
	{0xC0, 0xA6, 0x00}: vendorApple,
	{0xC0, 0xB6, 0x58}: vendorApple,
	{0xC0, 0xCC, 0xF8}: vendorApple,
	{0xC0, 0xCE, 0xCD}: vendorApple,
	{0xC0, 0xD0, 0x12}: vendorApple,
	{0xC0, 0xE8, 0x62}: vendorApple,
	{0xC0, 0xF2, 0xFB}: vendorApple,
	{0xC4, 0x0B, 0x31}: vendorApple,
	{0xC4, 0x12, 0x34}: vendorApple,
	{0xC4, 0x14, 0x11}: vendorApple,
	{0xC4, 0x2A, 0xD0}: vendorApple,
	{0xC4, 0x2C, 0x03}: vendorApple,
	{0xC4, 0x61, 0x8B}: vendorApple,
	{0xC4, 0x84, 0x66}: vendorApple,
	{0xC4, 0x91, 0x0C}: vendorApple,
	{0xC4, 0x98, 0x80}: vendorApple,
	{0xC4, 0xB3, 0x01}: vendorApple,
	{0xC4, 0xC3, 0x6B}: vendorApple,
	{0xC8, 0x1E, 0xE7}: vendorApple,
	{0xC8, 0x2A, 0x14}: vendorApple,
	{0xC8, 0x33, 0x4B}: vendorApple,
	{0xC8, 0x3C, 0x85}: vendorApple,
	{0xC8, 0x69, 0xCD}: vendorApple,
	{0xC8, 0x6F, 0x1D}: vendorApple,
	{0xC8, 0x85, 0x50}: vendorApple,
	{0xC8, 0xB1, 0xCD}: vendorApple,
	{0xC8, 0xB5, 0xB7}: vendorApple,
	{0xC8, 0xBC, 0xC8}: vendorApple,
	{0xC8, 0xD0, 0x83}: vendorApple,
	{0xC8, 0xE0, 0xEB}: vendorApple,
	{0xC8, 0xF6, 0x50}: vendorApple,
	{0xCC, 0x08, 0x8D}: vendorApple,
	{0xCC, 0x08, 0xE0}: vendorApple,
	{0xCC, 0x20, 0xE8}: vendorApple,
	{0xCC, 0x25, 0xEF}: vendorApple,
	{0xCC, 0x29, 0xF5}: vendorApple,
	{0xCC, 0x2D, 0xB7}: vendorApple,
	{0xCC, 0x44, 0x63}: vendorApple,
	{0xCC, 0x66, 0x0A}: vendorApple,
	{0xCC, 0x69, 0xFA}: vendorApple,
	{0xCC, 0x78, 0x5F}: vendorApple,
	{0xCC, 0xC7, 0x60}: vendorApple,
	{0xCC, 0xC9, 0x5D}: vendorApple,
	{0xCC, 0xD2, 0x81}: vendorApple,
	{0xD0, 0x03, 0x4B}: vendorApple,
	{0xD0, 0x23, 0xDB}: vendorApple,
	{0xD0, 0x25, 0x98}: vendorApple,
	{0xD0, 0x2B, 0x20}: vendorApple,
	{0xD0, 0x33, 0x11}: vendorApple,
	{0xD0, 0x3F, 0xAA}: vendorApple,
	{0xD0, 0x4F, 0x7E}: vendorApple,
	{0xD0, 0x65, 0x44}: vendorApple,
	{0xD0, 0x81, 0x7A}: vendorApple,
	{0xD0, 0xA6, 0x37}: vendorApple,
	{0xD0, 0xC5, 0xF3}: vendorApple,
	{0xD0, 0xD2, 0x3C}: vendorApple,
	{0xD0, 0xD2, 0xB0}: vendorApple,
	{0xD0, 0xE1, 0x40}: vendorApple,
	{0xD4, 0x46, 0xE1}: vendorApple,
	{0xD4, 0x57, 0x63}: vendorApple,
	{0xD4, 0x61, 0x9D}: vendorApple,
	{0xD4, 0x61, 0xDA}: vendorApple,
	{0xD4, 0x68, 0xAA}: vendorApple,
	{0xD4, 0x90, 0x9C}: vendorApple,
	{0xD4, 0x9A, 0x20}: vendorApple,
	{0xD4, 0xA3, 0x3D}: vendorApple,
	{0xD4, 0xDC, 0xCD}: vendorApple,
	{0xD4, 0xF4, 0x6F}: vendorApple,
	{0xD8, 0x00, 0x4D}: vendorApple,
	{0xD8, 0x1C, 0x79}: vendorApple,
	{0xD8, 0x1D, 0x72}: vendorApple,
	{0xD8, 0x30, 0x62}: vendorApple,
	{0xD8, 0x4C, 0x90}: vendorApple,
	{0xD8, 0x8F, 0x76}: vendorApple,
	{0xD8, 0x96, 0x95}: vendorApple,
	{0xD8, 0x9E, 0x3F}: vendorApple,
	{0xD8, 0xA2, 0x5E}: vendorApple,
	{0xD8, 0xBB, 0x2C}: vendorApple,
	{0xD8, 0xBE, 0x1F}: vendorApple,
	{0xD8, 0xCF, 0x9C}: vendorApple,
	{0xD8, 0xD1, 0xCB}: vendorApple,
	{0xD8, 0xDC, 0x40}: vendorApple,
	{0xD8, 0xDE, 0x3A}: vendorApple,
	{0xDC, 0x08, 0x0F}: vendorApple,
	{0xDC, 0x0C, 0x5C}: vendorApple,
	{0xDC, 0x2B, 0x2A}: vendorApple,
	{0xDC, 0x2B, 0x61}: vendorApple,
	{0xDC, 0x37, 0x14}: vendorApple,
	{0xDC, 0x41, 0x5F}: vendorApple,
	{0xDC, 0x52, 0x85}: vendorApple,
	{0xDC, 0x53, 0x92}: vendorApple,
	{0xDC, 0x56, 0xE7}: vendorApple,
	{0xDC, 0x86, 0xD8}: vendorApple,
	{0xDC, 0x9B, 0x9C}: vendorApple,
	{0xDC, 0xA4, 0xCA}: vendorApple,
	{0xDC, 0xA9, 0x04}: vendorApple,
	{0xDC, 0xB5, 0x4F}: vendorApple,
	{0xDC, 0xD3, 0xA2}: vendorApple,
	{0xDC, 0xF4, 0xCA}: vendorApple,
	{0xE0, 0x2B, 0x96}: vendorApple,
	{0xE0, 0x33, 0x8E}: vendorApple,
	{0xE0, 0x5F, 0x45}: vendorApple,
	{0xE0, 0x66, 0x78}: vendorApple,
	{0xE0, 0x6D, 0x17}: vendorApple,
	{0xE0, 0x89, 0x7E}: vendorApple,
	{0xE0, 0x92, 0x5C}: vendorApple,
	{0xE0, 0xAC, 0xCB}: vendorApple,
	{0xE0, 0xB5, 0x2D}: vendorApple,
	{0xE0, 0xB5, 0x5F}: vendorApple,
	{0xE0, 0xB9, 0xBA}: vendorApple,
	{0xE0, 0xC7, 0x67}: vendorApple,
	{0xE0, 0xC9, 0x7A}: vendorApple,
	{0xE0, 0xEB, 0x40}: vendorApple,
	{0xE0, 0xF5, 0xC6}: vendorApple,
	{0xE0, 0xF8, 0x47}: vendorApple,
	{0xE4, 0x25, 0xE7}: vendorApple,
	{0xE4, 0x2B, 0x34}: vendorApple,
	{0xE4, 0x50, 0xEB}: vendorApple,
	{0xE4, 0x76, 0x84}: vendorApple,
	{0xE4, 0x8B, 0x7F}: vendorApple,
	{0xE4, 0x90, 0xFD}: vendorApple,
	{0xE4, 0x98, 0xD6}: vendorApple,
	{0xE4, 0x9A, 0x79}: vendorApple,
	{0xE4, 0x9A, 0xDC}: vendorApple,
	{0xE4, 0xB2, 0xFB}: vendorApple,
	{0xE4, 0xC6, 0x3D}: vendorApple,
	{0xE4, 0xCE, 0x8F}: vendorApple,
	{0xE4, 0xE0, 0xA6}: vendorApple,
	{0xE4, 0xE4, 0xAB}: vendorApple,
	{0xE8, 0x04, 0x0B}: vendorApple,
	{0xE8, 0x06, 0x88}: vendorApple,
	{0xE8, 0x1C, 0xD8}: vendorApple,
	{0xE8, 0x36, 0x17}: vendorApple,
	{0xE8, 0x78, 0x65}: vendorApple,
	{0xE8, 0x7F, 0x95}: vendorApple,
	{0xE8, 0x80, 0x2E}: vendorApple,
	{0xE8, 0x81, 0x52}: vendorApple,
	{0xE8, 0x85, 0x4B}: vendorApple,
	{0xE8, 0x8D, 0x28}: vendorApple,
	{0xE8, 0xA7, 0x30}: vendorApple,
	{0xE8, 0xB2, 0xAC}: vendorApple,
	{0xE8, 0xFB, 0xE9}: vendorApple,
	{0xEC, 0x26, 0x51}: vendorApple,
	{0xEC, 0x2C, 0xE2}: vendorApple,
	{0xEC, 0x35, 0x86}: vendorApple,
	{0xEC, 0x85, 0x2F}: vendorApple,
	{0xEC, 0xAD, 0xB8}: vendorApple,
	{0xEC, 0xCE, 0xD7}: vendorApple,
	{0xF0, 0x18, 0x98}: vendorApple,
	{0xF0, 0x24, 0x75}: vendorApple,
	{0xF0, 0x2F, 0x4B}: vendorApple,
	{0xF0, 0x5C, 0xD5}: vendorApple,
	{0xF0, 0x76, 0x6F}: vendorApple,
	{0xF0, 0x78, 0x07}: vendorApple,
	{0xF0, 0x79, 0x60}: vendorApple,
	{0xF0, 0x98, 0x9D}: vendorApple,
	{0xF0, 0x99, 0xB6}: vendorApple,
	{0xF0, 0x99, 0xBF}: vendorApple,
	{0xF0, 0xA3, 0x5A}: vendorApple,
	{0xF0, 0xB0, 0xE7}: vendorApple,
	{0xF0, 0xB3, 0xEC}: vendorApple,
	{0xF0, 0xB4, 0x79}: vendorApple,
	{0xF0, 0xC1, 0xF1}: vendorApple,
	{0xF0, 0xC3, 0x71}: vendorApple,
	{0xF0, 0xCB, 0xA1}: vendorApple,
	{0xF0, 0xD1, 0xA9}: vendorApple,
	{0xF0, 0xDB, 0xE2}: vendorApple,
	{0xF0, 0xDB, 0xF8}: vendorApple,
	{0xF0, 0xDC, 0xE2}: vendorApple,
	{0xF0, 0xF6, 0x1C}: vendorApple,
	{0xF4, 0x06, 0x16}: vendorApple,
	{0xF4, 0x0E, 0x01}: vendorApple,
	{0xF4, 0x0F, 0x24}: vendorApple,
	{0xF4, 0x1B, 0xA1}: vendorApple,
	{0xF4, 0x31, 0xC3}: vendorApple,
	{0xF4, 0x34, 0xF0}: vendorApple,
	{0xF4, 0x37, 0xB7}: vendorApple,
	{0xF4, 0x5C, 0x89}: vendorApple,
	{0xF4, 0x65, 0xA6}: vendorApple,
	{0xF4, 0xAF, 0xE7}: vendorApple,
	{0xF4, 0xBE, 0xEC}: vendorApple,
	{0xF4, 0xD4, 0x88}: vendorApple,
	{0xF4, 0xDB, 0xE3}: vendorApple,
	{0xF4, 0xF1, 0x5A}: vendorApple,
	{0xF4, 0xF9, 0x51}: vendorApple,
	{0xF8, 0x03, 0x77}: vendorApple,
	{0xF8, 0x10, 0x93}: vendorApple,
	{0xF8, 0x1E, 0xDF}: vendorApple,
	{0xF8, 0x27, 0x93}: vendorApple,
	{0xF8, 0x2D, 0x7C}: vendorApple,
	{0xF8, 0x38, 0x80}: vendorApple,
	{0xF8, 0x4D, 0x89}: vendorApple,
	{0xF8, 0x4E, 0x73}: vendorApple,
	{0xF8, 0x62, 0x14}: vendorApple,
	{0xF8, 0x66, 0x5A}: vendorApple,
	{0xF8, 0x6F, 0xC1}: vendorApple,
	{0xF8, 0x87, 0xF1}: vendorApple,
	{0xF8, 0x95, 0xEA}: vendorApple,
	{0xF8, 0xB1, 0xDD}: vendorApple,
	{0xF8, 0xC3, 0xCC}: vendorApple,
	{0xF8, 0xE9, 0x4E}: vendorApple,
	{0xF8, 0xFF, 0xC2}: vendorApple,
	{0xFC, 0x18, 0x3C}: vendorApple,
	{0xFC, 0x1D, 0x43}: vendorApple,
	{0xFC, 0x25, 0x3F}: vendorApple,
	{0xFC, 0x2A, 0x9C}: vendorApple,
	{0xFC, 0x4E, 0xA4}: vendorApple,
	{0xFC, 0x66, 0xCF}: vendorApple,
	{0xFC, 0xAA, 0x81}: vendorApple,
	{0xFC, 0xB6, 0xD8}: vendorApple,
	{0xFC, 0xD8, 0x48}: vendorApple,
	{0xFC, 0xE9, 0x98}: vendorApple,
	{0xFC, 0xFC, 0x48}: vendorApple,
	// Google
	{0x00, 0x1A, 0x11}: vendorGoogle,
	{0x00, 0xF6, 0x20}: vendorGoogle,
	{0x08, 0x9E, 0x08}: vendorGoogle,
	{0x08, 0xB4, 0xB1}: vendorGoogle,
	{0x0C, 0xC4, 0x13}: vendorGoogle,
	{0x14, 0x22, 0x3B}: vendorGoogle,
	{0x14, 0xC1, 0x4E}: vendorGoogle,
	{0x1C, 0x53, 0xF9}: vendorGoogle,
	{0x1C, 0xF2, 0x9A}: vendorGoogle,
	{0x20, 0x1F, 0x3B}: vendorGoogle,
	{0x20, 0xDF, 0xB9}: vendorGoogle,
	{0x24, 0x05, 0x88}: vendorGoogle,
	{0x28, 0xBD, 0x89}: vendorGoogle,
	{0x30, 0xFD, 0x38}: vendorGoogle,
	{0x38, 0x8B, 0x59}: vendorGoogle,
	{0x3C, 0x28, 0x6D}: vendorGoogle,
	{0x3C, 0x5A, 0xB4}: vendorGoogle,
	{0x3C, 0x8D, 0x20}: vendorGoogle,
	{0x44, 0x07, 0x0B}: vendorGoogle,
	{0x44, 0xBB, 0x3B}: vendorGoogle,
	{0x48, 0xD6, 0xD5}: vendorGoogle,
	{0x54, 0x60, 0x09}: vendorGoogle,
	{0x58, 0x24, 0x29}: vendorGoogle,
	{0x58, 0xCB, 0x52}: vendorGoogle,
	{0x60, 0xB7, 0x6E}: vendorGoogle,
	{0x70, 0x3A, 0xCB}: vendorGoogle,
	{0x7C, 0x2E, 0xBD}: vendorGoogle,
	{0x7C, 0xD9, 0x5C}: vendorGoogle,
	{0x88, 0x3D, 0x24}: vendorGoogle,
	{0x88, 0x54, 0x1F}: vendorGoogle,
	{0x90, 0x0C, 0xC8}: vendorGoogle,
	{0x94, 0x95, 0xA0}: vendorGoogle,
	{0x94, 0xEB, 0x2C}: vendorGoogle,
	{0x98, 0xD2, 0x93}: vendorGoogle,
	{0xA4, 0x77, 0x33}: vendorGoogle,
	{0xAC, 0x67, 0x84}: vendorGoogle,
	{0xB0, 0x2A, 0x43}: vendorGoogle,
	{0xB0, 0x6A, 0x41}: vendorGoogle,
	{0xB0, 0xE4, 0xD5}: vendorGoogle,
	{0xCC, 0xA7, 0xC1}: vendorGoogle,
	{0xCC, 0xF4, 0x11}: vendorGoogle,
	{0xD4, 0xF5, 0x47}: vendorGoogle,
	{0xD8, 0x6C, 0x63}: vendorGoogle,
	{0xD8, 0x8C, 0x79}: vendorGoogle,
	{0xD8, 0xEB, 0x46}: vendorGoogle,
	{0xDA, 0xA1, 0x19}: vendorGoogle,
	{0xDC, 0xE5, 0x5B}: vendorGoogle,
	{0xE4, 0x5E, 0x1B}: vendorGoogle,
	{0xE4, 0xF0, 0x42}: vendorGoogle,
	{0xF0, 0x5C, 0x77}: vendorGoogle,
	{0xF0, 0x72, 0xEA}: vendorGoogle,
	{0xF0, 0xEF, 0x86}: vendorGoogle,
	{0xF4, 0x03, 0x04}: vendorGoogle,
	{0xF4, 0xF5, 0xD8}: vendorGoogle,
	{0xF4, 0xF5, 0xE8}: vendorGoogle,
	{0xF8, 0x0F, 0xF9}: vendorGoogle,
	{0xF8, 0x1A, 0x2B}: vendorGoogle,
	{0xF8, 0x8F, 0xCA}: vendorGoogle,
	// Samsung
	{0x00, 0x00, 0xF0}: vendorSamsung,
	{0x00, 0x07, 0xAB}: vendorSamsung,
	{0x00, 0x0D, 0xE5}: vendorSamsung,
	{0x00, 0x12, 0x47}: vendorSamsung,
	{0x00, 0x12, 0xFB}: vendorSamsung,
	{0x00, 0x13, 0x77}: vendorSamsung,
	{0x00, 0x15, 0x99}: vendorSamsung,
	{0x00, 0x15, 0xB9}: vendorSamsung,
	{0x00, 0x16, 0x32}: vendorSamsung,
	{0x00, 0x16, 0x6B}: vendorSamsung,
	{0x00, 0x16, 0x6C}: vendorSamsung,
	{0x00, 0x16, 0xDB}: vendorSamsung,
	{0x00, 0x17, 0xC9}: vendorSamsung,
	{0x00, 0x17, 0xD5}: vendorSamsung,
	{0x00, 0x18, 0xAF}: vendorSamsung,
	{0x00, 0x1A, 0x8A}: vendorSamsung,
	{0x00, 0x1B, 0x98}: vendorSamsung,
	{0x00, 0x1C, 0x43}: vendorSamsung,
	{0x00, 0x1D, 0x25}: vendorSamsung,
	{0x00, 0x1D, 0xF6}: vendorSamsung,
	{0x00, 0x1E, 0x7D}: vendorSamsung,
	{0x00, 0x1E, 0xE1}: vendorSamsung,
	{0x00, 0x1E, 0xE2}: vendorSamsung,
	{0x00, 0x1F, 0xCC}: vendorSamsung,
	{0x00, 0x1F, 0xCD}: vendorSamsung,
	{0x00, 0x21, 0x4C}: vendorSamsung,
	{0x00, 0x21, 0xD1}: vendorSamsung,
	{0x00, 0x21, 0xD2}: vendorSamsung,
	{0x00, 0x23, 0x39}: vendorSamsung,
	{0x00, 0x23, 0x3A}: vendorSamsung,
	{0x00, 0x23, 0x99}: vendorSamsung,
	{0x00, 0x23, 0xD6}: vendorSamsung,
	{0x00, 0x23, 0xD7}: vendorSamsung,
	{0x00, 0x24, 0x54}: vendorSamsung,
	{0x00, 0x24, 0x90}: vendorSamsung,
	{0x00, 0x24, 0x91}: vendorSamsung,
	{0x00, 0x24, 0xE9}: vendorSamsung,
	{0x00, 0x25, 0x38}: vendorSamsung,
	{0x00, 0x25, 0x66}: vendorSamsung,
	{0x00, 0x25, 0x67}: vendorSamsung,
	{0x00, 0x26, 0x5D}: vendorSamsung,
	{0x00, 0x26, 0x5F}: vendorSamsung,
	{0x00, 0x6F, 0x64}: vendorSamsung,
	{0x00, 0x72, 0x04}: vendorSamsung,
	{0x00, 0x73, 0xE0}: vendorSamsung,
	{0x00, 0x7C, 0x2D}: vendorSamsung,
	{0x00, 0x87, 0x01}: vendorSamsung,
	{0x00, 0xB5, 0xD0}: vendorSamsung,
	{0x00, 0xBF, 0x61}: vendorSamsung,
	{0x00, 0xC3, 0xF4}: vendorSamsung,
	{0x00, 0xE3, 0xB2}: vendorSamsung,
	{0x00, 0xF4, 0x6F}: vendorSamsung,
	{0x00, 0xFA, 0x21}: vendorSamsung,
	{0x04, 0x18, 0x0F}: vendorSamsung,
	{0x04, 0x1B, 0xBA}: vendorSamsung,
	{0x04, 0xB1, 0xA1}: vendorSamsung,
	{0x04, 0xB4, 0x29}: vendorSamsung,
	{0x04, 0xB9, 0xE3}: vendorSamsung,
	{0x04, 0xBA, 0x8D}: vendorSamsung,
	{0x04, 0xBD, 0xBF}: vendorSamsung,
	{0x04, 0xFE, 0x31}: vendorSamsung,
	{0x08, 0x08, 0xC2}: vendorSamsung,
	{0x08, 0x15, 0x2F}: vendorSamsung,
	{0x08, 0x21, 0xEF}: vendorSamsung,
	{0x08, 0x37, 0x3D}: vendorSamsung,
	{0x08, 0x3D, 0x88}: vendorSamsung,
	{0x08, 0x78, 0x08}: vendorSamsung,
	{0x08, 0x8C, 0x2C}: vendorSamsung,
	{0x08, 0xAE, 0xD6}: vendorSamsung,
	{0x08, 0xBF, 0xA0}: vendorSamsung,
	{0x08, 0xD4, 0x2B}: vendorSamsung,
	{0x08, 0xEC, 0xA9}: vendorSamsung,
	{0x08, 0xEE, 0x8B}: vendorSamsung,
	{0x08, 0xFC, 0x88}: vendorSamsung,
	{0x08, 0xFD, 0x0E}: vendorSamsung,
	{0x0C, 0x02, 0xBD}: vendorSamsung,
	{0x0C, 0x14, 0x20}: vendorSamsung,
	{0x0C, 0x2F, 0xB0}: vendorSamsung,
	{0x0C, 0x71, 0x5D}: vendorSamsung,
	{0x0C, 0x89, 0x10}: vendorSamsung,
	{0x0C, 0x8D, 0xCA}: vendorSamsung,
	{0x0C, 0xA8, 0xA7}: vendorSamsung,
	{0x0C, 0xB3, 0x19}: vendorSamsung,
	{0x0C, 0xDF, 0xA4}: vendorSamsung,
	{0x0C, 0xE0, 0xDC}: vendorSamsung,
	{0x10, 0x07, 0xB6}: vendorSamsung,
	{0x10, 0x1D, 0xC0}: vendorSamsung,
	{0x10, 0x29, 0xAB}: vendorSamsung,
	{0x10, 0x2B, 0x41}: vendorSamsung,
	{0x10, 0x30, 0x47}: vendorSamsung,
	{0x10, 0x39, 0x17}: vendorSamsung,
	{0x10, 0x3B, 0x59}: vendorSamsung,
	{0x10, 0x77, 0xB1}: vendorSamsung,
	{0x10, 0x89, 0xFB}: vendorSamsung,
	{0x10, 0x8E, 0xE0}: vendorSamsung,
	{0x10, 0x92, 0x66}: vendorSamsung,
	{0x10, 0xD3, 0x8A}: vendorSamsung,
	{0x10, 0xD5, 0x42}: vendorSamsung,
	{0x10, 0xE4, 0xC2}: vendorSamsung,
	{0x10, 0xEC, 0x81}: vendorSamsung,
	{0x14, 0x01, 0x52}: vendorSamsung,
	{0x14, 0x1F, 0x78}: vendorSamsung,
	{0x14, 0x32, 0xD1}: vendorSamsung,
	{0x14, 0x56, 0x8E}: vendorSamsung,
	{0x14, 0x89, 0xFD}: vendorSamsung,
	{0x14, 0x96, 0xE5}: vendorSamsung,
	{0x14, 0x9F, 0x3C}: vendorSamsung,
	{0x14, 0xA3, 0x64}: vendorSamsung,
	{0x14, 0xB4, 0x84}: vendorSamsung,
	{0x14, 0xBB, 0x6E}: vendorSamsung,
	{0x14, 0xF4, 0x2A}: vendorSamsung,
	{0x18, 0x16, 0xC9}: vendorSamsung,
	{0x18, 0x19, 0xD6}: vendorSamsung,
	{0x18, 0x1E, 0xB0}: vendorSamsung,
	{0x18, 0x21, 0x95}: vendorSamsung,
	{0x18, 0x22, 0x7E}: vendorSamsung,
	{0x18, 0x26, 0x66}: vendorSamsung,
	{0x18, 0x3A, 0x2D}: vendorSamsung,
	{0x18, 0x3F, 0x47}: vendorSamsung,
	{0x18, 0x46, 0x17}: vendorSamsung,
	{0x18, 0x4E, 0x16}: vendorSamsung,
	{0x18, 0x4E, 0xCB}: vendorSamsung,
	{0x18, 0x54, 0xCF}: vendorSamsung,
	{0x18, 0x5B, 0xB3}: vendorSamsung,
	{0x18, 0x67, 0xB0}: vendorSamsung,
	{0x18, 0x83, 0x31}: vendorSamsung,
	{0x18, 0x89, 0x5B}: vendorSamsung,
	{0x18, 0xAB, 0x1D}: vendorSamsung,
	{0x18, 0xBF, 0xB3}: vendorSamsung,
	{0x18, 0xCE, 0x94}: vendorSamsung,
	{0x18, 0xE2, 0xC2}: vendorSamsung,
	{0x1C, 0x23, 0x2C}: vendorSamsung,
	{0x1C, 0x3A, 0xDE}: vendorSamsung,
	{0x1C, 0x5A, 0x3E}: vendorSamsung,
	{0x1C, 0x62, 0xB8}: vendorSamsung,
	{0x1C, 0x66, 0xAA}: vendorSamsung,
	{0x1C, 0xAF, 0x05}: vendorSamsung,
	{0x1C, 0xE5, 0x7F}: vendorSamsung,
	{0x1C, 0xE6, 0x1D}: vendorSamsung,
	{0x20, 0x13, 0xE0}: vendorSamsung,
	{0x20, 0x2D, 0x07}: vendorSamsung,
	{0x20, 0x32, 0x6C}: vendorSamsung,
	{0x20, 0x55, 0x31}: vendorSamsung,
	{0x20, 0x5E, 0xF7}: vendorSamsung,
	{0x20, 0x6E, 0x9C}: vendorSamsung,
	{0x20, 0xD3, 0x90}: vendorSamsung,
	{0x20, 0xD5, 0xBF}: vendorSamsung,
	{0x20, 0xDB, 0xAB}: vendorSamsung,
	{0x24, 0x4B, 0x03}: vendorSamsung,
	{0x24, 0x4B, 0x81}: vendorSamsung,
	{0x24, 0x5A, 0xB5}: vendorSamsung,
	{0x24, 0x68, 0xB0}: vendorSamsung,
	{0x24, 0x92, 0x0E}: vendorSamsung,
	{0x24, 0xC6, 0x96}: vendorSamsung,
	{0x24, 0xDB, 0xED}: vendorSamsung,
	{0x24, 0xF5, 0xAA}: vendorSamsung,
	{0x24, 0xFC, 0xE5}: vendorSamsung,
	{0x28, 0x02, 0xD8}: vendorSamsung,
	{0x28, 0x27, 0xBF}: vendorSamsung,
	{0x28, 0x39, 0x5E}: vendorSamsung,
	{0x28, 0x83, 0x35}: vendorSamsung,
	{0x28, 0x98, 0x7B}: vendorSamsung,
	{0x28, 0xBA, 0xB5}: vendorSamsung,
	{0x28, 0xCC, 0x01}: vendorSamsung,
	{0x2C, 0x15, 0xBF}: vendorSamsung,
	{0x2C, 0x40, 0x53}: vendorSamsung,
	{0x2C, 0x44, 0x01}: vendorSamsung,
	{0x2C, 0xAE, 0x2B}: vendorSamsung,
	{0x2C, 0xBA, 0xBA}: vendorSamsung,
	{0x30, 0x19, 0x66}: vendorSamsung,
	{0x30, 0x6A, 0x85}: vendorSamsung,
	{0x30, 0x74, 0x67}: vendorSamsung,
	{0x30, 0x96, 0xFB}: vendorSamsung,
	{0x30, 0xC7, 0xAE}: vendorSamsung,
	{0x30, 0xCB, 0xF8}: vendorSamsung,
	{0x30, 0xCD, 0xA7}: vendorSamsung,
	{0x30, 0xD5, 0x87}: vendorSamsung,
	{0x30, 0xD6, 0xC9}: vendorSamsung,
	{0x34, 0x14, 0x5F}: vendorSamsung,
	{0x34, 0x2D, 0x0D}: vendorSamsung,
	{0x34, 0x31, 0x11}: vendorSamsung,
	{0x34, 0x82, 0xC5}: vendorSamsung,
	{0x34, 0x8A, 0x7B}: vendorSamsung,
	{0x34, 0xAA, 0x8B}: vendorSamsung,
	{0x34, 0xBE, 0x00}: vendorSamsung,
	{0x34, 0xC3, 0xAC}: vendorSamsung,
	{0x38, 0x01, 0x95}: vendorSamsung,
	{0x38, 0x0A, 0x94}: vendorSamsung,
	{0x38, 0x0B, 0x40}: vendorSamsung,
	{0x38, 0x16, 0xD1}: vendorSamsung,
	{0x38, 0x2D, 0xD1}: vendorSamsung,
	{0x38, 0x2D, 0xE8}: vendorSamsung,
	{0x38, 0x68, 0xA4}: vendorSamsung,
	{0x38, 0x6A, 0x77}: vendorSamsung,
	{0x38, 0x94, 0x96}: vendorSamsung,
	{0x38, 0x9A, 0xF6}: vendorSamsung,
	{0x38, 0xD4, 0x0B}: vendorSamsung,
	{0x38, 0xEC, 0xE4}: vendorSamsung,
	{0x3C, 0x05, 0x18}: vendorSamsung,
	{0x3C, 0x19, 0x5E}: vendorSamsung,
	{0x3C, 0x20, 0xF6}: vendorSamsung,
	{0x3C, 0x57, 0x6C}: vendorSamsung,
	{0x3C, 0x5A, 0x37}: vendorSamsung,
	{0x3C, 0x62, 0x00}: vendorSamsung,
	{0x3C, 0x8B, 0xFE}: vendorSamsung,
	{0x3C, 0xA1, 0x0D}: vendorSamsung,
	{0x3C, 0xBB, 0xFD}: vendorSamsung,
	{0x3C, 0xDC, 0xBC}: vendorSamsung,
	{0x3C, 0xF7, 0xA4}: vendorSamsung,
	{0x40, 0x16, 0x3B}: vendorSamsung,
	{0x40, 0xD3, 0xAE}: vendorSamsung,
	{0x44, 0x4E, 0x1A}: vendorSamsung,
	{0x44, 0x5C, 0xE9}: vendorSamsung,
	{0x44, 0x6D, 0x6C}: vendorSamsung,
	{0x44, 0x78, 0x3E}: vendorSamsung,
	{0x44, 0x8F, 0x17}: vendorSamsung,
	{0x44, 0xF4, 0x59}: vendorSamsung,
	{0x48, 0x13, 0x7E}: vendorSamsung,
	{0x48, 0x27, 0xEA}: vendorSamsung,
	{0x48, 0x44, 0xF7}: vendorSamsung,
	{0x48, 0x49, 0xC7}: vendorSamsung,
	{0x48, 0x51, 0x69}: vendorSamsung,
	{0x48, 0x79, 0x4D}: vendorSamsung,
	{0x48, 0x9D, 0xD1}: vendorSamsung,
	{0x48, 0xC7, 0x96}: vendorSamsung,
	{0x4C, 0x3C, 0x16}: vendorSamsung,
	{0x4C, 0xA5, 0x6D}: vendorSamsung,
	{0x4C, 0xBC, 0xA5}: vendorSamsung,
	{0x4C, 0xC9, 0x5E}: vendorSamsung,
	{0x4C, 0xDD, 0x31}: vendorSamsung,
	{0x50, 0x01, 0xBB}: vendorSamsung,
	{0x50, 0x32, 0x75}: vendorSamsung,
	{0x50, 0x3D, 0xA1}: vendorSamsung,
	{0x50, 0x49, 0xB0}: vendorSamsung,
	{0x50, 0x50, 0xA4}: vendorSamsung,
	{0x50, 0x56, 0xBF}: vendorSamsung,
	{0x50, 0x77, 0x05}: vendorSamsung,
	{0x50, 0x85, 0x69}: vendorSamsung,
	{0x50, 0x92, 0xB9}: vendorSamsung,
	{0x50, 0x9E, 0xA7}: vendorSamsung,
	{0x50, 0xA4, 0xC8}: vendorSamsung,
	{0x50, 0xB7, 0xC3}: vendorSamsung,
	{0x50, 0xC8, 0xE5}: vendorSamsung,
	{0x50, 0xF0, 0xD3}: vendorSamsung,
	{0x50, 0xF5, 0x20}: vendorSamsung,
	{0x50, 0xFC, 0x9F}: vendorSamsung,
	{0x54, 0x21, 0x9D}: vendorSamsung,
	{0x54, 0x3A, 0xD6}: vendorSamsung,
	{0x54, 0x40, 0xAD}: vendorSamsung,
	{0x54, 0x44, 0xA3}: vendorSamsung,
	{0x54, 0x92, 0xBE}: vendorSamsung,
	{0x54, 0x9B, 0x12}: vendorSamsung,
	{0x54, 0xB8, 0x02}: vendorSamsung,
	{0x54, 0xBD, 0x79}: vendorSamsung,
	{0x54, 0xD1, 0x7D}: vendorSamsung,
	{0x54, 0xF2, 0x01}: vendorSamsung,
	{0x54, 0xFA, 0x3E}: vendorSamsung,
	{0x54, 0xFC, 0xF0}: vendorSamsung,
	{0x58, 0xA6, 0x39}: vendorSamsung,
	{0x58, 0xB1, 0x0F}: vendorSamsung,
	{0x58, 0xC3, 0x8B}: vendorSamsung,
	{0x58, 0xC5, 0xCB}: vendorSamsung,
	{0x5C, 0x10, 0xC5}: vendorSamsung,
	{0x5C, 0x2E, 0x59}: vendorSamsung,
	{0x5C, 0x3C, 0x27}: vendorSamsung,
	{0x5C, 0x49, 0x7D}: vendorSamsung,
	{0x5C, 0x51, 0x81}: vendorSamsung,
	{0x5C, 0x86, 0x5C}: vendorSamsung,
	{0x5C, 0x99, 0x60}: vendorSamsung,
	{0x5C, 0xC1, 0xD7}: vendorSamsung,
	{0x5C, 0xCB, 0x99}: vendorSamsung,
	{0x5C, 0xE8, 0xEB}: vendorSamsung,
	{0x5C, 0xF6, 0xDC}: vendorSamsung,
	{0x60, 0x3A, 0xAF}: vendorSamsung,
	{0x60, 0x68, 0x4E}: vendorSamsung,
	{0x60, 0x6B, 0xBD}: vendorSamsung,
	{0x60, 0x77, 0xE2}: vendorSamsung,
	{0x60, 0x8E, 0x08}: vendorSamsung,
	{0x60, 0x8F, 0x5C}: vendorSamsung,
	{0x60, 0xA1, 0x0A}: vendorSamsung,
	{0x60, 0xA4, 0xD0}: vendorSamsung,
	{0x60, 0xAF, 0x6D}: vendorSamsung,
	{0x60, 0xC5, 0xAD}: vendorSamsung,
	{0x60, 0xD0, 0xA9}: vendorSamsung,
	{0x64, 0x03, 0x7F}: vendorSamsung,
	{0x64, 0x07, 0xF6}: vendorSamsung,
	{0x64, 0x1C, 0xAE}: vendorSamsung,
	{0x64, 0x1C, 0xB0}: vendorSamsung,
	{0x64, 0x6C, 0xB2}: vendorSamsung,
	{0x64, 0x77, 0x91}: vendorSamsung,
	{0x64, 0x7B, 0xCE}: vendorSamsung,
	{0x64, 0x89, 0xF1}: vendorSamsung,
	{0x64, 0xB3, 0x10}: vendorSamsung,
	{0x64, 0xB8, 0x53}: vendorSamsung,
	{0x64, 0xE7, 0xD8}: vendorSamsung,
	{0x68, 0x05, 0x71}: vendorSamsung,
	{0x68, 0x27, 0x37}: vendorSamsung,
	{0x68, 0x48, 0x98}: vendorSamsung,
	{0x68, 0x5A, 0xCF}: vendorSamsung,
	{0x68, 0x72, 0xC3}: vendorSamsung,
	{0x68, 0x7D, 0x6B}: vendorSamsung,
	{0x68, 0xBF, 0xC4}: vendorSamsung,
	{0x68, 0xE7, 0xC2}: vendorSamsung,
	{0x68, 0xEB, 0xAE}: vendorSamsung,
	{0x6C, 0x00, 0x6B}: vendorSamsung,
	{0x6C, 0x2F, 0x2C}: vendorSamsung,
	{0x6C, 0x2F, 0x8A}: vendorSamsung,
	{0x6C, 0x83, 0x36}: vendorSamsung,
	{0x6C, 0xB7, 0xF4}: vendorSamsung,
	{0x6C, 0xDD, 0xBC}: vendorSamsung,
	{0x6C, 0xF3, 0x73}: vendorSamsung,
	{0x70, 0x09, 0x71}: vendorSamsung,
	{0x70, 0x1F, 0x3C}: vendorSamsung,
	{0x70, 0x28, 0x8B}: vendorSamsung,
	{0x70, 0x2A, 0xD5}: vendorSamsung,
	{0x70, 0x5A, 0xAC}: vendorSamsung,
	{0x70, 0xB1, 0x3D}: vendorSamsung,
	{0x70, 0xCE, 0x8C}: vendorSamsung,
	{0x70, 0xF9, 0x27}: vendorSamsung,
	{0x70, 0xFD, 0x46}: vendorSamsung,
	{0x74, 0x45, 0x8A}: vendorSamsung,
	{0x74, 0x5F, 0x00}: vendorSamsung,
	{0x74, 0x9E, 0xF5}: vendorSamsung,
	{0x74, 0xEB, 0x80}: vendorSamsung,
	{0x78, 0x00, 0x9E}: vendorSamsung,
	{0x78, 0x1F, 0xDB}: vendorSamsung,
	{0x78, 0x23, 0x27}: vendorSamsung,
	{0x78, 0x25, 0xAD}: vendorSamsung,
	{0x78, 0x37, 0x16}: vendorSamsung,
	{0x78, 0x40, 0xE4}: vendorSamsung,
	{0x78, 0x46, 0xD4}: vendorSamsung,
	{0x78, 0x47, 0x1D}: vendorSamsung,
	{0x78, 0x52, 0x1A}: vendorSamsung,
	{0x78, 0x59, 0x5E}: vendorSamsung,
	{0x78, 0x9E, 0xD0}: vendorSamsung,
	{0x78, 0xA8, 0x73}: vendorSamsung,
	{0x78, 0xAB, 0xBB}: vendorSamsung,
	{0x78, 0xBD, 0xBC}: vendorSamsung,
	{0x78, 0xC3, 0xE9}: vendorSamsung,
	{0x78, 0xF7, 0xBE}: vendorSamsung,
	{0x7C, 0x0A, 0x3F}: vendorSamsung,
	{0x7C, 0x0B, 0xC6}: vendorSamsung,
	{0x7C, 0x1C, 0x68}: vendorSamsung,
	{0x7C, 0x23, 0x02}: vendorSamsung,
	{0x7C, 0x2E, 0xDD}: vendorSamsung,
	{0x7C, 0x38, 0xAD}: vendorSamsung,
	{0x7C, 0x64, 0x56}: vendorSamsung,
	{0x7C, 0x78, 0x7E}: vendorSamsung,
	{0x7C, 0x89, 0x56}: vendorSamsung,
	{0x7C, 0x8B, 0xB5}: vendorSamsung,
	{0x7C, 0x91, 0x22}: vendorSamsung,
	{0x7C, 0xF8, 0x54}: vendorSamsung,
	{0x7C, 0xF9, 0x0E}: vendorSamsung,
	{0x80, 0x18, 0xA7}: vendorSamsung,
	{0x80, 0x20, 0xFD}: vendorSamsung,
	{0x80, 0x31, 0xF0}: vendorSamsung,
	{0x80, 0x47, 0x86}: vendorSamsung,
	{0x80, 0x4E, 0x70}: vendorSamsung,
	{0x80, 0x4E, 0x81}: vendorSamsung,
	{0x80, 0x57, 0x19}: vendorSamsung,
	{0x80, 0x65, 0x6D}: vendorSamsung,
	{0x80, 0x7B, 0x3E}: vendorSamsung,
	{0x80, 0x86, 0xD9}: vendorSamsung,
	{0x80, 0x8A, 0xBD}: vendorSamsung,
	{0x80, 0x9F, 0xF5}: vendorSamsung,
	{0x80, 0xCE, 0xB9}: vendorSamsung,
	{0x84, 0x11, 0x9E}: vendorSamsung,
	{0x84, 0x25, 0x19}: vendorSamsung,
	{0x84, 0x25, 0xDB}: vendorSamsung,
	{0x84, 0x2E, 0x27}: vendorSamsung,
	{0x84, 0x37, 0xD5}: vendorSamsung,
	{0x84, 0x51, 0x81}: vendorSamsung,
	{0x84, 0x55, 0xA5}: vendorSamsung,
	{0x84, 0x98, 0x66}: vendorSamsung,
	{0x84, 0xA4, 0x66}: vendorSamsung,
	{0x84, 0xB5, 0x41}: vendorSamsung,
	{0x84, 0xC0, 0xEF}: vendorSamsung,
	{0x88, 0x29, 0x9C}: vendorSamsung,
	{0x88, 0x75, 0x98}: vendorSamsung,
	{0x88, 0x83, 0x22}: vendorSamsung,
	{0x88, 0x9B, 0x39}: vendorSamsung,
	{0x88, 0x9F, 0x6F}: vendorSamsung,
	{0x88, 0xA3, 0x03}: vendorSamsung,
	{0x88, 0xAD, 0xD2}: vendorSamsung,
	{0x88, 0xBD, 0x45}: vendorSamsung,
	{0x8C, 0x1A, 0xBF}: vendorSamsung,
	{0x8C, 0x71, 0xF8}: vendorSamsung,
	{0x8C, 0x77, 0x12}: vendorSamsung,
	{0x8C, 0x79, 0xF5}: vendorSamsung,
	{0x8C, 0x83, 0xE1}: vendorSamsung,
	{0x8C, 0xB0, 0xE9}: vendorSamsung,
	{0x8C, 0xBF, 0xA6}: vendorSamsung,
	{0x8C, 0xC8, 0xCD}: vendorSamsung,
	{0x8C, 0xDE, 0xE6}: vendorSamsung,
	{0x8C, 0xE5, 0xC0}: vendorSamsung,
	{0x8C, 0xEA, 0x48}: vendorSamsung,
	{0x90, 0x00, 0xDB}: vendorSamsung,
	{0x90, 0x06, 0x28}: vendorSamsung,
	{0x90, 0x63, 0x3B}: vendorSamsung,
	{0x90, 0x81, 0x75}: vendorSamsung,
	{0x90, 0x97, 0xF3}: vendorSamsung,
	{0x90, 0xB1, 0x44}: vendorSamsung,
	{0x90, 0xB6, 0x22}: vendorSamsung,
	{0x90, 0xEE, 0xC7}: vendorSamsung,
	{0x90, 0xF1, 0xAA}: vendorSamsung,
	{0x94, 0x01, 0xC2}: vendorSamsung,
	{0x94, 0x2D, 0xDC}: vendorSamsung,
	{0x94, 0x35, 0x0A}: vendorSamsung,
	{0x94, 0x51, 0x03}: vendorSamsung,
	{0x94, 0x63, 0xD1}: vendorSamsung,
	{0x94, 0x76, 0xB7}: vendorSamsung,
	{0x94, 0x7B, 0xE7}: vendorSamsung,
	{0x94, 0x8B, 0xC1}: vendorSamsung,
	{0x94, 0xB1, 0x0A}: vendorSamsung,
	{0x94, 0xD7, 0x71}: vendorSamsung,
	{0x98, 0x06, 0x3C}: vendorSamsung,
	{0x98, 0x1D, 0xFA}: vendorSamsung,
	{0x98, 0x2D, 0x68}: vendorSamsung,
	{0x98, 0x39, 0x8E}: vendorSamsung,
	{0x98, 0x52, 0xB1}: vendorSamsung,
	{0x98, 0x80, 0xEE}: vendorSamsung,
	{0x98, 0x83, 0x89}: vendorSamsung,
	{0x98, 0xB8, 0xBC}: vendorSamsung,
	{0x9C, 0x02, 0x98}: vendorSamsung,
	{0x9C, 0x2A, 0x83}: vendorSamsung,
	{0x9C, 0x3A, 0xAF}: vendorSamsung,
	{0x9C, 0x5F, 0xB0}: vendorSamsung,
	{0x9C, 0x65, 0xB0}: vendorSamsung,
	{0x9C, 0x8C, 0x6E}: vendorSamsung,
	{0x9C, 0xA5, 0x13}: vendorSamsung,
	{0x9C, 0xD3, 0x5B}: vendorSamsung,
	{0x9C, 0xE0, 0x63}: vendorSamsung,
	{0x9C, 0xE6, 0xE7}: vendorSamsung,
	{0xA0, 0x07, 0x98}: vendorSamsung,
	{0xA0, 0x10, 0x81}: vendorSamsung,
	{0xA0, 0x21, 0x95}: vendorSamsung,
	{0xA0, 0x27, 0xB6}: vendorSamsung,
	{0xA0, 0x60, 0x90}: vendorSamsung,
	{0xA0, 0x75, 0x91}: vendorSamsung,
	{0xA0, 0x82, 0x1F}: vendorSamsung,
	{0xA0, 0xAC, 0x69}: vendorSamsung,
	{0xA0, 0xB4, 0xA5}: vendorSamsung,
	{0xA0, 0xCB, 0xFD}: vendorSamsung,
	{0xA0, 0xD0, 0x5B}: vendorSamsung,
	{0xA0, 0xD7, 0x22}: vendorSamsung,
	{0xA0, 0xD7, 0xF3}: vendorSamsung,
	{0xA4, 0x07, 0xB6}: vendorSamsung,
	{0xA4, 0x30, 0x7A}: vendorSamsung,
	{0xA4, 0x6C, 0xF1}: vendorSamsung,
	{0xA4, 0x84, 0x31}: vendorSamsung,
	{0xA4, 0x9A, 0x58}: vendorSamsung,
	{0xA4, 0xC6, 0x9A}: vendorSamsung,
	{0xA4, 0xD9, 0x90}: vendorSamsung,
	{0xA4, 0xEB, 0xD3}: vendorSamsung,
	{0xA8, 0x06, 0x00}: vendorSamsung,
	{0xA8, 0x16, 0xD0}: vendorSamsung,
	{0xA8, 0x2B, 0xB9}: vendorSamsung,
	{0xA8, 0x30, 0xBC}: vendorSamsung,
	{0xA8, 0x34, 0x6A}: vendorSamsung,
	{0xA8, 0x51, 0x5B}: vendorSamsung,
	{0xA8, 0x76, 0x50}: vendorSamsung,
	{0xA8, 0x7C, 0x01}: vendorSamsung,
	{0xA8, 0x81, 0x95}: vendorSamsung,
	{0xA8, 0x87, 0xB3}: vendorSamsung,
	{0xA8, 0x9F, 0xBA}: vendorSamsung,
	{0xA8, 0xF2, 0x74}: vendorSamsung,
	{0xAC, 0x1E, 0x92}: vendorSamsung,
	{0xAC, 0x36, 0x13}: vendorSamsung,
	{0xAC, 0x5A, 0x14}: vendorSamsung,
	{0xAC, 0x6C, 0x90}: vendorSamsung,
	{0xAC, 0xAF, 0xB9}: vendorSamsung,
	{0xAC, 0xC3, 0x3A}: vendorSamsung,
	{0xAC, 0xEE, 0x9E}: vendorSamsung,
	{0xB0, 0x47, 0xBF}: vendorSamsung,
	{0xB0, 0x6F, 0xE0}: vendorSamsung,
	{0xB0, 0xC4, 0xE7}: vendorSamsung,
	{0xB0, 0xC5, 0x59}: vendorSamsung,
	{0xB0, 0xD0, 0x9C}: vendorSamsung,
	{0xB0, 0xDF, 0x3A}: vendorSamsung,
	{0xB0, 0xEC, 0x71}: vendorSamsung,
	{0xB4, 0x1A, 0x1D}: vendorSamsung,
	{0xB4, 0x3A, 0x28}: vendorSamsung,
	{0xB4, 0x62, 0x93}: vendorSamsung,
	{0xB4, 0x74, 0x43}: vendorSamsung,
	{0xB4, 0x9D, 0x02}: vendorSamsung,
	{0xB4, 0xBF, 0xF6}: vendorSamsung,
	{0xB4, 0xCE, 0x40}: vendorSamsung,
	{0xB4, 0xEF, 0x39}: vendorSamsung,
	{0xB8, 0x57, 0xD8}: vendorSamsung,
	{0xB8, 0x5A, 0x73}: vendorSamsung,
	{0xB8, 0x5E, 0x7B}: vendorSamsung,
	{0xB8, 0x6C, 0xE8}: vendorSamsung,
	{0xB8, 0xBB, 0xAF}: vendorSamsung,
	{0xB8, 0xBC, 0x5B}: vendorSamsung,
	{0xB8, 0xC6, 0x8E}: vendorSamsung,
	{0xB8, 0xD9, 0xCE}: vendorSamsung,
	{0xBC, 0x14, 0x85}: vendorSamsung,
	{0xBC, 0x20, 0xA4}: vendorSamsung,
	{0xBC, 0x44, 0x86}: vendorSamsung,
	{0xBC, 0x47, 0x60}: vendorSamsung,
	{0xBC, 0x54, 0x51}: vendorSamsung,
	{0xBC, 0x72, 0xB1}: vendorSamsung,
	{0xBC, 0x76, 0x5E}: vendorSamsung,
	{0xBC, 0x79, 0xAD}: vendorSamsung,
	{0xBC, 0x7A, 0xBF}: vendorSamsung,
	{0xBC, 0x7E, 0x8B}: vendorSamsung,
	{0xBC, 0x85, 0x1F}: vendorSamsung,
	{0xBC, 0xA5, 0x8B}: vendorSamsung,
	{0xBC, 0xB1, 0xF3}: vendorSamsung,
	{0xBC, 0xD1, 0x1F}: vendorSamsung,
	{0xBC, 0xE6, 0x3F}: vendorSamsung,
	{0xC0, 0x11, 0x73}: vendorSamsung,
	{0xC0, 0x17, 0x4D}: vendorSamsung,
	{0xC0, 0x23, 0x8D}: vendorSamsung,
	{0xC0, 0x3D, 0x03}: vendorSamsung,
	{0xC0, 0x48, 0xE6}: vendorSamsung,
	{0xC0, 0x65, 0x99}: vendorSamsung,
	{0xC0, 0x87, 0xEB}: vendorSamsung,
	{0xC0, 0x89, 0x97}: vendorSamsung,
	{0xC0, 0xBD, 0xC8}: vendorSamsung,
	{0xC0, 0xD2, 0xDD}: vendorSamsung,
	{0xC0, 0xD3, 0xC0}: vendorSamsung,
	{0xC0, 0xDC, 0xDA}: vendorSamsung,
	{0xC4, 0x18, 0xE9}: vendorSamsung,
	{0xC4, 0x42, 0x02}: vendorSamsung,
	{0xC4, 0x50, 0x06}: vendorSamsung,
	{0xC4, 0x57, 0x6E}: vendorSamsung,
	{0xC4, 0x5D, 0x83}: vendorSamsung,
	{0xC4, 0x62, 0xEA}: vendorSamsung,
	{0xC4, 0x73, 0x1E}: vendorSamsung,
	{0xC4, 0x88, 0xE5}: vendorSamsung,
	{0xC4, 0x93, 0xD9}: vendorSamsung,
	{0xC4, 0xAE, 0x12}: vendorSamsung,
	{0xC8, 0x14, 0x79}: vendorSamsung,
	{0xC8, 0x19, 0xF7}: vendorSamsung,
	{0xC8, 0x38, 0x70}: vendorSamsung,
	{0xC8, 0x41, 0x8A}: vendorSamsung,
	{0xC8, 0x51, 0x42}: vendorSamsung,
	{0xC8, 0x7E, 0x75}: vendorSamsung,
	{0xC8, 0xA8, 0x23}: vendorSamsung,
	{0xC8, 0xBD, 0x4D}: vendorSamsung,
	{0xC8, 0xBD, 0x69}: vendorSamsung,
	{0xC8, 0xD7, 0xB0}: vendorSamsung,
	{0xCC, 0x05, 0x1B}: vendorSamsung,
	{0xCC, 0x07, 0xAB}: vendorSamsung,
	{0xCC, 0x21, 0x19}: vendorSamsung,
	{0xCC, 0x46, 0x4E}: vendorSamsung,
	{0xCC, 0x6E, 0xA4}: vendorSamsung,
	{0xCC, 0xB1, 0x1A}: vendorSamsung,
	{0xCC, 0xF9, 0xE8}: vendorSamsung,
	{0xCC, 0xFE, 0x3C}: vendorSamsung,
	{0xD0, 0x03, 0xDF}: vendorSamsung,
	{0xD0, 0x17, 0x6A}: vendorSamsung,
	{0xD0, 0x1B, 0x49}: vendorSamsung,
	{0xD0, 0x31, 0x69}: vendorSamsung,
	{0xD0, 0x59, 0xE4}: vendorSamsung,
	{0xD0, 0x66, 0x7B}: vendorSamsung,
	{0xD0, 0x7F, 0xA0}: vendorSamsung,
	{0xD0, 0x87, 0xE2}: vendorSamsung,
	{0xD0, 0xB1, 0x28}: vendorSamsung,
	{0xD0, 0xC1, 0xB1}: vendorSamsung,
	{0xD0, 0xC2, 0x4E}: vendorSamsung,
	{0xD0, 0xD0, 0x03}: vendorSamsung,
	{0xD0, 0xDF, 0xC7}: vendorSamsung,
	{0xD0, 0xFC, 0xCC}: vendorSamsung,
	{0xD4, 0x11, 0xA3}: vendorSamsung,
	{0xD4, 0x7A, 0xE2}: vendorSamsung,
	{0xD4, 0x87, 0xD8}: vendorSamsung,
	{0xD4, 0x88, 0x90}: vendorSamsung,
	{0xD4, 0x8A, 0x39}: vendorSamsung,
	{0xD4, 0x9D, 0xC0}: vendorSamsung,
	{0xD4, 0xAE, 0x05}: vendorSamsung,
	{0xD4, 0xE6, 0xB7}: vendorSamsung,
	{0xD4, 0xE8, 0xB2}: vendorSamsung,
	{0xD8, 0x08, 0x31}: vendorSamsung,
	{0xD8, 0x0B, 0x9A}: vendorSamsung,
	{0xD8, 0x31, 0xCF}: vendorSamsung,
	{0xD8, 0x55, 0x75}: vendorSamsung,
	{0xD8, 0x57, 0xEF}: vendorSamsung,
	{0xD8, 0x5B, 0x2A}: vendorSamsung,
	{0xD8, 0x68, 0xC3}: vendorSamsung,
	{0xD8, 0x90, 0xE8}: vendorSamsung,
	{0xD8, 0xA3, 0x5C}: vendorSamsung,
	{0xD8, 0xC4, 0xE9}: vendorSamsung,
	{0xD8, 0xE0, 0xE1}: vendorSamsung,
	{0xDC, 0x44, 0xB6}: vendorSamsung,
	{0xDC, 0x66, 0x72}: vendorSamsung,
	{0xDC, 0x74, 0xA8}: vendorSamsung,
	{0xDC, 0x89, 0x83}: vendorSamsung,
	{0xDC, 0xCF, 0x96}: vendorSamsung,
	{0xDC, 0xDC, 0xE2}: vendorSamsung,
	{0xDC, 0xF7, 0x56}: vendorSamsung,
	{0xE0, 0x99, 0x71}: vendorSamsung,
	{0xE0, 0x9D, 0x13}: vendorSamsung,
	{0xE0, 0xAA, 0x96}: vendorSamsung,
	{0xE0, 0xC3, 0x77}: vendorSamsung,
	{0xE0, 0xCB, 0xEE}: vendorSamsung,
	{0xE0, 0xD0, 0x83}: vendorSamsung,
	{0xE0, 0xDB, 0x10}: vendorSamsung,
	{0xE4, 0x12, 0x1D}: vendorSamsung,
	{0xE4, 0x32, 0xCB}: vendorSamsung,
	{0xE4, 0x40, 0xE2}: vendorSamsung,
	{0xE4, 0x58, 0xB8}: vendorSamsung,
	{0xE4, 0x58, 0xE7}: vendorSamsung,
	{0xE4, 0x5D, 0x75}: vendorSamsung,
	{0xE4, 0x7C, 0xF9}: vendorSamsung,
	{0xE4, 0x7D, 0xBD}: vendorSamsung,
	{0xE4, 0x92, 0xFB}: vendorSamsung,
	{0xE4, 0xB0, 0x21}: vendorSamsung,
	{0xE4, 0xE0, 0xC5}: vendorSamsung,
	{0xE4, 0xF3, 0xC4}: vendorSamsung,
	{0xE4, 0xF8, 0xEF}: vendorSamsung,
	{0xE4, 0xFA, 0xED}: vendorSamsung,
	{0xE8, 0x03, 0x9A}: vendorSamsung,
	{0xE8, 0x11, 0x32}: vendorSamsung,
	{0xE8, 0x3A, 0x12}: vendorSamsung,
	{0xE8, 0x4E, 0x84}: vendorSamsung,
	{0xE8, 0x6D, 0xCB}: vendorSamsung,
	{0xE8, 0x7F, 0x6B}: vendorSamsung,
	{0xE8, 0x93, 0x09}: vendorSamsung,
	{0xE8, 0xB4, 0xC8}: vendorSamsung,
	{0xE8, 0xE5, 0xD6}: vendorSamsung,
	{0xEC, 0x10, 0x7B}: vendorSamsung,
	{0xEC, 0x7C, 0xB6}: vendorSamsung,
	{0xEC, 0xAA, 0x25}: vendorSamsung,
	{0xEC, 0xE0, 0x9B}: vendorSamsung,
	{0xF0, 0x08, 0xF1}: vendorSamsung,
	{0xF0, 0x39, 0x65}: vendorSamsung,
	{0xF0, 0x5A, 0x09}: vendorSamsung,
	{0xF0, 0x5B, 0x7B}: vendorSamsung,
	{0xF0, 0x6B, 0xCA}: vendorSamsung,
	{0xF0, 0x70, 0x4F}: vendorSamsung,
	{0xF0, 0x72, 0x8C}: vendorSamsung,
	{0xF0, 0x8A, 0x76}: vendorSamsung,
	{0xF0, 0xCD, 0x31}: vendorSamsung,
	{0xF0, 0xE7, 0x7E}: vendorSamsung,
	{0xF0, 0xEE, 0x10}: vendorSamsung,
	{0xF0, 0xF5, 0x64}: vendorSamsung,
	{0xF4, 0x0E, 0x22}: vendorSamsung,
	{0xF4, 0x42, 0x8F}: vendorSamsung,
	{0xF4, 0x71, 0x90}: vendorSamsung,
	{0xF4, 0x7B, 0x5E}: vendorSamsung,
	{0xF4, 0x7D, 0xEF}: vendorSamsung,
	{0xF4, 0x9F, 0x54}: vendorSamsung,
	{0xF4, 0xC2, 0x48}: vendorSamsung,
	{0xF4, 0xD9, 0xFB}: vendorSamsung,
	{0xF4, 0xFE, 0xFB}: vendorSamsung,
	{0xF8, 0x3F, 0x51}: vendorSamsung,
	{0xF8, 0x77, 0xB8}: vendorSamsung,
	{0xF8, 0x84, 0xF2}: vendorSamsung,
	{0xF8, 0x8F, 0x07}: vendorSamsung,
	{0xF8, 0xD0, 0xBD}: vendorSamsung,
	{0xF8, 0xE6, 0x1A}: vendorSamsung,
	{0xF8, 0xF1, 0xE6}: vendorSamsung,
	{0xFA, 0x63, 0xE1}: vendorSamsung,
	{0xFC, 0x03, 0x9F}: vendorSamsung,
	{0xFC, 0x19, 0x10}: vendorSamsung,
	{0xFC, 0x42, 0x03}: vendorSamsung,
	{0xFC, 0x64, 0x3A}: vendorSamsung,
	{0xFC, 0x8F, 0x90}: vendorSamsung,
	{0xFC, 0xA1, 0x3E}: vendorSamsung,
	{0xFC, 0xA6, 0x21}: vendorSamsung,
	{0xFC, 0xAA, 0xB6}: vendorSamsung,
	{0xFC, 0xC7, 0x34}: vendorSamsung,
	{0xFC, 0xDE, 0x90}: vendorSamsung,
	{0xFC, 0xF1, 0x36}: vendorSamsung,
}
