/// <reference types="node" />
import { NativeResource } from "./native_resource";
import { TlsVersion, SocketType, SocketDomain } from '../common/io';
import { Readable } from 'stream';
export { TlsVersion, SocketType, SocketDomain } from '../common/io';
/**
 * Convert a native error code into a human-readable string
 * @param error_code - An error code returned from a native API call, or delivered
 * via callback.
 * @returns Long-form description of the error
 * @see CrtError
 *
 * nodejs only.
 *
 * @category System
 */
export declare function error_code_to_string(error_code: number): string;
/**
 * Convert a native error code into a human-readable identifier
 * @param error_code - An error code returned from a native API call, or delivered
 * via callback.
 * @return error name as a string
 * @see CrtError
 *
 * nodejs only.
 *
 * @category System
 */
export declare function error_code_to_name(error_code: number): string;
/**
 * The amount of detail that will be logged
 * @category Logging
 */
export declare enum LogLevel {
    /** No logging whatsoever. Equivalent to never calling {@link enable_logging}. */
    NONE = 0,
    /** Only fatals. In practice, this will not do much, as the process will log and then crash (intentionally) if a fatal condition occurs */
    FATAL = 1,
    /** Only errors */
    ERROR = 2,
    /** Only warnings and errors */
    WARN = 3,
    /** Information about connection/stream creation/destruction events */
    INFO = 4,
    /** Enough information to debug the chain of events a given network connection encounters */
    DEBUG = 5,
    /** Everything. Only use this if you really need to know EVERY single call */
    TRACE = 6
}
/**
 * Enables logging of the native AWS CRT libraries.
 * @param level - The logging level to filter to. It is not possible to log less than WARN.
 *
 * nodejs only.
 * @category Logging
 */
export declare function enable_logging(level: LogLevel): void;
/**
 * Returns true if ALPN is available on this platform natively
 * @return true if ALPN is supported natively, false otherwise
 *
 * nodejs only.
 * @category TLS
*/
export declare function is_alpn_available(): boolean;
/**
 * Wraps a ```Readable``` for reading by native code, used to stream
 *  data into the AWS CRT libraries.
 *
 * nodejs only.
 * @category IO
 */
export declare class InputStream extends NativeResource {
    private source;
    constructor(source: Readable);
}
/**
 * Represents native resources required to bootstrap a client connection
 * Things like a host resolver, event loop group, etc. There should only need
 * to be 1 of these per application, in most cases.
 *
 * nodejs only.
 * @category IO
 */
export declare class ClientBootstrap extends NativeResource {
    constructor();
}
/**
 * Standard Berkeley socket style options.
 *
 * nodejs only.
 * @category Network
*/
export declare class SocketOptions extends NativeResource {
    constructor(type?: SocketType, domain?: SocketDomain, connect_timeout_ms?: number, keepalive?: boolean, keep_alive_interval_sec?: number, keep_alive_timeout_sec?: number, keep_alive_max_failed_probes?: number);
}
/**
 * Options for creating a {@link ClientTlsContext} or {@link ServerTlsContext}.
 *
 * nodejs only.
 * @category TLS
 */
export declare class TlsContextOptions {
    /** Minimum version of TLS to support. Uses OS/system default if unspecified. */
    min_tls_version: TlsVersion;
    /** Path to a single file with all trust anchors in it, in PEM format */
    ca_filepath?: string;
    /** Path to directory containing trust anchors. Only used on Unix-style systems. */
    ca_dirpath?: string;
    /** String with all trust anchors in it, in PEM format */
    certificate_authority?: string;
    /** List of ALPN protocols to be used on platforms which support ALPN */
    alpn_list: string[];
    /** Path to certificate, in PEM format */
    certificate_filepath?: string;
    /** Certificate, in PEM format */
    certificate?: string;
    /** Path to private key, in PEM format */
    private_key_filepath?: string;
    /** Private key, in PEM format */
    private_key?: string;
    /** Path to certificate, in PKCS#12 format. Currently, only supported on OSX */
    pkcs12_filepath?: string;
    /** Password for PKCS#12. Currently, only supported on OSX. */
    pkcs12_password?: string;
    /** PKCS#11 options. Currently, only supported on Unix */
    pkcs11_options?: TlsContextOptions.Pkcs11Options;
    /**
     * In client mode, this turns off x.509 validation. Don't do this unless you are testing.
     * It is much better to just override the default trust store and pass the self-signed
     * certificate as the ca_file argument.
     *
     * In server mode (ServerTlsContext), this defaults to false. If you want to enforce mutual TLS on the server,
     * set this to true.
     */
    verify_peer: boolean;
    /**
     * Overrides the default system trust store.
     * @param ca_dirpath - Only used on Unix-style systems where all trust anchors are
     * stored in a directory (e.g. /etc/ssl/certs).
     * @param ca_filepath - Single file containing all trust CAs, in PEM format
     */
    override_default_trust_store_from_path(ca_dirpath?: string, ca_filepath?: string): void;
    /**
     * Overrides the default system trust store.
     * @param certificate_authority - String containing all trust CAs, in PEM format
     */
    override_default_trust_store(certificate_authority: string): void;
    /**
     * Create options configured for mutual TLS in client mode,
     * with client certificate and private key provided as in-memory strings.
     * @param certificate - Client certificate file contents, in PEM format
     * @param private_key - Client private key file contents, in PEM format
     *
     * @returns newly configured TlsContextOptions object
     */
    static create_client_with_mtls(certificate: string, private_key: string): TlsContextOptions;
    /**
     * Create options configured for mutual TLS in client mode,
     * with client certificate and private key provided via filepath.
     * @param certificate_filepath - Path to client certificate, in PEM format
     * @param private_key_filepath - Path to private key, in PEM format
     *
     * @returns newly configured TlsContextOptions object
     */
    static create_client_with_mtls_from_path(certificate_filepath: string, private_key_filepath: string): TlsContextOptions;
    /**
     * Create options for mutual TLS in client mode,
     * with client certificate and private key bundled in a single PKCS#12 file.
     * @param pkcs12_filepath - Path to PKCS#12 file containing client certificate and private key.
     * @param pkcs12_password - PKCS#12 password
     *
     * @returns newly configured TlsContextOptions object
    */
    static create_client_with_mtls_pkcs12_from_path(pkcs12_filepath: string, pkcs12_password: string): TlsContextOptions;
    /**
     * @deprecated Renamed [[create_client_with_mtls_pkcs12_from_path]]
     */
    static create_client_with_mtls_pkcs_from_path(pkcs12_filepath: string, pkcs12_password: string): TlsContextOptions;
    /**
     * Create options configured for mutual TLS in client mode,
     * using a PKCS#11 library for private key operations.
     *
     * NOTE: This configuration only works on Unix devices.
     *
     * @param options - PKCS#11 options
     *
     * @returns newly configured TlsContextOptions object
     */
    static create_client_with_mtls_pkcs11(options: TlsContextOptions.Pkcs11Options): TlsContextOptions;
    /**
     * Creates TLS context with peer verification disabled, along with a certificate and private key
     * @param certificate_filepath - Path to certificate, in PEM format
     * @param private_key_filepath - Path to private key, in PEM format
     *
     * @returns newly configured TlsContextOptions object
     */
    static create_server_with_mtls_from_path(certificate_filepath: string, private_key_filepath: string): TlsContextOptions;
    /**
     * Creates TLS context with peer verification disabled, along with a certificate and private key
     * in PKCS#12 format
     * @param pkcs12_filepath - Path to certificate, in PKCS#12 format
     * @param pkcs12_password - PKCS#12 Password
     *
     * @returns newly configured TlsContextOptions object
     */
    static create_server_with_mtls_pkcs_from_path(pkcs12_filepath: string, pkcs12_password: string): TlsContextOptions;
}
export declare namespace TlsContextOptions {
    /**
     * Options for TLS using a PKCS#11 library for private key operations.
     *
     * Unix only. nodejs only.
     *
     * @see [[TlsContextOptions.create_client_with_mtls_pkcs11]]
     */
    type Pkcs11Options = {
        /**
         * Use this PKCS#11 library.
         */
        pkcs11_lib: Pkcs11Lib;
        /**
         * Use this PIN to log the user into the PKCS#11 token. Pass `null`
         * to log into a token with a "protected authentication path".
         */
        user_pin: null | string;
        /**
         * Specify the slot ID containing a PKCS#11 token. If not specified, the token
         * will be chosen based on other criteria (such as [[token_label]]).
         */
        slot_id?: number;
        /**
         * Specify the label of the PKCS#11 token to use. If not specified, the token
         * will be chosen based on other criteria (such as [[slot_id]]).
         */
        token_label?: string;
        /**
         * Specify the label of the private key object on the PKCS#11 token. If not
         * specified, the key will be chosen based on other criteria (such as being the
         * only available private key on the token).
         */
        private_key_object_label?: string;
        /**
         * Use this X.509 certificate (file on disk).
         * The certificate must be PEM-formatted.
         * The certificate may be specified by other means instead
         * (ex: [[cert_file_contents]])
         */
        cert_file_path?: string;
        /**
         * Use this X.509 certificate (contents in memory).
         * The certificate must be PEM-formatted.
         * The certificate may be specified by other means instead
         * (ex: [[cert_file_path]])
         */
        cert_file_contents?: string;
    };
}
/**
 * Abstract base TLS context used for client/server TLS communications over sockets.
 *
 * @see ClientTlsContext
 * @see ServerTlsContext
 *
 * nodejs only.
 * @category TLS
 */
export declare abstract class TlsContext extends NativeResource {
    constructor(ctx_opt: TlsContextOptions);
}
/**
 * TLS context used for client TLS communications over sockets. If no
 * options are supplied, the context will default to enabling peer verification
 * only.
 *
 * nodejs only.
 * @category TLS
 */
export declare class ClientTlsContext extends TlsContext {
    constructor(ctx_opt?: TlsContextOptions);
}
/**
 * TLS context used for server TLS communications over sockets. If no
 * options are supplied, the context will default to disabling peer verification
 * only.
 *
 * nodejs only.
 * @category TLS
 */
export declare class ServerTlsContext extends TlsContext {
    constructor(ctx_opt?: TlsContextOptions);
}
/**
 * TLS options that are unique to a given connection using a shared TlsContext.
 *
 * nodejs only.
 * @category TLS
 */
export declare class TlsConnectionOptions extends NativeResource {
    readonly tls_ctx: TlsContext;
    readonly server_name?: string | undefined;
    readonly alpn_list: string[];
    constructor(tls_ctx: TlsContext, server_name?: string | undefined, alpn_list?: string[]);
}
/**
 * Handle to a loaded PKCS#11 library.
 *
 * For most use cases, a single instance of Pkcs11Lib should be used
 * for the lifetime of your application.
 *
 * nodejs only.
 * @category TLS
 */
export declare class Pkcs11Lib extends NativeResource {
    /**
     * @param path - Path to PKCS#11 library.
     * @param behavior - Specifies how `C_Initialize()` and `C_Finalize()`
     *                   will be called on the PKCS#11 library.
     */
    constructor(path: string, behavior?: Pkcs11Lib.InitializeFinalizeBehavior);
    /**
     * Release the PKCS#11 library immediately, without waiting for the GC.
     */
    close(): void;
}
export declare namespace Pkcs11Lib {
    /**
     * Controls `C_Initialize()` and `C_Finalize()` are called on the PKCS#11 library.
     */
    enum InitializeFinalizeBehavior {
        /**
         * Default behavior that accommodates most use cases.
         *
         * `C_Initialize()` is called on creation, and "already-initialized"
         * errors are ignored. `C_Finalize()` is never called, just in case
         * another part of your application is still using the PKCS#11 library.
         */
        DEFAULT = 0,
        /**
         * Skip calling `C_Initialize()` and `C_Finalize()`.
         *
         * Use this if your application has already initialized the PKCS#11 library,
         * and you do not want `C_Initialize()` called again.
         */
        OMIT = 1,
        /**
         * `C_Initialize()` is called on creation and `C_Finalize()` is called on cleanup.
         *
         * If `C_Initialize()` reports that's it's already initialized, this is
         * treated as an error. Use this if you need perfect cleanup (ex: running
         * valgrind with --leak-check).
         */
        STRICT = 2
    }
}
