"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.aws_sign_request = exports.AwsCredentialsProvider = exports.AwsSignedBodyHeaderType = exports.AwsSignedBodyValue = exports.AwsSignatureType = exports.AwsSigningAlgorithm = void 0;
const binding_1 = __importDefault(require("./binding"));
const error_1 = require("./error");
/**
 * AWS signing algorithm enumeration.
 *
 * @module aws-crt
 * @category Auth
 */
var AwsSigningAlgorithm;
(function (AwsSigningAlgorithm) {
    /** Use the Aws signature version 4 signing process to sign the request */
    AwsSigningAlgorithm[AwsSigningAlgorithm["SigV4"] = 0] = "SigV4";
})(AwsSigningAlgorithm = exports.AwsSigningAlgorithm || (exports.AwsSigningAlgorithm = {}));
/**
 * AWS signature type enumeration.
 *
 * @category Auth
 */
var AwsSignatureType;
(function (AwsSignatureType) {
    /** Sign an http request and apply the signing results as headers */
    AwsSignatureType[AwsSignatureType["HttpRequestViaHeaders"] = 0] = "HttpRequestViaHeaders";
    /** Sign an http request and apply the signing results as query params */
    AwsSignatureType[AwsSignatureType["HttpRequestViaQueryParams"] = 1] = "HttpRequestViaQueryParams";
    /** Sign an http request payload chunk */
    AwsSignatureType[AwsSignatureType["HttpRequestChunk"] = 2] = "HttpRequestChunk";
    /** Sign an event stream event */
    AwsSignatureType[AwsSignatureType["HttpRequestEvent"] = 3] = "HttpRequestEvent";
})(AwsSignatureType = exports.AwsSignatureType || (exports.AwsSignatureType = {}));
/**
 * Values for use with {@link AwsSigningConfig.signed_body_value}.
 *
 * Some services use special values (e.g. 'UNSIGNED-PAYLOAD') when the body
 * is not being signed in the usual way.
 *
 * @category Auth
 */
var AwsSignedBodyValue;
(function (AwsSignedBodyValue) {
    /** Use the SHA-256 of the empty string as the canonical request payload value */
    AwsSignedBodyValue["EmptySha256"] = "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855";
    /** Use the literal string 'UNSIGNED-PAYLOAD' as the canonical request payload value  */
    AwsSignedBodyValue["UnsignedPayload"] = "UNSIGNED-PAYLOAD";
    /** Use the literal string 'STREAMING-AWS4-HMAC-SHA256-PAYLOAD' as the canonical request payload value  */
    AwsSignedBodyValue["StreamingAws4HmacSha256Payload"] = "STREAMING-AWS4-HMAC-SHA256-PAYLOAD";
    /** Use the literal string 'STREAMING-AWS4-HMAC-SHA256-EVENTS' as the canonical request payload value  */
    AwsSignedBodyValue["StreamingAws4HmacSha256Events"] = "STREAMING-AWS4-HMAC-SHA256-EVENTS";
})(AwsSignedBodyValue = exports.AwsSignedBodyValue || (exports.AwsSignedBodyValue = {}));
/**
 * AWS signed body header enumeration.
 *
 * @category Auth
 */
var AwsSignedBodyHeaderType;
(function (AwsSignedBodyHeaderType) {
    /** Do not add a header containing the canonical request payload value */
    AwsSignedBodyHeaderType[AwsSignedBodyHeaderType["None"] = 0] = "None";
    /** Add the X-Amz-Content-Sha256 header with the canonical request payload value */
    AwsSignedBodyHeaderType[AwsSignedBodyHeaderType["XAmzContentSha256"] = 1] = "XAmzContentSha256";
})(AwsSignedBodyHeaderType = exports.AwsSignedBodyHeaderType || (exports.AwsSignedBodyHeaderType = {}));
/**
 * Credentials providers source the AwsCredentials needed to sign an authenticated AWS request.
 *
 * @module aws-crt
 * @category Auth
 */
/* Subclass for the purpose of exposing a non-NativeHandle based API */
class AwsCredentialsProvider extends binding_1.default.AwsCredentialsProvider {
    static newDefault(bootstrap) {
        return super.newDefault(bootstrap.native_handle());
    }
}
exports.AwsCredentialsProvider = AwsCredentialsProvider;
/**
 * Perform AWS HTTP request signing.
 *
 * The {@link HttpRequest} is transformed asynchronously,
 * according to the {@link AwsSigningConfig}.
 *
 * When signing:
 *  1.  It is good practice to use a new config for each signature,
 *      or the date might get too old.
 *
 *  2.  Do not add the following headers to requests before signing, they may be added by the signer:
 *      x-amz-content-sha256,
 *      X-Amz-Date,
 *      Authorization
 *
 *  3.  Do not add the following query params to requests before signing, they may be added by the signer:
 *      X-Amz-Signature,
 *      X-Amz-Date,
 *      X-Amz-Credential,
 *      X-Amz-Algorithm,
 *      X-Amz-SignedHeaders
 * @param request The HTTP request to sign.
 * @param config Configuration for signing.
 * @returns A Future whose result will be the signed
 *       {@link HttpRequest}. The future will contain an exception
 *       if the signing process fails.
 *
 * @module aws-crt
 * @category Auth
 */
function aws_sign_request(request, config) {
    return __awaiter(this, void 0, void 0, function* () {
        return new Promise((resolve, reject) => {
            try {
                /* Note: if the body of request has not fully loaded, it will lead to an endless loop.
                 * User should set the signed_body_value of config to prevent this endless loop in this case */
                binding_1.default.aws_sign_request(request, config, (error_code) => {
                    if (error_code == 0) {
                        resolve(request);
                    }
                    else {
                        reject(new error_1.CrtError(error_code));
                    }
                });
            }
            catch (error) {
                reject(error);
            }
        });
    });
}
exports.aws_sign_request = aws_sign_request;
//# sourceMappingURL=auth.js.map