"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TlsConnectionOptions = exports.ServerTlsContext = exports.ClientTlsContext = exports.TlsContext = exports.TlsContextOptions = exports.SocketOptions = exports.ClientBootstrap = exports.InputStream = exports.is_alpn_available = exports.enable_logging = exports.LogLevel = exports.error_code_to_name = exports.error_code_to_string = void 0;
const binding_1 = __importDefault(require("./binding"));
const native_resource_1 = require("./native_resource");
const io_1 = require("../common/io");
var io_2 = require("../common/io");
Object.defineProperty(exports, "TlsVersion", { enumerable: true, get: function () { return io_2.TlsVersion; } });
Object.defineProperty(exports, "SocketType", { enumerable: true, get: function () { return io_2.SocketType; } });
Object.defineProperty(exports, "SocketDomain", { enumerable: true, get: function () { return io_2.SocketDomain; } });
/**
 * Convert a native error code into a human-readable string
 * @param error_code - An error code returned from a native API call, or delivered
 * via callback.
 * @see CrtError
 *
 * nodejs only.
 */
function error_code_to_string(error_code) {
    return binding_1.default.error_code_to_string(error_code);
}
exports.error_code_to_string = error_code_to_string;
/**
 * Convert a native error code into a human-readable identifier
 * @param error_code - An error code returned from a native API call, or delivered
 * via callback.
 * @see CrtError
 *
 * nodejs only.
 */
function error_code_to_name(error_code) {
    return binding_1.default.error_code_to_name(error_code);
}
exports.error_code_to_name = error_code_to_name;
/**
 * The amount of detail that will be logged
 * @module aws-crt
 * @category Logging
 */
var LogLevel;
(function (LogLevel) {
    /** No logging whatsoever. Equivalent to never calling {@link enable_logging}. */
    LogLevel[LogLevel["NONE"] = 0] = "NONE";
    /** Only fatals. In practice, this will not do much, as the process will log and then crash (intentionally) if a fatal condition occurs */
    LogLevel[LogLevel["FATAL"] = 1] = "FATAL";
    /** Only errors */
    LogLevel[LogLevel["ERROR"] = 2] = "ERROR";
    /** Only warnings and errors */
    LogLevel[LogLevel["WARN"] = 3] = "WARN";
    /** Information about connection/stream creation/destruction events */
    LogLevel[LogLevel["INFO"] = 4] = "INFO";
    /** Enough information to debug the chain of events a given network connection encounters */
    LogLevel[LogLevel["DEBUG"] = 5] = "DEBUG";
    /** Everything. Only use this if you really need to know EVERY single call */
    LogLevel[LogLevel["TRACE"] = 6] = "TRACE";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * Enables logging of the native AWS CRT libraries.
 * @param level - The logging level to filter to. It is not possible to log less than WARN.
 *
 * nodejs only.
 * @module aws-crt
 * @category Logging
 */
function enable_logging(level) {
    binding_1.default.io_logging_enable(level);
}
exports.enable_logging = enable_logging;
/**
 * Returns true if ALPN is available on this platform natively
 * @return true if ALPN is supported natively, false otherwise
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
*/
function is_alpn_available() {
    return binding_1.default.is_alpn_available();
}
exports.is_alpn_available = is_alpn_available;
/**
 * Wraps a ```Readable``` for reading by native code, used to stream
 *  data into the AWS CRT libraries.
 *
 * nodejs only.
 * @module aws-crt
 * @category I/O
 */
class InputStream extends native_resource_1.NativeResource {
    constructor(source) {
        super(binding_1.default.io_input_stream_new(16 * 1024));
        this.source = source;
        this.source.on('data', (data) => {
            data = Buffer.isBuffer(data) ? data : new Buffer(data.toString(), 'utf8');
            binding_1.default.io_input_stream_append(this.native_handle(), data);
        });
        this.source.on('end', () => {
            binding_1.default.io_input_stream_append(this.native_handle(), undefined);
        });
    }
}
exports.InputStream = InputStream;
/**
 * Represents native resources required to bootstrap a client connection
 * Things like a host resolver, event loop group, etc. There should only need
 * to be 1 of these per application, in most cases.
 *
 * nodejs only.
 * @module aws-crt
 * @category I/O
 */
class ClientBootstrap extends native_resource_1.NativeResource {
    constructor() {
        super(binding_1.default.io_client_bootstrap_new());
    }
}
exports.ClientBootstrap = ClientBootstrap;
/**
 * Standard Berkeley socket style options.
 *
 * nodejs only.
 * @module aws-crt
 * @category I/O
*/
class SocketOptions extends native_resource_1.NativeResource {
    constructor(type = io_1.SocketType.STREAM, domain = io_1.SocketDomain.IPV6, connect_timeout_ms = 5000, keepalive = false, keep_alive_interval_sec = 0, keep_alive_timeout_sec = 0, keep_alive_max_failed_probes = 0) {
        super(binding_1.default.io_socket_options_new(type, domain, connect_timeout_ms, keep_alive_interval_sec, keep_alive_timeout_sec, keep_alive_max_failed_probes, keepalive));
    }
}
exports.SocketOptions = SocketOptions;
/**
 * Options for creating a {@link ClientTlsContext} or {@link ServerTlsContext}.
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
 */
class TlsContextOptions {
    constructor() {
        /** Minimum version of TLS to support. Uses OS/system default if unspecified. */
        this.min_tls_version = io_1.TlsVersion.Default;
        /** List of ALPN protocols to be used on platforms which support ALPN */
        this.alpn_list = [];
        /**
         * In client mode, this turns off x.509 validation. Don't do this unless you are testing.
         * It is much better to just override the default trust store and pass the self-signed
         * certificate as the ca_file argument.
         *
         * In server mode, this defaults to false. If you want to enforce mutual TLS on the server,
         * set this to true.
         */
        this.verify_peer = false;
    }
    /**
     * Overrides the default system trust store.
     * @param ca_dirpath - Only used on Unix-style systems where all trust anchors are
     * stored in a directory (e.g. /etc/ssl/certs).
     * @param ca_filepath - Single file containing all trust CAs, in PEM format
     */
    override_default_trust_store_from_path(ca_dirpath, ca_filepath) {
        this.ca_dirpath = ca_dirpath;
        this.ca_filepath = ca_filepath;
    }
    /**
     * Overrides the default system trust store.
     * @param certificate_authority - String containing all trust CAs, in PEM format
     */
    override_default_trust_store(certificate_authority) {
        this.certificate_authority = certificate_authority;
    }
    /**
     * Creates a client with secure-by-default options, along with a client cert and private key
     * @param certificate - Client certificate, in PEM format
     * @param private_key - Client private key, in PEM format
     */
    static create_client_with_mtls(certificate, private_key) {
        let opt = new TlsContextOptions();
        opt.certificate = certificate;
        opt.private_key = private_key;
        opt.verify_peer = true;
        return opt;
    }
    /**
     * Creates a client with secure-by-default options, along with a client cert and private key
     * @param certificate_filepath - Path to client certificate, in PEM format
     * @param private_key_filepath - Path to private key, in PEM format
     */
    static create_client_with_mtls_from_path(certificate_filepath, private_key_filepath) {
        let opt = new TlsContextOptions();
        opt.certificate_filepath = certificate_filepath;
        opt.private_key_filepath = private_key_filepath;
        opt.verify_peer = true;
        return opt;
    }
    /**
     * Creates a TLS context with secure-by-default options, along with a client cert and password
     * @param pkcs12_filepath - Path to client certificate in PKCS#12 format
     * @param pkcs12_password - PKCS#12 password
    */
    static create_client_with_mtls_pkcs_from_path(pkcs12_filepath, pkcs12_password) {
        let opt = new TlsContextOptions();
        opt.pkcs12_filepath = pkcs12_filepath;
        opt.pkcs12_password = pkcs12_password;
        opt.verify_peer = true;
        return opt;
    }
    /**
     * Creates TLS context with peer verification disabled, along with a certificate and private key
     * @param certificate_filepath - Path to certificate, in PEM format
     * @param private_key_filepath - Path to private key, in PEM format
     *
     */
    static create_server_with_mtls_from_path(certificate_filepath, private_key_filepath) {
        let opt = new TlsContextOptions();
        opt.certificate_filepath = certificate_filepath;
        opt.private_key_filepath = private_key_filepath;
        opt.verify_peer = false;
        return opt;
    }
    /**
     * Creates TLS context with peer verification disabled, along with a certificate and private key
     * in PKCS#12 format
     * @param pkcs12_filepath - Path to certificate, in PKCS#12 format
     * @param pkcs12_password - PKCS#12 Password
     *
     */
    static create_server_with_mtls_pkcs_from_path(pkcs12_filepath, pkcs12_password) {
        let opt = new TlsContextOptions();
        opt.pkcs12_filepath = pkcs12_filepath;
        opt.pkcs12_password = pkcs12_password;
        opt.verify_peer = false;
        return opt;
    }
}
exports.TlsContextOptions = TlsContextOptions;
/**
 * TLS context used for client/server TLS communications over sockets.
 *
 * @see ClientTlsContext
 * @see ServerTlsContext
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
 */
class TlsContext extends native_resource_1.NativeResource {
    constructor(ctx_opt) {
        super(binding_1.default.io_tls_ctx_new(ctx_opt.min_tls_version, ctx_opt.ca_filepath, ctx_opt.ca_dirpath, ctx_opt.certificate_authority, (ctx_opt.alpn_list && ctx_opt.alpn_list.length > 0) ? ctx_opt.alpn_list.join(';') : undefined, ctx_opt.certificate_filepath, ctx_opt.certificate, ctx_opt.private_key_filepath, ctx_opt.private_key, ctx_opt.pkcs12_filepath, ctx_opt.pkcs12_password, ctx_opt.verify_peer));
    }
}
exports.TlsContext = TlsContext;
/**
 * TLS context used for client TLS communications over sockets. If no
 * options are supplied, the context will default to enabling peer verification
 * only.
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
 */
class ClientTlsContext extends TlsContext {
    constructor(ctx_opt) {
        if (!ctx_opt) {
            ctx_opt = new TlsContextOptions();
            ctx_opt.verify_peer = true;
        }
        super(ctx_opt);
    }
}
exports.ClientTlsContext = ClientTlsContext;
/**
 * TLS context used for server TLS communications over sockets. If no
 * options are supplied, the context will default to disabling peer verification
 * only.
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
 */
class ServerTlsContext extends TlsContext {
    constructor(ctx_opt) {
        if (!ctx_opt) {
            ctx_opt = new TlsContextOptions();
            ctx_opt.verify_peer = false;
        }
        super(ctx_opt);
    }
}
exports.ServerTlsContext = ServerTlsContext;
/**
 * TLS options that are unique to a given connection using a shared TlsContext.
 *
 * nodejs only.
 * @module aws-crt
 * @category TLS
 */
class TlsConnectionOptions extends native_resource_1.NativeResource {
    constructor(tls_ctx, server_name, alpn_list = []) {
        super(binding_1.default.io_tls_connection_options_new(tls_ctx.native_handle(), server_name, (alpn_list && alpn_list.length > 0) ? alpn_list.join(';') : undefined));
        this.tls_ctx = tls_ctx;
        this.server_name = server_name;
        this.alpn_list = alpn_list;
    }
}
exports.TlsConnectionOptions = TlsConnectionOptions;
//# sourceMappingURL=io.js.map