// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModuleClient = void 0;
const dbg = require("debug");
const debug = dbg('azure-iot-device:ModuleClient');
const debugErrors = dbg('azure-iot-device:ModuleClients:Errors');
const fs = require("fs");
const azure_iot_common_1 = require("azure-iot-common");
const internal_client_1 = require("./internal_client");
const azure_iot_common_2 = require("azure-iot-common");
const sak_authentication_provider_1 = require("./sak_authentication_provider");
const sas_authentication_provider_1 = require("./sas_authentication_provider");
const iotedge_authentication_provider_1 = require("./iotedge_authentication_provider");
const device_method_1 = require("./device_method");
function safeCallback(callback, error, result) {
    if (callback)
        callback(error, result);
}
/**
 * IoT Hub device client used to connect a device with an Azure IoT hub.
 *
 * Users of the SDK should call one of the factory methods,
 * {@link azure-iot-device.Client.fromConnectionString|fromConnectionString}
 * or {@link azure-iot-device.Client.fromSharedAccessSignature|fromSharedAccessSignature}
 * to create an IoT Hub device client.
 */
class ModuleClient extends internal_client_1.InternalClient {
    /**
     * @private
     * @constructor
     * @param {Object}  transport         An object that implements the interface
     *                                    expected of a transport object, e.g.,
     *                                    {@link azure-iot-device-mqtt.Mqtt|Mqtt}.
     * @param {Object}  restApiClient     the RestApiClient object to use for HTTP calls
     */
    constructor(transport, methodClient) {
        super(transport, undefined);
        this._userRegisteredInputMessageListener = false;
        this._methodClient = methodClient;
        /* Codes_SRS_NODE_MODULE_CLIENT_18_012: [ The `inputMessage` event shall be emitted when an inputMessage is received from the IoT Hub service. ]*/
        /* Codes_SRS_NODE_MODULE_CLIENT_18_013: [ The `inputMessage` event parameters shall be the inputName for the message and a `Message` object. ]*/
        this._transport.on('inputMessage', (inputName, msg) => {
            this.emit('inputMessage', inputName, msg);
        });
        this.on('removeListener', () => {
            if (this.listenerCount('inputMessage') === 0) {
                this._userRegisteredInputMessageListener = false;
                /* Codes_SRS_NODE_MODULE_CLIENT_18_015: [ The client shall stop listening for messages from the service whenever the last listener unsubscribes from the `inputMessage` event. ]*/
                debug('in removeListener, disabling input messages');
                this._disableInputMessages((err) => {
                    if (err) {
                        debugErrors('in removeListener, error disabling input messages: ' + err);
                        this.emit('error', err);
                    }
                    else {
                        debug('removeListener successfully disabled input messages.');
                    }
                });
            }
        });
        this.on('newListener', (eventName) => {
            if (eventName === 'inputMessage') {
                //
                // We want to always retain that the we want to have this feature enabled because the API (.on) doesn't really
                // provide for the capability to say it failed.  It can certainly fail because a network operation is required to
                // enable.
                // By saving this off, we are strictly honoring that the feature is enabled.  If it doesn't turn on we signal via
                // the emitted 'error' that something bad happened.
                // But if we ever again attain a connected state, this feature will be operational.
                //
                this._userRegisteredInputMessageListener = true;
                /* Codes_SRS_NODE_MODULE_CLIENT_18_014: [ The client shall start listening for messages from the service whenever there is a listener subscribed to the `inputMessage` event. ]*/
                debug('in newListener, enabling input messages');
                this._enableInputMessages((err) => {
                    if (err) {
                        debugErrors('in newListener, error enabling input messages: ' + err);
                        this.emit('error', err);
                    }
                    else {
                        debug('in newListener, successfully enabled input messages');
                    }
                });
            }
        });
        this._moduleDisconnectHandler = (err) => {
            if (err) {
                debugErrors('transport disconnect event: ' + err);
            }
            else {
                debug('transport disconnect event:  no error');
            }
            if (err && this._retryPolicy.shouldRetry(err)) {
                if (this._userRegisteredInputMessageListener) {
                    debug('re-enabling input message link');
                    this._enableInputMessages((err) => {
                        if (err) {
                            debugErrors('Error re-enabling input messages: ' + err);
                            /*Codes_SRS_NODE_MODULE_CLIENT_16_102: [If the retry policy fails to reestablish the C2D functionality a `disconnect` event shall be emitted with a `results.Disconnected` object.]*/
                            this.emit('disconnect', new azure_iot_common_1.results.Disconnected(err));
                        }
                    });
                }
            }
        };
        /*Codes_SRS_NODE_MODULE_CLIENT_16_045: [If the transport successfully establishes a connection the `open` method shall subscribe to the `disconnect` event of the transport.]*/
        this._transport.on('disconnect', this._moduleDisconnectHandler);
    }
    sendOutputEvent(outputName, message, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            const retryOp = new azure_iot_common_1.RetryOperation('sendOutputEvent', this._retryPolicy, this._maxOperationTimeout);
            retryOp.retry((opCallback) => {
                /* Codes_SRS_NODE_MODULE_CLIENT_18_010: [ The `sendOutputEvent` method shall send the event indicated by the `message` argument via the transport associated with the Client instance. ]*/
                this._transport.sendOutputEvent(outputName, message, opCallback);
            }, (err, result) => {
                /*Codes_SRS_NODE_MODULE_CLIENT_18_018: [ When the `sendOutputEvent` method completes, the `callback` function shall be invoked with the same arguments as the underlying transport method's callback. ]*/
                /*Codes_SRS_NODE_MODULE_CLIENT_18_019: [ The `sendOutputEvent` method shall not throw if the `callback` is not passed. ]*/
                safeCallback(_callback, err, result);
            });
        }, callback);
    }
    sendOutputEventBatch(outputName, messages, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            const retryOp = new azure_iot_common_1.RetryOperation('sendOutputEventBatch', this._retryPolicy, this._maxOperationTimeout);
            retryOp.retry((opCallback) => {
                /* Codes_SRS_NODE_MODULE_CLIENT_18_011: [ The `sendOutputEventBatch` method shall send the list of events (indicated by the `messages` argument) via the transport associated with the Client instance. ]*/
                this._transport.sendOutputEventBatch(outputName, messages, opCallback);
            }, (err, result) => {
                /*Codes_SRS_NODE_MODULE_CLIENT_18_021: [ When the `sendOutputEventBatch` method completes the `_callback` function shall be invoked with the same arguments as the underlying transport method's callback. ]*/
                /*Codes_SRS_NODE_MODULE_CLIENT_18_022: [ The `sendOutputEventBatch` method shall not throw if the `_callback` is not passed. ]*/
                safeCallback(_callback, err, result);
            });
        }, callback);
    }
    close(closeCallback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            this._transport.removeListener('disconnect', this._moduleDisconnectHandler);
            super.close(_callback);
        }, closeCallback);
    }
    _invokeMethod(deviceId, moduleIdOrMethodParams, methodParamsOrCallback, callback) {
        /*Codes_SRS_NODE_MODULE_CLIENT_16_093: [`invokeMethod` shall throw a `ReferenceError` if the `deviceId` argument is falsy.]*/
        if (!deviceId) {
            throw new ReferenceError('deviceId cannot be \'' + deviceId + '\'');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_16_094: [`invokeMethod` shall throw a `ReferenceError` if the `moduleIdOrMethodParams` argument is falsy.]*/
        if (!moduleIdOrMethodParams) {
            throw new ReferenceError('The second parameter cannot be \'' + moduleIdOrMethodParams + '\'');
        }
        const actualModuleId = typeof moduleIdOrMethodParams === 'string' ? moduleIdOrMethodParams : null;
        const actualMethodParams = typeof moduleIdOrMethodParams === 'object' ? moduleIdOrMethodParams : methodParamsOrCallback;
        const actualCallback = typeof methodParamsOrCallback === 'function' ? methodParamsOrCallback : callback;
        /*Codes_SRS_NODE_MODULE_CLIENT_16_095: [`invokeMethod` shall throw a `ReferenceError` if the `deviceId` and `moduleIdOrMethodParams` are strings and the `methodParamsOrCallback` argument is falsy.]*/
        if (!actualMethodParams || typeof actualMethodParams !== 'object') {
            throw new ReferenceError('methodParams cannot be \'' + actualMethodParams + '\'');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_16_096: [`invokeMethod` shall throw a `ArgumentError` if the `methodName` property of the `MethodParams` argument is falsy.]*/
        if (!actualMethodParams.methodName) {
            throw new azure_iot_common_2.errors.ArgumentError('the name property of the methodParams argument cannot be \'' + actualMethodParams.methodName + '\'');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_16_097: [`invokeMethod` shall call the `invokeMethod` API of the `MethodClient` API that was created for the `ModuleClient` instance.]*/
        this._methodClient.invokeMethod(deviceId, actualModuleId, actualMethodParams, actualCallback);
    }
    invokeMethod(deviceId, moduleIdOrMethodParams, methodParamsOrCallback, callback) {
        if (callback) {
            return this._invokeMethod(deviceId, moduleIdOrMethodParams, methodParamsOrCallback, callback);
        }
        else if (typeof methodParamsOrCallback === 'function') {
            return this._invokeMethod(deviceId, moduleIdOrMethodParams, methodParamsOrCallback);
        }
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => this._invokeMethod(deviceId, moduleIdOrMethodParams, methodParamsOrCallback, _callback));
    }
    /**
     * Registers a callback for a method named `methodName`.
     *
     * @param methodName Name of the method that will be handled by the callback
     * @param callback Function that shall be called whenever a method request for the method called `methodName` is received.
     */
    onMethod(methodName, callback) {
        this._onDeviceMethod(methodName, callback);
    }
    setOptions(options, done) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_MODULE_CLIENT_16_098: [The `setOptions` method shall call the `setOptions` method with the `options` argument on the `MethodClient` object of the `ModuleClient`.]*/
            if (this._methodClient) {
                this._methodClient.setOptions(options);
            }
            /*Codes_SRS_NODE_MODULE_CLIENT_16_042: [The `setOptions` method shall throw a `ReferenceError` if the options object is falsy.]*/
            /*Codes_SRS_NODE_MODULE_CLIENT_16_043: [The `_callback` callback shall be invoked with no parameters when it has successfully finished setting the client and/or transport options.]*/
            /*Codes_SRS_NODE_MODULE_CLIENT_16_044: [The `_callback` callback shall be invoked with a standard javascript `Error` object and no result object if the client could not be configured as requested.]*/
            super.setOptions(options, _callback);
        }, done);
    }
    _disableInputMessages(callback) {
        debug('disabling input messages');
        this._transport.disableInputMessages((err) => {
            if (!err) {
                debug('disabled input messages');
            }
            else {
                debugErrors('Error while disabling input messages: ' + err);
            }
            callback(err);
        });
    }
    _enableInputMessages(callback) {
        debug('enabling input messages');
        const retryOp = new azure_iot_common_1.RetryOperation('_enableInputMessages', this._retryPolicy, this._maxOperationTimeout);
        retryOp.retry((opCallback) => {
            this._transport.enableInputMessages(opCallback);
        }, (err) => {
            if (!err) {
                debug('enabled input messages');
            }
            else {
                debugErrors('Error while enabling input messages: ' + err);
            }
            callback(err);
        });
    }
    /**
     * Creates an IoT Hub device client from the given connection string using the given transport type.
     *
     * @param {String}    connStr        A connection string which encapsulates "device connect" permissions on an IoT hub.
     * @param {Function}  transportCtor  A transport constructor.
     *
     * @throws {ReferenceError}          If the connStr parameter is falsy.
     *
     * @returns {module:azure-iot-device.ModuleClient}
     */
    static fromConnectionString(connStr, transportCtor) {
        /*Codes_SRS_NODE_MODULE_CLIENT_05_003: [The fromConnectionString method shall throw ReferenceError if the connStr argument is falsy.]*/
        if (!connStr)
            throw new ReferenceError('connStr is \'' + connStr + '\'');
        const cn = azure_iot_common_1.ConnectionString.parse(connStr);
        /*Codes_SRS_NODE_MODULE_CLIENT_16_087: [The `fromConnectionString` method shall create a new `SharedAccessKeyAuthorizationProvider` object with the connection string passed as argument if it contains a SharedAccessKey parameter and pass this object to the transport constructor.]*/
        let authenticationProvider;
        if (cn.SharedAccessKey) {
            authenticationProvider = sak_authentication_provider_1.SharedAccessKeyAuthenticationProvider.fromConnectionString(connStr);
        }
        else {
            /*Codes_SRS_NODE_MODULE_CLIENT_16_001: [The `fromConnectionString` method shall throw a `NotImplementedError` if the connection string does not contain a `SharedAccessKey` field because x509 authentication is not supported yet for modules.]*/
            throw new azure_iot_common_2.errors.NotImplementedError('ModuleClient only supports SAS Token authentication');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_05_006: [The fromConnectionString method shall return a new instance of the Client object, as by a call to new Client(new transportCtor(...)).]*/
        return new ModuleClient(new transportCtor(authenticationProvider), new device_method_1.MethodClient(authenticationProvider));
    }
    /**
     * Creates an IoT Hub module client from the given shared access signature using the given transport type.
     *
     * @param {String}    sharedAccessSignature      A shared access signature which encapsulates "device
     *                                  connect" permissions on an IoT hub.
     * @param {Function}  Transport     A transport constructor.
     *
     * @throws {ReferenceError}         If the connStr parameter is falsy.
     *
     * @returns {module:azure-iothub.Client}
     */
    static fromSharedAccessSignature(sharedAccessSignature, transportCtor) {
        /*Codes_SRS_NODE_MODULE_CLIENT_16_029: [The fromSharedAccessSignature method shall throw a ReferenceError if the sharedAccessSignature argument is falsy.] */
        if (!sharedAccessSignature)
            throw new ReferenceError('sharedAccessSignature is \'' + sharedAccessSignature + '\'');
        /*Codes_SRS_NODE_MODULE_CLIENT_16_088: [The `fromSharedAccessSignature` method shall create a new `SharedAccessSignatureAuthorizationProvider` object with the shared access signature passed as argument, and pass this object to the transport constructor.]*/
        const authenticationProvider = sas_authentication_provider_1.SharedAccessSignatureAuthenticationProvider.fromSharedAccessSignature(sharedAccessSignature);
        /*Codes_SRS_NODE_MODULE_CLIENT_16_030: [The fromSharedAccessSignature method shall return a new instance of the Client object] */
        return new ModuleClient(new transportCtor(authenticationProvider), new device_method_1.MethodClient(authenticationProvider));
    }
    /**
     * Creates an IoT Hub module client from the given authentication method and using the given transport type.
     * @param authenticationProvider  Object used to obtain the authentication parameters for the IoT hub.
     * @param transportCtor           Transport protocol used to connect to IoT hub.
     */
    static fromAuthenticationProvider(authenticationProvider, transportCtor) {
        /*Codes_SRS_NODE_MODULE_CLIENT_16_089: [The `fromAuthenticationProvider` method shall throw a `ReferenceError` if the `authenticationProvider` argument is falsy.]*/
        if (!authenticationProvider) {
            throw new ReferenceError('authenticationMethod cannot be \'' + authenticationProvider + '\'');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_16_092: [The `fromAuthenticationProvider` method shall throw a `ReferenceError` if the `transportCtor` argument is falsy.]*/
        if (!transportCtor) {
            throw new ReferenceError('transportCtor cannot be \'' + transportCtor + '\'');
        }
        /*Codes_SRS_NODE_MODULE_CLIENT_16_090: [The `fromAuthenticationProvider` method shall pass the `authenticationProvider` object passed as argument to the transport constructor.]*/
        /*Codes_SRS_NODE_MODULE_CLIENT_16_091: [The `fromAuthenticationProvider` method shall return a `Client` object configured with a new instance of a transport created using the `transportCtor` argument.]*/
        return new ModuleClient(new transportCtor(authenticationProvider), new device_method_1.MethodClient(authenticationProvider));
    }
    static fromEnvironment(transportCtor, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            // Codes_SRS_NODE_MODULE_CLIENT_13_033: [ The fromEnvironment method shall throw a ReferenceError if the callback argument is falsy or is not a function. ]
            if (!_callback || typeof (_callback) !== 'function') {
                throw new ReferenceError('callback cannot be \'' + _callback + '\'');
            }
            // Codes_SRS_NODE_MODULE_CLIENT_13_026: [ The fromEnvironment method shall invoke callback with a ReferenceError if the transportCtor argument is falsy. ]
            if (!transportCtor) {
                _callback(new ReferenceError('transportCtor cannot be \'' + transportCtor + '\''));
                return;
            }
            // Codes_SRS_NODE_MODULE_CLIENT_13_028: [ The fromEnvironment method shall delegate to ModuleClient.fromConnectionString if an environment variable called EdgeHubConnectionString or IotHubConnectionString exists. ]
            // if the environment has a value for EdgeHubConnectionString then we use that
            const connectionString = process.env.EdgeHubConnectionString || process.env.IotHubConnectionString;
            if (connectionString) {
                ModuleClient._fromEnvironmentNormal(connectionString, transportCtor, _callback);
            }
            else {
                ModuleClient._fromEnvironmentEdge(transportCtor, _callback);
            }
        }, callback);
    }
    static _fromEnvironmentEdge(transportCtor, callback) {
        // make sure all the environment variables we need have been provided
        const validationError = ModuleClient.validateEnvironment();
        if (validationError) {
            callback(validationError);
            return;
        }
        const authConfig = {
            workloadUri: process.env.IOTEDGE_WORKLOADURI,
            deviceId: process.env.IOTEDGE_DEVICEID,
            moduleId: process.env.IOTEDGE_MODULEID,
            iothubHostName: process.env.IOTEDGE_IOTHUBHOSTNAME,
            authScheme: process.env.IOTEDGE_AUTHSCHEME,
            gatewayHostName: process.env.IOTEDGE_GATEWAYHOSTNAME,
            generationId: process.env.IOTEDGE_MODULEGENERATIONID
        };
        // Codes_SRS_NODE_MODULE_CLIENT_13_032: [ The fromEnvironment method shall create a new IotEdgeAuthenticationProvider object and pass this to the transport constructor. ]
        const authenticationProvider = new iotedge_authentication_provider_1.IotEdgeAuthenticationProvider(authConfig);
        // get trust bundle
        authenticationProvider.getTrustBundle((err, ca) => {
            if (err) {
                callback(err);
            }
            else {
                const transport = new transportCtor(authenticationProvider);
                // Codes_SRS_NODE_MODULE_CLIENT_13_035: [ If the client is running in IoTEdge mode then the IotEdgeAuthenticationProvider.getTrustBundle method shall be invoked to retrieve the CA cert and the returned value shall be set as the CA cert for the transport via the transport's setOptions method passing in the CA value for the ca property in the options object. ]
                transport.setOptions({ ca });
                const methodClient = new device_method_1.MethodClient(authenticationProvider);
                methodClient.setOptions({ ca });
                // Codes_SRS_NODE_MODULE_CLIENT_13_031: [ The fromEnvironment method shall invoke the callback with a new instance of the ModuleClient object. ]
                callback(null, new ModuleClient(transport, methodClient));
            }
        });
    }
    static _fromEnvironmentNormal(connectionString, transportCtor, callback) {
        let ca = '';
        if (process.env.EdgeModuleCACertificateFile) {
            fs.readFile(process.env.EdgeModuleCACertificateFile, 'utf8', (err, data) => {
                if (err) {
                    callback(err);
                }
                else {
                    // Codes_SRS_NODE_MODULE_CLIENT_13_034: [ If the client is running in a non-IoTEdge mode and an environment variable named EdgeModuleCACertificateFile exists then its file contents shall be set as the CA cert for the transport via the transport's setOptions method passing in the CA as the value for the ca property in the options object. ]
                    ca = data;
                    const moduleClient = ModuleClient.fromConnectionString(connectionString, transportCtor);
                    moduleClient.setOptions({ ca }, (err) => {
                        if (err) {
                            callback(err);
                        }
                        else {
                            callback(null, moduleClient);
                        }
                    });
                }
            });
        }
        else {
            callback(null, ModuleClient.fromConnectionString(connectionString, transportCtor));
        }
    }
    static validateEnvironment() {
        // Codes_SRS_NODE_MODULE_CLIENT_13_029: [ If environment variables EdgeHubConnectionString and IotHubConnectionString do not exist then the following environment variables must be defined: IOTEDGE_WORKLOADURI, IOTEDGE_DEVICEID, IOTEDGE_MODULEID, IOTEDGE_IOTHUBHOSTNAME, IOTEDGE_AUTHSCHEME and IOTEDGE_MODULEGENERATIONID. ]
        const keys = [
            'IOTEDGE_WORKLOADURI',
            'IOTEDGE_DEVICEID',
            'IOTEDGE_MODULEID',
            'IOTEDGE_IOTHUBHOSTNAME',
            'IOTEDGE_AUTHSCHEME',
            'IOTEDGE_MODULEGENERATIONID'
        ];
        for (const key of keys) {
            if (!process.env[key]) {
                return new ReferenceError(`Environment variable ${key} was not provided.`);
            }
        }
        // Codes_SRS_NODE_MODULE_CLIENT_13_030: [ The value for the environment variable IOTEDGE_AUTHSCHEME must be sasToken. ]
        // we only support sas token auth scheme at this time
        if (process.env.IOTEDGE_AUTHSCHEME.toLowerCase() !== 'sastoken') {
            return new ReferenceError(`Authentication scheme ${process.env.IOTEDGE_AUTHSCHEME} is not a supported scheme.`);
        }
    }
}
exports.ModuleClient = ModuleClient;
//# sourceMappingURL=module_client.js.map