"use strict";
/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for
 * license information.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DigitalTwinClient = void 0;
const iotHubGatewayServiceAPIs_1 = require("../pl/iotHubGatewayServiceAPIs");
const promise_utils_1 = require("../common-core/promise_utils");
/**
 * @private
 * Helper function to create extended result type
 */
function createResultWithHttpOperationResponse(result, response) {
    if (result) {
        result._response = response;
    }
    return result;
}
/**
 * @export
 * @class DigitalTwinClient    Main class to implement Azure IoT Digital Twin Client API
 */
class DigitalTwinClient {
    /**
     * Constructor which also creates an instance of the Protocol Layer Client used by the DigitalTwinClient.
     *
     * @param {IoTHubTokenCredentials} creds    The IoTHub token credentials used for creating the Protocol Layer client.
     * @memberof DigitalTwinClient
     */
    constructor(creds) {
        /**
         * @private
         * The Azure IoT service's API version.
         */
        this._apiVersion = '2021-04-12';
        /*Code_SRS_NODE_DIGITAL_TWIN_CLIENT_12_001: [** The `DigitalTwinClient` creates an instance of the DigitalTwinClient passing IoTHubTokenCredentials class as an argument.]*/
        this._creds = creds;
        this._pl = new iotHubGatewayServiceAPIs_1.IotHubGatewayServiceAPIs(this._creds, {
            baseUri: 'https://' + this._creds.getHubName(),
            apiVersion: this._apiVersion,
            deserializationContentTypes: {
                json: [
                    'application/ld+json',
                    'application/json',
                    'text/json'
                ]
            }
        });
    }
    getDigitalTwin(digitalTwinId, callback) {
        /*Codes_SRS_NODE_DIGITAL_TWIN_CLIENT_12_002: [The `getDigitalTwin` method shall call the `getDigitalTwin` method of the protocol layer with the given argument.]*/
        /*Codes_SRS_NODE_DIGITAL_TWIN_CLIENT_12_003: [The `getDigitalTwin` method shall call the callback with an error parameter if a callback is passed..]*/
        /*Codes_SRS_NODE_DIGITAL_TWIN_CLIENT_12_004: [The `getDigitalTwin` method shall return error if the method of the protocol layer failed.]*/
        /*Codes_SRS_NODE_DIGITAL_TWIN_CLIENT_12_020: [The `getDigitalTwin` method shall return a promise if there is no callback passed.]*/
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            this._pl.digitalTwin.getDigitalTwin(digitalTwinId, (err, result, _request, response) => {
                _callback(err, result, response);
            });
        }, (result, response) => createResultWithHttpOperationResponse(result, response), callback);
    }
    updateDigitalTwin(digitalTwinId, patch, eTagOrCallback, callback) {
        const actualCallback = typeof eTagOrCallback === 'function' ? eTagOrCallback : callback;
        const actualEtag = typeof eTagOrCallback !== 'function' ? eTagOrCallback : undefined;
        const options = actualEtag ? { ifMatch: actualEtag } : undefined;
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            this._pl.digitalTwin.updateDigitalTwin(digitalTwinId, patch, options, (err, result, _request, response) => {
                _callback(err, result, response);
            });
        }, (result, response) => createResultWithHttpOperationResponse(result, response), actualCallback);
    }
    /**
     * @method invokeComponentCommand                                         module: azure-iot-digitaltwins-service.DigitalTwinClient.invokeComponentCommand
     * @description                                                           Invoke a command on an component of a particular device and get the result of it.
     * @param {string} digitalTwinId                                          The digital twin Id of the given device.
     * @param {string} componentName                                          The component's name.
     * @param {string} commandName                                            The command's name.
     * @param {string|any} argument                                           The argument of a command.
     * @param {DigitalTwinInvokeComponentCommandOptionalParams} options       The optional parameter to set options including connectionTimeoutInSeconds and responseTimeoutInSeconds.
     *                                                                        The responseTimeoutInSeconds must be within [5; 300]
     * @returns DigitalTwinInvokeComponentCommandResponse                     The result of the invoked command containing the result, status code, request ID and the parsed HttpResponse.
     * @memberof DigitalTwinClient
     */
    invokeComponentCommand(digitalTwinId, componentName, commandName, argument, options) {
        return this._pl.digitalTwin.invokeComponentCommand(digitalTwinId, componentName, commandName, argument, options);
    }
    /**
     * @method invokeCommand                                                  module: azure-iot-digitaltwins-service.DigitalTwinClient.invokeCommand
     * @description                                                           Invoke a command on an component of a particular device and get the result of it.
     * @param {string} digitalTwinId                                          The digital twin Id of the given device.
     * @param {string|any} argument                                           The argument of a command.
     * @param {DigitalTwinInvokeRootLevelCommandOptionalParams} options       The optional parameter to set options including connectionTimeoutInSeconds and responseTimeoutInSeconds.
     *                                                                        The responseTimeoutInSeconds must be within [5; 300]
     * @returns DigitalTwinInvokeRootLevelCommandResponse                     The result of the invoked command containing the result, status code, request ID and the parsed HttpResponse.
     * @memberof DigitalTwinClient
     */
    invokeCommand(digitalTwinId, commandName, argument, options) {
        return this._pl.digitalTwin.invokeRootLevelCommand(digitalTwinId, commandName, argument, options);
    }
}
exports.DigitalTwinClient = DigitalTwinClient;
//# sourceMappingURL=digital_twin_client.js.map