// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.createJobStatusResponse = exports.JobClient = void 0;
const https_1 = require("https");
const ConnectionString = require("./connection_string");
const SharedAccessSignature = require("./shared_access_signature");
const rest_api_client_1 = require("./common-http/rest_api_client");
const device_method_1 = require("./device_method");
const query_1 = require("./query");
const promise_utils_1 = require("./common-core/promise_utils");
const version_1 = require("./version");
// eslint-disable-next-line @typescript-eslint/no-var-requires
const packageJson = require('../package.json');
const defaultMaxExecutionTimeInSeconds = 3600;
/**
 * Provides methods to create, update, monitor and cancel long-running jobs on an IoT Hub instance, as well as query existing jobs.
 * The Jobs API in Azure IoT Hub allows to schedule direct method calls and twin updates on multiple devices.
 *
 * SDK users are expected to create {@link azure-iothub.JobClient} instances using the factory methods {@link azure-iothub.JobClient.fromConnectionString} and {@link azure-iothub.JobClient.fromSharedAccessSignature}.
 */
class JobClient {
    /**
     * @private
     * @constructor
     * @param {RestApiClient}     restApiClient   The HTTP registry client used to execute REST API calls.@constructor
     * @throws {ReferenceError}   If the restApiClient argument is falsy.
     */
    constructor(restApiClient) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_001: [The `JobClient` constructor shall throw a `ReferenceError` if `restApiClient` is falsy.]*/
        if (!restApiClient)
            throw new ReferenceError('restApiClient cannot be \'' + restApiClient + '\'');
        this._restApiClient = restApiClient;
        if (this._restApiClient.setOptions) {
            this._restApiClient.setOptions({ http: { agent: new https_1.Agent({ keepAlive: true }) } });
        }
    }
    getJob(jobId, done) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_006: [The `getJob` method shall throw a `ReferenceError` if `jobId` is `null`, `undefined` or an empty string.]*/
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            if (jobId === undefined || jobId === null || jobId === '')
                throw new ReferenceError('jobId cannot be \'' + jobId + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_007: [The `getJob` method shall construct the HTTP request as follows:
            ```
            GET /jobs/v2/<jobId>?api-version=<version>
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            User-Agent: <sdk-name>/<sdk-version>
            ```]*/
            const path = '/jobs/v2/' + jobId + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, createJobStatusResponse, done);
    }
    /**
     * @method            module:azure-iothub.JobClient#createQuery
     * @description       Creates a query that can be used to return pages of existing job based on type and status.
     *
     * @param {String}    jobType     The type that should be used to filter results.
     * @param {String}    jobStatus   The status that should be used to filter results.
     * @param {Number}    pageSize    The number of elements to return per page.
     */
    createQuery(jobType, jobStatus, pageSize) {
        return new query_1.Query(this._getJobsFunc(jobType, jobStatus, pageSize));
    }
    cancelJob(jobId, done) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_008: [The `cancelJob` method shall throw a `ReferenceError` if `jobId` is `null`, `undefined` or an empty string.]*/
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            if (jobId === undefined || jobId === null || jobId === '')
                throw new ReferenceError('jobId cannot be \'' + jobId + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_009: [The `cancelJob` method shall construct the HTTP request as follows:
            ```
            POST /jobs/v2/<jobId>/cancel?api-version=<version>
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            User-Agent: <sdk-name>/<sdk-version>
            ```]*/
            const path = '/jobs/v2/' + jobId + '/cancel' + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('POST', path, null, null, _callback);
        }, createJobStatusResponse, done);
    }
    scheduleDeviceMethod(jobId, queryCondition, methodParams, jobStartTime, maxExecutionTimeInSeconds, done) {
        const callback = (typeof jobStartTime === 'function') ? jobStartTime : ((typeof maxExecutionTimeInSeconds === 'function') ? maxExecutionTimeInSeconds : done);
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_JOB_CLIENT_16_013: [The `scheduleDeviceMethod` method shall throw a `ReferenceError` if `jobId` is `null`, `undefined` or an empty string.]*/
            if (jobId === undefined || jobId === null || jobId === '')
                throw new ReferenceError('jobId cannot be \'' + jobId + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_014: [The `scheduleDeviceMethod` method shall throw a `ReferenceError` if `queryCondition` is falsy.]*/
            if (!queryCondition)
                throw new ReferenceError('queryCondition cannot be \'' + queryCondition + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_029: [The `scheduleDeviceMethod` method shall throw a `ReferenceError` if `methodParams` is falsy.*/
            if (!methodParams)
                throw new ReferenceError('methodParams cannot be \'' + methodParams + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_015: [The `scheduleDeviceMethod` method shall throw a `ReferenceError` if `methodParams.methodName` is `null`, `undefined` or an empty string.]*/
            if (methodParams.methodName === undefined || methodParams.methodName === null || methodParams.methodName === '')
                throw new ReferenceError('methodParams.methodName cannot be \'' + methodParams.methodName + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_018: [If `jobStartTime` is a function, `jobStartTime` shall be considered the callback and a `TypeError` shall be thrown if `maxExecutionTimeInSeconds` and/or `_callback` are not `undefined`.]*/
            if (typeof jobStartTime === 'function') {
                if (maxExecutionTimeInSeconds || done) {
                    throw new TypeError('The callback must be the last parameter');
                }
                else {
                    _callback = jobStartTime;
                    jobStartTime = new Date();
                    maxExecutionTimeInSeconds = defaultMaxExecutionTimeInSeconds;
                }
                /*Codes_SRS_NODE_JOB_CLIENT_16_019: [If `maxExecutionTimeInSeconds` is a function, `maxExecutionTimeInSeconds` shall be considered the callback and a `TypeError` shall be thrown if `_callback` is not `undefined`.]*/
            }
            else if (typeof maxExecutionTimeInSeconds === 'function') {
                if (done) {
                    throw new TypeError('The callback must be the last parameter');
                }
                else {
                    _callback = maxExecutionTimeInSeconds;
                    maxExecutionTimeInSeconds = defaultMaxExecutionTimeInSeconds;
                }
            }
            /*Codes_SRS_NODE_JOB_CLIENT_16_030: [The `scheduleDeviceMethod` method shall use the `DeviceMethod.defaultPayload` value if `methodParams.payload` is `undefined`.]*/
            /*Codes_SRS_NODE_JOB_CLIENT_16_031: [The `scheduleDeviceMethod` method shall use the `DeviceMethod.defaultTimeout` value if `methodParams.responseTimeoutInSeconds` is falsy.]*/
            const fullMethodParams = {
                methodName: methodParams.methodName,
                payload: methodParams.payload || device_method_1.DeviceMethod.defaultPayload,
                responseTimeoutInSeconds: methodParams.responseTimeoutInSeconds || device_method_1.DeviceMethod.defaultResponseTimeout
            };
            /*Codes_SRS_NODE_JOB_CLIENT_16_020: [The `scheduleDeviceMethod` method shall construct the HTTP request as follows:
            ```
            PUT /jobs/v2/<jobId>?api-version=<version>
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            User-Agent: <sdk-name>/<sdk-version>
      
            {
              jobId: '<jobId>',
              type: 'scheduleDirectRequest', // TBC
              cloudToDeviceMethod: {
                methodName: '<methodName>',
                payload: <payload>,           // valid JSON object
                timeoutInSeconds: methodTimeoutInSeconds // Number
              },
              queryCondition: '<queryCondition>', // if the query parameter is a string
              startTime: <jobStartTime>,          // as an ISO-8601 date string
              maxExecutionTimeInSeconds: <maxExecutionTimeInSeconds>        // format TBD
            }
            ```]*/
            const jobDesc = {
                jobId: jobId,
                type: 'scheduleDeviceMethod',
                cloudToDeviceMethod: fullMethodParams,
                startTime: jobStartTime ? jobStartTime.toISOString() : null,
                maxExecutionTimeInSeconds: maxExecutionTimeInSeconds
            };
            if (typeof queryCondition === 'string') {
                jobDesc.queryCondition = queryCondition;
            }
            else {
                throw new TypeError('queryCondition must be a sql WHERE clause string');
            }
            this._scheduleJob(jobDesc, _callback);
        }, createJobStatusResponse, callback);
    }
    scheduleTwinUpdate(jobId, queryCondition, patch, jobStartTime, maxExecutionTimeInSeconds, done) {
        const callback = (typeof jobStartTime === 'function') ? jobStartTime : ((typeof maxExecutionTimeInSeconds === 'function') ? maxExecutionTimeInSeconds : done);
        return (0, promise_utils_1.tripleValueCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_JOB_CLIENT_16_021: [The `scheduleTwinUpdate` method shall throw a `ReferenceError` if `jobId` is `null`, `undefined` or an empty string.]*/
            if (jobId === undefined || jobId === null || jobId === '')
                throw new ReferenceError('jobId cannot be \'' + jobId + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_022: [The `scheduleTwinUpdate` method shall throw a `ReferenceError` if `query` is falsy.]*/
            if (!queryCondition)
                throw new ReferenceError('queryCondition cannot be \'' + queryCondition + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_023: [The `scheduleTwinUpdate` method shall throw a `ReferenceError` if `patch` is falsy.]*/
            if (!patch)
                throw new ReferenceError('patch cannot be \'' + patch + '\'');
            /*Codes_SRS_NODE_JOB_CLIENT_16_024: [If `jobStartTime` is a function, `jobStartTime` shall be considered the callback and a `TypeError` shall be thrown if `maxExecutionTimeInSeconds` and/or `_callback` are not `undefined`.]*/
            if (typeof jobStartTime === 'function') {
                if (maxExecutionTimeInSeconds || done) {
                    throw new TypeError('The callback must be the last parameter');
                }
                else {
                    _callback = jobStartTime;
                    jobStartTime = new Date();
                    maxExecutionTimeInSeconds = defaultMaxExecutionTimeInSeconds;
                }
                /*Codes_SRS_NODE_JOB_CLIENT_16_025: [If `maxExecutionTimeInSeconds` is a function, `maxExecutionTimeInSeconds` shall be considered the callback and a `TypeError` shall be thrown if `_callback` is not `undefined`.]*/
            }
            else if (typeof maxExecutionTimeInSeconds === 'function') {
                if (done) {
                    throw new TypeError('The callback must be the last parameter');
                }
                else {
                    _callback = maxExecutionTimeInSeconds;
                    maxExecutionTimeInSeconds = defaultMaxExecutionTimeInSeconds;
                }
            }
            const jobDesc = {
                jobId: jobId,
                type: 'scheduleUpdateTwin',
                updateTwin: patch,
                startTime: jobStartTime ? jobStartTime.toISOString() : null,
                maxExecutionTimeInSeconds: maxExecutionTimeInSeconds
            };
            if (typeof queryCondition === 'string') {
                jobDesc.queryCondition = queryCondition;
            }
            else {
                throw new TypeError('queryCondition must be a sql WHERE clause string');
            }
            /*Codes_SRS_NODE_JOB_CLIENT_16_026: [The `scheduleTwinUpdate` method shall construct the HTTP request as follows:
            ```
            PUT /jobs/v2/<jobId>?api-version=<version>
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            User-Agent: <sdk-name>/<sdk-version>
      
            {
              jobId: '<jobId>',
              type: 'scheduleTwinUpdate', // TBC
              updateTwin: <patch>                 // Valid JSON object
              queryCondition: '<queryCondition>', // if the query parameter is a string
              startTime: <jobStartTime>,          // as an ISO-8601 date string
              maxExecutionTimeInSeconds: <maxExecutionTimeInSeconds>        // format TBD
            }
            ```]*/
            this._scheduleJob(jobDesc, _callback);
        }, createJobStatusResponse, callback);
    }
    _getJobsFunc(jobType, jobStatus, pageSize) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_035: [The `_getJobsFunc` function shall return a function that can be used by the `Query` object to get a new page of results]*/
        return (continuationToken, done) => {
            /*Codes_SRS_NODE_JOB_CLIENT_16_012: [The `_getJobsFunc` method shall construct the HTTP request as follows:
            ```
            GET /jobs/v2/query?api-version=<version>[&jobType=<jobType>][&jobStatus=<jobStatus>][&pageSize=<pageSize>][&continuationToken=<continuationToken>]
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            User-Agent: <sdk-name>/<sdk-version>
            ```]*/
            const jobStatusQueryParam = jobStatus ? '&jobStatus=' + encodeURIComponent(jobStatus) : '';
            const jobTypeQueryParam = jobType ? '&jobType=' + encodeURIComponent(jobType) : '';
            const path = '/jobs/v2/query' + (0, version_1.versionQueryString)() + jobStatusQueryParam + jobTypeQueryParam;
            const headers = {};
            if (continuationToken) {
                headers['x-ms-continuation'] = continuationToken;
            }
            if (pageSize) {
                headers['x-ms-max-item-count'] = pageSize;
            }
            this._restApiClient.executeApiCall('GET', path, headers, null, done);
        };
    }
    _scheduleJob(jobDesc, done) {
        const path = '/jobs/v2/' + encodeURIComponent(jobDesc.jobId.toString()) + (0, version_1.versionQueryString)();
        const headers = {
            'Content-Type': 'application/json; charset=utf-8'
        };
        /*Codes_SRS_NODE_JOB_CLIENT_16_027: [The method shall call the `done` callback with a single argument that is a standard Javascript `Error` object if the request failed.]*/
        /*Codes_SRS_NODE_JOB_CLIENT_16_028: [The method shall call the `done` callback with a `null` error argument, a result and a transport-specific response object if the request was successful.]*/
        this._restApiClient.executeApiCall('PUT', path, headers, jobDesc, done);
    }
    /**
     * @method          module:azure-iothub.JobClient.fromConnectionString
     * @description     Constructs a JobClient object from the given connection string.
     * @static
     *
     * @param   {String}          connectionString       A connection string which encapsulates the
     *                                                   appropriate (read and/or write) Registry
     *                                                   permissions.
     *
     * @throws  {ReferenceError}  If the connectionString argument is falsy.
     *
     * @returns {module:azure-iothub.JobClient}
     */
    static fromConnectionString(connectionString) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_002: [The `fromConnectionString` method shall throw a `ReferenceError` if `connectionString` is falsy.]*/
        if (!connectionString)
            throw new ReferenceError('connectionString cannot be \'' + connectionString + '\'');
        const cn = ConnectionString.parse(connectionString);
        const config = {
            host: cn.HostName,
            sharedAccessSignature: SharedAccessSignature.create(cn.HostName, cn.SharedAccessKeyName, cn.SharedAccessKey, Date.now()),
            tokenCredential: undefined
        };
        /*Codes_SRS_NODE_JOB_CLIENT_16_003: [The `fromConnectionString` method shall return a new `JobClient` instance.]*/
        return new JobClient(new rest_api_client_1.RestApiClient(config, packageJson.name + '/' + packageJson.version));
    }
    /**
     * @method            module:azure-iothub.JobClient.fromSharedAccessSignature
     * @description       Constructs a JobClient object from the given shared access signature.
     * @static
     *
     * @param {String}    sharedAccessSignature     A shared access signature which encapsulates
     *                                              the appropriate (read and/or write) Registry
     *                                              permissions.
     *
     * @throws  {ReferenceError}  If the sharedAccessSignature argument is falsy.
     *
     * @returns {module:azure-iothub.JobClient}
     */
    static fromSharedAccessSignature(sharedAccessSignature) {
        /*Codes_SRS_NODE_JOB_CLIENT_16_004: [The `fromSharedAccessSignature` method shall throw a `ReferenceError` if `sharedAccessSignature` is falsy.]*/
        if (!sharedAccessSignature)
            throw new ReferenceError('sharedAccessSignature cannot be \'' + sharedAccessSignature + '\'');
        const sas = SharedAccessSignature.parse(sharedAccessSignature);
        const config = {
            host: sas.sr,
            sharedAccessSignature: sharedAccessSignature,
            tokenCredential: undefined
        };
        /*Codes_SRS_NODE_JOB_CLIENT_16_005: [The `fromSharedAccessSignature` method shall return a new `JobClient` instance.]*/
        return new JobClient(new rest_api_client_1.RestApiClient(config, packageJson.name + '/' + packageJson.version));
    }
    /**
     * @method            module:azure-iothub.JobClient.fromTokenCredential
     * @description       Constructs a JobClient object from the given shared Azure TokenCredential.
     * @static
     *
     * @param {String}    hostName                  Host name of the Azure service.
     * @param {String}    tokenCredential           An Azure TokenCredential used to authenticate
     *                                              with the Azure  service
     *
     * @throws  {ReferenceError}  If the tokenCredential argument is falsy.
     *
     * @returns {module:azure-iothub.JobClient}
     */
    static fromTokenCredential(hostName, tokenCredential) {
        const config = {
            host: hostName,
            tokenCredential,
            tokenScope: 'https://iothubs.azure.net/.default'
        };
        return new JobClient(new rest_api_client_1.RestApiClient(config, packageJson.name + '/' + packageJson.version));
    }
}
exports.JobClient = JobClient;
function createJobStatusResponse(jobStatus, response) {
    return { jobStatus: jobStatus, response: response };
}
exports.createJobStatusResponse = createJobStatusResponse;
//# sourceMappingURL=job_client.js.map