"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs = __importStar(require("fs"));
const http = __importStar(require("http"));
const https = __importStar(require("https"));
const basic_auth_1 = __importDefault(require("basic-auth"));
const url = __importStar(require("url"));
const TD = __importStar(require("@node-wot/td-tools"));
const core_1 = require("@node-wot/core");
const oauth_token_validation_1 = __importDefault(require("./oauth-token-validation"));
const slugify_1 = __importDefault(require("slugify"));
const acceptLanguageParser = __importStar(require("accept-language-parser"));
class HttpServer {
    constructor(config = {}) {
        var _a;
        this.PROPERTY_DIR = "properties";
        this.ACTION_DIR = "actions";
        this.EVENT_DIR = "events";
        this.OBSERVABLE_DIR = "observable";
        this.port = 8080;
        this.address = undefined;
        this.baseUri = undefined;
        this.httpSecurityScheme = "NoSec";
        this.validOAuthClients = /.*/g;
        this.server = null;
        this.things = new Map();
        this.servient = null;
        if (typeof config !== "object") {
            throw new Error(`HttpServer requires config object (got ${typeof config})`);
        }
        if (config.port !== undefined) {
            this.port = config.port;
        }
        const environmentObj = ["WOT_PORT", "PORT"]
            .map((envVar) => {
            return { key: envVar, value: process.env[envVar] };
        })
            .find((envObj) => envObj.value != null);
        if (environmentObj) {
            console.info("[binding-http]", `HttpServer Port Overridden to ${environmentObj.value} by Environment Variable ${environmentObj.key}`);
            this.port = +environmentObj.value;
        }
        if (config.address !== undefined) {
            this.address = config.address;
        }
        if (config.baseUri !== undefined) {
            this.baseUri = config.baseUri;
        }
        if (config.serverKey && config.serverCert) {
            const options = {};
            options.key = fs.readFileSync(config.serverKey);
            options.cert = fs.readFileSync(config.serverCert);
            this.scheme = "https";
            this.server = https.createServer(options, (req, res) => {
                this.handleRequest(req, res);
            });
        }
        else {
            this.scheme = "http";
            this.server = http.createServer((req, res) => {
                this.handleRequest(req, res);
            });
        }
        if (config.security) {
            switch (config.security.scheme) {
                case "nosec":
                    this.httpSecurityScheme = "NoSec";
                    break;
                case "basic":
                    this.httpSecurityScheme = "Basic";
                    break;
                case "digest":
                    this.httpSecurityScheme = "Digest";
                    break;
                case "bearer":
                    this.httpSecurityScheme = "Bearer";
                    break;
                case "oauth2":
                    {
                        this.httpSecurityScheme = "OAuth";
                        const oAuthConfig = config.security;
                        this.validOAuthClients = new RegExp((_a = oAuthConfig.allowedClients) !== null && _a !== void 0 ? _a : ".*");
                        this.oAuthValidator = (0, oauth_token_validation_1.default)(oAuthConfig.method);
                    }
                    break;
                default:
                    throw new Error(`HttpServer does not support security scheme '${config.security.scheme}`);
            }
        }
    }
    start(servient) {
        console.info("[binding-http]", `HttpServer starting on ${this.address !== undefined ? this.address + " " : ""}port ${this.port}`);
        return new Promise((resolve, reject) => {
            this.servient = servient;
            this.server.setTimeout(60 * 60 * 1000, () => {
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} timed out connection`);
            });
            this.server.keepAliveTimeout = 0;
            this.server.once("error", (err) => {
                reject(err);
            });
            this.server.once("listening", () => {
                this.server.on("error", (err) => {
                    console.error("[binding-http]", `HttpServer on port ${this.port} failed: ${err.message}`);
                });
                resolve();
            });
            this.server.listen(this.port, this.address);
        });
    }
    stop() {
        console.info("[binding-http]", `HttpServer stopping on port ${this.getPort()}`);
        return new Promise((resolve, reject) => {
            this.server.once("error", (err) => {
                reject(err);
            });
            this.server.once("close", () => {
                resolve();
            });
            this.server.close();
        });
    }
    getServer() {
        return this.server;
    }
    getPort() {
        if (this.server.address() && typeof this.server.address() === "object") {
            return this.server.address().port;
        }
        else {
            return -1;
        }
    }
    getHttpSecurityScheme() {
        return this.httpSecurityScheme;
    }
    updateInteractionNameWithUriVariablePattern(interactionName, uriVariables) {
        if (uriVariables && Object.keys(uriVariables).length > 0) {
            let pattern = "{?";
            let index = 0;
            for (const key in uriVariables) {
                if (index !== 0) {
                    pattern += ",";
                }
                pattern += encodeURIComponent(key);
                index++;
            }
            pattern += "}";
            return encodeURIComponent(interactionName) + pattern;
        }
        else {
            return encodeURIComponent(interactionName);
        }
    }
    expose(thing, tdTemplate) {
        return __awaiter(this, void 0, void 0, function* () {
            let urlPath = (0, slugify_1.default)(thing.title, { lower: true });
            if (this.things.has(urlPath)) {
                urlPath = core_1.Helpers.generateUniqueName(urlPath);
            }
            if (this.getPort() !== -1) {
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} exposes '${thing.title}' as unique '/${urlPath}'`);
                this.things.set(urlPath, thing);
                if (this.baseUri !== undefined) {
                    const base = this.baseUri.concat("/", encodeURIComponent(urlPath));
                    console.info("[binding-http]", "HttpServer TD hrefs using baseUri " + this.baseUri);
                    this.addEndpoint(thing, tdTemplate, base);
                }
                else {
                    for (const address of core_1.Helpers.getAddresses()) {
                        const base = this.scheme + "://" + address + ":" + this.getPort() + "/" + encodeURIComponent(urlPath);
                        this.addEndpoint(thing, tdTemplate, base);
                    }
                    if (this.scheme === "https") {
                        this.fillSecurityScheme(thing);
                    }
                }
            }
        });
    }
    destroy(thingId) {
        console.debug("[binding-http]", `HttpServer on port ${this.getPort()} destroying thingId '${thingId}'`);
        return new Promise((resolve, reject) => {
            let removedThing;
            for (const name of Array.from(this.things.keys())) {
                const expThing = this.things.get(name);
                if ((expThing === null || expThing === void 0 ? void 0 : expThing.id) === thingId) {
                    this.things.delete(name);
                    removedThing = expThing;
                }
            }
            if (removedThing) {
                console.info("[binding-http]", `HttpServer succesfully destroyed '${removedThing.title}'`);
            }
            else {
                console.info("[binding-http]", `HttpServer failed to destroy thing with thingId '${thingId}'`);
            }
            resolve(removedThing !== undefined);
        });
    }
    addEndpoint(thing, tdTemplate, base) {
        for (const type of core_1.ContentSerdes.get().getOfferedMediaTypes()) {
            let allReadOnly = true;
            let allWriteOnly = true;
            let anyProperties = false;
            for (const propertyName in thing.properties) {
                anyProperties = true;
                if (!thing.properties[propertyName].readOnly) {
                    allReadOnly = false;
                }
                else if (!thing.properties[propertyName].writeOnly) {
                    allWriteOnly = false;
                }
            }
            if (anyProperties) {
                const href = base + "/" + this.PROPERTY_DIR;
                const form = new TD.Form(href, type);
                if (allReadOnly) {
                    form.op = ["readallproperties", "readmultipleproperties"];
                }
                else if (allWriteOnly) {
                    form.op = ["writeallproperties", "writemultipleproperties"];
                }
                else {
                    form.op = [
                        "readallproperties",
                        "readmultipleproperties",
                        "writeallproperties",
                        "writemultipleproperties",
                    ];
                }
                if (!thing.forms) {
                    thing.forms = [];
                }
                thing.forms.push(form);
            }
            for (const propertyName in thing.properties) {
                const propertyNamePattern = this.updateInteractionNameWithUriVariablePattern(propertyName, thing.properties[propertyName].uriVariables);
                const href = base + "/" + this.PROPERTY_DIR + "/" + propertyNamePattern;
                const form = new TD.Form(href, type);
                core_1.ProtocolHelpers.updatePropertyFormWithTemplate(form, tdTemplate, propertyName);
                if (thing.properties[propertyName].readOnly) {
                    form.op = ["readproperty"];
                    const hform = form;
                    if (hform["htv:methodName"] === undefined) {
                        hform["htv:methodName"] = "GET";
                    }
                }
                else if (thing.properties[propertyName].writeOnly) {
                    form.op = ["writeproperty"];
                    const hform = form;
                    if (hform["htv:methodName"] === undefined) {
                        hform["htv:methodName"] = "PUT";
                    }
                }
                else {
                    form.op = ["readproperty", "writeproperty"];
                }
                thing.properties[propertyName].forms.push(form);
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} assigns '${href}' to Property '${propertyName}'`);
                if (thing.properties[propertyName].observable) {
                    const href = base +
                        "/" +
                        this.PROPERTY_DIR +
                        "/" +
                        encodeURIComponent(propertyName) +
                        "/" +
                        this.OBSERVABLE_DIR;
                    const form = new TD.Form(href, type);
                    form.op = ["observeproperty", "unobserveproperty"];
                    form.subprotocol = "longpoll";
                    thing.properties[propertyName].forms.push(form);
                    console.debug("[binding-http]", `HttpServer on port ${this.getPort()} assigns '${href}' to observable Property '${propertyName}'`);
                }
            }
            for (const actionName in thing.actions) {
                const actionNamePattern = this.updateInteractionNameWithUriVariablePattern(actionName, thing.actions[actionName].uriVariables);
                const href = base + "/" + this.ACTION_DIR + "/" + actionNamePattern;
                const form = new TD.Form(href, type);
                core_1.ProtocolHelpers.updateActionFormWithTemplate(form, tdTemplate, actionName);
                form.op = ["invokeaction"];
                const hform = form;
                if (hform["htv:methodName"] === undefined) {
                    hform["htv:methodName"] = "POST";
                }
                thing.actions[actionName].forms.push(form);
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} assigns '${href}' to Action '${actionName}'`);
            }
            for (const eventName in thing.events) {
                const eventNamePattern = this.updateInteractionNameWithUriVariablePattern(eventName, thing.events[eventName].uriVariables);
                const href = base + "/" + this.EVENT_DIR + "/" + eventNamePattern;
                const form = new TD.Form(href, type);
                core_1.ProtocolHelpers.updateEventFormWithTemplate(form, tdTemplate, eventName);
                form.subprotocol = "longpoll";
                form.op = ["subscribeevent", "unsubscribeevent"];
                thing.events[eventName].forms.push(form);
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} assigns '${href}' to Event '${eventName}'`);
            }
        }
    }
    checkCredentials(thing, req) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            console.debug("[binding-http]", `HttpServer on port ${this.getPort()} checking credentials for '${thing.id}'`);
            const creds = this.servient.getCredentials(thing.id);
            switch (this.httpSecurityScheme) {
                case "NoSec":
                    return true;
                case "Basic": {
                    const basic = (0, basic_auth_1.default)(req);
                    const basicCreds = creds;
                    return (creds !== undefined &&
                        basic !== undefined &&
                        basic.name === basicCreds.username &&
                        basic.pass === basicCreds.password);
                }
                case "Digest":
                    return false;
                case "OAuth": {
                    const oAuthScheme = thing.securityDefinitions[thing.security[0]];
                    const scopes = (_a = oAuthScheme.scopes) !== null && _a !== void 0 ? _a : [];
                    let valid = false;
                    try {
                        valid = yield this.oAuthValidator.validate(req, scopes, this.validOAuthClients);
                    }
                    catch (error) {
                        console.error("OAuth authorization error; sending unauthorized response error");
                        console.error("this was possibly caused by a misconfiguration of the server");
                        console.error(error);
                    }
                    return valid;
                }
                case "Bearer": {
                    if (req.headers.authorization === undefined)
                        return false;
                    const auth = req.headers.authorization.split(" ");
                    const bearerCredentials = creds;
                    return auth[0] === "Bearer" && creds !== undefined && auth[1] === bearerCredentials.token;
                }
                default:
                    return false;
            }
        });
    }
    fillSecurityScheme(thing) {
        if (thing.securityDefinitions) {
            const secCandidate = Object.keys(thing.securityDefinitions).find((key) => {
                let scheme = thing.securityDefinitions[key].scheme;
                scheme = scheme === "oauth2" ? scheme.split("2")[0] : scheme;
                return scheme === this.httpSecurityScheme.toLowerCase();
            });
            if (!secCandidate) {
                throw new Error("Servient does not support thing security schemes. Current scheme supported: " +
                    this.httpSecurityScheme +
                    " secCandidate " +
                    Object.keys(thing.securityDefinitions).join(", "));
            }
            const selectedSecurityScheme = thing.securityDefinitions[secCandidate];
            thing.securityDefinitions = {};
            thing.securityDefinitions[secCandidate] = selectedSecurityScheme;
            thing.security = [secCandidate];
        }
        else {
            thing.securityDefinitions = {
                noSec: { scheme: "nosec" },
            };
            thing.security = ["noSec"];
        }
    }
    handleRequest(req, res) {
        return __awaiter(this, void 0, void 0, function* () {
            const requestUri = url.parse(req.url);
            console.debug("[binding-http]", `HttpServer on port ${this.getPort()} received '${req.method} ${requestUri.pathname}' from ${core_1.Helpers.toUriLiteral(req.socket.remoteAddress)}:${req.socket.remotePort}`);
            res.on("finish", () => {
                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} replied with '${res.statusCode}' to ${core_1.Helpers.toUriLiteral(req.socket.remoteAddress)}:${req.socket.remotePort}`);
            });
            function respondUnallowedMethod(res, allowed) {
                if (!allowed.includes("OPTIONS")) {
                    allowed += ", OPTIONS";
                }
                if (req.method === "OPTIONS" && req.headers.origin && req.headers["access-control-request-method"]) {
                    console.debug("[binding-http]", `HttpServer received an CORS preflight request from ${core_1.Helpers.toUriLiteral(req.socket.remoteAddress)}:${req.socket.remotePort}`);
                    res.setHeader("Access-Control-Allow-Methods", allowed);
                    res.setHeader("Access-Control-Allow-Headers", "content-type, authorization, *");
                    res.writeHead(200);
                    res.end();
                }
                else {
                    res.setHeader("Allow", allowed);
                    res.writeHead(405);
                    res.end("Method Not Allowed");
                }
            }
            res.setHeader("Access-Control-Allow-Origin", "*");
            const contentTypeHeader = req.headers["content-type"];
            let contentType = Array.isArray(contentTypeHeader) ? contentTypeHeader[0] : contentTypeHeader;
            if (req.method === "PUT" || req.method === "POST") {
                if (!contentType) {
                    console.warn("[binding-http]", `HttpServer on port ${this.getPort()} received no Content-Type from ${core_1.Helpers.toUriLiteral(req.socket.remoteAddress)}:${req.socket.remotePort}`);
                    contentType = core_1.ContentSerdes.DEFAULT;
                }
                else if (core_1.ContentSerdes.get().getSupportedMediaTypes().indexOf(core_1.ContentSerdes.getMediaType(contentType)) < 0) {
                    res.writeHead(415);
                    res.end("Unsupported Media Type");
                    return;
                }
            }
            let segments;
            try {
                segments = decodeURI(requestUri.pathname).split("/");
            }
            catch (ex) {
                console.warn("[binding-http]", `HttpServer on port ${this.getPort()} cannot decode URI for '${requestUri.pathname}'`);
                res.writeHead(400);
                res.end("decodeURI error for " + requestUri.pathname);
                return;
            }
            if (segments[1] === "") {
                if (req.method === "GET") {
                    res.setHeader("Content-Type", core_1.ContentSerdes.DEFAULT);
                    res.writeHead(200);
                    const list = [];
                    for (const address of core_1.Helpers.getAddresses()) {
                        for (const name of Array.from(this.things.keys())) {
                            if (name) {
                                list.push(this.scheme +
                                    "://" +
                                    core_1.Helpers.toUriLiteral(address) +
                                    ":" +
                                    this.getPort() +
                                    "/" +
                                    encodeURIComponent(name));
                            }
                        }
                    }
                    res.end(JSON.stringify(list));
                }
                else {
                    respondUnallowedMethod(res, "GET");
                }
                return;
            }
            else {
                const thing = this.things.get(segments[1]);
                if (thing) {
                    if (segments.length === 2 || segments[2] === "") {
                        if (req.method === "GET") {
                            const td = thing.getThingDescription();
                            if (req.headers["accept-language"] && req.headers["accept-language"] !== "*") {
                                if (thing.titles) {
                                    const supportedLanguagesArray = [];
                                    for (const lang in thing.titles) {
                                        supportedLanguagesArray.push(lang);
                                    }
                                    const prefLang = acceptLanguageParser.pick(supportedLanguagesArray, req.headers["accept-language"], { loose: true });
                                    if (prefLang) {
                                        console.debug("[binding-http]", `TD language negotiation through the Accept-Language header field of HTTP leads to "${prefLang}"`);
                                        this.resetMultiLangThing(td, prefLang);
                                    }
                                }
                            }
                            res.setHeader("Content-Type", core_1.ContentSerdes.TD);
                            res.writeHead(200);
                            res.end(JSON.stringify(td));
                        }
                        else {
                            respondUnallowedMethod(res, "GET");
                        }
                        return;
                    }
                    else {
                        if (this.httpSecurityScheme !== "NoSec" && !(yield this.checkCredentials(thing, req))) {
                            res.setHeader("WWW-Authenticate", `${this.httpSecurityScheme} realm="${thing.id}"`);
                            res.writeHead(401);
                            res.end();
                            return;
                        }
                        if (segments[2] === this.PROPERTY_DIR) {
                            if (segments.length === 3) {
                                if (req.method === "GET") {
                                    try {
                                        const propMap = yield thing.handleReadAllProperties({
                                            formIndex: 0,
                                        });
                                        res.setHeader("Content-Type", "application/json");
                                        res.writeHead(200);
                                        const recordResponse = {};
                                        for (const key of propMap.keys()) {
                                            const content = propMap.get(key);
                                            const data = yield core_1.ProtocolHelpers.readStreamFully(content.body);
                                            recordResponse[key] = data.toString();
                                        }
                                        res.end(JSON.stringify(recordResponse));
                                    }
                                    catch (err) {
                                        console.error("[binding-http]", `HttpServer on port ${this.getPort()} got internal error on invoke '${requestUri.pathname}': ${err.message}`);
                                        res.writeHead(500);
                                        res.end(err.message);
                                    }
                                }
                                else {
                                    respondUnallowedMethod(res, "GET");
                                }
                                return;
                            }
                            else {
                                const property = thing.properties[segments[3]];
                                if (property) {
                                    const options = {
                                        formIndex: core_1.ProtocolHelpers.findRequestMatchingFormIndex(property.forms, this.scheme, req.url, contentType),
                                    };
                                    const uriVariables = core_1.Helpers.parseUrlParameters(req.url, thing.uriVariables, property.uriVariables);
                                    if (!this.isEmpty(uriVariables)) {
                                        options.uriVariables = uriVariables;
                                    }
                                    if (req.method === "GET") {
                                        if (segments[4] === this.OBSERVABLE_DIR) {
                                            const listener = (value) => __awaiter(this, void 0, void 0, function* () {
                                                try {
                                                    value.body.pipe(res);
                                                }
                                                catch (err) {
                                                    if ((err === null || err === void 0 ? void 0 : err.code) === "ERR_HTTP_HEADERS_SENT") {
                                                        thing.handleUnsubscribeEvent(segments[3], listener, options);
                                                        return;
                                                    }
                                                    console.warn("[binding-http]", `HttpServer on port ${this.getPort()} cannot process data for Event '${segments[3]}: ${err.message}'`);
                                                    res.writeHead(500);
                                                    res.end("Invalid Event Data");
                                                }
                                            });
                                            yield thing.handleObserveProperty(segments[3], listener, options);
                                            res.on("finish", () => {
                                                console.debug("[binding-http]", `HttpServer on port ${this.getPort()} closed connection`);
                                                thing.handleUnobserveProperty(segments[3], listener, options);
                                            });
                                            res.setTimeout(60 * 60 * 1000, () => thing.handleUnobserveProperty(segments[3], listener, options));
                                        }
                                        else {
                                            try {
                                                const content = yield thing.handleReadProperty(segments[3], options);
                                                res.setHeader("Content-Type", content.type);
                                                res.writeHead(200);
                                                content.body.pipe(res);
                                            }
                                            catch (err) {
                                                console.error("[binding-http]", `HttpServer on port ${this.getPort()} got internal error on read '${requestUri.pathname}': ${err.message}`);
                                                res.writeHead(500);
                                                res.end(err.message);
                                            }
                                            return;
                                        }
                                    }
                                    else if (req.method === "PUT") {
                                        if (!property.readOnly) {
                                            try {
                                                yield thing.handleWriteProperty(segments[3], { body: req, type: contentType }, options);
                                                res.writeHead(204);
                                                res.end("Changed");
                                            }
                                            catch (err) {
                                                console.error("[binding-http]", `HttpServer on port ${this.getPort()} got internal error on invoke '${requestUri.pathname}': ${err.message}`);
                                                res.writeHead(500);
                                                res.end(err.message);
                                            }
                                        }
                                        else {
                                            respondUnallowedMethod(res, "GET, PUT");
                                        }
                                        return;
                                    }
                                }
                            }
                        }
                        else if (segments[2] === this.ACTION_DIR) {
                            const action = thing.actions[segments[3]];
                            if (action) {
                                if (req.method === "POST") {
                                    const options = {
                                        formIndex: core_1.ProtocolHelpers.findRequestMatchingFormIndex(action.forms, this.scheme, req.url, contentType),
                                    };
                                    const uriVariables = core_1.Helpers.parseUrlParameters(req.url, thing.uriVariables, action.uriVariables);
                                    if (!this.isEmpty(uriVariables)) {
                                        options.uriVariables = uriVariables;
                                    }
                                    try {
                                        const output = yield thing.handleInvokeAction(segments[3], { body: req, type: contentType }, options);
                                        if (output) {
                                            res.setHeader("Content-Type", output.type);
                                            res.writeHead(200);
                                            output.body.pipe(res);
                                        }
                                        else {
                                            res.writeHead(200);
                                            res.end();
                                        }
                                    }
                                    catch (err) {
                                        console.error("[binding-http]", `HttpServer on port ${this.getPort()} got internal error on invoke '${requestUri.pathname}': ${err.message}`);
                                        res.writeHead(500);
                                        res.end(err.message);
                                    }
                                }
                                else {
                                    respondUnallowedMethod(res, "POST");
                                }
                                return;
                            }
                        }
                        else if (segments[2] === this.EVENT_DIR) {
                            const event = thing.events[segments[3]];
                            if (event) {
                                if (req.method === "GET") {
                                    const options = {
                                        formIndex: core_1.ProtocolHelpers.findRequestMatchingFormIndex(event.forms, this.scheme, req.url, contentType),
                                    };
                                    const uriVariables = core_1.Helpers.parseUrlParameters(req.url, thing.uriVariables, event.uriVariables);
                                    if (!this.isEmpty(uriVariables)) {
                                        options.uriVariables = uriVariables;
                                    }
                                    const listener = (value) => __awaiter(this, void 0, void 0, function* () {
                                        try {
                                            res.setHeader("Content-Type", value.type);
                                            res.writeHead(200);
                                            value.body.pipe(res);
                                        }
                                        catch (err) {
                                            if ((err === null || err === void 0 ? void 0 : err.code) === "ERR_HTTP_HEADERS_SENT") {
                                                thing.handleUnsubscribeEvent(segments[3], listener, options);
                                                return;
                                            }
                                            console.warn("[binding-http]", `HttpServer on port ${this.getPort()} cannot process data for Event '${segments[3]}: ${err.message}'`);
                                            res.writeHead(500);
                                            res.end("Invalid Event Data");
                                        }
                                    });
                                    yield thing.handleSubscribeEvent(segments[3], listener, options);
                                    res.on("finish", () => {
                                        console.debug("[binding-http]", `HttpServer on port ${this.getPort()} closed Event connection`);
                                        thing.handleUnsubscribeEvent(segments[3], listener, options);
                                    });
                                    res.setTimeout(60 * 60 * 1000, () => thing.handleUnsubscribeEvent(segments[3], listener, options));
                                }
                                else if (req.method === "HEAD") {
                                    res.writeHead(202);
                                    res.end();
                                }
                                else {
                                    respondUnallowedMethod(res, "GET");
                                }
                                return;
                            }
                        }
                    }
                }
            }
            res.writeHead(404);
            res.end("Not Found");
        });
    }
    isEmpty(obj) {
        for (const key in obj) {
            if (Object.prototype.hasOwnProperty.call(obj, key))
                return false;
        }
        return true;
    }
    resetMultiLangThing(thing, prefLang) {
        if (thing["@context"] && Array.isArray(thing["@context"])) {
            const arrayContext = thing["@context"];
            let languageSet = false;
            for (const arrayEntry of arrayContext) {
                if (arrayEntry instanceof Object) {
                    if (arrayEntry["@language"] !== undefined) {
                        arrayEntry["@language"] = prefLang;
                        languageSet = true;
                    }
                }
            }
            if (!languageSet) {
                arrayContext.push({
                    "@language": prefLang,
                });
            }
        }
        if (thing.titles) {
            for (const titleLang in thing.titles) {
                if (titleLang.startsWith(prefLang)) {
                    thing.title = thing.titles[titleLang];
                }
            }
        }
        if (thing.descriptions) {
            for (const titleLang in thing.descriptions) {
                if (titleLang.startsWith(prefLang)) {
                    thing.description = thing.descriptions[titleLang];
                }
            }
        }
        delete thing.titles;
        delete thing.descriptions;
        this.resetMultiLangInteraction(thing.properties, prefLang);
        this.resetMultiLangInteraction(thing.actions, prefLang);
        this.resetMultiLangInteraction(thing.events, prefLang);
    }
    resetMultiLangInteraction(interactions, prefLang) {
        if (interactions) {
            for (const interName in interactions) {
                delete interactions[interName].title;
                delete interactions[interName].description;
                if (interactions[interName].titles) {
                    for (const titleLang in interactions[interName].titles) {
                        if (titleLang.startsWith(prefLang)) {
                            interactions[interName].title = interactions[interName].titles[titleLang];
                        }
                    }
                }
                if (interactions[interName].descriptions) {
                    for (const descLang in interactions[interName].descriptions) {
                        if (descLang.startsWith(prefLang)) {
                            interactions[interName].description = interactions[interName].descriptions[descLang];
                        }
                    }
                }
                delete interactions[interName].titles;
                delete interactions[interName].descriptions;
            }
        }
    }
}
exports.default = HttpServer;
//# sourceMappingURL=http-server.js.map