/*
 * parse.c : config file & command line parser
 * version 0.0.3 - Copyright (C) 2002-2003 Alessandro Fausto
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *	Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer.
 *
 *	Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in the
 *	documentation and/or other materials provided with the distribution.
 *
 *	Neither the name of the author nor the names of the contributors may
 *	be used to endorse or promote products derived from this software 
 *	without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * To contact me write at:
 *             fareale@libero.it
 *   for bugs: fareale.bug@libero.it
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <limits.h>
#include <sys/utsname.h>

#include "debug.h"
#include "copying.h"      // license text
#include "parse.h"      
#include "command.h"

#define DEBUGNAME         "PARSE: "

// VARIABLES 
// 
extern char * pchProgramName ;
extern char * pchEventDevice ;

inline void ltrim( char * pchText)
{
	if( pchText) {
		char * pchTxt = pchText ;
    		while( pchTxt[ 0] == ' ') pchTxt++ ;
    		strcpy( pchText, pchTxt) ;
  	}
}

inline void rtrim( char * pchText)
{
  	if( pchText)
  	{
    		int i=strlen( pchText) ;
    		while( (pchText[ i] == ' ') && (i > 0) ) i-- ;
    		pchText[ i] = '\0' ;
 	}
}

inline void trim( char * pchText) 
{
  	ltrim( pchText) ;
  	rtrim( pchText) ;
}

int readCommandLine( int argc, char *argv[])
{
  	int i, iResult = 1 ;
  	char * pchCommandTranslations[] = {
      "--device",          "device=%s",       "select event device\n",
      "--verbose",         "log=verbose",     "print more info\n",
      "--debug",           "log=debug",       "print debug info\n",
      "--quiet",           "log=none",        "do not print info\n",
      "--copying",         "copying",         "print copying banner and exit\n",
      "--version",         "version",         "print version banner and exit\n",
      "--help",            NULL,              "print this help and exit\n",
      NULL,                NULL,              NULL } ;
      
	pchProgramName = argv[ 0] ;

  	for( i = 1 ; i < argc ; i++) {
		int iCmd ;

		/* TODO : translate --help to help command */
    		if( strcmp( argv[ i], "--help") == 0 ) {
      			printf( "Use %s <option>\n", pchProgramName) ;
      			for( iCmd=0; pchCommandTranslations[ iCmd] != NULL ; iCmd+=3)
        			printf( "%s\t: %s",
			                pchCommandTranslations[ iCmd],
			                pchCommandTranslations[ iCmd+2]) ;
      
      			return 0 ;
    		}
	   
		for( iCmd=0; pchCommandTranslations[ iCmd] != NULL ; iCmd+=3)
      			if( strcmp( argv[ i], pchCommandTranslations[ iCmd]) == 0)
				break ;

    		if( pchCommandTranslations[ iCmd] == NULL) {
      			printf( "%s unknow options, please use %s --help\n",
   	       			argv[ i], pchProgramName) ;
			iResult = 0 ; 
		} else {
		      	char achCommand[ 1024] ;
      
	      		d2printf( "command line command %s found\n", pchCommandTranslations[ iCmd+1]) ;
	      
	      		vsnprintf( achCommand, sizeof( achCommand), 
			   	   pchCommandTranslations[ iCmd+1], 
				   &argv[ i+1]) ;
			
			d2printf( "Command %s found\n", achCommand) ;
      			switch( ParseCommand( achCommand)) {
				// print error & exit (next line)
				case -1: dprintf( "ERROR\n") ;  
				// exit  
        			case 0:  return 0 ;             
      			}
    		}
  	}
  	return 1 ;
}

int readConfigurationFile( const char * pchConfigFile)
{
  	FILE * fConfig = NULL ;
  	int    iResult = 1;
 	char   achLine[ 2048] ;

  	fConfig = fopen( pchConfigFile, "rt") ;
  	if( !fConfig)
		dprintf( "Configuration file %s not found\n", pchConfigFile) ;
	else {
		int iLineCount ;
		for( iLineCount = 1 ; (!feof( fConfig)) && (iResult > 0); iLineCount++) {
			int iLen = -1 ;
      
//      fgets( achLine, sizeof( achLine), fConfig) ;
			do {
				if( (achLine[ ++iLen] = fgetc( fConfig)) == '\r') iLen-- ;
			} while( !feof( fConfig) && 
		 	         iLen < sizeof( achLine)  &&	
	       			 (achLine[ iLen] != '\n') &&
	       			 (achLine[ iLen] != '\0') ) ;
			
			achLine[ iLen] = '\0' ; // remove trailing \n
      
			d2printf( "line is %s\n", achLine) ;
      			if( iLen > 0) {
			        iResult = ParseCommand( achLine) ;
			        if( iResult == -1) {
					printf( "Config file error at line # %d! : %s \n",
					          iLineCount, achLine) ; 
				        iResult = 0 ;
			        }
			        if( iResult == 0) break ;
		      	}
	    	}
    		fclose( fConfig) ;
  	}
  	return iResult ;
}

int writeConfigurationFile( const char * pchConfigFile)
{
  FILE * fConfig = NULL ;
  int    iResult = 1;
  char   achLine[ 2048] ;

  fConfig = fopen( pchConfigFile, "wt") ;
  if( !fConfig)
  {
    printf( "Error when open %s to write configuration data\n", pchConfigFile) ;
    return 0 ;
  }
  else
  {
    int i ;
    fprintf( fConfig, "device=%s\n", pchEventDevice) ;

    fprintf( fConfig, "log=%s\n", 
	     (iLogLevel > 1) ? "debug" : (iLogLevel == 1) ? "verbose" : "normal") ;
    
    d2printf( "command number is %d\n", getNumberofEntry()) ;
    for( i = getNumberofEntry() - 1 ; i >= 0 ; i--)
    {
      const struct CMD * tmp ;
      
      d2printf( "write command number %d\n", i) ;
      if( (tmp = getCommand( i)) != NULL ) 
      {
        d2printf( "write command number %d\n", i) ;
	fprintf( fConfig, "command=%03d,%c%c%c,%s\n", 
		 tmp->code, 
                 (tmp->mode & ON_PRESS) ? 'P' : ' ',
                 (tmp->mode & ON_RELEASE) ? 'R' : ' ',
                 (tmp->mode & ON_REPEAT) ? 'H' : ' ',
		 tmp->command) ;
      }
    }
    
    fclose( fConfig) ;
  }

  return 1 ;
}

// Parse config file tokens
int ParseCommand( char * pchCommandLine)
{
  ltrim( pchCommandLine) ;

  if( (pchCommandLine) && (pchCommandLine[ 0] != '#') )
  {
    char * pchCommand ;
    char * pchValue ;
    
    pchCommand = strtok( pchCommandLine, "=") ;
    pchValue = strtok( NULL, "=") ;

    rtrim( pchCommand) ;      // remove rigth spaces
    trim( pchValue) ;         // remove trailing spaces

    d2printf( "Command string is '%s'\t Value is '%s'\n",
             pchCommand, pchValue) ;
  
    if( strcmp( pchCommand, "help") == 0 ) 
    {
      // IGNORE HELP REQUEST FROM CONFIGURATION FILE
      return 1 ;
    }
    
    if( strcmp( pchCommand, "command") == 0) 
    {
      char * command = NULL ;
      unsigned short code = 0, mode = 0 ;

      pchValue = strtok( pchValue, ",") ; 
      d2printf( "Value is '%s'\n", pchValue) ;
      if( (pchValue == NULL ) || ( sscanf( pchValue, "%d", &code) != 1) )
	return -1 ;
      
      pchValue = strtok( NULL, ",") ;
      d2printf( "Value is '%s'\n", pchValue) ;
      if( pchValue == NULL) return -1 ;
      
      if( strchr( pchValue, 'P') != NULL) mode |= ON_PRESS ;
      if( strchr( pchValue, 'R') != NULL) mode |= ON_RELEASE ;
      if( strchr( pchValue, 'H') != NULL) mode |= ON_REPEAT ;
      if( mode == 0) return -1 ;
      
      pchValue += strlen( pchValue) + 1 ;
      d2printf( "Value is '%s'\n", pchValue) ;
      if( pchValue == NULL) return -1 ;

      command = pchValue ;
      
      trim( command) ;
      if( command[ 0] == '\0') return -1 ;
      
      return addCommand( command, code, mode) ;
    }
   
    if( strcmp( pchCommand, "device") == 0 )
    {
      if( (pchValue != NULL) && 
          (strncmp( pchValue, "/dev/input/event", 10) == 0) )
      {
        pchEventDevice = strdup( pchValue) ;
        return 1 ;
      }
      printf( "Option 'device' expects a /dev/input/eventX argument\n");
      return -1 ;
    }

    if( strcmp( pchCommand, "log") == 0 ) 
    {
      if( strcmp( pchValue, "none") == 0) iLogLevel = -1 ;
      else if( strcmp( pchValue, "normal") == 0) iLogLevel = 0 ;
      else if( strcmp( pchValue, "verbose") == 0) iLogLevel = 1 ;
      else if( strcmp( pchValue, "debug") == 0) iLogLevel = 2 ;
      else return -1 ;
      return 1 ;
    }

    if( strcmp( pchCommand, "copying") == 0 ) 
    {
      puts( achCopying) ;
      return 0 ;
    }

    if( strcmp( pchCommand, "version") == 0 ) 
    {
      struct utsname uts;
    
      uname(&uts);
      printf( "Running on %s/%s kernel %s\n",
              uts.sysname, uts.machine, uts.release) ;

      if (0 == geteuid() && 0 != getuid()) 
        printf("WARNING: Installed suid root!\n") ;
    
      return 0 ;
    }

    return -1 ; // comando sconosciuto !!!
  }

  return 1 ; // nessun comando o linea di commento :)
}

