/*
 *  Database record class for DAAP sharing
 *
 *  Copyright (C) 2008 W. Michael Petullo <mike@flyn.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "dmapd-daap-record.h"
#include "dmapd-daap-record-factory.h"
#include "dmapd-dmap-db.h"
#include "av-meta-reader.h"
#include "util.h"

static const char *_unknown = "Unknown";

struct DmapdDmapAvRecordPrivate {
	char *location;
	char *original_location; /* Original file, not transcoded file. */
	GArray *hash;
	guint64 filesize;
	const char *format;      /* Format, possibly after transcoding. */
	gint mediakind;
	char *title;
	const char *album;
	const char *sort_album;
	const char *artist;
	const char *sort_artist;
	const char *genre;
	gboolean has_video;
	gint rating;
	gint32 duration;
	gint32 track;
	gint32 year;
	gint32 firstseen;
	gint32 mtime;
	gint32 disc;
	gint32 bitrate;
};

enum {
	PROP_0,
	PROP_LOCATION,
	PROP_ORIGINAL_LOCATION,
	PROP_HASH,
	PROP_TITLE,
	PROP_RATING,
	PROP_FILESIZE,
	PROP_ALBUM,
	PROP_SORT_ALBUM,
	PROP_ARTIST,
	PROP_SORT_ARTIST,
	PROP_GENRE,
	PROP_FORMAT,
	PROP_MEDIAKIND,
	PROP_DURATION,
	PROP_TRACK,
	PROP_YEAR,
	PROP_FIRSTSEEN,
	PROP_MTIME,
	PROP_DISC,
	PROP_BITRATE,
	PROP_HAS_VIDEO
};

static void
_set_property (GObject *object,
               guint prop_id,
               const GValue *value,
               GParamSpec *pspec)
{
	DmapdDmapAvRecord *record = DMAPD_DMAP_AV_RECORD (object);

	switch (prop_id) {
		const char *str;
		case PROP_LOCATION:
			g_free (record->priv->location);
			record->priv->location = g_value_dup_string (value);
			break;
		case PROP_ORIGINAL_LOCATION:
			g_free (record->priv->original_location);
			record->priv->original_location = g_value_dup_string (value);
			break;
		case PROP_HASH:
			if (record->priv->hash) {
				g_array_unref(record->priv->hash);
			}
                        record->priv->hash = g_array_ref(g_value_get_boxed(value));
                        break;
		case PROP_TITLE:
			g_free (record->priv->title);
			record->priv->title = g_value_dup_string (value);
			break;
		case PROP_ALBUM:
			util_stringleton_unref (record->priv->album);
			record->priv->album = util_stringleton_ref (g_value_get_string(value));
			break;
		case PROP_SORT_ALBUM:
			str = g_value_get_string(value);
			util_stringleton_unref (record->priv->sort_album);
			record->priv->sort_album = str ? util_stringleton_ref (str) : NULL;
			break;
		case PROP_ARTIST:
			util_stringleton_unref (record->priv->artist);
			record->priv->artist = util_stringleton_ref (g_value_get_string(value));
			break;
		case PROP_SORT_ARTIST:
			str = g_value_get_string(value);
			util_stringleton_unref (record->priv->sort_artist);
			record->priv->sort_artist = str ? util_stringleton_ref (str) : NULL;
			break;
		case PROP_GENRE:
			util_stringleton_unref (record->priv->genre);
			record->priv->genre = util_stringleton_ref (g_value_get_string(value));
			break;
		case PROP_FORMAT:
			util_stringleton_unref (record->priv->format);
			record->priv->format = util_stringleton_ref (g_value_get_string(value));
			break;
		case PROP_MEDIAKIND:
			record->priv->mediakind = g_value_get_enum (value);
			break;
		case PROP_RATING:
			record->priv->rating = g_value_get_int (value);
			break;
		case PROP_FILESIZE:
			record->priv->filesize = g_value_get_uint64 (value);
			break;
		case PROP_DURATION:
			record->priv->duration = g_value_get_int (value);
			break;
		case PROP_TRACK:
			record->priv->track = g_value_get_int (value);
			break;
		case PROP_YEAR:
			record->priv->year = g_value_get_int (value);
			break;
		case PROP_FIRSTSEEN:
			record->priv->firstseen = g_value_get_int (value);
			break;
		case PROP_MTIME:
			record->priv->mtime = g_value_get_int (value);
			break;
		case PROP_DISC:
			record->priv->disc = g_value_get_int (value);
			break;
		case PROP_BITRATE:
			record->priv->bitrate = g_value_get_int (value);
			break;
		case PROP_HAS_VIDEO:
			record->priv->has_video = g_value_get_boolean (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object,
							   prop_id,
							   pspec);
			break;
	}
}

static void
_get_property (GObject *object,
               guint prop_id,
               GValue *value,
               GParamSpec *pspec)
{
	DmapdDmapAvRecord *record = DMAPD_DMAP_AV_RECORD (object);

	switch (prop_id) {
		case PROP_LOCATION:
			g_value_set_static_string (value, record->priv->location);
			break;
		case PROP_ORIGINAL_LOCATION:
			g_value_set_static_string (value, record->priv->original_location);
			break;
		case PROP_HASH:
                        g_value_set_boxed (value, g_array_ref(record->priv->hash));
                        break;
		case PROP_TITLE:
			g_value_set_static_string (value, record->priv->title);
			break;
		case PROP_ALBUM:
			g_value_set_static_string (value, record->priv->album);
			break;
		case PROP_SORT_ALBUM:
			g_value_set_static_string (value, record->priv->sort_album);
			break;
		case PROP_ARTIST:
			g_value_set_static_string (value, record->priv->artist);
			break;
		case PROP_SORT_ARTIST:
			g_value_set_static_string (value, record->priv->sort_artist);
			break;
		case PROP_GENRE:
			g_value_set_static_string (value, record->priv->genre);
			break;
		case PROP_FORMAT:
			g_value_set_static_string (value, record->priv->format);
			break;
		case PROP_MEDIAKIND:
			g_value_set_enum (value, record->priv->mediakind);
			break;
		case PROP_RATING:
			g_value_set_int (value, record->priv->rating);
			break;
		case PROP_FILESIZE:
			g_value_set_uint64 (value, record->priv->filesize);
			break;
		case PROP_DURATION:
			g_value_set_int (value, record->priv->duration);
			break;
		case PROP_TRACK:
			g_value_set_int (value, record->priv->track);
			break;
		case PROP_YEAR:
			g_value_set_int (value, record->priv->year);
			break;
		case PROP_FIRSTSEEN:
			g_value_set_int (value, record->priv->firstseen);
			break;
		case PROP_MTIME:
			g_value_set_int (value, record->priv->mtime);
			break;
		case PROP_DISC:
			g_value_set_int (value, record->priv->disc);
			break;
		case PROP_BITRATE:
			g_value_set_int (value, record->priv->bitrate);
			break;
		case PROP_HAS_VIDEO:
			g_value_set_boolean (value, record->priv->has_video);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object,
							   prop_id,
							   pspec);
			break;
	}
}

gboolean
dmapd_daap_record_itunes_compat (DmapAvRecord *record)
{
	const gchar *format = DMAPD_DMAP_AV_RECORD (record)->priv->format;

	return !strcmp (format, "mp3") ? TRUE : FALSE;
}

GInputStream *
dmapd_daap_record_read (DmapAvRecord *record, GError **error)
{
	GFile *file;
	GInputStream *fnval = NULL;

	file = g_file_new_for_uri (DMAPD_DMAP_AV_RECORD (record)->priv->location);
	g_assert (file);
	fnval = G_INPUT_STREAM (g_file_read (file, NULL, error));

	return fnval;
}

static GArray *
_to_blob (DmapRecord *record)
{
	/* FIXME: not endian safe. Don't write with one endianness and
	 * read with another
	 */
	
	DmapdDmapAvRecordPrivate *priv = DMAPD_DMAP_AV_RECORD (record)->priv;
	GArray *blob = g_array_new (FALSE, FALSE, 1);

	/* NOTE: do not store ID in the blob. */

	g_assert (priv->location);
	g_assert (priv->hash);
	g_assert (priv->format);
	g_assert (priv->title);
	g_assert (priv->album);
	g_assert (priv->artist);
	g_assert (priv->genre);

	util_blob_add_string (blob, VERSION);
	util_blob_add_string (blob, priv->location);

	util_blob_add_atomic (blob, (const guint8 *) &(priv->hash->len), sizeof (priv->hash->len));
	g_array_append_vals (blob, priv->hash->data, priv->hash->len);

	util_blob_add_atomic (blob, (const guint8 *) &(priv->filesize),
			   sizeof (priv->filesize));

	util_blob_add_string (blob, priv->format);
	util_blob_add_string (blob, priv->title);
	util_blob_add_string (blob, priv->album);
	util_blob_add_string (blob, priv->artist);
	util_blob_add_string (blob, priv->genre);

	util_blob_add_atomic   (blob, (const guint8 *) &(priv->has_video),
			   sizeof (priv->has_video));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->mediakind),
			   sizeof (priv->mediakind));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->rating),
			   sizeof (priv->rating));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->duration),
			   sizeof (priv->duration));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->track),
			   sizeof (priv->track));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->year),
			   sizeof (priv->year));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->firstseen),
			   sizeof (priv->firstseen));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->mtime),
			   sizeof (priv->mtime));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->disc),
			   sizeof (priv->disc));
	util_blob_add_atomic   (blob, (const guint8 *) &(priv->bitrate),
			   sizeof (priv->bitrate));
	
	return blob;
}

static gboolean
_set_from_blob (DmapRecord *_record, GArray *blob)
{
	struct stat buf;
	gboolean fnval = FALSE;
	DmapdDmapAvRecord *record = NULL;
	GFile *file = NULL;
	gchar *path = NULL;
	GFileInputStream *stream = NULL;
	GError *error = NULL;
	gchar *ptr = blob->data;

	char *version;
	guint size;
	guint64 filesize;
	char *location;
	char *format;
	GArray *hash = NULL;
	guchar hash2[DMAP_HASH_SIZE];
	char *title;
	char *songalbum;
	char *songartist;
	char *songgenre;
	gboolean has_video;
	gint mediakind;
	gint rating;
	gint32 duration;
	gint32 track;
	gint32 year;
	gint32 firstseen;
	gint32 mtime;
	gint32 disc;
	gint32 bitrate;

	version = (char *) ptr;
	ptr += strlen (version) + 1;
	if (strcmp (version, VERSION)) {
		g_warning ("Cache written by wrong dmapd version");
		goto _done;
	}

	location = (char *) ptr;
        ptr += strlen (location) + 1;

	size = *((guint *) ptr);
	if (DMAP_HASH_SIZE != size) {
                g_warning ("Improper hash size in cache");
                goto _done;
        }
        ptr += sizeof (size);

        hash = g_array_sized_new (FALSE, FALSE, 1, size);
	if (NULL == hash) {
		g_warning ("Error allocating array for hash");
		goto _done;
	}
        g_array_append_vals (hash, ptr, size);
        ptr += size;

	filesize = *(guint64 *) ptr;
        ptr += sizeof (record->priv->filesize);

	format = (char *) ptr;
        ptr += strlen ((char *) ptr) + 1;

	title = (char *) ptr;
        ptr += strlen ((char *) ptr) + 1;

	songalbum = (char *) ptr;
        ptr += strlen ((char *) ptr) + 1;

	songartist = (char *) ptr;
        ptr += strlen ((char *) ptr) + 1;

	songgenre = (char *) ptr;
        ptr += strlen ((char *) ptr) + 1;

	has_video = *(gint *) ptr;
        ptr += sizeof (record->priv->has_video);

	mediakind = *(gint *) ptr;
	ptr += sizeof (record->priv->mediakind);

	rating = *(gint *) ptr;
        ptr += sizeof (record->priv->rating);

	duration = *(gint32 *) ptr;
        ptr += sizeof (record->priv->duration );

	track = *(gint32 *) ptr;
        ptr += sizeof (record->priv->track);

	year = *(gint32 *) ptr;
        ptr += sizeof (record->priv->year);

	firstseen = *(gint32 *) ptr;
        ptr += sizeof (record->priv->firstseen);

	mtime = *(gint32 *) ptr;
        ptr += sizeof (record->priv->mtime);

	disc = *(gint32 *) ptr;
        ptr += sizeof (record->priv->disc);

	bitrate = *(gint32 *) ptr;

	file = g_file_new_for_uri (location);
        if (NULL == location) {
                g_warning ("Could not open %s", location);
                goto _done;
        }	

	stream = g_file_read (file, NULL, &error);
	if (NULL == stream) {
                g_warning ("Could not read %s: %s", location, error->message);
                goto _done;
	}

	if (FALSE /* FIXME: calculate_media_hashes */) {
		if (! util_hash_file (location, hash2)
		 || memcmp (hash->data, hash2, DMAP_HASH_SIZE)) {
			g_warning ("Media file hash has changed since being cached");
			goto _done;
		}
	} else {
		path = g_file_get_path(file);
		if (NULL == path) {
			g_warning ("Could not convert %s to local path; assuming modified", location);
			goto _done;
		}
		if (stat (path, &buf) == -1) {
			g_warning ("Unable to determine mtime of %s; assuming modified", path);
			goto _done;
		} else {
			if (buf.st_mtime > mtime) {
				g_warning ("Media file mtime has changed since being cached");
				goto _done;
			}
		}
	}

	record = DMAPD_DMAP_AV_RECORD (_record);

	g_object_set (record, "year", year,
	                      "filesize", filesize,
	                      "location", location,
                              "hash", hash,
	                      "format", format,
	                      "title", title,
	                      "songalbum", songalbum,
	                      "songartist", songartist,
	                      "songgenre", songgenre,
	                      "has-video", has_video,
	                      "mediakind", mediakind,
	                      "rating", rating,
	                      "duration", duration,
	                      "track", track,
	                      "firstseen", firstseen,
	                      "mtime", mtime,
	                      "disc", disc,
	                      "bitrate", bitrate, NULL);

	fnval = TRUE;

_done:
	if (NULL != file) {
		g_object_unref (file);
	}

	if (NULL != path) {
		g_free (path);
	}

	if (NULL != stream) {
		g_object_unref (stream);
	}

	if (NULL != hash) {
		g_array_unref (hash);
	}

	return fnval;
}

static void
_finalize (GObject *object);

static void
dmapd_daap_record_class_init (DmapdDmapAvRecordClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

	gobject_class->set_property = _set_property;
	gobject_class->get_property = _get_property;
	gobject_class->finalize     = _finalize;

	g_object_class_override_property (gobject_class, PROP_LOCATION, "location");
	g_object_class_override_property (gobject_class, PROP_HASH, "hash");
	g_object_class_override_property (gobject_class, PROP_TITLE, "title");
	g_object_class_override_property (gobject_class, PROP_ALBUM, "songalbum");
	g_object_class_override_property (gobject_class, PROP_SORT_ALBUM, "sort-album");
	g_object_class_override_property (gobject_class, PROP_ARTIST, "songartist");
	g_object_class_override_property (gobject_class, PROP_SORT_ARTIST, "sort-artist");
	g_object_class_override_property (gobject_class, PROP_GENRE, "songgenre");
	g_object_class_override_property (gobject_class, PROP_FORMAT, "format");
	g_object_class_override_property (gobject_class, PROP_RATING, "rating");
	g_object_class_override_property (gobject_class, PROP_FILESIZE, "filesize");
	g_object_class_override_property (gobject_class, PROP_DURATION, "duration");
	g_object_class_override_property (gobject_class, PROP_TRACK, "track");
	g_object_class_override_property (gobject_class, PROP_YEAR, "year");
	g_object_class_override_property (gobject_class, PROP_FIRSTSEEN, "firstseen");
	g_object_class_override_property (gobject_class, PROP_MTIME, "mtime");
	g_object_class_override_property (gobject_class, PROP_DISC, "disc");
	g_object_class_override_property (gobject_class, PROP_BITRATE, "bitrate");
	g_object_class_override_property (gobject_class, PROP_HAS_VIDEO, "has-video");
	g_object_class_override_property (gobject_class, PROP_MEDIAKIND, "mediakind");

	g_object_class_install_property (gobject_class,
	                                 PROP_ORIGINAL_LOCATION,
	                                 g_param_spec_string ("original-location",
	                                                      "original-location",
	                                                      "original-location",
	                                                      NULL,
	                                                      G_PARAM_READWRITE));
}

static void
_dmap_av_record_iface_init (gpointer iface, G_GNUC_UNUSED gpointer data)
{
	DmapAvRecordInterface *daap_record = iface;

	g_assert (G_TYPE_FROM_INTERFACE (daap_record) == DMAP_TYPE_AV_RECORD);

	daap_record->itunes_compat = dmapd_daap_record_itunes_compat;
	daap_record->read = dmapd_daap_record_read;
}

static void
_dmap_record_iface_init (gpointer iface, G_GNUC_UNUSED gpointer data)
{
	DmapRecordInterface *dmap_record = iface;

	g_assert (G_TYPE_FROM_INTERFACE (dmap_record) == DMAP_TYPE_RECORD);

	dmap_record->to_blob = _to_blob;
	dmap_record->set_from_blob = _set_from_blob;
}


G_DEFINE_TYPE_WITH_CODE (DmapdDmapAvRecord, dmapd_daap_record, G_TYPE_OBJECT, 
			 G_IMPLEMENT_INTERFACE (DMAP_TYPE_AV_RECORD, _dmap_av_record_iface_init)
			 G_IMPLEMENT_INTERFACE (DMAP_TYPE_RECORD, _dmap_record_iface_init)
                         G_ADD_PRIVATE (DmapdDmapAvRecord))

static void
dmapd_daap_record_init (DmapdDmapAvRecord *record)
{
	record->priv = dmapd_daap_record_get_instance_private(record);
}

static void
_finalize (GObject *object)
{
	DmapdDmapAvRecord *record = DMAPD_DMAP_AV_RECORD (object);

	g_debug ("Free'ing record");

	g_free (record->priv->location);
	g_free (record->priv->original_location);
	g_free (record->priv->title);

	util_stringleton_unref (record->priv->format);
	util_stringleton_unref (record->priv->album);
	util_stringleton_unref (record->priv->artist);
	util_stringleton_unref (record->priv->genre);

	G_OBJECT_CLASS (dmapd_daap_record_parent_class)->finalize (object);
}

DmapdDmapAvRecord *
dmapd_daap_record_new (const char *path, AvMetaReader *reader, GError **error)
{
	DmapdDmapAvRecord *record = NULL;
	guchar hash_buf[DMAP_HASH_SIZE];
	struct stat buf;
	char *title = NULL;
	char *location = NULL;
	GArray *hash = NULL;

	if (path) {
		location = g_filename_to_uri (path, NULL, error);
		if (NULL == location) {
			goto _done;
		}

		title = g_path_get_basename (path);
		hash = g_array_sized_new (FALSE, FALSE, 1, DMAP_HASH_SIZE);

		if (FALSE /* FIXME: calculate_media_hashes */) {
			if (! util_hash_file (location, hash_buf)) {
				g_set_error(error,
				            DMAP_ERROR,
				            DMAP_STATUS_FAILED,
				           "Unable to hash %s", location);
				goto _done;
			}
		} else {
			memset(hash_buf, 0x00, DMAP_HASH_SIZE);
		}

		g_array_append_vals (hash, hash_buf, DMAP_HASH_SIZE);

		record = DMAPD_DMAP_AV_RECORD (g_object_new (TYPE_DMAPD_DMAP_AV_RECORD, NULL));

		if (stat (path, &buf) == -1) {
			g_warning ("Unable to determine size of %s", path);
		} else {
			g_object_set (record,
			             "filesize",
				     (guint64) buf.st_size,
				     "mtime",
				      buf.st_mtime,
				      NULL);
		}

		g_object_set (record, "location",    location,
		                      "hash",        hash,
		                      "title",       title,
		                      "songartist",  _unknown,
		                      "songalbum",   _unknown,
		                      "songgenre",   _unknown,
		                      "format",      _unknown,
		                      "mediakind",   DMAP_MEDIA_KIND_MUSIC,
		                      "year",        1985,
		                      "disc",        1,
		                       NULL);

		if (! av_meta_reader_read (AV_META_READER (reader),
		                           DMAP_AV_RECORD (record),
		                           path,
		                           error)) {
			g_object_unref (record);
			record = NULL;
			goto _done;
		}

		record->priv->rating = 5;	/* FIXME */
		record->priv->firstseen = 1;	/* FIXME */
		record->priv->bitrate = 128;	/* FIXME, from codec decoder */
	} else {
		record = DMAPD_DMAP_AV_RECORD (g_object_new (TYPE_DMAPD_DMAP_AV_RECORD, NULL));
	}

_done:

	g_free (location);
	g_free (title);

	if (NULL != hash) {
		g_array_unref (hash);
	}

	return record;
}

#ifdef HAVE_CHECK

START_TEST(_test_add_lookup)
{
	guint id;
        guint64 filesize;
        char *location;
        char *format;
        char *title;
        char *album;
        char *artist;
        char *genre;
        gint rating;
        gint32 duration;
        gint32 track;
        gint32 year;
        gint32 firstseen;
        gint32 mtime;
        gint32 disc;
        gint32 bitrate;
        gboolean has_video;
	DmapdDmapAvRecordFactory *factory;
	DmapRecord *record1, *record2;

	factory = g_object_new (TYPE_DMAPD_DMAP_RECORD_FACTORY, NULL);

	DmapDb *db = DMAP_DB (util_object_from_module (TYPE_DMAPD_DMAP_DB,
	                                          DEFAULT_MODULEDIR,
						 "ghashtable",
						 "record-factory",
						  factory,
						  NULL));

	record1 = DMAP_RECORD (
		g_object_new (TYPE_DMAPD_DMAP_AV_RECORD,
			      "location", "aaaaaaaaaaaaaaaaaaaa",
			      "title", "b",
			      "songalbum", "c",
			      "songartist", "d",
			      "songgenre", "e",
			      "format", "f",
			      "rating", 1,
			      "filesize", 2L,
			      "duration", 3,
			      "track", 4,
			      "year", 5,
			      "firstseen", 6,
			      "mtime", 7,
			      "disc", 8,
			      "bitrate", 9,
			      "has-video", TRUE,
			      NULL));

	id = dmap_db_add (db, record1, NULL);

	record2 = dmap_db_lookup_by_id (db, id);

	g_object_get (record2, "location", &location,
			       "title", &title,
			       "songalbum", &album,
			       "songartist", &artist,
			       "songgenre", &genre,
			       "format", &format,
			       "rating", &rating,
			       "filesize", &filesize,
			       "duration", &duration,
			       "track", &track,
			       "year", &year,
			       "firstseen", &firstseen,
			       "mtime", &mtime,
			       "disc", &disc,
			       "bitrate", &bitrate,
			       "has-video", &has_video,
			       NULL);

	fail_unless (! strcmp (location,    "aaaaaaaaaaaaaaaaaaaa"));
	fail_unless (! strcmp (title,       "b"));
	fail_unless (! strcmp (album,       "c"));
	fail_unless (! strcmp (artist,      "d"));
	fail_unless (! strcmp (genre,       "e"));
	fail_unless (! strcmp (format,      "f"));
	fail_unless (rating    == 1);
	fail_unless (filesize  == 2);
	fail_unless (duration  == 3);
	fail_unless (track     == 4);
	fail_unless (year      == 5);
	fail_unless (firstseen == 6);
	fail_unless (mtime     == 7);
	fail_unless (disc      == 8);
	fail_unless (bitrate   == 9);
	fail_unless (has_video == TRUE);

	g_object_unref (db);
	g_object_unref (factory);
}
END_TEST

Suite *dmapd_test_daap_record_suite(void)
{
        Suite *s = suite_create("dmapd-test-daap-record");

	TCase *tc_add_lookup = tcase_create("test_add_lookup");
	tcase_add_test(tc_add_lookup, _test_add_lookup);
	suite_add_tcase(s, tc_add_lookup);

	return s;
}

#endif
