#ifndef DRV_TAPI_H
#define DRV_TAPI_H
/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi.h
   Contains TAPI functions declaration und structures.
*/


/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_api.h"
#include <lib_fifo.h>
#include <lib_bufferpool.h>
#include "drv_tapi_io.h"
#include "drv_tapi_event.h"


/* ============================= */
/* Global defines                */
/* ============================= */

#define VMMC_IOC_MAGIC 'M'
#define VINETIC_IOC_MAGIC 'V'
#define SVIP_IOC_MAGIC 'S'
#define DUS_IOC_MAGIC 'D'

/** Maximum parameters for the hook state machine */
#define TAPI_MIN_FLASH                80
#define TAPI_MAX_FLASH               200
#define TAPI_MIN_FLASH_MAKE          200
#define TAPI_MIN_DIGIT_LOW            30
#define TAPI_MAX_DIGIT_LOW            80
#define TAPI_MIN_DIGIT_HIGH           30
#define TAPI_MAX_DIGIT_HIGH           80
#define TAPI_MIN_OFF_HOOK             40
#define TAPI_MIN_ON_HOOK             400
#define TAPI_MIN_INTERDIGIT          300

/** Maximum number of allowed LL drivers */
#ifndef TAPI_MAX_LL_DRIVERS
#define TAPI_MAX_LL_DRIVERS            5
#endif /* TAPI_MAX_LL_DRIVERS */

#ifndef TAPI_TONE_MAXRES
/** Maximum tone generators that can run in parallel: 3 modules ALM, PCM,
    COD and CONF and 2 tone generators */
#define TAPI_TONE_MAXRES 3
#endif

#ifndef IFX_TAPI_EVENT_POOL_INITIAL_SIZE
/** Initial number of events structures allocated by the driver. If more
    event structures are needed the pool grows automatically in steps of
    IFX_TAPI_EVENT_POOL_GROW_SIZE */
#define IFX_TAPI_EVENT_POOL_INITIAL_SIZE 50
#endif

#ifndef IFX_TAPI_EVENT_POOL_GROW_SIZE
/** Number of events that the event structure pool grows every time it gets
    depleted. */
#define IFX_TAPI_EVENT_POOL_GROW_SIZE 50
#endif

#ifndef IFX_TAPI_EVENT_POOL_GROW_LIMIT
#define IFX_TAPI_EVENT_POOL_GROW_LIMIT           500
#endif

#ifndef IFX_TAPI_EVENT_FIFO_SIZE
/** Event Fifo Size */
   #define IFX_TAPI_EVENT_FIFO_SIZE               10
#endif /* IFX_TAPI_EVENT_FIFO_SIZE */

#ifdef TAPI_EXT_KEYPAD
/** This info is used to set flag "nDtmfInfo" for DTMF incase of external
    key event */
enum DTMF_INFO
{
/*
   DTMF_EV_OOB_DEAFULT = 0x01,
   DTMF_EV_OOB_NO      = 0x02,
   DTMF_EV_OOB_ONLY    = 0x04,
   DTMF_EV_OOB_ALL     = 0x08,
   DTMF_EV_OOB_BLOCK   = 0x10,
*/
   DTMF_EV_LOCAL_PLAY  = 0x80
};
#endif /* TAPI_EXT_KEYPAD */

/* Number of tones for internal driver purposes */
#define TAPI_MAX_RESERVED_TONES       10
/* Maximum number of tones codes, comprising the maximum user tones and the
   maximum internal reserved tones. */
#define TAPI_MAX_TONE_CODE           (IFX_TAPI_TONE_INDEX_MAX + \
                                      TAPI_MAX_RESERVED_TONES)

/* =================================== */
/* Global typedef forward declarations */
/* =================================== */
typedef struct _TAPI_DEV      TAPI_DEV;
typedef struct _TAPI_CHANNEL  TAPI_CHANNEL;


/* ================================= */
/* Defines for tapi operating modes  */
/* ================================= */

/** Bit 0 used for packets and bit 1 for events. */
typedef enum _TAPI_WORKING_MODE
{
   /** Interrupt mode for packets */
   TAPI_INTERRUPT_MODE_PACKETS = 0,
   /** Interrupt mode for events */
   TAPI_INTERRUPT_MODE_EVENTS = 0,
   /** Polling mode for packets */
   TAPI_POLLING_MODE_PACKETS = 1,
   /** Polling mode for events */
   TAPI_POLLING_MODE_EVENTS = 2
} TAPI_WORKING_MODE;

/* =============================== */
/* Defines for CID operations      */
/* =============================== */

/* Default indexes for CID alert tones */
enum _TAPI_CID_ALERTTONE_INDEX_DEFAULT
{
   /* Alert Tone default index, onhook */
   TAPI_CID_ATONE_INDEXDEF_ONHOOK       = IFX_TAPI_TONE_INDEX_MAX + 1,
   /* Alert Tone default index, offhook */
   TAPI_CID_ATONE_INDEXDEF_OFFHOOK      = IFX_TAPI_TONE_INDEX_MAX + 2,
   /* Alert Tone default index, AS NTT */
   TAPI_CID_ATONE_INDEXDEF_OFFHOOKNTT   = IFX_TAPI_TONE_INDEX_MAX + 3,
   /* define others cid reserved tones,
      should be in the range of reserved tones */
   TAPI_CID_ATONE_INDEXDEF_MAX          = TAPI_MAX_TONE_CODE
};


/* include definitions of the low level driver interface which require
   some of the definitions above */
#include "drv_tapi_ll_interface.h"
/* include for KPI requires the TAPI_CHANNEL so include it here */
#ifdef KPI_SUPPORT
#include "drv_tapi_kpi.h"
#endif /* KPI_SUPPORT */

typedef struct _IFX_TAPI_HL_DRV_CTX
{
   /* Pointer to the driver context provided by the LL driver. A value of
      NULL indicates that this struct is unused. */
   IFX_TAPI_DRV_CTX_t *pDrvCtx;
#ifdef LINUX
   /* buffer to store the registered device driver name, Linux references to
      this string, e.g. on cat /proc/devices - please don't remove /Olaf */
   IFX_char_t          registeredDrvName[20];
#endif /* LINUX */
} IFX_TAPI_HL_DRV_CTX_t;

/* ============================= */
/** Structure for operation control
    \internal */
/* ============================= */
typedef struct
{
   /* interrupt routine save hookstatus here */
   IFX_uint8_t               bHookState;
   /* polarity status from the line; 1 = reversed; 0 = normal */
   IFX_uint8_t               nPolarity;
   /* automatic battery switch; 1 = automatic; 0 = normal */
   IFX_uint8_t               nBatterySw;
   /* last line feed mode; is set after ringing stops */
   IFX_uint8_t               nLineMode;
   /* Last line type; changed when new line type changed with success. */
   IFX_TAPI_LINE_TYPE_t      nLineType;
   /** number of the assigned daa instance (LineType FXO),
       handled by the daa abstraction driver */
   IFX_uint16_t              nDAA;
   /** status of DAA channel initialisation
       (GPIO reservation, etc is done only once) */
   IFX_boolean_t             bDaaInitialized;
} TAPI_OPCONTROL_DATA_t;

/* ============================= */
/* Structure for CID data        */
/* ============================= */
struct TAPI_CID_DATA;
typedef struct TAPI_CID_DATA TAPI_CID_DATA_t;


/* ============================= */
/* Structure for ring data       */
/* ============================= */
struct TAPI_RING_DATA;
typedef struct TAPI_RING_DATA TAPI_RING_DATA_t;


/* ============================= */
/* Structure for pcm data        */
/* ============================= */
typedef struct
{
   /* configuration data for pcm services */
   IFX_TAPI_PCM_CFG_t   PCMConfig;
   /* save activation status */
   IFX_boolean_t        bTimeSlotActive;
   /* save configuration status */
   IFX_boolean_t        bCfgSuccess;
}TAPI_PCM_DATA_t;


/* ============================= */
/* Structure for tone data       */
/*                               */
/* ============================= */
typedef struct
{
   /* network tone on/off */
   IFX_boolean_t     bNetTone;
   /* off time duration */
   IFX_uint32_t      nOffTime;
   /* on time duration */
   IFX_uint32_t      nOnTime;
   /* tone playing state */
   IFX_uint32_t      nToneState;
   IFX_boolean_t     bUseSecondTG;
}TAPI_TG_TONE_DATA_t;


/** This structure is used by IFX_TAPI_TONE_PLAY and IFX_TAPI_TONE_STOP
    commands to enable/disable tone playout.*/
typedef struct
{
   /** Device index */
   IFX_uint16_t dev;
   /** Channel "module" index */
   IFX_uint16_t ch;
   /** Tone index to enable. For disable, index is set to zero. */
   IFX_uint32_t index;
   /** Firmware module where to play the tone */
   IFX_TAPI_MODULE_TYPE_t module;
   /** Tone generation towards external
       (e.g. analog line, pcm bus, RTP inband, etc.). */
   IFX_uint16_t external:1;
   /** Tone generation towards internal (connected firmware module). */
   IFX_uint16_t internal:1;
   /** Generate event IFX_TAPI_EVENT_TONE_GEN_END  1=yes, 0=no */
   IFX_uint16_t sendEndEvent : 1;
   /** Reserved */
   IFX_uint16_t reserved:14;
} IFX_TAPI_TONE_PLAY_t;


/* ============================= */
/* Structure for dial data       */
/*                               */
/* ============================= */
struct TAPI_DIAL_DATA;
typedef struct TAPI_DIAL_DATA TAPI_DIAL_DATA_t;

/* ============================= */
/* Structure for metering data   */
/*                               */
/* ============================= */
struct TAPI_METER_DATA;
typedef struct TAPI_METER_DATA TAPI_METER_DATA_t;

/* ============================= */
/* Structure for internal tone   */
/* coefficients table            */
/* ============================= */
typedef enum
{
   TAPI_TONE_TYPE_NONE,
   TAPI_TONE_TYPE_SIMPLE,
   TAPI_TONE_TYPE_COMP,
   TAPI_TONE_TYPE_PREDEF,
   TAPI_TONE_TYPE_DUAL
}TAPI_TONE_TYPE;

typedef struct
{
   TAPI_TONE_TYPE          type;
   union
   {
      IFX_int32_t          nPredefined;
      IFX_TAPI_TONE_DUAL_t       dual;
      IFX_TAPI_TONE_SIMPLE_t     simple;
      IFX_TAPI_TONE_COMPOSED_t   composed;
   } tone;
}COMPLEX_TONE;

/* ============================= */
/* Structure for complex tone    */
/* data                          */
/* ============================= */
typedef struct
{
   /* type of tone sequence SIMPLE or COMPOSED */
   TAPI_TONE_TYPE          nType;
   IFX_uint32_t            nSimpleMaxReps;
   /* current repetition counter for simple tone sequence */
   IFX_uint32_t            nSimpleCurrReps;
   /* maximum number of repetitions for composed tone sequence */
   IFX_uint32_t            nComposedMaxReps;
   /* current repetition counter for composed tone sequence */
   IFX_uint32_t            nComposedCurrReps;
   /* pause time in ms betweensimple tones */
   IFX_uint32_t            nPauseTime;
   /* alternate voice timein ms between composed tones */
   IFX_uint32_t            nAlternateVoiceTime;
   /* current simple tone codeplaying */
   IFX_uint32_t            nSimpleToneCode;
   /* current composed tone code playing */
   IFX_uint32_t            nComposedToneCode;
   /* maximum simple toneswithin composed tone */
   IFX_uint32_t            nMaxToneCount;
   /* current simple tone within composed tone */
   IFX_uint32_t            nToneCounter;
   /* complex tone playing state */
   TAPI_CMPLX_TONE_STATE_t nToneState;
   /** Specifies the capability of the tone generator regarding tone
       sequence support. See \ref IFX_TAPI_TONE_RESSEQ_t for details. */
   IFX_TAPI_TONE_RESSEQ_t sequenceCap;
   /** which direction to play the tone (net or local) */
   TAPI_TONE_DST dst;
   /** stores the index of the tone currently played */
   IFX_int32_t       nToneIndex;
   /** send IFX_TAPI_EVENT_TONE_GEN_END ? */
   IFX_boolean_t           bSendToneGenEndTapiEvent;
}TAPI_TONE_DATA_t;


/** tone resource structure as time argument */
typedef struct
{
   /* timer id for voice path establisment */
   Timer_ID          Tone_Timer;
   TAPI_CHANNEL      *pTapiCh;
   /** Resource number of the tone play unit */
   IFX_uint32_t      nRes;
}TAPI_TONERES;

/* ============================== */
/* Structure for                  */
/* TAPI-event handling data       */
/* ============================== */
struct IFX_TAPI_EVENT_HANDLER_DATA;
typedef struct IFX_TAPI_EVENT_HANDLER_DATA IFX_TAPI_EVENT_HANDLER_DATA_t;

/* ============================== */
/* Structure for statistic data   */
/*                                */
/* ============================== */
struct TAPI_STAT_DATA;
typedef struct TAPI_STAT_DATA TAPI_STAT_DATA_t;


/* ============================= */
/* channel specific structure    */
/* ============================= */
struct _TAPI_CHANNEL
{
   /* channel number */
   /* ATTENTION, nChannel must be the first element */
   IFX_uint8_t                   nChannel;
   /* pointer to the Low level driver channel */
   IFX_TAPI_LL_CH_t             *pLLChannel;
   /* pointer to the tapi device structure */
   TAPI_DEV                     *pTapiDevice;
   /* semaphore used only in blocking read access,
      in this case given from interrupt context */
   TAPI_OS_event_t               semReadBlock;
   /* wakeup queue for select on read */
   TAPI_OS_drvSelectQueue_t      wqRead;
   /* wakeup queue for select on write */
   TAPI_OS_drvSelectQueue_t      wqWrite;
   /* stores the current fax status */
   volatile IFX_boolean_t        bFaxDataRequest;
   /* stores the current linear channel status */
   volatile IFX_boolean_t        bLinChannelActive;
   volatile IFX_boolean_t        bLinDataRequest;

   /* PS: moved from low level channel structure */
   /* flags for different purposes, see CH_FLAGS */
   volatile IFX_uint32_t                  nFlags;

   /* channel is initialized */
   IFX_boolean_t                 bInitialized;

   /* locking semaphore for protecting data */
   TAPI_OS_mutex_t               semTapiChDataLock;

   /* overall channel protection ( read/write/ioctl level)
   PS: Avoid nested locking of this mutex. It can lead to a deadlock */
   TAPI_OS_mutex_t               semTapiChSingleIoctlAccess;

   /* data structures for services */
   TAPI_LEC_DATA_t               TapiLecAlmData;
   TAPI_LEC_DATA_t               TapiLecPcmData;
   TAPI_METER_DATA_t             *pTapiMeterData;
   TAPI_RING_DATA_t              *pTapiRingData;
   TAPI_OPCONTROL_DATA_t         TapiOpControlData;
   TAPI_PCM_DATA_t               TapiPCMData;
   TAPI_TG_TONE_DATA_t           TapiTgToneData;
   TAPI_DIAL_DATA_t              *pTapiDialData;
   TAPI_TONERES                  *pToneRes;
   TAPI_CID_DATA_t               *pTapiCidData;

   /* complex tone data */
   TAPI_TONE_DATA_t              TapiComplexToneData [TAPI_TONE_MAXRES];
   /* Handler of event dispatcher. */
   IFX_TAPI_EVENT_HANDLER_DATA_t *pEventHandler;

#ifdef TAPI_EXT_KEYPAD
   /** for external keypad (INCA-IP2 only) */
   IFX_uint8_t                   nDtmfInfo;
#endif /* TAPI_EXT_KEYPAD */

   /* In Use counter */
   IFX_uint16_t                  nInUse;

#ifdef TAPI_PACKET
   FIFO_ID*                      pUpStreamFifo[IFX_TAPI_STREAM_MAX];
#endif /* TAPI_PACKET */

#ifdef KPI_SUPPORT
   /** Control structure for the Kernel Packet Interface (KPI) */
   IFX_TAPI_KPI_STREAM_SWITCH    pKpiStream[IFX_TAPI_KPI_STREAM_MAX];
#endif /* KPI_SUPPORT */

#ifdef TAPI_STATISTICS
   TAPI_STAT_DATA_t              *pTapiStatData;
#endif /* TAPI_STATISTICS */
};

/* ============================= */
/* tapi structure                */
/* ============================= */
struct _TAPI_DEV
{
   /* channel number IFX_TAPI_DEVICE_CH_NUMBER indicates the control device */
   /* ATTENTION, nChannel must be the first element */
   IFX_uint8_t               nChannel;
   /* pointer to LL device structure */
   IFX_TAPI_LL_DEV_t        *pLLDev;
   /* link to the device driver context */
   IFX_TAPI_DRV_CTX_t       *pDevDrvCtx;

   /* Number of channels for which memory is allocated in the array below.
     This does not reflect the number of analog, signaling, pcm or coder
     channels. They are reported in the nResource struct below. */
   IFX_uint8_t               nMaxChannel;
   /* array of tapi channel structures */
   TAPI_CHANNEL             *pTapiChanelArray;
   /* struct with counters how many resources are available and nitialised */
   IFX_TAPI_RESOURCE        nResource;

   /* usage counter, counts the number of open fds */
   IFX_uint16_t              nInUse;
   /* already opened or not */
   IFX_boolean_t             bInitialized;
#ifdef TAPI_ONE_DEVNODE
   /** if set to IFX_TRUE an open was performed on a single dev node.
      This is important for the select wait */
   IFX_boolean_t             bSingleFd;
#endif /* TAPI_ONE_DEVNODE */

   /* TODO: wqEvent and bNeedWakeup should be in a separate structure */
   /* Event wakeup queue for select, this one is used to report all kind of
      device and channel events to the application. Its something like VxWorks
      version of select() call. */
   TAPI_OS_drvSelectQueue_t  wqEvent;

   /* Additional flag for vxWorks if a real wakeup is required */
   volatile IFX_boolean_t    bNeedWakeup;

   /* overall channel protection (ioctl level)
   PS: Avoid nested locking of this mutex. It can lead to a deadlock */
   TAPI_OS_mutex_t           semTapiDevSingleIoctlAccess;

#ifdef TAPI_POLL
   IFX_TAPI_POLL_CONFIG_t *pPollCfg;
   /* Flag for operting mode of tapi (interrupt - default or polling). */
   TAPI_WORKING_MODE             fWorkingMode;
#ifdef TAPI_PACKET
   /* Device specific fifo for downstream direction. */
   FIFO_ID*                  pDownStreamFifo;
#endif /* TAPI_PACKET */
#endif /* TAPI_POLL */

   /** unique TAPI device ID [0,1,...] */
   IFX_int32_t              nDevID;

   /* flag for SmartSLIC - required for special handling of FXO ioctls */
   IFX_boolean_t            bSmartSlicFxo;

   /** last error code and error stack */
   IFX_TAPI_Error_t error;
};


/* internal structure used by the event dispatcher to transport the event
   details from timer/interrupt context into process context
   Note:
   this structure requires the definition of the TAPI_CHANNEL above
   while dependencies to drv_tapi_event.h exist in parallel, the
   best place (although not nice) for this structure is here.
*/
typedef struct
{
#ifdef LINUX
   /* !!! important, work struct/tq_struct must be the first element,
          because we need to cast it later on to its surrounding structure
          IFX_TAPI_EXT_EVENT_PARAM_t */
#ifdef LINUX_2_6
   struct work_struct tapiWs;
#else
   struct tq_struct   tapiTq;
#endif /* LINUX_2_6 */
#endif /* LINUX */
   void             (*pFunc) (void*);
   TAPI_CHANNEL     *pChannel;
   IFX_TAPI_EVENT_t tapiEvent;
} IFX_TAPI_EXT_EVENT_PARAM_t;

/* ============================= */
/* Global variables declaration  */
/* ============================= */
/* Declarations for debug interface */
DECLARE_TRACE_GROUP  (TAPI_DRV);
/* global high level driver context used in system interface only */
extern struct _IFX_TAPI_HL_DRV_CTX gHLDrvCtx [TAPI_MAX_LL_DRIVERS];

extern const IFX_char_t TAPI_WHATVERSION[];
#ifdef HAVE_CONFIG_H
extern const IFX_char_t DRV_TAPI_WHICHCONFIG[];
#endif /* HAVE_CONFIG_H */


/* ======================================== */
/**  Driver identification                  */
/* ======================================== */
extern IFX_int32_t TAPI_Phone_Get_Version      (IFX_char_t * version_string);
extern IFX_int32_t TAPI_Phone_Check_Version    (IFX_TAPI_VERSION_t const *vers);

/* ======================================== */
/**  TAPI Initialisation                    */
/* ======================================== */
extern IFX_int32_t IFX_TAPI_Driver_Start       (IFX_void_t);
extern IFX_void_t  IFX_TAPI_Driver_Stop        (IFX_void_t);

extern IFX_int32_t IFX_TAPI_Event_On_Driver_Start (IFX_void_t);
extern IFX_void_t  IFX_TAPI_Event_On_Driver_Stop  (IFX_void_t);

extern IFX_return_t TAPI_OS_RegisterLLDrv (
                        IFX_TAPI_DRV_CTX_t *pLLDrvCtx,
                        IFX_TAPI_HL_DRV_CTX_t *pHLDrvCtx);
extern IFX_return_t TAPI_OS_UnregisterLLDrv (
                        IFX_TAPI_DRV_CTX_t *pLLDrvCtx,
                        IFX_TAPI_HL_DRV_CTX_t *pHLDrvCtx);

extern IFX_TAPI_DRV_CTX_t* IFX_TAPI_DeviceDriverContextGet (
                        IFX_int32_t Major);

extern IFX_int32_t IFX_TAPI_DeviceStart (
                        TAPI_DEV *pTapiDev,
                        IFX_TAPI_DEV_START_CFG_t *pDevStartCfg);
extern IFX_int32_t IFX_TAPI_DeviceStop (
                        TAPI_DEV *pTapiDev);

/* ======================================== */
/**  TAPI Services                          */
/* ======================================== */

/* Driver initialisation */
extern IFX_int32_t IFX_TAPI_Phone_Init         (TAPI_DEV *pTapiDev, IFX_TAPI_CH_INIT_t const *pInit);

/* Line services */
extern IFX_int32_t TAPI_Phone_Set_LineType     (TAPI_CHANNEL *pChannel, IFX_TAPI_LINE_TYPE_CFG_t *pCfg);
extern IFX_int32_t TAPI_Phone_Set_Linefeed     (TAPI_CHANNEL *pChannel, IFX_int32_t nMode);
extern IFX_return_t TAPI_FXO_Register_DAA      (TAPI_CHANNEL *pChannel, IFX_int32_t nDAA);
extern IFX_return_t TAPI_FXO_Init_DAA          (TAPI_CHANNEL *pChannel);

/* Ringing Services */
extern IFX_int32_t IFX_TAPI_Ring_Initialise    (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Ring_Cleanup       (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_Prepare       (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_Start         (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_Stop          (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_DoBlocking    (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_IsActive      (TAPI_CHANNEL *pChannel);
extern IFX_int32_t IFX_TAPI_Ring_SetCadence    (TAPI_CHANNEL *pChannel, IFX_uint32_t nCadence);
extern IFX_int32_t IFX_TAPI_Ring_SetCadenceHighRes (TAPI_CHANNEL *pChannel, IFX_TAPI_RING_CADENCE_t const *pCadence);
extern IFX_int32_t IFX_TAPI_Ring_SetConfig     (TAPI_CHANNEL *pChannel, IFX_TAPI_RING_CFG_t const *pRingConfig);
extern IFX_int32_t IFX_TAPI_Ring_GetConfig     (TAPI_CHANNEL *pChannel, IFX_TAPI_RING_CFG_t *pRingConfig);
extern IFX_int32_t IFX_TAPI_Ring_SetMaxRings   (TAPI_CHANNEL *pChannel, IFX_uint32_t nArg);
extern IFX_int32_t IFX_TAPI_Ring_Engine_Start  (TAPI_CHANNEL *pChannel, IFX_boolean_t bStartWithInitial);
#ifdef TAPI_CID
extern IFX_int32_t IFX_TAPI_Ring_CalculateRingTiming  (TAPI_CHANNEL *pChannel, IFX_uint32_t *pCadenceRingBurst, IFX_uint32_t *pCadenceRingPause);
#endif /* TAPI_CID */

/* Operation Control Services */
extern IFX_int32_t TAPI_Phone_Hookstate        (TAPI_CHANNEL *pChannel, IFX_int32_t *pState);
extern IFX_int32_t TAPI_Phone_LecMode_Alm_Set  (TAPI_CHANNEL *pChannel, IFX_TAPI_WLEC_CFG_t *pLecConf);
extern IFX_int32_t TAPI_Phone_LecMode_Alm_Get  (TAPI_CHANNEL *pChannel, IFX_TAPI_WLEC_CFG_t *pWLecConf);
extern IFX_int32_t TAPI_Phone_LecMode_Pcm_Set  (TAPI_CHANNEL *pChannel, IFX_TAPI_WLEC_CFG_t *pLecConf);
extern IFX_int32_t TAPI_Phone_LecMode_Pcm_Get  (TAPI_CHANNEL *pChannel, IFX_TAPI_WLEC_CFG_t *pLecConf);
extern IFX_int32_t TAPI_Phone_Dtmf_RxCoeff_Cfg (TAPI_CHANNEL *pChannel, IFX_boolean_t bRW, IFX_TAPI_DTMF_RX_CFG_t *pDtmfRxCoeff);
extern IFX_int32_t IFX_TAPI_PowerSave_Init     (IFX_void_t);
extern IFX_void_t  IFX_TAPI_PowerSave_Exit     (IFX_void_t);

/* Tone Services */
extern IFX_int32_t TAPI_Phone_Tone_Predef_Config (IFX_void_t);
extern IFX_int32_t IFX_TAPI_Tone_Initialise    (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Tone_Cleanup       (TAPI_CHANNEL *pChannel);
extern IFX_int32_t TAPI_Phone_Tone_Play        (TAPI_CHANNEL *pChannel,
                                                IFX_int32_t nToneIndex, TAPI_TONE_DST dst);
extern IFX_int32_t TAPI_Phone_Tone_Play_Unprot (TAPI_CHANNEL *pChannel,
                                                IFX_TAPI_TONE_PLAY_t *pTone);
extern IFX_int32_t TAPI_Phone_Tone_Stop        (TAPI_CHANNEL *pChannel,
                                                IFX_int32_t nToneIndex,
                                                TAPI_TONE_DST nDirection);
extern IFX_int32_t TAPI_DECT_Tone_Stop         (TAPI_CHANNEL *pChannel);

extern IFX_int32_t TAPI_Phone_Tone_Ringback    (TAPI_CHANNEL *pChannel);
extern IFX_int32_t TAPI_Phone_Tone_Busy        (TAPI_CHANNEL *pChannel);
extern IFX_int32_t TAPI_Phone_Tone_Dial        (TAPI_CHANNEL *pChannel);

extern IFX_int32_t TAPI_Phone_Tone_Set_Level   (TAPI_CHANNEL *pChannel, IFX_TAPI_PREDEF_TONE_LEVEL_t const *pToneLevel);
extern IFX_int32_t TAPI_Phone_Tone_Get_State   (TAPI_CHANNEL *pChannel, IFX_uint32_t *pToneState);

extern IFX_int32_t TAPI_Phone_Tone_TableConf   (IFX_TAPI_TONE_t const *pTone);
extern IFX_int32_t TAPI_Phone_Add_SimpleTone   (IFX_TAPI_TONE_SIMPLE_t const *pSimpleTone);
extern IFX_int32_t TAPI_Phone_Add_ComposedTone (IFX_TAPI_TONE_COMPOSED_t const *pComposedTone);
extern IFX_uint32_t IFX_TAPI_Tone_DurationGet  (IFX_uint32_t nToneIndex);

extern IFX_int32_t TAPI_Phone_Tone_Set_On_Time (TAPI_CHANNEL *pChannel, IFX_uint32_t nTime);
extern IFX_int32_t TAPI_Phone_Tone_Set_Off_Time(TAPI_CHANNEL *pChannel, IFX_uint32_t nTime);
extern IFX_int32_t TAPI_Phone_Tone_Get_On_Time (TAPI_CHANNEL *pChannel, IFX_uint32_t *pOnTime);
extern IFX_int32_t TAPI_Phone_Tone_Get_Off_Time(TAPI_CHANNEL *pChannel, IFX_uint32_t *pOffTime);

extern IFX_int32_t TAPI_Phone_DetectToneStart  (TAPI_CHANNEL *pChannel, IFX_TAPI_TONE_CPTD_t const *signal);
extern IFX_int32_t TAPI_Phone_DetectToneStop   (TAPI_CHANNEL *pChannel);

/* PCM Services */
extern IFX_int32_t TAPI_Phone_PCM_IF_Set_Config (TAPI_DEV *pTAPIDev, IFX_TAPI_PCM_IF_CFG_t const *pPCMif);
extern IFX_int32_t TAPI_Phone_PCM_Set_Config    (TAPI_CHANNEL *pChannel, IFX_TAPI_PCM_CFG_t const *pPCMConfig);
extern IFX_int32_t TAPI_Phone_PCM_Get_Config    (TAPI_CHANNEL *pChannel, IFX_TAPI_PCM_CFG_t *pPCMConfig);
extern IFX_int32_t TAPI_Phone_PCM_Set_Activation(TAPI_CHANNEL *pChannel, IFX_uint32_t nMode);
extern IFX_int32_t TAPI_Phone_PCM_Get_Activation(TAPI_CHANNEL *pChannel, IFX_boolean_t *pbAct);
#ifdef TAPI_HDLC
extern IFX_int32_t TAPI_Phone_PCM_HDLC_Set      (TAPI_CHANNEL *pChannel, IFX_TAPI_PCM_HDLC_CFG_t const *pHdlcCfg);
#endif /* TAPI_HDLC */
extern IFX_int32_t TAPI_Phone_PCM_Loop_Set      (TAPI_CHANNEL *pChannel, IFX_TAPI_PCM_LOOP_CFG_t const *pLoopCfg);

/* Dial Services */
extern IFX_int32_t IFX_TAPI_Dial_Initialise    (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Dial_Cleanup       (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Dial_HookEvent     (TAPI_CHANNEL *pChannel, IFX_uint8_t bHookState);
extern IFX_int32_t IFX_TAPI_Dial_SetValidationTime (TAPI_CHANNEL *pChannel, IFX_TAPI_LINE_HOOK_VT_t const *pTime);

#ifdef TAPI_METERING
/* Metering Services */
extern IFX_int32_t IFX_TAPI_Meter_Initialise   (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Meter_Cleanup      (TAPI_CHANNEL *pChannel);
extern IFX_boolean_t TAPI_Phone_Meter_IsActive (TAPI_CHANNEL *pChannel);
extern IFX_int32_t TAPI_Phone_Meter_Stop       (TAPI_CHANNEL *pChannel);
extern IFX_int32_t TAPI_Phone_Meter_Burst      (TAPI_CHANNEL *pChannel, IFX_TAPI_METER_BURST_t const *pMeterBurst);
#endif /* TAPI_METERING */

#ifdef TAPI_VOICE
/* Connection services */
extern IFX_int32_t TAPI_Data_Channel_Add       (TAPI_CHANNEL *pChannel, IFX_TAPI_MAP_DATA_t const *pMap);
extern IFX_int32_t TAPI_Data_Channel_Remove    (TAPI_CHANNEL *pChannel, IFX_TAPI_MAP_DATA_t const *pMap);
#endif /* TAPI_VOICE */

#ifdef TAPI_STATISTICS
extern IFX_int32_t IFX_TAPI_Stat_Initialise    (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Stat_Cleanup       (TAPI_CHANNEL *pChannel);
extern IFX_void_t  IFX_TAPI_Stat_Reset         (TAPI_CHANNEL *pChannel);
extern IFX_uint32_t IFX_TAPI_Stat_Get          (TAPI_CHANNEL *pChannel,
                                                IFX_TAPI_STREAM_t stream,
                                                TAPI_STAT_COUNTER_t counter);
#endif /* TAPI_STATISTICS */

extern IFX_int32_t IFX_TAPI_Module_Find_Connected_Data_Channel (
                                                TAPI_CHANNEL *pChannel,
                                                IFX_TAPI_MAP_TYPE_t nModType,
                                                TAPI_CHANNEL **pTapiCh);

/* FXO services */
extern IFX_int32_t TAPI_FXO_Ioctl              (TAPI_CHANNEL *pChannel,
                                                IFX_uint32_t cmd,
                                                IFX_ulong_t arg);

/* Event packet generation */
extern IFX_return_t TAPI_EVENT_PKT_EV_Generate (TAPI_CHANNEL *pChannel,
                              IFX_TAPI_PKT_EV_GENERATE_t *pPacketEvent);



/* ======================================== */
/**  Event Handling                         */
/* ======================================== */
extern IFX_int32_t  IFX_TAPI_EventDispatcher_Init  (TAPI_CHANNEL *);
extern IFX_int32_t  IFX_TAPI_EventDispatcher_Exit  (TAPI_CHANNEL * pChannel);
extern IFX_int32_t  IFX_TAPI_EventFifoGet          (TAPI_CHANNEL *, IFX_TAPI_EVENT_t *);
extern IFX_uint8_t  IFX_TAPI_EventFifoEmpty        (TAPI_CHANNEL *);
extern IFX_int32_t  IFX_TAPI_Event_SetMask         (TAPI_CHANNEL *pChannel, IFX_TAPI_EVENT_t *pEvent, IFX_uint32_t value);
extern IFX_int32_t  TAPI_Phone_GetEvent            (TAPI_DEV *pTapiDev, IFX_TAPI_EVENT_t *pEvent);
extern IFX_int32_t  IFX_TAPI_Event_Dispatch_ProcessCtx (IFX_TAPI_EXT_EVENT_PARAM_t *pParam);

#ifdef TAPI_FAX_T38
extern IFX_void_t TAPI_FaxT38_Event_Update     (TAPI_CHANNEL *pChannel, IFX_uint8_t status, IFX_uint8_t error);
#endif /* TAPI_FAX_T38 */


/* ======================================== */
/**  Timer functions                        */
/* ======================================== */
extern Timer_ID          TAPI_Create_Timer     (TIMER_ENTRY pTimerEntry, IFX_ulong_t nArgument);
extern IFX_boolean_t     TAPI_SetTime_Timer    (Timer_ID Timer, IFX_uint32_t nTime, IFX_boolean_t bPeriodically, IFX_boolean_t bRestart);
extern IFX_boolean_t     TAPI_Delete_Timer     (Timer_ID Timer);
extern IFX_boolean_t     TAPI_Stop_Timer       (Timer_ID Timer);


/* ======================================== */
/**  Other functions                        */
/* ======================================== */
extern IFX_boolean_t ptr_chk(IFX_void_t *ptr, const IFX_char_t *pPtrName);

extern IFX_void_t TAPI_Tone_Step_Completed(TAPI_CHANNEL* pChannel,
                                           IFX_uint8_t utgNum);
extern IFX_return_t TAPI_DeferWork (IFX_void_t* pFunc, IFX_void_t* pParam);

#ifdef TAPI_EXT_KEYPAD
extern TAPI_CHANNEL * IFX_TAPI_IncaChannelContextGet(IFX_void_t);
#endif /* TAPI_EXT_KEYPAD */


#endif  /* DRV_TAPI_H */
