/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

*******************************************************************************/

/**
   \file drv_tapi_dial.c
   Implements the hook state machine for the analog line.
*/

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_tapi.h"
#include "drv_tapi_errno.h"

/*lint -save -esym(749, TAPI_HOOK_STATE_PULSE_H_FLASH_VAL) */
/* ============================= */
/* Local macros and definitions  */
/* ============================= */


/* ============================= */
/* Type declarations             */
/* ============================= */

/** States of the hookstate finite state machine. */
typedef enum
{
   /* phone is offhook, possibly waiting for interdigit timer */
   TAPI_HOOK_STATE_OFFHOOK,
   /* phone is offhook, waiting for next dialing pulse */
   TAPI_HOOK_STATE_PULSE_H_CONFIRM,
   /* phone has gone offhook: it may just be noise */
   TAPI_HOOK_STATE_OFFHOOK_VAL,
   /* phone has gone offhook while collecting pulses */
   TAPI_HOOK_STATE_PULSE_H_VAL,
   /* phone has gone offhook with an overlap of flash hook min time and digit
      low max time. In this state both digit 1 and flash are reported */
   TAPI_HOOK_STATE_PULSE_H_FLASH_VAL,
   /* phone is onhook, and no timers are running */
   TAPI_HOOK_STATE_ONHOOK,
   /* phone has remained onhook long enough to be low pulse: wait for next offhook to confirm */
   TAPI_HOOK_STATE_PULSE_L_CONFIRM,
   /* overlap of flash hook min time and digit low max time. In this
      state both digit 1 and flash are reported */
   TAPI_HOOK_STATE_PULSE_L_FLASH_CONFIRM,
   /* phone onhook (too long to be digit pulse): could be flash or final onhook */
   TAPI_HOOK_STATE_FLASH_WAIT,
   /* phone onhook (long enough to be flash): wait for next offhook to confirm this */
   TAPI_HOOK_STATE_FLASH_CONFIRM,
   /* validation of the flash hook time after off hook */
   TAPI_HOOK_STATE_FLASH_VAL,
   /* phone onhook (long enough to be final onhook): wait for onhook timer to confirm */
   TAPI_HOOK_STATE_ONHOOK_CONFIRM,
   /* phone has gone onhook: it may just be noise */
   TAPI_HOOK_STATE_ONHOOK_VAL,
   /* phone has gone onhook during pulse dialing: wait to validate */
   TAPI_HOOK_STATE_DIAL_L_VAL
} TAPI_HOOK_STATE;

/** Data of the hook state machine. */
struct TAPI_DIAL_DATA
{
   /* state of the hookstate Finite State Machine */
   TAPI_HOOK_STATE   nHookState;
   /* timer id for dial service */
   Timer_ID          DialTimerID;
   /* number of hook changes */
   IFX_uint8_t       nHookChanges;
   /* a flash hook might has been detected in case of on overlap
      with digit validation time */
   IFX_boolean_t     bProbablyFlash;
   /* indicates that pulse start event was sent, reset by onhook or flashhook */
   IFX_boolean_t     bPulseStartSent;
   /* stores the validation timer settings */
   IFX_TAPI_LINE_HOOK_VT_t       TapiHookOffTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiHookOnTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiHookFlashTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiHookFlashMakeTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiDigitLowTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiDigitHighTime;
   IFX_TAPI_LINE_HOOK_VT_t       TapiInterDigitTime;
};


/* ============================= */
/* Local function declarations   */
/* ============================= */
static IFX_void_t ifx_tapi_dial_OnTimer(Timer_ID Timer, IFX_ulong_t nArg);


/* ============================= */
/* Local function definitions    */
/* ============================= */

/**
   Hook state machine timer callback function.

   The timer is used to verify the duration of various on- and offhook events:
     - Offhook long enough to be confirmed (not just line noise)
     - Offhook too long to be high pulse
     - Offhook long enough to indicate pulse digit completion (interdigit timer)
     - Onhook long enough to be confirmed (not just line noise)
     - Onhook too long to be a digit pulse (it may be flash or final onhook)
     - Onhook long enough to be a flash
     - Onhook too long to be a flash (it may be final onhook)
     - Onhook long enough to be a final onhook

   \param  Timer        TimerID of timer that exipres.
   \param  nArg         Argument of timer. This argument is a pointer to the
                        TAPI_CHANNEL structure.
*/
static IFX_void_t ifx_tapi_dial_OnTimer(Timer_ID Timer, IFX_ulong_t nArg)
{
   TAPI_CHANNEL *pChannel = (TAPI_CHANNEL *) nArg;
   TAPI_DIAL_DATA_t *pTapiDialData = pChannel->pTapiDialData;
   IFX_uint8_t     nPulseDigit;
  /* Event information. */
   IFX_TAPI_EVENT_t tapiEvent;

   /* to silence the code checkers */
   Timer = Timer;

   switch (pTapiDialData->nHookState)
   {
      case TAPI_HOOK_STATE_OFFHOOK:
         /* Interdigit timer expiry indicates a complete pulse digit has
            been collected */
         nPulseDigit = pTapiDialData->nHookChanges;
         if (nPulseDigit == 10)
            nPulseDigit = 0xB;  /* digit 0 is represented by 0xB */
         if ((nPulseDigit == 1)&&
             (pTapiDialData->bProbablyFlash == IFX_TRUE))
         {
            /* FLASH event. Put pulse event into the event fifo.*/
            memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
            tapiEvent.id = IFX_TAPI_EVENT_FXS_FLASH;
            IFX_TAPI_Event_Dispatch(pChannel,&tapiEvent);

            TAPI_Stop_Timer (pTapiDialData->DialTimerID);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
            pTapiDialData->bPulseStartSent = IFX_FALSE;
         }

         /* PULSE event. Put pulse event into the event fifo. */
         memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
         tapiEvent.id = IFX_TAPI_EVENT_PULSE_DIGIT;
         tapiEvent.data.pulse.digit = (nPulseDigit & 0xff);
         IFX_TAPI_Event_Dispatch(pChannel,&tapiEvent);

         /* reset probably flash flag */
         pTapiDialData->bProbablyFlash = IFX_FALSE;
         /* reset the pulse counter, ready for next digit */
         pTapiDialData->nHookChanges = 0;

         /* NB: the FSM remains in the same state, OFFHOOK */
         break;

      case TAPI_HOOK_STATE_PULSE_H_CONFIRM:
         /* digit_h_max expiry indicates offhook has lasted too long to be
            high pulse */
         TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiInterDigitTime.nMinTime -
            pTapiDialData->TapiDigitHighTime.nMaxTime),
            IFX_FALSE, IFX_FALSE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         break;

      case TAPI_HOOK_STATE_OFFHOOK_VAL:
         /* Timer indicates offhook has lasted long enough to be validated */
         /* IFX_TRUE => hook event, put into fifo */
         memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
         tapiEvent.id = IFX_TAPI_EVENT_FXS_OFFHOOK;
         IFX_TAPI_Event_Dispatch(pChannel,&tapiEvent);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         pTapiDialData->bPulseStartSent = IFX_FALSE;
         break;

      case TAPI_HOOK_STATE_PULSE_H_VAL:
         /* digit_h_min timer indicates offhook has lasted long enough to
            be validated */
         TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiDigitHighTime.nMaxTime -
            pTapiDialData->TapiDigitHighTime.nMinTime),
            IFX_FALSE, IFX_FALSE);
         /* A pulse has been detected (not yet a complete pulse digit) */
         pTapiDialData->nHookChanges++;
         pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_H_CONFIRM;

         if ((pTapiDialData->nHookChanges == 1) &&
             (pTapiDialData->bPulseStartSent == IFX_FALSE))
         {
            /* Report the first validated pulse */
            /* Use this to stop a dialtone right after the first pulse. */
            memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
            tapiEvent.id = IFX_TAPI_EVENT_PULSE_START;
            IFX_TAPI_Event_Dispatch(pChannel,&tapiEvent);
            /* Do this only for the first pulse after offhook or flash. */
            pTapiDialData->bPulseStartSent = IFX_TRUE;
         }
         break;

      case TAPI_HOOK_STATE_PULSE_L_CONFIRM:
         /* Timer digit_l_max expires: onhook has lasted too long to be
            a pulse reset the pulse counter: pulse digit collection, if in
            progress, has been aborted */
         if (pTapiDialData->TapiHookFlashTime.nMinTime >
             pTapiDialData->TapiDigitLowTime.nMaxTime)
         {
            TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
               (IFX_uint32_t)(pTapiDialData->TapiHookFlashTime.nMinTime -
               pTapiDialData->TapiDigitLowTime.nMaxTime),
               IFX_FALSE, IFX_FALSE);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_FLASH_WAIT;
         }
         else
         {
            if (pTapiDialData->TapiHookFlashTime.nMinTime ==
                pTapiDialData->TapiDigitLowTime.nMaxTime)
            {
               /* special case no time to wait in state L_FLASH_CONFIRM */
               TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
                  (IFX_uint32_t)(pTapiDialData->TapiHookFlashTime.nMaxTime -
                  pTapiDialData->TapiDigitLowTime.nMaxTime),
                  IFX_FALSE, IFX_FALSE);
               pTapiDialData->nHookState = TAPI_HOOK_STATE_FLASH_CONFIRM;
            }
            else
            {
               /* If TapiHookFlashTime.nMinTime <= TapiDigitLowTime.nMaxTime,
                * skip over state TAPI_HOOK_STATE_FLASH_WAIT.           */
               TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
                  (IFX_uint32_t)(pTapiDialData->TapiDigitLowTime.nMaxTime -
                  pTapiDialData->TapiHookFlashTime.nMinTime),
                  IFX_FALSE, IFX_FALSE);
               pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_L_FLASH_CONFIRM;

            }
         }
         break;

      case TAPI_HOOK_STATE_FLASH_WAIT:
         /* Timer flash_min expires: the onhook has lasted long enough
            to be a flash */
         TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiHookFlashTime.nMaxTime -
            pTapiDialData->TapiHookFlashTime.nMinTime),
            IFX_FALSE, IFX_FALSE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_FLASH_CONFIRM;
         break;

      case TAPI_HOOK_STATE_FLASH_CONFIRM:
         /* Timer flash_max expires: onhook has lasted too long to be a flash */
         if (pTapiDialData->TapiHookOnTime.nMinTime > pTapiDialData->TapiHookFlashTime.nMaxTime)
         {
            TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
               (IFX_uint32_t)(pTapiDialData->TapiHookOnTime.nMinTime -
               pTapiDialData->TapiHookFlashTime.nMaxTime),
               IFX_FALSE, IFX_FALSE);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK_CONFIRM;
         }
         else
         {
            /* If TapiHookOnTime <= TapiHookFlashTime.nMaxTime,
             * skip over state TAPI_HOOK_STATE_ONHOOK_CONFIRM: offhook is
             already confirmed.  */
            /* IFX_TRUE => hook event */
            memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
            tapiEvent.id = IFX_TAPI_EVENT_FXS_ONHOOK;
            IFX_TAPI_Event_Dispatch(pChannel, &tapiEvent);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK;
         }
         break;

      case TAPI_HOOK_STATE_FLASH_VAL:
         /* Still Offhook indicates completion of flash */
         memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
         tapiEvent.id = IFX_TAPI_EVENT_FXS_FLASH;
         IFX_TAPI_Event_Dispatch(pChannel,&tapiEvent);
         TAPI_Stop_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         pTapiDialData->bPulseStartSent = IFX_FALSE;
         break;

      case TAPI_HOOK_STATE_ONHOOK_CONFIRM:
         /* Timer onhook expires: onhook has lasted long enough to be final */
         /* IFX_TRUE => hook event */
         memset(&tapiEvent, 0, sizeof(IFX_TAPI_EVENT_t));
         tapiEvent.id = IFX_TAPI_EVENT_FXS_ONHOOK;
         IFX_TAPI_Event_Dispatch(pChannel, &tapiEvent);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK;
         break;

      case TAPI_HOOK_STATE_ONHOOK_VAL:
         /* digit_l_min expires: onhook has lasted long enough to be a
            low pulse (not noise). Now reset the pulse counter,
            ready for next digit */
         pTapiDialData->nHookChanges = 0;
         /* NOTE: the "break" statement has been intentionally omitted */
         /*lint -fallthrough */

      case TAPI_HOOK_STATE_DIAL_L_VAL:
         /* digit_l_min expires: onhook has lasted long enough to be a
            certain low pulse (not noise). The next state is the overlap with
            flash hook */
         if (pTapiDialData->TapiDigitLowTime.nMaxTime <
             pTapiDialData->TapiHookFlashTime.nMinTime)
         {
            /* no overlap of hook flash min and pulse max time */
            TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
               (IFX_uint32_t)(pTapiDialData->TapiDigitLowTime.nMaxTime -
               pTapiDialData->TapiDigitLowTime.nMinTime),
               IFX_FALSE, IFX_FALSE);
         }
         else
         {
            TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
               (IFX_uint32_t)(pTapiDialData->TapiHookFlashTime.nMinTime -
               pTapiDialData->TapiDigitLowTime.nMinTime),
               IFX_FALSE, IFX_FALSE);
         }
         pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_L_CONFIRM;
         break;

      case TAPI_HOOK_STATE_PULSE_L_FLASH_CONFIRM:
         /* Overlap time of flash hook min and digit low max time ended */
         TAPI_SetTime_Timer(pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiHookFlashTime.nMaxTime -
            pTapiDialData->TapiDigitLowTime.nMaxTime),
            IFX_FALSE, IFX_FALSE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_FLASH_CONFIRM;

         break;

      default:
         /* all other states represent error cases */
         break;
   }

   return;
}


/* ============================= */
/* Global function definitions   */
/* ============================= */

/**
   Initialise the analog line state machines on the given channel.

   Initialise the data structures and resources needed for the
   hook state machine.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.

   \return
     - \ref IFX_SUCCESS: if successful
     - \ref IFX_ERROR: in case of an error
*/
IFX_int32_t IFX_TAPI_Dial_Initialise(TAPI_CHANNEL *pChannel)
{
   TAPI_DIAL_DATA_t  *pTapiDialData  = pChannel->pTapiDialData;

   /* check if channel has the required analog or audio module */
   if ((pChannel->nChannel >=  pChannel->pTapiDevice->nResource.AlmCount) &&
       (pChannel->nChannel >=  pChannel->pTapiDevice->nResource.AudioCount))
   {
      /* no analog or audio module -> nothing to do  --  this is not an error */
      return IFX_SUCCESS;
   }

   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   /* allocate data storage for the hook state machine on the channel
      if not already existing */
   if (pTapiDialData == IFX_NULL)
   {
      /* allocate data storage */
      if ((pTapiDialData = TAPI_OS_Malloc (sizeof(*pTapiDialData))) == IFX_NULL)
      {
         TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
         return IFX_ERROR;
      }
      /* Store pointer to data in the channel or we lose it on exit. */
      pChannel->pTapiDialData = pTapiDialData;
      memset (pTapiDialData, 0x00, sizeof(*pTapiDialData));
   }
   /* from here on pTapiDialData and pChannel->pTapiDialData are valid */

   /* create hook-state-machine validation timer if not already existing */
   if (pTapiDialData->DialTimerID == 0)
   {
      pTapiDialData->DialTimerID =
         TAPI_Create_Timer((TIMER_ENTRY)ifx_tapi_dial_OnTimer,
                           (IFX_uintptr_t)pChannel);
      if (pTapiDialData->DialTimerID == 0)
      {
         TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
         IFX_TAPI_Dial_Cleanup(pChannel);
         return IFX_ERROR;
      }
   }
   /* set default values for the validation timers */
   pTapiDialData->TapiDigitLowTime.nMinTime      = TAPI_MIN_DIGIT_LOW;
   pTapiDialData->TapiDigitLowTime.nMaxTime      = TAPI_MAX_DIGIT_LOW;
   pTapiDialData->TapiDigitHighTime.nMinTime     = TAPI_MIN_DIGIT_HIGH;
   pTapiDialData->TapiDigitHighTime.nMaxTime     = TAPI_MAX_DIGIT_HIGH;
   pTapiDialData->TapiHookFlashTime.nMinTime     = TAPI_MIN_FLASH;
   pTapiDialData->TapiHookFlashTime.nMaxTime     = TAPI_MAX_FLASH;
   pTapiDialData->TapiHookFlashMakeTime.nMinTime = TAPI_MIN_FLASH_MAKE;
   pTapiDialData->TapiInterDigitTime.nMinTime    = TAPI_MIN_INTERDIGIT;
   pTapiDialData->TapiHookOffTime.nMinTime       = TAPI_MIN_OFF_HOOK;
   pTapiDialData->TapiHookOnTime.nMinTime        = TAPI_MIN_ON_HOOK;
   /* start hook state FSM in onhook state */
   pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK;
   /* in onhook state the next pulse is the first pulse to be sent */
   pTapiDialData->bPulseStartSent = IFX_FALSE;

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);

   return IFX_SUCCESS;
}


/**
   Cleanup the analog line state machines on the given channel.

   Free the resources needed for the hook state machine.

   \param   pChannel    Pointer to TAPI_CHANNEL structure.
*/
IFX_void_t IFX_TAPI_Dial_Cleanup(TAPI_CHANNEL *pChannel)
{
   TAPI_DIAL_DATA_t  *pTapiDialData  = pChannel->pTapiDialData;

   TAPI_OS_MutexGet (&pChannel->semTapiChDataLock);

   if (pTapiDialData != IFX_NULL)
   {
      /* unconditionally destruct the hook state machine timer if existing */
      if (pTapiDialData->DialTimerID != 0)
      {
         TAPI_Delete_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->DialTimerID = 0;
      }

      /* free the data storage on the channel */
      TAPI_OS_Free (pTapiDialData);
      pChannel->pTapiDialData = IFX_NULL;
   }

   TAPI_OS_MutexRelease (&pChannel->semTapiChDataLock);
}


/**
   Trigger an event in the hook-state-machine.

   \param pChannel    - handle to TAPI_CHANNEL structure
   \param bHookState  - IFX_TRUE : off hook
                        IFX_FALSE: on hook
   \return
     None
*/
IFX_void_t IFX_TAPI_Dial_HookEvent (TAPI_CHANNEL * pChannel,
                                    IFX_uint8_t bHookState)
{
   TAPI_DIAL_DATA_t *pTapiDialData = pChannel->pTapiDialData;

   /* make sure dial is initialised on this channel */
   if (pTapiDialData == IFX_NULL)
   {
      /* silently ignore event - this case can happen during startup or
         shutdown when an interrupt occures but the structures do not exist. */
      return;
   }

   if (bHookState == IFX_TRUE)
   {
      /* phone has gone offhook */
      switch (pTapiDialData->nHookState)
      {
      case TAPI_HOOK_STATE_ONHOOK:
         /* Set timer to verify this offhook isn't just some line noise */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiHookOffTime.nMinTime),
            IFX_FALSE, IFX_FALSE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK_VAL;
         break;

      case TAPI_HOOK_STATE_PULSE_L_CONFIRM:
         /* Offhook may indicate completion of low pulse: set timer to validate
          */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiDigitHighTime.nMinTime),
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_H_VAL;
         break;

      case TAPI_HOOK_STATE_PULSE_L_FLASH_CONFIRM:
         /* overlap of hook flash min and pulse max time. Confirmation needed:
            - interdigit time and hook flash min time ended: report digit 1 and
            flash hook - next pulse occurs: report only digit */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiDigitHighTime.nMinTime),
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_H_VAL;
         pTapiDialData->bProbablyFlash = IFX_TRUE;
         break;

      case TAPI_HOOK_STATE_FLASH_WAIT:
         /* Offhook arrives too soon for flash: go to offhook (no event) */
         TAPI_Stop_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         break;

      case TAPI_HOOK_STATE_FLASH_CONFIRM:
         if (pTapiDialData->TapiHookFlashMakeTime.nMinTime == 0)
         {
            IFX_TAPI_EVENT_t tapiEvent;
            /* Offhook indicates completion of flash */
            memset (&tapiEvent, 0, sizeof (IFX_TAPI_EVENT_t));
            tapiEvent.id = IFX_TAPI_EVENT_FXS_FLASH;
            IFX_TAPI_Event_Dispatch (pChannel, &tapiEvent);
            TAPI_Stop_Timer (pTapiDialData->DialTimerID);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
            pTapiDialData->bPulseStartSent = IFX_FALSE;
         }
         else
         {
            /* validation time specified so validate */
            TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
               (IFX_uint32_t)(pTapiDialData->TapiHookFlashMakeTime.nMinTime),
               IFX_FALSE, IFX_TRUE);
            pTapiDialData->nHookState = TAPI_HOOK_STATE_FLASH_VAL;
         }
         break;

      case TAPI_HOOK_STATE_ONHOOK_CONFIRM:
         /* Offhook arrives too soon for final onhook: go to offhook (no event)
          */
         TAPI_Stop_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         break;

      case TAPI_HOOK_STATE_ONHOOK_VAL:
         /* Offhook while validating onhook: stop validation timer and return
            to OFFHOOK */
         /* If collecting digits, abort */
         pTapiDialData->nHookChanges = 0;
         TAPI_Stop_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_OFFHOOK;
         break;

      case TAPI_HOOK_STATE_DIAL_L_VAL:
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)(pTapiDialData->TapiDigitHighTime.nMinTime),
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_PULSE_H_VAL;
         break;

      default:
         /* all other states represent an error condition */
         break;
      }
   }
   else
   {
      /* phone has gone onhook */
      switch (pTapiDialData->nHookState)
      {
      case TAPI_HOOK_STATE_FLASH_VAL:
      case TAPI_HOOK_STATE_OFFHOOK:
         /* Set timer to confirm whether this onhook is not just result of line
            noise */
         /* Restart=IFX_TRUE because interdigit timer could be running */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)pTapiDialData->TapiDigitLowTime.nMinTime,
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK_VAL;
         break;

      case TAPI_HOOK_STATE_PULSE_H_CONFIRM:
         /* Set timer to confirm whether this onhook is not just result of line
            noise */
         /* Restart=IFX_TRUE because DigitHigh.nMaxTime timer is running */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)pTapiDialData->TapiDigitLowTime.nMinTime,
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_DIAL_L_VAL;
         break;

      case TAPI_HOOK_STATE_PULSE_H_VAL:
         /* Pulse duration too short */
         TAPI_SetTime_Timer (pTapiDialData->DialTimerID,
            (IFX_uint32_t)pTapiDialData->TapiDigitLowTime.nMinTime,
            IFX_FALSE, IFX_TRUE);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_DIAL_L_VAL;
         break;

      case TAPI_HOOK_STATE_OFFHOOK_VAL:
         TAPI_Stop_Timer (pTapiDialData->DialTimerID);
         pTapiDialData->nHookState = TAPI_HOOK_STATE_ONHOOK;
         break;

      default:
         /* all other states represent an error condition */
         break;
      }
   }

   return;
}


/**
   Sets the validation timer for hook, pulse digit and hook flash.

   \param pChannel        - handle to TAPI_CHANNEL structure
   \param pTime           - type of validation setting, min and max time in milliseconds

   \return
   IFX_SUCCESS or IFX_ERROR

   \remark
   For timers that do not distinguish between a min and max value, both
   min and max values are set equal.
*/
IFX_int32_t IFX_TAPI_Dial_SetValidationTime(TAPI_CHANNEL *pChannel,
                                            IFX_TAPI_LINE_HOOK_VT_t const *pTime)
{
   TAPI_DIAL_DATA_t *pTapiDialData = pChannel->pTapiDialData;

   /* check if dial is initialised on this channel */
   if (pTapiDialData == IFX_NULL)
   {
      /* errmsg: Service not supported on called channel context */
      RETURN_STATUS (TAPI_statusInvalidCh, 0);
   }

   /* zero timer values are not allowed */
   if (pTime->nMinTime == 0 || pTime->nMaxTime == 0)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
            ("DRV_ERROR: zero validation timer values not allowed for channel %d\n",
                                     pChannel->nChannel));
      RETURN_STATUS (TAPI_statusParam, 0);
   }

   /* check whether min timer value > max timer value */
   if (pTime->nMinTime > pTime->nMaxTime)
   {
      TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
           ("DRV_ERROR: min value %d > max value %d for channel %d\n",
             pTime->nMinTime, pTime->nMaxTime, pChannel->nChannel));
      RETURN_STATUS (TAPI_statusParam, 0);
   }

   /* configure the validation timers at system startup */
   switch (pTime->nType)
   {
      case IFX_TAPI_LINE_HOOK_VT_HOOKOFF_TIME:
         pTapiDialData->TapiHookOffTime.nType    = pTime->nType;
         pTapiDialData->TapiHookOffTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiHookOffTime.nMaxTime = pTime->nMaxTime;
         break;

      case IFX_TAPI_LINE_HOOK_VT_HOOKON_TIME:
         pTapiDialData->TapiHookOnTime.nType    = pTime->nType;
         pTapiDialData->TapiHookOnTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiHookOnTime.nMaxTime = pTime->nMaxTime;
         break;

      case IFX_TAPI_LINE_HOOK_VT_HOOKFLASH_TIME:
         pTapiDialData->TapiHookFlashTime.nType    = pTime->nType;
         pTapiDialData->TapiHookFlashTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiHookFlashTime.nMaxTime = pTime->nMaxTime;
         break;

      case IFX_TAPI_LINE_HOOK_VT_DIGITLOW_TIME:
         pTapiDialData->TapiDigitLowTime.nType    = pTime->nType;
         pTapiDialData->TapiDigitLowTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiDigitLowTime.nMaxTime = pTime->nMaxTime;
         break;

      case IFX_TAPI_LINE_HOOK_VT_DIGITHIGH_TIME:
         pTapiDialData->TapiDigitHighTime.nType    = pTime->nType;
         pTapiDialData->TapiDigitHighTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiDigitHighTime.nMaxTime = pTime->nMaxTime;
         break;

      case IFX_TAPI_LINE_HOOK_VT_INTERDIGIT_TIME:
         pTapiDialData->TapiInterDigitTime.nType    = pTime->nType;
         pTapiDialData->TapiInterDigitTime.nMinTime = pTime->nMinTime;
         pTapiDialData->TapiInterDigitTime.nMaxTime = pTime->nMaxTime;
         break;

      default:
         TRACE(TAPI_DRV,DBG_LEVEL_HIGH,
               ("DRV_ERROR: unknown validation type 0x%x\n",pTime->nType));
         RETURN_STATUS (TAPI_statusParam, 0);
   }

   return IFX_SUCCESS;
}
/*lint -restore */
