/******************************************************************************

                              Copyright (c) 2009
                            Lantiq Deutschland GmbH
                     Am Campeon 3; 85579 Neubiberg, Germany

  For licensing information, see the file 'LICENSE' in the root folder of
  this software module.

******************************************************************************/

/**
   \file drv_tapi_qos.c
   Implements the handling of the QOS driver registration and ioctls.
*/

/* ============================= */
/* Check if feature is enabled   */
/* ============================= */
#ifdef HAVE_CONFIG_H
#include <drv_config.h>
#endif

#ifdef QOS_SUPPORT

#ifndef LINUX
#error This feature is actually only available under Linux.
#endif

#ifndef KPI_SUPPORT
#error This feature requires KPI to be enabled.
#endif

/* ============================= */
/* Includes                      */
/* ============================= */
#include "drv_tapi.h"
#include "drv_tapi_qos_io.h"
#include "drv_tapi_qos_ll_interface.h"

/* ============================= */
/* Local definitions             */
/* ============================= */

/* ============================= */
/* Local structures              */
/* ============================= */

/* ============================= */
/* Local variable definition     */
/* ============================= */
/** Pointer to registered QOS driver context. */
static IFX_TAPI_DRV_CTX_QOS_t *gpQosCtx = IFX_NULL;
/** QOS driver context protection */
static TAPI_OS_mutex_t        semProtectQosCtx;

/** global variable used to configure via insmod option */
extern IFX_int32_t block_egress_tasklet;

/* ============================= */
/* Global variable definition    */
/* ============================= */

/* ============================= */
/* Local function declaration    */
/* ============================= */

static IFX_return_t ifx_tapi_QOS_SessionStart(TAPI_CHANNEL* const pCh,
                                           const QOS_INIT_SESSION* const pInit);
static IFX_return_t ifx_tapi_QOS_SessionStop(TAPI_CHANNEL* const pCh);
static IFX_int32_t  ifx_tapi_QOS_Cleanup(IFX_void_t);


/* ============================= */
/* Local function definition     */
/* ============================= */

/** This service creates a new session on a channel.

    Multiple sessions can be created on a channel but only one session may be
    active at the same time.

   \param  pCh       Handle to TAPI_CHANNEL structure.
   \param  pInit     Handle to QOS_INIT_SESSION structure.

   \return Returns value as follows:
      - \ref IFX_SUCCESS: if successful
      - \ref IFX_ERROR: in case of an error
*/
static IFX_return_t ifx_tapi_QOS_SessionStart(TAPI_CHANNEL* const pCh,
                                            const QOS_INIT_SESSION* const pInit)
{
   IFX_int16_t       nKpiChannel;

   IFX_return_t ret = IFX_SUCCESS;

   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("Qos_StartSession\n"));

   /* caller of this static function passes only safe parameters */
   TAPI_ASSERT(pCh != IFX_NULL);
   TAPI_ASSERT(pInit != IFX_NULL);

   /* protect channel-data from mutual access */
   TAPI_OS_MutexGet (&pCh->semTapiChDataLock);

   nKpiChannel = IFX_TAPI_KPI_ChGet(pCh, IFX_TAPI_KPI_STREAM_COD);

   /* release channel-data protection */
   TAPI_OS_MutexRelease (&pCh->semTapiChDataLock);

   /* Is KPI already configured on this channel? */
   if ((nKpiChannel & 0xF000) != IFX_TAPI_KPI_UDP)
   {
      /* Get a free channel in the QOS driver. We will then map the traffic
         of this channel to this KPI channel in KPI group IFX_TAPI_KPI_UDP. */
      nKpiChannel = gpQosCtx->getFreeChannel();
      if (nKpiChannel >= 0)
      {
         IFX_TAPI_KPI_CH_CFG_t  cfg;

         /* Configure KPI */
         cfg.nStream = IFX_TAPI_KPI_STREAM_COD;
         cfg.nKpiCh = IFX_TAPI_KPI_UDP | nKpiChannel;
         ret = IFX_TAPI_KPI_ChCfgSet (pCh, &cfg);
      }
      else
      {
         ret = IFX_ERROR;
      }
   }
   else
   {
      /* Remove KPI group from channel */
      nKpiChannel &= 0x0FFF;
   }

   if (ret == IFX_SUCCESS)
   {
      /* initiate a session */
      ret = gpQosCtx->start((IFX_TAPI_KPI_CH_t)nKpiChannel,
                            pInit->srcAddr, pInit->srcPort,
                            pInit->destAddr, pInit->destPort);
   }

   if (ret != IFX_SUCCESS)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("QOS session NOT started.\n"));
      ret = IFX_ERROR;
   }

   return ret;
} /* Qos_StartSession() */


/** Deactivate and delete a session on a channel.

   \param  pCh       Handle to TAPI_CHANNEL structure.

   \return Returns value as follows:
      - \ref IFX_SUCCESS: if successful
      - \ref IFX_ERROR: in case of an error
*/
static IFX_return_t ifx_tapi_QOS_SessionStop(TAPI_CHANNEL* const pCh)
{
   IFX_TAPI_KPI_CH_t nKpiChannel;
   IFX_return_t ret = IFX_ERROR;

   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("Qos_StopSession()\n"));

   /* caller of this static function passes only safe parameters */
   TAPI_ASSERT(pCh != IFX_NULL);

   /* protect channel-data from mutual access */
   TAPI_OS_MutexGet (&pCh->semTapiChDataLock);

   nKpiChannel = IFX_TAPI_KPI_ChGet(pCh, IFX_TAPI_KPI_STREAM_COD);

   /* release channel-data protection */
   TAPI_OS_MutexRelease (&pCh->semTapiChDataLock);

   /* Is KPI configured for UDP group on this channel? */
   if ((nKpiChannel & 0xF000) == IFX_TAPI_KPI_UDP)
   {
      IFX_TAPI_KPI_CH_CFG_t  cfg;

      /* Remove KPI group from channel */
      nKpiChannel &= 0x0FFF;
      /* deactivate a session */
      ret = gpQosCtx->stop(nKpiChannel);

      /* Unconfigure KPI */
      cfg.nStream = IFX_TAPI_KPI_STREAM_COD;
      cfg.nKpiCh = IFX_TAPI_KPI_GROUP0;
      ret = IFX_TAPI_KPI_ChCfgSet (pCh, &cfg);
   }

   if (ret != IFX_SUCCESS)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("QOS session NOT stopped.\n"));
      ret = IFX_ERROR;
   }

   return ret;
}


/**
   Cleanup every ressource used by QOS.

   \return Returns value as follows:
      - \ref IFX_SUCCESS: if successful
      - \ref IFX_ERROR: in case of an error
*/
static IFX_int32_t ifx_tapi_QOS_Cleanup(IFX_void_t)
{
   IFX_int32_t  ret;

   TRACE(TAPI_DRV, DBG_LEVEL_LOW, ("Qos_Cleanup()\n"));

   /* Clear all filter tables. */
   ret = gpQosCtx->clean();

   return ret;
}


/* ============================= */
/* Global function definition    */
/* ============================= */

/**
   Initialise the QOS service

   \return
   - IFX_SUCCESS  in all cases
*/
IFX_return_t IFX_TAPI_QOS_Init (IFX_void_t)
{
   /* create QOS driver context protection semaphore */
   TAPI_OS_MutexInit (&semProtectQosCtx);

   return IFX_SUCCESS;
}


/**
   Clean-up the QOS service

   \return none
*/
IFX_void_t IFX_TAPI_QOS_Cleanup (IFX_void_t)
{
   /* delete QOS driver context protection semaphore */
   TAPI_OS_MutexDelete (&semProtectQosCtx);
}


/**
   Register a QOS driver in TAPI

   This function is exported to be called by the QOS driver.

   \param  pQosCtx      Pointer to QOS driver context. IFX_NULL to unregister.

   \return IFX_SUCCESS or IFX_ERROR.
*/
IFX_return_t IFX_TAPI_QOS_DrvRegister (IFX_TAPI_DRV_CTX_QOS_t *pQosCtx)
{

   if (pQosCtx != IFX_NULL)
   {
      IFX_size_t l = strlen(pQosCtx->InterfaceVersion);

      /* Make sure the interface versions are matching before registering. */
      if ((l != strlen(DRV_QOS_INTERFACE_VER_STR)) ||
         strncmp (pQosCtx->InterfaceVersion, DRV_QOS_INTERFACE_VER_STR, l))
      {
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("TAPI: ATTENTION - mismatch of QOS driver Interface\n"));
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("TAPI: please check that drv_tapi and drv_%s driver match.\n",
             pQosCtx->drvName));
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("Version set in QOS Driver = %.10s\n", pQosCtx->InterfaceVersion));
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
            ("Version expected by TAPI  = %.10s\n", DRV_QOS_INTERFACE_VER_STR));
         return IFX_ERROR;
      }
   }

   /* protect access to global variable to prevent deregistering while an
      ioctl is called */
   TAPI_OS_MutexGet (&semProtectQosCtx);

   /* store the driver context */
   gpQosCtx = pQosCtx;

   /* optional: register KPI egress tasklet - if implemented */
   if (pQosCtx != IFX_NULL && pQosCtx->pQosEgressTasklet)
   {
      if (!block_egress_tasklet)
         IFX_TAPI_KPI_EgressTaskletRegister(IFX_TAPI_KPI_UDP,
                                            pQosCtx->pQosEgressTasklet);
   }

   TAPI_OS_MutexRelease (&semProtectQosCtx);

   return IFX_SUCCESS;
}


/**
   Handles the ioctls for QOS support on a TAPI channel.

   This function is called from the OS Ioctl to process qos actions.

   \param  pChanDev     Pointer to TAPI_CHANNEL structure.
   \param  qosCmd       QOS command id.
   \param  qosArg       QOS argument.

   \return IFX_SUCCESS or IFX_ERROR.
*/
IFX_int32_t IFX_TAPI_Qos_Ctrl(IFX_void_t *pChanDev,
                              IFX_uint32_t qosCmd,
                              IFX_ulong_t qosArg)
{
   TAPI_CHANNEL* pCh = (TAPI_CHANNEL *)pChanDev;
   IFX_int32_t ret = IFX_ERROR;
   QOS_INIT_SESSION _init;

   if (IFX_NULL == pCh)
   {
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("Invalid input argument(s). (File: %s, line: %d)\n",
            __FILE__, __LINE__));
      return ret;
   }

   /* protect access to global variable to prevent deregistering while an
      ioctl is called */
   TAPI_OS_MutexGet (&semProtectQosCtx);

   if (gpQosCtx == IFX_NULL)
   {
      TAPI_OS_MutexRelease (&semProtectQosCtx);
      TRACE(TAPI_DRV, DBG_LEVEL_HIGH,
           ("No QOS driver registered - configuration failed\n"));
      return ret;
   }

   switch (qosCmd)
   {
      case FIO_QOS_START:
         TAPI_OS_CpyUsr2Kern (&_init, (IFX_void_t*)qosArg,
                              sizeof(QOS_INIT_SESSION));
         ret = ifx_tapi_QOS_SessionStart(pCh, &_init);
      break;
      case FIO_QOS_ACTIVATE:
         /* "activate" is now already done with "start" */
         ret = IFX_SUCCESS;
      break;
      case FIO_QOS_STOP:
         ret = ifx_tapi_QOS_SessionStop(pCh);
      break;
      case FIO_QOS_CLEAN:
         ret = ifx_tapi_QOS_Cleanup();
      break;
      default:
         /* Unknown QOS command. */
         TRACE(TAPI_DRV, DBG_LEVEL_HIGH, ("Unknown QOS command.\n"));
      break;
   }

   TAPI_OS_MutexRelease (&semProtectQosCtx);

   return ret;
}

#endif /* QOS_SUPPORT */
