// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { v4 as uuid } from "uuid";
import { Constants, TokenType, defaultCancellableLock, isSasTokenProvider } from "@azure/core-amqp";
import { logger } from "./log";
import { getRetryAttemptTimeoutInMs } from "./util/retries";
/**
 * Describes the base class for entities like EventHub Sender, Receiver and Management link.
 * @internal
 */
export class LinkEntity {
    /**
     * Creates a new LinkEntity instance.
     * @hidden
     * @param context - The connection context.
     * @param options - Options that can be provided while creating the LinkEntity.
     */
    constructor(context, options) {
        /**
         * Indicates whether the link is in the process of connecting
         * (establishing) itself. Default value: `false`.
         */
        this.isConnecting = false;
        if (!options)
            options = {};
        this._context = context;
        this.address = options.address || "";
        this.audience = options.audience || "";
        this.name = `${options.name}-${uuid()}`;
        this.partitionId = options.partitionId;
    }
    /**
     * Negotiates cbs claim for the LinkEntity.
     * @hidden
     * @returns Promise<void>
     */
    async _negotiateClaim({ abortSignal, setTokenRenewal, timeoutInMs }) {
        // Acquire the lock and establish a cbs session if it does not exist on the connection.
        // Although node.js is single threaded, we need a locking mechanism to ensure that a
        // race condition does not happen while creating a shared resource (in this case the
        // cbs session, since we want to have exactly 1 cbs session per connection).
        logger.verbose("[%s] Acquiring cbs lock: '%s' for creating the cbs session while creating the %s: " +
            "'%s' with address: '%s'.", this._context.connectionId, this._context.cbsSession.cbsLock, this._type, this.name, this.address);
        const startTime = Date.now();
        if (!this._context.cbsSession.isOpen()) {
            await defaultCancellableLock.acquire(this._context.cbsSession.cbsLock, () => {
                return this._context.cbsSession.init({
                    abortSignal,
                    timeoutInMs: timeoutInMs - (Date.now() - startTime)
                });
            }, {
                abortSignal,
                timeoutInMs
            });
        }
        let tokenObject;
        let tokenType;
        if (isSasTokenProvider(this._context.tokenCredential)) {
            tokenObject = this._context.tokenCredential.getToken(this.audience);
            tokenType = TokenType.CbsTokenTypeSas;
            // renew sas token in every 45 minutess
            this._tokenTimeoutInMs = (3600 - 900) * 1000;
        }
        else {
            const aadToken = await this._context.tokenCredential.getToken(Constants.aadEventHubsScope);
            if (!aadToken) {
                throw new Error(`Failed to get token from the provided "TokenCredential" object`);
            }
            tokenObject = aadToken;
            tokenType = TokenType.CbsTokenTypeJwt;
            this._tokenTimeoutInMs = tokenObject.expiresOnTimestamp - Date.now() - 2 * 60 * 1000;
        }
        logger.verbose("[%s] %s: calling negotiateClaim for audience '%s'.", this._context.connectionId, this._type, this.audience);
        // Acquire the lock to negotiate the CBS claim.
        logger.verbose("[%s] Acquiring cbs lock: '%s' for cbs auth for %s: '%s' with address '%s'.", this._context.connectionId, this._context.negotiateClaimLock, this._type, this.name, this.address);
        await defaultCancellableLock.acquire(this._context.negotiateClaimLock, () => {
            return this._context.cbsSession.negotiateClaim(this.audience, tokenObject.token, tokenType, { abortSignal, timeoutInMs: timeoutInMs - (Date.now() - startTime) });
        }, {
            abortSignal,
            timeoutInMs: timeoutInMs - (Date.now() - startTime)
        });
        logger.verbose("[%s] Negotiated claim for %s '%s' with with address: %s", this._context.connectionId, this._type, this.name, this.address);
        if (setTokenRenewal) {
            await this._ensureTokenRenewal();
        }
    }
    /**
     * Ensures that the token is renewed within the predefined renewal margin.
     * @hidden
     */
    _ensureTokenRenewal() {
        if (!this._tokenTimeoutInMs) {
            return;
        }
        // Clear the existing token renewal timer.
        // This scenario can happen if the connection goes down and is brought back up
        // before the `nextRenewalTimeout` was reached.
        if (this._tokenRenewalTimer) {
            clearTimeout(this._tokenRenewalTimer);
        }
        this._tokenRenewalTimer = setTimeout(async () => {
            try {
                await this._negotiateClaim({
                    setTokenRenewal: true,
                    abortSignal: undefined,
                    timeoutInMs: getRetryAttemptTimeoutInMs(undefined)
                });
            }
            catch (err) {
                logger.verbose("[%s] %s '%s' with address %s, an error occurred while renewing the token: %O", this._context.connectionId, this._type, this.name, this.address, err);
            }
        }, this._tokenTimeoutInMs);
        logger.verbose("[%s] %s '%s' with address %s, has next token renewal in %d milliseconds @(%s).", this._context.connectionId, this._type, this.name, this.address, this._tokenTimeoutInMs, new Date(Date.now() + this._tokenTimeoutInMs).toString());
    }
    /**
     * Closes the Sender|Receiver link and it's underlying session and also removes it from the
     * internal map.
     * @hidden
     * @param link - The Sender or Receiver link that needs to be closed and
     * removed.
     */
    async _closeLink(link) {
        clearTimeout(this._tokenRenewalTimer);
        if (link) {
            try {
                // Closing the link and its underlying session if the link is open. This should also
                // remove them from the internal map.
                await link.close();
                logger.verbose("[%s] %s '%s' with address '%s' closed.", this._context.connectionId, this._type, this.name, this.address);
            }
            catch (err) {
                logger.verbose("[%s] An error occurred while closing the %s '%s' with address '%s': %O", this._context.connectionId, this._type, this.name, this.address, err);
            }
        }
    }
    /**
     * Provides the current type of the LinkEntity.
     * @returns The entity type.
     */
    get _type() {
        let result = "LinkEntity";
        if (this.constructor && this.constructor.name) {
            result = this.constructor.name;
        }
        return result;
    }
}
//# sourceMappingURL=linkEntity.js.map