// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { isAmqpAnnotatedMessage, toRheaMessage } from "./eventData";
import { message } from "rhea-promise";
import { throwTypeErrorIfParameterMissing } from "./util/error";
import { TRACEPARENT_PROPERTY, instrumentEventData } from "./diagnostics/instrumentEventData";
import { convertTryAddOptionsForCompatibility, createMessageSpan } from "./diagnostics/tracing";
import { isDefined, isObjectWithProperties } from "./util/typeGuards";
/**
 * The amount of bytes to reserve as overhead for a small message.
 */
const smallMessageOverhead = 5;
/**
 * The amount of bytes to reserve as overhead for a large message.
 */
const largeMessageOverhead = 8;
/**
 * The maximum number of bytes that a message may be to be considered small.
 */
const smallMessageMaxBytes = 255;
/**
 * Checks if the provided eventDataBatch is an instance of `EventDataBatch`.
 * @param eventDataBatch - The instance of `EventDataBatch` to verify.
 * @internal
 */
export function isEventDataBatch(eventDataBatch) {
    return (isObjectWithProperties(eventDataBatch, ["count", "sizeInBytes", "tryAdd"]) &&
        typeof eventDataBatch.tryAdd === "function" &&
        typeof eventDataBatch.count === "number" &&
        typeof eventDataBatch.sizeInBytes === "number");
}
/**
 * An internal class representing a batch of events which can be used to send events to Event Hub.
 *
 * @internal
 */
export class EventDataBatchImpl {
    /**
     * EventDataBatch should not be constructed using `new EventDataBatch()`
     * Use the `createBatch()` method on your `EventHubProducer` instead.
     * @internal
     */
    constructor(context, maxSizeInBytes, partitionKey, partitionId) {
        /**
         * Encoded amqp messages.
         */
        this._encodedMessages = [];
        /**
         * List of 'message' span contexts.
         */
        this._spanContexts = [];
        this._context = context;
        this._maxSizeInBytes = maxSizeInBytes;
        this._partitionKey = isDefined(partitionKey) ? String(partitionKey) : partitionKey;
        this._partitionId = isDefined(partitionId) ? String(partitionId) : partitionId;
        this._sizeInBytes = 0;
        this._count = 0;
    }
    /**
     * The maximum size of the batch, in bytes.
     * @readonly
     */
    get maxSizeInBytes() {
        return this._maxSizeInBytes;
    }
    /**
     * The partitionKey set during `EventDataBatch` creation. This value is hashed to
     * produce a partition assignment when the producer is created without a `partitionId`
     * @readonly
     */
    get partitionKey() {
        return this._partitionKey;
    }
    /**
     * The partitionId set during `EventDataBatch` creation.
     * If this value is set then partitionKey can not be set.
     * @readonly
     */
    get partitionId() {
        return this._partitionId;
    }
    /**
     * Size of the `EventDataBatch` instance after the events added to it have been
     * encoded into a single AMQP message.
     * @readonly
     */
    get sizeInBytes() {
        return this._sizeInBytes;
    }
    /**
     * Number of events in the `EventDataBatch` instance.
     * @readonly
     */
    get count() {
        return this._count;
    }
    /**
     * Gets the "message" span contexts that were created when adding events to the batch.
     * @internal
     */
    get _messageSpanContexts() {
        return this._spanContexts;
    }
    /**
     * Generates an AMQP message that contains the provided encoded events and annotations.
     * @param encodedEvents - The already encoded events to include in the AMQP batch.
     * @param annotations - The message annotations to set on the batch.
     */
    _generateBatch(encodedEvents, annotations) {
        const batchEnvelope = {
            body: message.data_sections(encodedEvents)
        };
        if (annotations) {
            batchEnvelope.message_annotations = annotations;
        }
        return message.encode(batchEnvelope);
    }
    /**
     * Generates the single AMQP message which is the result of encoding all the events
     * added into the `EventDataBatch` instance.
     *
     * This is not meant for the user to use directly.
     *
     * When the `EventDataBatch` instance is passed to the `send()` method on the `EventHubProducer`,
     * this single batched AMQP message is what gets sent over the wire to the service.
     * @readonly
     */
    _generateMessage() {
        return this._generateBatch(this._encodedMessages, this._batchAnnotations);
    }
    /**
     * Tries to add an event data to the batch if permitted by the batch's size limit.
     * **NOTE**: Always remember to check the return value of this method, before calling it again
     * for the next event.
     *
     * @param eventData -  An individual event data object.
     * @returns A boolean value indicating if the event data has been added to the batch or not.
     */
    tryAdd(eventData, options = {}) {
        var _a;
        throwTypeErrorIfParameterMissing(this._context.connectionId, "tryAdd", "eventData", eventData);
        options = convertTryAddOptionsForCompatibility(options);
        // check if the event has already been instrumented
        const previouslyInstrumented = Boolean((_a = (isAmqpAnnotatedMessage(eventData)
            ? eventData.applicationProperties
            : eventData.properties)) === null || _a === void 0 ? void 0 : _a[TRACEPARENT_PROPERTY] // Event Data maps properties to applicationProperties.
        );
        let spanContext;
        if (!previouslyInstrumented) {
            const { span: messageSpan } = createMessageSpan(options, this._context.config);
            eventData = instrumentEventData(eventData, messageSpan);
            spanContext = messageSpan.spanContext();
            messageSpan.end();
        }
        // Convert EventData to RheaMessage.
        const amqpMessage = toRheaMessage(eventData, this._partitionKey);
        const encodedMessage = message.encode(amqpMessage);
        let currentSize = this._sizeInBytes;
        // The first time an event is added, we need to calculate
        // the overhead of creating an AMQP batch, including the
        // message_annotations that are taken from the 1st message.
        if (this.count === 0) {
            if (amqpMessage.message_annotations) {
                this._batchAnnotations = amqpMessage.message_annotations;
            }
            // Figure out the overhead of creating a batch by generating an empty batch
            // with the expected batch annotations.
            currentSize += this._generateBatch([], this._batchAnnotations).length;
        }
        const messageSize = encodedMessage.length;
        const messageOverhead = messageSize <= smallMessageMaxBytes ? smallMessageOverhead : largeMessageOverhead;
        currentSize += messageSize + messageOverhead;
        // Check if the size of the batch exceeds the maximum allowed size
        // once we add the new event to it.
        if (currentSize > this._maxSizeInBytes) {
            return false;
        }
        // The event will fit in the batch, so it is now safe to store it.
        this._encodedMessages.push(encodedMessage);
        if (spanContext) {
            this._spanContexts.push(spanContext);
        }
        this._sizeInBytes = currentSize;
        this._count++;
        return true;
    }
}
//# sourceMappingURL=eventDataBatch.js.map