// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { SpanStatusCode, SpanKind } from "@azure/core-tracing";
import { createConnectionContext } from "./connectionContext";
import { instrumentEventData, TRACEPARENT_PROPERTY } from "./diagnostics/instrumentEventData";
import { createMessageSpan } from "./diagnostics/tracing";
import { EventDataBatchImpl, isEventDataBatch } from "./eventDataBatch";
import { EventHubSender } from "./eventHubSender";
import { logErrorStackTrace, logger } from "./log";
import { throwErrorIfConnectionClosed, throwTypeErrorIfParameterMissing } from "./util/error";
import { isCredential, isDefined } from "./util/typeGuards";
import { createEventHubSpan } from "./diagnostics/tracing";
/**
 * The `EventHubProducerClient` class is used to send events to an Event Hub.
 *
 * There are multiple ways to create an `EventHubProducerClient`
 * - Use the connection string from the SAS policy created for your Event Hub instance.
 * - Use the connection string from the SAS policy created for your Event Hub namespace,
 * and the name of the Event Hub instance
 * - Use the full namespace like `<yournamespace>.servicebus.windows.net`, and a credentials object.
 *
 * Optionally, you can also pass an options bag to configure the retry policy or proxy settings.
 *
 */
export class EventHubProducerClient {
    constructor(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4 // eslint-disable-line @azure/azure-sdk/ts-naming-options
    ) {
        this._context = createConnectionContext(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4);
        if (typeof eventHubNameOrOptions2 !== "string") {
            this._clientOptions = eventHubNameOrOptions2 || {};
        }
        else if (!isCredential(credentialOrOptions3)) {
            this._clientOptions = credentialOrOptions3 || {};
        }
        else {
            this._clientOptions = options4 || {};
        }
        this._sendersMap = new Map();
    }
    /**
     * @readonly
     * The name of the Event Hub instance for which this client is created.
     */
    get eventHubName() {
        return this._context.config.entityPath;
    }
    /**
     * @readonly
     * The fully qualified namespace of the Event Hub instance for which this client is created.
     * This is likely to be similar to <yournamespace>.servicebus.windows.net.
     */
    get fullyQualifiedNamespace() {
        return this._context.config.host;
    }
    /**
     * Creates an instance of `EventDataBatch` to which one can add events until the maximum supported size is reached.
     * The batch can be passed to the {@link sendBatch} method of the `EventHubProducerClient` to be sent to Azure Event Hubs.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubProducerClient(connectionString);
     * let batch = await client.createBatch();
     * for (let i = 0; i < messages.length; i++) {
     *  if (!batch.tryAdd(messages[i])) {
     *    await client.sendBatch(batch);
     *    batch = await client.createBatch();
     *    if (!batch.tryAdd(messages[i])) {
     *      throw new Error("Message too big to fit")
     *    }
     *    if (i === messages.length - 1) {
     *      await client.sendBatch(batch);
     *    }
     *   }
     * }
     * ```
     *
     * @param options -  Configures the behavior of the batch.
     * - `partitionKey`  : A value that is hashed and used by the Azure Event Hubs service to determine the partition to which
     * the events need to be sent.
     * - `partitionId`   : Id of the partition to which the batch of events need to be sent.
     * - `maxSizeInBytes`: The upper limit for the size of batch. The `tryAdd` function will return `false` after this limit is reached.
     * - `abortSignal`   : A signal the request to cancel the operation.
     * @returns Promise<EventDataBatch>
     * @throws Error if both `partitionId` and `partitionKey` are set in the options.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal in the options.
     */
    async createBatch(options = {}) {
        throwErrorIfConnectionClosed(this._context);
        if (isDefined(options.partitionId) && isDefined(options.partitionKey)) {
            throw new Error("partitionId and partitionKey cannot both be set when creating a batch");
        }
        let sender = this._sendersMap.get("");
        if (!sender) {
            sender = EventHubSender.create(this._context);
            this._sendersMap.set("", sender);
        }
        let maxMessageSize = await sender.getMaxMessageSize({
            retryOptions: this._clientOptions.retryOptions,
            abortSignal: options.abortSignal
        });
        if (options.maxSizeInBytes) {
            if (options.maxSizeInBytes > maxMessageSize) {
                const error = new Error(`Max message size (${options.maxSizeInBytes} bytes) is greater than maximum message size (${maxMessageSize} bytes) on the AMQP sender link.`);
                logger.warning(`[${this._context.connectionId}] ${error.message}`);
                logErrorStackTrace(error);
                throw error;
            }
            maxMessageSize = options.maxSizeInBytes;
        }
        return new EventDataBatchImpl(this._context, maxMessageSize, options.partitionKey, options.partitionId);
    }
    async sendBatch(batch, options = {}) {
        throwErrorIfConnectionClosed(this._context);
        throwTypeErrorIfParameterMissing(this._context.connectionId, "sendBatch", "batch", batch);
        let partitionId;
        let partitionKey;
        // link message span contexts
        let spanContextsToLink = [];
        if (isEventDataBatch(batch)) {
            // For batches, partitionId and partitionKey would be set on the batch.
            partitionId = batch.partitionId;
            partitionKey = batch.partitionKey;
            const unexpectedOptions = options;
            if (unexpectedOptions.partitionKey && partitionKey !== unexpectedOptions.partitionKey) {
                throw new Error(`The partitionKey (${unexpectedOptions.partitionKey}) set on sendBatch does not match the partitionKey (${partitionKey}) set when creating the batch.`);
            }
            if (unexpectedOptions.partitionId && unexpectedOptions.partitionId !== partitionId) {
                throw new Error(`The partitionId (${unexpectedOptions.partitionId}) set on sendBatch does not match the partitionId (${partitionId}) set when creating the batch.`);
            }
            spanContextsToLink = batch._messageSpanContexts;
        }
        else {
            if (!Array.isArray(batch)) {
                batch = [batch];
            }
            // For arrays of events, partitionId and partitionKey would be set in the options.
            const expectedOptions = options;
            partitionId = expectedOptions.partitionId;
            partitionKey = expectedOptions.partitionKey;
            for (let i = 0; i < batch.length; i++) {
                const event = batch[i];
                if (!event.properties || !event.properties[TRACEPARENT_PROPERTY]) {
                    const { span: messageSpan } = createMessageSpan(options, this._context.config);
                    // since these message spans are created from same context as the send span,
                    // these message spans don't need to be linked.
                    // replace the original event with the instrumented one
                    batch[i] = instrumentEventData(batch[i], messageSpan);
                    messageSpan.end();
                }
            }
        }
        if (isDefined(partitionId) && isDefined(partitionKey)) {
            throw new Error(`The partitionId (${partitionId}) and partitionKey (${partitionKey}) cannot both be specified.`);
        }
        if (isDefined(partitionId)) {
            partitionId = String(partitionId);
        }
        if (isDefined(partitionKey)) {
            partitionKey = String(partitionKey);
        }
        let sender = this._sendersMap.get(partitionId || "");
        if (!sender) {
            sender = EventHubSender.create(this._context, partitionId);
            this._sendersMap.set(partitionId || "", sender);
        }
        const sendSpan = this._createSendSpan(options, spanContextsToLink);
        try {
            const result = await sender.send(batch, Object.assign(Object.assign({}, options), { partitionId,
                partitionKey, retryOptions: this._clientOptions.retryOptions }));
            sendSpan.setStatus({ code: SpanStatusCode.OK });
            return result;
        }
        catch (error) {
            sendSpan.setStatus({
                code: SpanStatusCode.ERROR,
                message: error.message
            });
            throw error;
        }
        finally {
            sendSpan.end();
        }
    }
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    async close() {
        await this._context.close();
        for (const pair of this._sendersMap) {
            await pair[1].close();
        }
        this._sendersMap.clear();
    }
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options = {}) {
        return this._context.managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options = {}) {
        return this._context
            .managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }))
            .then((eventHubProperties) => {
            return eventHubProperties.partitionIds;
        });
    }
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId, options = {}) {
        return this._context.managementSession.getPartitionProperties(partitionId, Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
    _createSendSpan(operationOptions, spanContextsToLink = []) {
        const links = spanContextsToLink.map((context) => {
            return {
                context
            };
        });
        const { span } = createEventHubSpan("send", operationOptions, this._context.config, {
            kind: SpanKind.CLIENT,
            links
        });
        return span;
    }
}
//# sourceMappingURL=eventHubProducerClient.js.map