// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { v4 as uuid } from "uuid";
import { Constants, RequestResponseLink, RetryOperationType, defaultCancellableLock, isSasTokenProvider, retry, translate } from "@azure/core-amqp";
import { ReceiverEvents, SenderEvents, generate_uuid } from "rhea-promise";
import { LinkEntity } from "./linkEntity";
import { logErrorStackTrace, logger } from "./log";
import { getRetryAttemptTimeoutInMs } from "./util/retries";
import { throwErrorIfConnectionClosed, throwTypeErrorIfParameterMissing } from "./util/error";
import { SpanStatusCode } from "@azure/core-tracing";
import { createEventHubSpan } from "./diagnostics/tracing";
/**
 * @internal
 * Descibes the EventHubs Management Client that talks
 * to the $management endpoint over AMQP connection.
 */
export class ManagementClient extends LinkEntity {
    /**
     * Instantiates the management client.
     * @hidden
     * @param context - The connection context.
     * @param address - The address for the management endpoint. For IotHub it will be
     * `/messages/events/$management`.
     */
    constructor(context, options) {
        super(context, {
            address: options && options.address ? options.address : Constants.management,
            audience: options && options.audience ? options.audience : context.config.getManagementAudience()
        });
        this.managementLock = `${Constants.managementRequestKey}-${uuid()}`;
        /**
         * The reply to Guid for the management client.
         */
        this.replyTo = uuid();
        this._context = context;
        this.entityPath = context.config.entityPath;
    }
    /**
     * Gets the security token for the management application properties.
     * @internal
     */
    async getSecurityToken() {
        if (isSasTokenProvider(this._context.tokenCredential)) {
            // the security_token has the $management address removed from the end of the audience
            // expected audience: sb://fully.qualified.namespace/event-hub-name/$management
            const audienceParts = this.audience.split("/");
            // for management links, address should be '$management'
            if (audienceParts[audienceParts.length - 1] === this.address) {
                audienceParts.pop();
            }
            const audience = audienceParts.join("/");
            return this._context.tokenCredential.getToken(audience);
        }
        // aad credentials use the aad scope
        return this._context.tokenCredential.getToken(Constants.aadEventHubsScope);
    }
    /**
     * Provides the eventhub runtime information.
     * @hidden
     */
    async getEventHubProperties(options = {}) {
        throwErrorIfConnectionClosed(this._context);
        const { span: clientSpan } = createEventHubSpan("getEventHubProperties", options, this._context.config);
        try {
            const securityToken = await this.getSecurityToken();
            const request = {
                body: Buffer.from(JSON.stringify([])),
                message_id: uuid(),
                reply_to: this.replyTo,
                application_properties: {
                    operation: Constants.readOperation,
                    name: this.entityPath,
                    type: `${Constants.vendorString}:${Constants.eventHub}`,
                    security_token: securityToken === null || securityToken === void 0 ? void 0 : securityToken.token
                }
            };
            const info = await this._makeManagementRequest(request, Object.assign(Object.assign({}, options), { requestName: "getHubRuntimeInformation" }));
            const runtimeInfo = {
                name: info.name,
                createdOn: new Date(info.created_at),
                partitionIds: info.partition_ids
            };
            logger.verbose("[%s] The hub runtime info is: %O", this._context.connectionId, runtimeInfo);
            clientSpan.setStatus({ code: SpanStatusCode.OK });
            return runtimeInfo;
        }
        catch (error) {
            clientSpan.setStatus({
                code: SpanStatusCode.ERROR,
                message: error.message
            });
            logger.warning(`An error occurred while getting the hub runtime information: ${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
            logErrorStackTrace(error);
            throw error;
        }
        finally {
            clientSpan.end();
        }
    }
    /**
     * Provides information about the specified partition.
     * @hidden
     * @param partitionId - Partition ID for which partition information is required.
     */
    async getPartitionProperties(partitionId, options = {}) {
        throwErrorIfConnectionClosed(this._context);
        throwTypeErrorIfParameterMissing(this._context.connectionId, "getPartitionProperties", "partitionId", partitionId);
        partitionId = String(partitionId);
        const { span: clientSpan } = createEventHubSpan("getPartitionProperties", options, this._context.config);
        try {
            const securityToken = await this.getSecurityToken();
            const request = {
                body: Buffer.from(JSON.stringify([])),
                message_id: uuid(),
                reply_to: this.replyTo,
                application_properties: {
                    operation: Constants.readOperation,
                    name: this.entityPath,
                    type: `${Constants.vendorString}:${Constants.partition}`,
                    partition: `${partitionId}`,
                    security_token: securityToken === null || securityToken === void 0 ? void 0 : securityToken.token
                }
            };
            const info = await this._makeManagementRequest(request, Object.assign(Object.assign({}, options), { requestName: "getPartitionInformation" }));
            const partitionInfo = {
                beginningSequenceNumber: info.begin_sequence_number,
                eventHubName: info.name,
                lastEnqueuedOffset: info.last_enqueued_offset,
                lastEnqueuedOnUtc: new Date(info.last_enqueued_time_utc),
                lastEnqueuedSequenceNumber: info.last_enqueued_sequence_number,
                partitionId: info.partition,
                isEmpty: info.is_partition_empty
            };
            logger.verbose("[%s] The partition info is: %O.", this._context.connectionId, partitionInfo);
            clientSpan.setStatus({ code: SpanStatusCode.OK });
            return partitionInfo;
        }
        catch (error) {
            clientSpan.setStatus({
                code: SpanStatusCode.ERROR,
                message: error.message
            });
            logger.warning(`An error occurred while getting the partition information: ${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
            logErrorStackTrace(error);
            throw error;
        }
        finally {
            clientSpan.end();
        }
    }
    /**
     * Closes the AMQP management session to the Event Hub for this client,
     * returning a promise that will be resolved when disconnection is completed.
     * @hidden
     */
    async close() {
        try {
            // Always clear the timeout, as the isOpen check may report
            // false without ever having cleared the timeout otherwise.
            clearTimeout(this._tokenRenewalTimer);
            if (this._isMgmtRequestResponseLinkOpen()) {
                const mgmtLink = this._mgmtReqResLink;
                this._mgmtReqResLink = undefined;
                await mgmtLink.close();
                logger.info("Successfully closed the management session.");
            }
        }
        catch (err) {
            const msg = `An error occurred while closing the management session: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`;
            logger.warning(msg);
            logErrorStackTrace(err);
            throw new Error(msg);
        }
    }
    async _init({ abortSignal, timeoutInMs }) {
        try {
            if (!this._isMgmtRequestResponseLinkOpen()) {
                // Wait for the connectionContext to be ready to open the link.
                await this._context.readyToOpenLink();
                await this._negotiateClaim({ setTokenRenewal: false, abortSignal, timeoutInMs });
                const rxopt = {
                    source: { address: this.address },
                    name: this.replyTo,
                    target: { address: this.replyTo },
                    onSessionError: (context) => {
                        const id = context.connection.options.id;
                        const ehError = translate(context.session.error);
                        logger.verbose("[%s] An error occurred on the session for request/response links for " +
                            "$management: %O", id, ehError);
                    }
                };
                const sropt = {
                    target: { address: this.address }
                };
                logger.verbose("[%s] Creating sender/receiver links on a session for $management endpoint with " +
                    "srOpts: %o, receiverOpts: %O.", this._context.connectionId, sropt, rxopt);
                this._mgmtReqResLink = await RequestResponseLink.create(this._context.connection, sropt, rxopt, { abortSignal });
                this._mgmtReqResLink.sender.on(SenderEvents.senderError, (context) => {
                    const id = context.connection.options.id;
                    const ehError = translate(context.sender.error);
                    logger.verbose("[%s] An error occurred on the $management sender link.. %O", id, ehError);
                });
                this._mgmtReqResLink.receiver.on(ReceiverEvents.receiverError, (context) => {
                    const id = context.connection.options.id;
                    const ehError = translate(context.receiver.error);
                    logger.verbose("[%s] An error occurred on the $management receiver link.. %O", id, ehError);
                });
                logger.verbose("[%s] Created sender '%s' and receiver '%s' links for $management endpoint.", this._context.connectionId, this._mgmtReqResLink.sender.name, this._mgmtReqResLink.receiver.name);
                await this._ensureTokenRenewal();
            }
        }
        catch (err) {
            const translatedError = translate(err);
            logger.warning(`[${this._context.connectionId}] An error occured while establishing the $management links: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Helper method to make the management request
     * @param request - The AMQP message to send
     * @param options - The options to use when sending a request over a $management link
     */
    async _makeManagementRequest(request, options = {}) {
        const retryOptions = options.retryOptions || {};
        try {
            const abortSignal = options && options.abortSignal;
            const sendOperationPromise = async () => {
                let count = 0;
                const retryTimeoutInMs = getRetryAttemptTimeoutInMs(options.retryOptions);
                let timeTakenByInit = 0;
                if (!this._isMgmtRequestResponseLinkOpen()) {
                    logger.verbose("[%s] Acquiring lock to get the management req res link.", this._context.connectionId);
                    const initOperationStartTime = Date.now();
                    try {
                        await defaultCancellableLock.acquire(this.managementLock, () => {
                            const acquireLockEndTime = Date.now();
                            const timeoutInMs = retryTimeoutInMs - (acquireLockEndTime - initOperationStartTime);
                            return this._init({ abortSignal, timeoutInMs });
                        }, { abortSignal, timeoutInMs: retryTimeoutInMs });
                    }
                    catch (err) {
                        const translatedError = translate(err);
                        logger.warning("[%s] An error occurred while creating the management link %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
                        logErrorStackTrace(translatedError);
                        throw translatedError;
                    }
                    timeTakenByInit = Date.now() - initOperationStartTime;
                }
                const remainingOperationTimeoutInMs = retryTimeoutInMs - timeTakenByInit;
                const sendRequestOptions = {
                    abortSignal: options.abortSignal,
                    requestName: options.requestName,
                    timeoutInMs: remainingOperationTimeoutInMs
                };
                count++;
                if (count !== 1) {
                    // Generate a new message_id every time after the first attempt
                    request.message_id = generate_uuid();
                }
                else if (!request.message_id) {
                    // Set the message_id in the first attempt only if it is not set
                    request.message_id = generate_uuid();
                }
                return this._mgmtReqResLink.sendRequest(request, sendRequestOptions);
            };
            const config = Object.defineProperties({
                operation: sendOperationPromise,
                operationType: RetryOperationType.management,
                abortSignal: abortSignal,
                retryOptions: retryOptions
            }, {
                connectionId: {
                    enumerable: true,
                    get: () => {
                        return this._context.connectionId;
                    }
                }
            });
            return (await retry(config)).body;
        }
        catch (err) {
            const translatedError = translate(err);
            logger.warning("[%s] An error occurred during send on management request-response link with address '%s': %s", this._context.connectionId, this.address, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    _isMgmtRequestResponseLinkOpen() {
        return this._mgmtReqResLink && this._mgmtReqResLink.isOpen();
    }
}
//# sourceMappingURL=managementClient.js.map