'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var abortController = require('@azure/abort-controller');
var logger$1 = require('@azure/logger');
var coreAuth = require('@azure/core-auth');
var coreAmqp = require('@azure/core-amqp');
var rheaPromise = require('rhea-promise');
var buffer = require('buffer');
var isBuffer = require('is-buffer');
var coreTracing = require('@azure/core-tracing');
var uuid = require('uuid');
var os = require('os');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

function _interopNamespace(e) {
    if (e && e.__esModule) return e;
    var n = Object.create(null);
    if (e) {
        Object.keys(e).forEach(function (k) {
            if (k !== 'default') {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () { return e[k]; }
                });
            }
        });
    }
    n["default"] = e;
    return Object.freeze(n);
}

var isBuffer__default = /*#__PURE__*/_interopDefaultLegacy(isBuffer);
var os__namespace = /*#__PURE__*/_interopNamespace(os);

// Copyright (c) Microsoft Corporation.
/**
 * Helper TypeGuard that checks if something is defined or not.
 * @param thing - Anything
 * @internal
 */
function isDefined(thing) {
    return typeof thing !== "undefined" && thing !== null;
}
/**
 * Helper TypeGuard that checks if the input is an object with the specified properties.
 * @param thing - Anything.
 * @param properties - The name of the properties that should appear in the object.
 * @internal
 */
function isObjectWithProperties(thing, properties) {
    if (!isDefined(thing) || typeof thing !== "object") {
        return false;
    }
    for (const property of properties) {
        if (!objectHasProperty(thing, property)) {
            return false;
        }
    }
    return true;
}
/**
 * Helper TypeGuard that checks if the input is an object with the specified property.
 * @param thing - Any object.
 * @param property - The name of the property that should appear in the object.
 * @internal
 */
function objectHasProperty(thing, property) {
    return typeof thing === "object" && property in thing;
}
/**
 * Typeguard that checks if the input is a credential type the clients accept.
 * @param thing - Any object.
 * @internal
 */
function isCredential(thing) {
    return coreAuth.isTokenCredential(thing) || coreAuth.isNamedKeyCredential(thing) || coreAuth.isSASCredential(thing);
}

// Copyright (c) Microsoft Corporation.
/**
 * The `@azure/logger` configuration for this package.
 * This will output logs using the `azure:event-hubs` namespace prefix.
 */
const logger = logger$1.createClientLogger("event-hubs");
/**
 * Logs the error's stack trace to "verbose" if a stack trace is available.
 * @param error - Error containing a stack trace.
 * @internal
 */
function logErrorStackTrace(error) {
    if (isObjectWithProperties(error, ["stack"]) && error.stack) {
        logger.verbose(error.stack);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The `PartitionProcessor` is responsible for processing events received from Event Hubs when using `EventProcessor`
 *
 * The EventProcessor creates a new instance of the PartitionProcessor for each partition of the event hub it starts processing. When you extend the `PartitionProcessor` in order to customize it as you see fit,
 * - Override the `processEvents()` method to add the code to process the received events. This is also a good place to update the checkpoints using the `updateCheckpoint()` method
 * - Optionally override the `processError()` method to handle any error that might have occurred when processing the events.
 * - Optionally override the `initialize()` method to implement any set up related tasks you would want to carry out before starting to receive events from the partition
 * - Optionally override the `close()` method to implement any tear down or clean up tasks you would want to carry out.
 * @internal
 */
class PartitionProcessor {
    constructor(_eventHandlers, _checkpointStore, _context) {
        this._eventHandlers = _eventHandlers;
        this._checkpointStore = _checkpointStore;
        this._context = _context;
    }
    /**
     * Information on the last enqueued event in the partition that is being processed.
     * This property is updated by the `EventProcessor` if the `trackLastEnqueuedEventProperties` option is set to true
     * when creating an instance of EventProcessor
     * @readonly
     */
    get lastEnqueuedEventProperties() {
        return this._lastEnqueuedEventProperties;
    }
    /**
     * Information on the last enqueued event in the partition that is being processed.
     * This property is updated by the `EventProcessor` if the `trackLastEnqueuedEventProperties` option is set to true
     * when creating an instance of EventProcessor
     */
    set lastEnqueuedEventProperties(properties) {
        this._lastEnqueuedEventProperties = properties;
    }
    /**
     * The fully qualified namespace from where the current partition is being processed. It is set by the `EventProcessor`
     * @readonly
     */
    get fullyQualifiedNamespace() {
        return this._context.fullyQualifiedNamespace;
    }
    /**
     * The name of the consumer group from where the current partition is being processed. It is set by the `EventProcessor`
     * @readonly
     */
    get consumerGroup() {
        return this._context.consumerGroup;
    }
    /**
     * The name of the event hub to which the current partition belongs. It is set by the `EventProcessor`
     * @readonly
     */
    get eventHubName() {
        return this._context.eventHubName;
    }
    /**
     * The identifier of the Event Hub partition that is being processed. It is set by the `EventProcessor`
     * @readonly
     */
    get partitionId() {
        return this._context.partitionId;
    }
    /**
     * The unique identifier of the `EventProcessor` that has spawned the current instance of `PartitionProcessor`. This is set by the `EventProcessor`
     */
    get eventProcessorId() {
        return this._context.eventProcessorId;
    }
    /**
     * This method is called when the `EventProcessor` takes ownership of a new partition and before any
     * events are received.
     */
    async initialize() {
        if (this._eventHandlers.processInitialize) {
            await this._eventHandlers.processInitialize(this);
        }
    }
    /**
     * This method is called before the partition processor is closed by the EventProcessor.
     *
     * @param reason - The reason for closing this partition processor.
     */
    async close(reason) {
        if (this._eventHandlers.processClose) {
            await this._eventHandlers.processClose(reason, this);
        }
    }
    /**
     * This method is called when new events are received.
     *
     * This is also a good place to update checkpoints as appropriate.
     *
     * @param event - The received events to be processed.
     */
    async processEvents(events) {
        await this._eventHandlers.processEvents(events, this);
    }
    /**
     * This method is called when an error occurs while receiving events from Event Hubs.
     *
     * @param error - The error to be processed.
     */
    async processError(error) {
        if (this._eventHandlers.processError) {
            try {
                await this._eventHandlers.processError(error, this);
            }
            catch (err) {
                logger.verbose(`Error thrown from user's processError handler : ${err}`);
            }
        }
    }
    /**
     * Updates the checkpoint using the event data.
     *
     * A checkpoint is meant to represent the last successfully processed event by the user from a particular
     * partition of a consumer group in an Event Hub instance.
     *
     * @param eventData - The event that you want to update the checkpoint with.
     */
    async updateCheckpoint(eventData) {
        const checkpoint = {
            fullyQualifiedNamespace: this._context.fullyQualifiedNamespace,
            eventHubName: this._context.eventHubName,
            consumerGroup: this._context.consumerGroup,
            partitionId: this._context.partitionId,
            sequenceNumber: eventData.sequenceNumber,
            offset: eventData.offset,
        };
        await this._checkpointStore.updateCheckpoint(checkpoint);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 * Gets the expression to be set as the filter clause when creating the receiver
 * @returns filterExpression
 */
function getEventPositionFilter(eventPosition) {
    let result;
    // order of preference
    if (isDefined(eventPosition.offset)) {
        result = eventPosition.isInclusive
            ? `${coreAmqp.Constants.offsetAnnotation} >= '${eventPosition.offset}'`
            : `${coreAmqp.Constants.offsetAnnotation} > '${eventPosition.offset}'`;
    }
    else if (isDefined(eventPosition.sequenceNumber)) {
        result = eventPosition.isInclusive
            ? `${coreAmqp.Constants.sequenceNumberAnnotation} >= '${eventPosition.sequenceNumber}'`
            : `${coreAmqp.Constants.sequenceNumberAnnotation} > '${eventPosition.sequenceNumber}'`;
    }
    else if (isDefined(eventPosition.enqueuedOn)) {
        const time = eventPosition.enqueuedOn instanceof Date
            ? eventPosition.enqueuedOn.getTime()
            : eventPosition.enqueuedOn;
        result = `${coreAmqp.Constants.enqueuedTimeAnnotation} > '${time}'`;
    }
    if (!result) {
        throw coreAmqp.translate({
            condition: coreAmqp.ErrorNameConditionMapper.ArgumentError,
            description: "No starting position was set in the EventPosition.",
        });
    }
    return result;
}
/**
 * Gets the `EventPosition` corresponding to the location of the the first event present in the partition.
 * Pass this position to the `EventHubConsumerClient.subscribe()` method to begin receiving events from the
 * first event in the partition which has not expired due to the retention policy.
 */
const earliestEventPosition = {
    offset: -1,
};
/**
 * Gets the `EventPosition` corresponding to the end of the partition.
 * Pass this position to the `EventHubConsumerClient.subscribe()` method to begin receiving events from the
 * event that is enqueued right after the method call.
 * @returns EventPosition
 */
const latestEventPosition = {
    offset: "@latest",
};
/**
 * @internal
 */
function validateEventPositions(position) {
    if (!isDefined(position)) {
        return;
    }
    const keys = Object.keys(position);
    if (!keys.length) {
        throw new TypeError("Invalid value for EventPosition found. Pass an object with either of offset, sequenceNumber or enqueuedOn properties set.");
    }
    if (isEventPosition(position)) {
        validateEventPosition(position);
        return;
    }
    const positions = position;
    for (let i = 0; i < keys.length; i++) {
        if (Object.prototype.hasOwnProperty.call(positions, keys[i])) {
            validateEventPosition(positions[keys[i]]);
        }
    }
}
/**
 * Determines whether a position is an EventPosition.
 * Does not validate that the position is allowed.
 * @internal
 */
function isEventPosition(position) {
    if (!position) {
        return false;
    }
    if (objectHasProperty(position, "offset") && isDefined(position.offset)) {
        return true;
    }
    if (objectHasProperty(position, "sequenceNumber") && isDefined(position.sequenceNumber)) {
        return true;
    }
    if (objectHasProperty(position, "enqueuedOn") && isDefined(position.enqueuedOn)) {
        return true;
    }
    return false;
}
function validateEventPosition(position) {
    if (!isDefined(position)) {
        return;
    }
    const offsetPresent = isDefined(position.offset);
    const sequenceNumberPresent = isDefined(position.sequenceNumber);
    const enqueuedOnPresent = isDefined(position.enqueuedOn);
    if ((offsetPresent && sequenceNumberPresent) ||
        (offsetPresent && enqueuedOnPresent) ||
        (enqueuedOnPresent && sequenceNumberPresent)) {
        throw new TypeError("Invalid value for EventPosition found. Set only one of offset, sequenceNumber or enqueuedOn properties.");
    }
    if (!offsetPresent && !enqueuedOnPresent && !sequenceNumberPresent) {
        throw new TypeError("Invalid value for EventPosition found. Pass an object with either of offset, sequenceNumber or enqueuedOn properties set.");
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * An enum representing the different reasons for an `EventHubConsumerClient` to stop processing
 * events from a partition in a consumer group of an Event Hub.
 */
exports.CloseReason = void 0;
(function (CloseReason) {
    /**
     * Ownership of the partition was lost or transitioned to a new processor instance.
     */
    CloseReason["OwnershipLost"] = "OwnershipLost";
    /**
     * The EventProcessor was shutdown.
     */
    CloseReason["Shutdown"] = "Shutdown";
})(exports.CloseReason || (exports.CloseReason = {}));

// Copyright (c) Microsoft Corporation.
/** @internal */
const dataSectionTypeCode = 0x75;
/** @internal */
const sequenceSectionTypeCode = 0x76;
/** @internal */
const valueSectionTypeCode = 0x77;
/**
 * The default data transformer that will be used by the Azure SDK.
 * @internal
 */
const defaultDataTransformer = {
    /**
     * A function that takes the body property from an EventData object
     * and returns an encoded body (some form of AMQP type).
     *
     * @param body - The AMQP message body
     * @param bodyType - The AMQP section to story the body in.
     * @returns The encoded AMQP message body as an AMQP Data/Sequence/Value section.
     */
    encode(body, bodyType) {
        let result;
        // string, undefined, null, boolean, array, object, number should end up here
        // coercing undefined to null as that will ensure that null value will be given to the
        // customer on receive.
        if (body === undefined)
            body = null;
        if (bodyType === "value") {
            // TODO: Expose value_section from `rhea` similar to the data_section and sequence_section.
            // Right now there isn't a way to create a value section officially.
            result = rheaPromise.message.data_section(body);
            result.typecode = valueSectionTypeCode;
        }
        else if (bodyType === "sequence") {
            result = rheaPromise.message.sequence_section(body);
        }
        else if (isBuffer__default["default"](body) || body instanceof Uint8Array) {
            result = rheaPromise.message.data_section(body);
        }
        else if (body === null && bodyType === "data") {
            result = rheaPromise.message.data_section(null);
        }
        else {
            try {
                const bodyStr = JSON.stringify(body);
                result = rheaPromise.message.data_section(buffer.Buffer.from(bodyStr, "utf8"));
            }
            catch (err) {
                const msg = `An error occurred while executing JSON.stringify() on the given body ` +
                    body +
                    `${err ? err.stack : JSON.stringify(err)}`;
                logger.warning("[encode] " + msg);
                logErrorStackTrace(err);
                throw new Error(msg);
            }
        }
        return result;
    },
    /**
     * A function that takes the body property from an AMQP message, which can come from either
     * the 'data', 'value' or 'sequence' sections of an AMQP message.
     *
     * If the body is not a JSON string the the raw contents will be returned, along with the bodyType
     * indicating which part of the AMQP message the body was decoded from.
     *
     * @param body - The AMQP message body as received from rhea.
     * @param skipParsingBodyAsJson - Boolean to skip running JSON.parse() on message body when body type is `content`.
     * @returns The decoded/raw body and the body type.
     */
    decode(body, skipParsingBodyAsJson) {
        try {
            if (isRheaAmqpSection(body)) {
                switch (body.typecode) {
                    case dataSectionTypeCode:
                        return {
                            body: skipParsingBodyAsJson ? body.content : tryToJsonDecode(body.content),
                            bodyType: "data",
                        };
                    case sequenceSectionTypeCode:
                        return { body: body.content, bodyType: "sequence" };
                    case valueSectionTypeCode:
                        return { body: body.content, bodyType: "value" };
                }
            }
            else {
                if (isBuffer__default["default"](body)) {
                    return { body: skipParsingBodyAsJson ? body : tryToJsonDecode(body), bodyType: "data" };
                }
                return { body, bodyType: "value" };
            }
        }
        catch (err) {
            logger.verbose("[decode] An error occurred while decoding the received message body. The error is: %O", err);
            throw err;
        }
    },
};
/**
 * Attempts to decode 'body' as a JSON string. If it fails it returns body
 * verbatim.
 *
 * @param body - An AMQP message body.
 * @returns A JSON decoded object, or body if body was not a JSON string.
 *
 * @internal
 */
function tryToJsonDecode(body) {
    let processedBody = body;
    try {
        // Trying to stringify and JSON.parse() anything else will fail flat and we shall return
        // the original type back
        const bodyStr = processedBody.toString("utf8");
        processedBody = JSON.parse(bodyStr);
    }
    catch (err) {
        logger.verbose("[decode] An error occurred while trying JSON.parse() on the received body. The error is %O", err);
    }
    return processedBody;
}
/** @internal */
function isRheaAmqpSection(possibleSection) {
    return (possibleSection != null &&
        typeof possibleSection.typecode === "number" &&
        (possibleSection.typecode === dataSectionTypeCode ||
            possibleSection.typecode === valueSectionTypeCode ||
            possibleSection.typecode === sequenceSectionTypeCode));
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * @internal
 */
const packageJsonInfo = {
    name: "@azure/event-hubs",
    version: "5.8.0",
};
/**
 * @internal
 */
const idempotentProducerAmqpPropertyNames = {
    capability: "com.microsoft:idempotent-producer",
    epoch: "com.microsoft:producer-epoch",
    producerId: "com.microsoft:producer-id",
    producerSequenceNumber: "com.microsoft:producer-sequence-number",
};
/**
 * @internal
 */
const PENDING_PUBLISH_SEQ_NUM_SYMBOL = Symbol.for("@azure/event-hubs.pendingPublishSequenceNumber");

// Copyright (c) Microsoft Corporation.
/**
 * The {@link TracingClient} that is used to add tracing spans.
 */
const tracingClient = coreTracing.createTracingClient({
    namespace: "Microsoft.EventHub",
    packageName: packageJsonInfo.name,
    packageVersion: packageJsonInfo.version,
});
/**
 * Creates {@link TracingSpanOptions} from the provided data.
 * @param eventHubConfig - The configuration object containing initial attributes to set on the span.
 * @param spanKind - The {@link TracingSpanKind} for the newly created span.
 * @returns a {@link TracingSpanOptions} that can be passed to a {@link TracingClient}
 */
function toSpanOptions(eventHubConfig, spanKind) {
    const spanOptions = {
        spanAttributes: {
            "message_bus.destination": eventHubConfig.entityPath,
            "peer.address": eventHubConfig.host,
        },
    };
    if (spanKind) {
        spanOptions.spanKind = spanKind;
    }
    return spanOptions;
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 */
const TRACEPARENT_PROPERTY = "Diagnostic-Id";
/**
 * Populates the `EventData` with `SpanContext` info to support trace propagation.
 * Creates and returns a copy of the passed in `EventData` unless the `EventData`
 * has already been instrumented.
 * @param eventData - The `EventData` or `AmqpAnnotatedMessage` to instrument.
 * @param span - The `Span` containing the context to propagate tracing information.
 */
function instrumentEventData(eventData, options, entityPath, host) {
    var _a, _b;
    const props = isAmqpAnnotatedMessage(eventData)
        ? eventData.applicationProperties
        : eventData.properties;
    // check if the event has already been instrumented
    const previouslyInstrumented = Boolean(props === null || props === void 0 ? void 0 : props[TRACEPARENT_PROPERTY]);
    if (previouslyInstrumented) {
        return { event: eventData, spanContext: undefined };
    }
    const { span: messageSpan, updatedOptions } = tracingClient.startSpan("message", options, toSpanOptions({ entityPath, host }, "producer"));
    try {
        if (!messageSpan.isRecording()) {
            return {
                event: eventData,
                spanContext: undefined,
            };
        }
        const traceParent = tracingClient.createRequestHeaders((_a = updatedOptions.tracingOptions) === null || _a === void 0 ? void 0 : _a.tracingContext)["traceparent"];
        if (traceParent) {
            const copiedProps = Object.assign({}, props);
            // create a copy so the original isn't modified
            if (isAmqpAnnotatedMessage(eventData)) {
                eventData = Object.assign(Object.assign({}, eventData), { applicationProperties: copiedProps });
            }
            else {
                eventData = Object.assign(Object.assign({}, eventData), { properties: copiedProps });
            }
            copiedProps[TRACEPARENT_PROPERTY] = traceParent;
        }
        return {
            event: eventData,
            spanContext: (_b = updatedOptions.tracingOptions) === null || _b === void 0 ? void 0 : _b.tracingContext,
        };
    }
    finally {
        messageSpan.end();
    }
}
/**
 * Extracts the `SpanContext` from an `EventData` if the context exists.
 * @param eventData - An individual `EventData` object.
 * @internal
 */
function extractSpanContextFromEventData(eventData) {
    if (!eventData.properties || !eventData.properties[TRACEPARENT_PROPERTY]) {
        return;
    }
    const diagnosticId = eventData.properties[TRACEPARENT_PROPERTY];
    return tracingClient.parseTraceparentHeader(diagnosticId);
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 * Logs and throws Error if the current AMQP connection is closed.
 * @param context - The ConnectionContext associated with the current AMQP connection.
 */
function throwErrorIfConnectionClosed(context) {
    if (context && context.wasConnectionCloseCalled) {
        const errorMessage = "The underlying AMQP connection is closed.";
        const error = new Error(errorMessage);
        logger.warning(`[${context.connectionId}] ${error.name}: ${error.message}`);
        logErrorStackTrace(error);
        throw error;
    }
}
/**
 * @internal
 * Logs and Throws TypeError if given parameter is undefined or null
 * @param connectionId - Id of the underlying AMQP connection used for logging
 * @param methodName - Name of the method that was passed the parameter
 * @param parameterName - Name of the parameter to check
 * @param parameterValue - Value of the parameter to check
 */
function throwTypeErrorIfParameterMissing(connectionId, methodName, parameterName, parameterValue) {
    if (!isDefined(parameterValue)) {
        const error = new TypeError(`${methodName} called without required argument "${parameterName}"`);
        logger.warning(`[${connectionId}] ${error.name}: ${error.message}`);
        logErrorStackTrace(error);
        throw error;
    }
}
/**
 * Maps the amqp error conditions to the Error names.
 * @internal
 */
var ConditionErrorNameMapper;
(function (ConditionErrorNameMapper) {
    /**
     * Indicates that a sequenc enumber was out of order.
     */
    ConditionErrorNameMapper["com.microsoft:out-of-order-sequence"] = "SequenceOutOfOrderError";
    /**
     * Error is thrown when two or more instances connect to the same partition
     * with different epoch values.
     */
    ConditionErrorNameMapper["com.microsoft:producer-epoch-stolen"] = "ProducerDisconnectedError";
})(ConditionErrorNameMapper || (ConditionErrorNameMapper = {}));
/**
 * @internal
 */
const nonRetryableErrors = new Set([
    "ProducerDisconnectedError",
    "SequenceOutOfOrderError",
]);
/**
 * @internal
 */
function translateError(err) {
    const translatedError = coreAmqp.translate(err);
    // If we're not dealing with a messaging error, or the original error wasn't an AMQP error,
    // or we have a resolved code on the messaging error, just return the translated error.
    if (!coreAmqp.isMessagingError(translatedError) || !rheaPromise.isAmqpError(err) || translatedError.code) {
        return translatedError;
    }
    const amqpError = err;
    const condition = amqpError.condition;
    // If we don't have a condition, we can't map the condition to a code.
    if (!condition) {
        return translatedError;
    }
    // Attempt to resolve codes core-amqp doesn't know about.
    translatedError.code =
        ConditionErrorNameMapper[condition];
    if (translatedError.code) {
        translatedError.retryable = !nonRetryableErrors.has(translatedError.code);
    }
    return translatedError;
}
/**
 * @internal
 */
const idempotentAlreadyPublished = "These events have already been successfully published. When idempotent publishing is enabled, events that were acknowledged by the Event Hubs service may not be published again.";
/**
 * @internal
 */
const idempotentSomeAlreadyPublished = "1 or more of these events have already been successfully published. When idempotent publishing is enabled, events that were acknowledged by the Event Hubs service may not be published again.";
/**
 * @internal
 */
function validateProducerPartitionSettings({ enableIdempotentRetries, partitionId, partitionKey, }) {
    if (enableIdempotentRetries && (isDefined(partitionKey) || !isDefined(partitionId))) {
        throw new Error(`The "partitionId" must be supplied and "partitionKey" must not be provided when the EventHubProducerClient has "enableIdempotentRetries" set to true.`);
    }
    if (isDefined(partitionId) && isDefined(partitionKey)) {
        throw new Error(`The partitionId (${partitionId}) and partitionKey (${partitionKey}) cannot both be specified.`);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The amount of bytes to reserve as overhead for a small message.
 */
const smallMessageOverhead = 5;
/**
 * The amount of bytes to reserve as overhead for a large message.
 */
const largeMessageOverhead = 8;
/**
 * The maximum number of bytes that a message may be to be considered small.
 */
const smallMessageMaxBytes = 255;
/**
 * Checks if the provided eventDataBatch is an instance of `EventDataBatch`.
 * @param eventDataBatch - The instance of `EventDataBatch` to verify.
 * @internal
 */
function isEventDataBatch(eventDataBatch) {
    return (isObjectWithProperties(eventDataBatch, ["count", "sizeInBytes", "tryAdd"]) &&
        typeof eventDataBatch.tryAdd === "function" &&
        typeof eventDataBatch.count === "number" &&
        typeof eventDataBatch.sizeInBytes === "number");
}
/**
 * An internal class representing a batch of events which can be used to send events to Event Hub.
 *
 * @internal
 */
class EventDataBatchImpl {
    /**
     * EventDataBatch should not be constructed using `new EventDataBatch()`
     * Use the `createBatch()` method on your `EventHubProducer` instead.
     * @internal
     */
    constructor(context, maxSizeInBytes, isIdempotent, partitionKey, partitionId) {
        /**
         * Encoded amqp messages.
         */
        this._encodedMessages = [];
        /**
         * List of 'message' span contexts.
         */
        this._spanContexts = [];
        this._context = context;
        this._maxSizeInBytes = maxSizeInBytes;
        this._isIdempotent = isIdempotent;
        this._partitionKey = isDefined(partitionKey) ? String(partitionKey) : partitionKey;
        this._partitionId = isDefined(partitionId) ? String(partitionId) : partitionId;
        this._sizeInBytes = 0;
        this._count = 0;
    }
    /**
     * The maximum size of the batch, in bytes.
     * @readonly
     */
    get maxSizeInBytes() {
        return this._maxSizeInBytes;
    }
    /**
     * The partitionKey set during `EventDataBatch` creation. This value is hashed to
     * produce a partition assignment when the producer is created without a `partitionId`
     * @readonly
     */
    get partitionKey() {
        return this._partitionKey;
    }
    /**
     * The partitionId set during `EventDataBatch` creation.
     * If this value is set then partitionKey can not be set.
     * @readonly
     */
    get partitionId() {
        return this._partitionId;
    }
    /**
     * Size of the `EventDataBatch` instance after the events added to it have been
     * encoded into a single AMQP message.
     * @readonly
     */
    get sizeInBytes() {
        return this._sizeInBytes;
    }
    /**
     * Number of events in the `EventDataBatch` instance.
     * @readonly
     */
    get count() {
        return this._count;
    }
    /**
     * The publishing sequence number assigned to the first event in the batch at the time
     * the batch was successfully published.
     * If the producer was not configured to apply sequence numbering or if the batch
     * has not yet been successfully published, the value will be `undefined`.
     */
    get startingPublishedSequenceNumber() {
        return this._startingPublishSequenceNumber;
    }
    /**
     * Gets the "message" span contexts that were created when adding events to the batch.
     * @internal
     */
    get _messageSpanContexts() {
        return this._spanContexts;
    }
    /**
     * Generates an AMQP message that contains the provided encoded events and annotations.
     * @param encodedEvents - The already encoded events to include in the AMQP batch.
     * @param annotations - The message annotations to set on the batch.
     * @param publishingProps - Idempotent publishing properties used to decorate the events in the batch while sending.
     */
    _generateBatch(encodedEvents, annotations, publishingProps) {
        if (this._isIdempotent && publishingProps) {
            // We need to decode the encoded events, add the idempotent annotations, and re-encode them.
            // We can't lazily encode the events because we rely on `message.encode` to capture the
            // byte length of anything not in `event.body`.
            // Events can't be decorated ahead of time because the publishing properties aren't known
            // until the events are being sent to the service.
            const decodedEvents = encodedEvents.map(rheaPromise.message.decode);
            const decoratedEvents = this._decorateRheaMessagesWithPublishingProps(decodedEvents, publishingProps);
            encodedEvents = decoratedEvents.map(rheaPromise.message.encode);
        }
        const batchEnvelope = {
            body: rheaPromise.message.data_sections(encodedEvents),
        };
        if (annotations) {
            batchEnvelope.message_annotations = annotations;
        }
        return rheaPromise.message.encode(batchEnvelope);
    }
    /**
     * Uses the publishingProps to add idempotent properties as message annotations to rhea messages.
     */
    _decorateRheaMessagesWithPublishingProps(events, publishingProps) {
        if (!this._isIdempotent) {
            return events;
        }
        const { lastPublishedSequenceNumber = 0, ownerLevel, producerGroupId } = publishingProps;
        const startingSequenceNumber = lastPublishedSequenceNumber + 1;
        for (let i = 0; i < events.length; i++) {
            const event = events[i];
            populateIdempotentMessageAnnotations(event, {
                isIdempotentPublishingEnabled: this._isIdempotent,
                ownerLevel,
                producerGroupId,
                publishSequenceNumber: startingSequenceNumber + i,
            });
        }
        this._pendingStartingSequenceNumber = startingSequenceNumber;
        return events;
    }
    /**
     * Annotates a rhea message with placeholder idempotent properties if the batch is idempotent.
     * This is necessary so that we can accurately calculate the size of the batch while adding events.
     * Placeholder values are used because real values won't be known until we attempt to send the batch.
     */
    _decorateRheaMessageWithPlaceholderIdempotencyProps(event) {
        if (!this._isIdempotent) {
            return event;
        }
        if (!event.message_annotations) {
            event.message_annotations = {};
        }
        // Set placeholder values for these annotations.
        populateIdempotentMessageAnnotations(event, {
            isIdempotentPublishingEnabled: this._isIdempotent,
            ownerLevel: 0,
            publishSequenceNumber: 0,
            producerGroupId: 0,
        });
        return event;
    }
    /**
     * Generates the single AMQP message which is the result of encoding all the events
     * added into the `EventDataBatch` instance.
     *
     * This is not meant for the user to use directly.
     *
     * When the `EventDataBatch` instance is passed to the `send()` method on the `EventHubProducer`,
     * this single batched AMQP message is what gets sent over the wire to the service.
     * @readonly
     */
    _generateMessage(publishingProps) {
        return this._generateBatch(this._encodedMessages, this._batchAnnotations, publishingProps);
    }
    /**
     * Sets startingPublishSequenceNumber to the pending publish sequence number.
     */
    _commitPublish() {
        this._startingPublishSequenceNumber = this._pendingStartingSequenceNumber;
    }
    /**
     * Tries to add an event data to the batch if permitted by the batch's size limit.
     * **NOTE**: Always remember to check the return value of this method, before calling it again
     * for the next event.
     *
     * @param eventData -  An individual event data object.
     * @returns A boolean value indicating if the event data has been added to the batch or not.
     */
    tryAdd(eventData, options = {}) {
        throwTypeErrorIfParameterMissing(this._context.connectionId, "tryAdd", "eventData", eventData);
        const { entityPath, host } = this._context.config;
        const { event: instrumentedEvent, spanContext } = instrumentEventData(eventData, options, entityPath, host);
        // Convert EventData to RheaMessage.
        const amqpMessage = toRheaMessage(instrumentedEvent, this._partitionKey);
        const originalAnnotations = amqpMessage.message_annotations && Object.assign({}, amqpMessage.message_annotations);
        this._decorateRheaMessageWithPlaceholderIdempotencyProps(amqpMessage);
        const encodedMessage = rheaPromise.message.encode(amqpMessage);
        let currentSize = this._sizeInBytes;
        // The first time an event is added, we need to calculate
        // the overhead of creating an AMQP batch, including the
        // message_annotations that are taken from the 1st message.
        if (this.count === 0) {
            if (originalAnnotations) {
                this._batchAnnotations = originalAnnotations;
            }
            // Figure out the overhead of creating a batch by generating an empty batch
            // with the expected batch annotations.
            currentSize += this._generateBatch([], this._batchAnnotations).length;
        }
        const messageSize = encodedMessage.length;
        const messageOverhead = messageSize <= smallMessageMaxBytes ? smallMessageOverhead : largeMessageOverhead;
        currentSize += messageSize + messageOverhead;
        // Check if the size of the batch exceeds the maximum allowed size
        // once we add the new event to it.
        if (currentSize > this._maxSizeInBytes) {
            return false;
        }
        // The event will fit in the batch, so it is now safe to store it.
        this._encodedMessages.push(encodedMessage);
        if (spanContext) {
            this._spanContexts.push(spanContext);
        }
        this._sizeInBytes = currentSize;
        this._count++;
        return true;
    }
}

// Copyright (c) Microsoft Corporation.
const messagePropertiesMap = {
    message_id: "messageId",
    user_id: "userId",
    to: "to",
    subject: "subject",
    reply_to: "replyTo",
    correlation_id: "correlationId",
    content_type: "contentType",
    content_encoding: "contentEncoding",
    absolute_expiry_time: "absoluteExpiryTime",
    creation_time: "creationTime",
    group_id: "groupId",
    group_sequence: "groupSequence",
    reply_to_group_id: "replyToGroupId",
};
/**
 * Converts the AMQP message to an EventData.
 * @param msg - The AMQP message that needs to be converted to EventData.
 * @param skipParsingBodyAsJson - Boolean to skip running JSON.parse() on message body when body type is `content`.
 * @internal
 */
function fromRheaMessage(msg, skipParsingBodyAsJson) {
    const rawMessage = coreAmqp.AmqpAnnotatedMessage.fromRheaMessage(msg);
    const { body, bodyType } = defaultDataTransformer.decode(msg.body, skipParsingBodyAsJson);
    rawMessage.bodyType = bodyType;
    const data = {
        body,
        getRawAmqpMessage() {
            return rawMessage;
        },
    };
    if (msg.message_annotations) {
        for (const annotationKey of Object.keys(msg.message_annotations)) {
            switch (annotationKey) {
                case coreAmqp.Constants.partitionKey:
                    data.partitionKey = msg.message_annotations[annotationKey];
                    break;
                case coreAmqp.Constants.sequenceNumber:
                    data.sequenceNumber = msg.message_annotations[annotationKey];
                    break;
                case coreAmqp.Constants.enqueuedTime:
                    data.enqueuedTimeUtc = new Date(msg.message_annotations[annotationKey]);
                    break;
                case coreAmqp.Constants.offset:
                    data.offset = msg.message_annotations[annotationKey];
                    break;
                default:
                    if (!data.systemProperties) {
                        data.systemProperties = {};
                    }
                    data.systemProperties[annotationKey] = convertDatesToNumbers(msg.message_annotations[annotationKey]);
                    break;
            }
        }
    }
    if (msg.application_properties) {
        data.properties = convertDatesToNumbers(msg.application_properties);
    }
    if (msg.delivery_annotations) {
        data.lastEnqueuedOffset = msg.delivery_annotations.last_enqueued_offset;
        data.lastSequenceNumber = msg.delivery_annotations.last_enqueued_sequence_number;
        data.lastEnqueuedTime = new Date(msg.delivery_annotations.last_enqueued_time_utc);
        data.retrievalTime = new Date(msg.delivery_annotations.runtime_info_retrieval_time_utc);
    }
    const messageProperties = Object.keys(messagePropertiesMap);
    for (const messageProperty of messageProperties) {
        if (!data.systemProperties) {
            data.systemProperties = {};
        }
        if (msg[messageProperty] != null) {
            data.systemProperties[messagePropertiesMap[messageProperty]] = convertDatesToNumbers(msg[messageProperty]);
        }
    }
    if (msg.content_type != null) {
        data.contentType = msg.content_type;
    }
    if (msg.correlation_id != null) {
        data.correlationId = msg.correlation_id;
    }
    if (msg.message_id != null) {
        data.messageId = msg.message_id;
    }
    return data;
}
/**
 * Converts an EventData object to an AMQP message.
 * @param data - The EventData object that needs to be converted to an AMQP message.
 * @param partitionKey - An optional key to determine the partition that this event should land in.
 * @internal
 */
function toRheaMessage(data, partitionKey) {
    var _a, _b;
    let rheaMessage;
    if (isAmqpAnnotatedMessage(data)) {
        rheaMessage = Object.assign(Object.assign({}, coreAmqp.AmqpAnnotatedMessage.toRheaMessage(data)), { body: defaultDataTransformer.encode(data.body, (_a = data.bodyType) !== null && _a !== void 0 ? _a : "data") });
    }
    else {
        let bodyType = "data";
        if (typeof data.getRawAmqpMessage === "function") {
            /*
              If the event is being round-tripped, then we respect the `bodyType` of the
              underlying AMQP message.
            */
            bodyType = (_b = data.getRawAmqpMessage().bodyType) !== null && _b !== void 0 ? _b : "data";
        }
        rheaMessage = {
            body: defaultDataTransformer.encode(data.body, bodyType),
        };
        // As per the AMQP 1.0 spec If the message-annotations or delivery-annotations section is omitted,
        // it is equivalent to a message-annotations section containing anempty map of annotations.
        rheaMessage.message_annotations = {};
        if (data.properties) {
            rheaMessage.application_properties = data.properties;
        }
        if (isDefined(partitionKey)) {
            rheaMessage.message_annotations[coreAmqp.Constants.partitionKey] = partitionKey;
            // Event Hub service cannot route messages to a specific partition based on the partition key
            // if AMQP message header is an empty object. Hence we make sure that header is always present
            // with atleast one property. Setting durable to true, helps us achieve that.
            rheaMessage.durable = true;
        }
        if (data.contentType != null) {
            rheaMessage.content_type = data.contentType;
        }
        if (data.correlationId != null) {
            rheaMessage.correlation_id = data.correlationId;
        }
        if (data.messageId != null) {
            if (typeof data.messageId === "string" &&
                data.messageId.length > coreAmqp.Constants.maxMessageIdLength) {
                throw new Error(`Length of 'messageId' property on the event cannot be greater than ${coreAmqp.Constants.maxMessageIdLength} characters.`);
            }
            rheaMessage.message_id = data.messageId;
        }
    }
    return rheaMessage;
}
/**
 * @internal
 */
function isAmqpAnnotatedMessage(possible) {
    return (isObjectWithProperties(possible, ["body", "bodyType"]) &&
        !objectHasProperty(possible, "getRawAmqpMessage"));
}
/**
 * Converts any Date objects into a number representing date.getTime().
 * Recursively checks for any Date objects in arrays and objects.
 * @internal
 */
function convertDatesToNumbers(thing) {
    // fast exit
    if (!isDefined(thing))
        return thing;
    // When 'thing' is a Date, return the number representation
    if (typeof thing === "object" &&
        objectHasProperty(thing, "getTime") &&
        typeof thing.getTime === "function") {
        return thing.getTime();
    }
    /*
      Examples:
      [0, 'foo', new Date(), { nested: new Date()}]
    */
    if (Array.isArray(thing)) {
        return thing.map(convertDatesToNumbers);
    }
    /*
      Examples:
      { foo: new Date(), children: { nested: new Date() }}
    */
    if (typeof thing === "object" && isDefined(thing)) {
        thing = Object.assign({}, thing);
        for (const key of Object.keys(thing)) {
            thing[key] = convertDatesToNumbers(thing[key]);
        }
    }
    return thing;
}
/**
 * Populates a rhea message with idempotent producer properties.
 * @internal
 */
function populateIdempotentMessageAnnotations(rheaMessage, { isIdempotentPublishingEnabled, ownerLevel, producerGroupId, publishSequenceNumber, }) {
    if (!isIdempotentPublishingEnabled) {
        return;
    }
    const messageAnnotations = rheaMessage.message_annotations || {};
    if (!rheaMessage.message_annotations) {
        rheaMessage.message_annotations = messageAnnotations;
    }
    if (isDefined(ownerLevel)) {
        messageAnnotations[idempotentProducerAmqpPropertyNames.epoch] = rheaPromise.types.wrap_short(ownerLevel);
    }
    if (isDefined(producerGroupId)) {
        messageAnnotations[idempotentProducerAmqpPropertyNames.producerId] =
            rheaPromise.types.wrap_long(producerGroupId);
    }
    if (isDefined(publishSequenceNumber)) {
        messageAnnotations[idempotentProducerAmqpPropertyNames.producerSequenceNumber] =
            rheaPromise.types.wrap_int(publishSequenceNumber);
    }
}
/**
 * Commits the pending publish sequence number events.
 * EventDataBatch exposes this as `startingPublishSequenceNumber`,
 * EventData not in a batch exposes this as `publishedSequenceNumber`.
 * @internal
 */
function commitIdempotentSequenceNumbers(events) {
    if (isEventDataBatch(events)) {
        events._commitPublish();
    }
    else {
        // For each event, set the `publishedSequenceNumber` equal to the sequence number
        // we set when we attempted to send the events to the service.
        for (const event of events) {
            event._publishedSequenceNumber = event[PENDING_PUBLISH_SEQ_NUM_SYMBOL];
            delete event[PENDING_PUBLISH_SEQ_NUM_SYMBOL];
        }
    }
}
/**
 * Rolls back any pending publish sequence number in the events.
 * @internal
 */
function rollbackIdempotentSequenceNumbers(events) {
    if (isEventDataBatch(events)) ;
    else {
        for (const event of events) {
            delete event[PENDING_PUBLISH_SEQ_NUM_SYMBOL];
        }
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 */
function getRetryAttemptTimeoutInMs(retryOptions) {
    const timeoutInMs = !isDefined(retryOptions) ||
        typeof retryOptions.timeoutInMs !== "number" ||
        !isFinite(retryOptions.timeoutInMs) ||
        retryOptions.timeoutInMs < coreAmqp.Constants.defaultOperationTimeoutInMs
        ? coreAmqp.Constants.defaultOperationTimeoutInMs
        : retryOptions.timeoutInMs;
    return timeoutInMs;
}

// Copyright (c) Microsoft Corporation.
/**
 * Describes the base class for entities like EventHub Sender, Receiver and Management link.
 * @internal
 */
class LinkEntity {
    /**
     * Creates a new LinkEntity instance.
     * @param context - The connection context.
     * @param options - Options that can be provided while creating the LinkEntity.
     */
    constructor(context, options) {
        /**
         * Indicates whether the link is in the process of connecting
         * (establishing) itself. Default value: `false`.
         */
        this.isConnecting = false;
        if (!options)
            options = {};
        this._context = context;
        this.address = options.address || "";
        this.audience = options.audience || "";
        this.name = `${options.name}-${uuid.v4()}`;
        this.partitionId = options.partitionId;
    }
    /**
     * Negotiates cbs claim for the LinkEntity.
     * @returns Promise<void>
     */
    async _negotiateClaim({ abortSignal, setTokenRenewal, timeoutInMs, }) {
        // Acquire the lock and establish a cbs session if it does not exist on the connection.
        // Although node.js is single threaded, we need a locking mechanism to ensure that a
        // race condition does not happen while creating a shared resource (in this case the
        // cbs session, since we want to have exactly 1 cbs session per connection).
        logger.verbose("[%s] Acquiring cbs lock: '%s' for creating the cbs session while creating the %s: " +
            "'%s' with address: '%s'.", this._context.connectionId, this._context.cbsSession.cbsLock, this._type, this.name, this.address);
        const startTime = Date.now();
        if (!this._context.cbsSession.isOpen()) {
            await coreAmqp.defaultCancellableLock.acquire(this._context.cbsSession.cbsLock, () => {
                return this._context.cbsSession.init({
                    abortSignal,
                    timeoutInMs: timeoutInMs - (Date.now() - startTime),
                });
            }, {
                abortSignal,
                timeoutInMs,
            });
        }
        let tokenObject;
        let tokenType;
        if (coreAmqp.isSasTokenProvider(this._context.tokenCredential)) {
            tokenObject = this._context.tokenCredential.getToken(this.audience);
            tokenType = coreAmqp.TokenType.CbsTokenTypeSas;
            // renew sas token in every 45 minutess
            this._tokenTimeoutInMs = (3600 - 900) * 1000;
        }
        else {
            const aadToken = await this._context.tokenCredential.getToken(coreAmqp.Constants.aadEventHubsScope);
            if (!aadToken) {
                throw new Error(`Failed to get token from the provided "TokenCredential" object`);
            }
            tokenObject = aadToken;
            tokenType = coreAmqp.TokenType.CbsTokenTypeJwt;
            this._tokenTimeoutInMs = tokenObject.expiresOnTimestamp - Date.now() - 2 * 60 * 1000;
        }
        logger.verbose("[%s] %s: calling negotiateClaim for audience '%s'.", this._context.connectionId, this._type, this.audience);
        // Acquire the lock to negotiate the CBS claim.
        logger.verbose("[%s] Acquiring cbs lock: '%s' for cbs auth for %s: '%s' with address '%s'.", this._context.connectionId, this._context.negotiateClaimLock, this._type, this.name, this.address);
        await coreAmqp.defaultCancellableLock.acquire(this._context.negotiateClaimLock, () => {
            return this._context.cbsSession.negotiateClaim(this.audience, tokenObject.token, tokenType, { abortSignal, timeoutInMs: timeoutInMs - (Date.now() - startTime) });
        }, {
            abortSignal,
            timeoutInMs: timeoutInMs - (Date.now() - startTime),
        });
        logger.verbose("[%s] Negotiated claim for %s '%s' with with address: %s", this._context.connectionId, this._type, this.name, this.address);
        if (setTokenRenewal) {
            await this._ensureTokenRenewal();
        }
    }
    /**
     * Ensures that the token is renewed within the predefined renewal margin.
     */
    _ensureTokenRenewal() {
        if (!this._tokenTimeoutInMs) {
            return;
        }
        // Clear the existing token renewal timer.
        // This scenario can happen if the connection goes down and is brought back up
        // before the `nextRenewalTimeout` was reached.
        if (this._tokenRenewalTimer) {
            clearTimeout(this._tokenRenewalTimer);
        }
        this._tokenRenewalTimer = setTimeout(async () => {
            try {
                await this._negotiateClaim({
                    setTokenRenewal: true,
                    abortSignal: undefined,
                    timeoutInMs: getRetryAttemptTimeoutInMs(undefined),
                });
            }
            catch (err) {
                logger.verbose("[%s] %s '%s' with address %s, an error occurred while renewing the token: %O", this._context.connectionId, this._type, this.name, this.address, err);
            }
        }, this._tokenTimeoutInMs);
        logger.verbose("[%s] %s '%s' with address %s, has next token renewal in %d milliseconds @(%s).", this._context.connectionId, this._type, this.name, this.address, this._tokenTimeoutInMs, new Date(Date.now() + this._tokenTimeoutInMs).toString());
    }
    /**
     * Closes the Sender|Receiver link and it's underlying session and also removes it from the
     * internal map.
     * @param link - The Sender or Receiver link that needs to be closed and
     * removed.
     */
    async _closeLink(link) {
        clearTimeout(this._tokenRenewalTimer);
        if (link) {
            try {
                // Closing the link and its underlying session if the link is open. This should also
                // remove them from the internal map.
                await link.close();
                logger.verbose("[%s] %s '%s' with address '%s' closed.", this._context.connectionId, this._type, this.name, this.address);
            }
            catch (err) {
                logger.verbose("[%s] An error occurred while closing the %s '%s' with address '%s': %O", this._context.connectionId, this._type, this.name, this.address, err);
            }
        }
    }
    /**
     * Provides the current type of the LinkEntity.
     * @returns The entity type.
     */
    get _type() {
        let result = "LinkEntity";
        if (this.constructor && this.constructor.name) {
            result = this.constructor.name;
        }
        return result;
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * Describes the EventHubReceiver that will receive event data from EventHub.
 * @internal
 */
class EventHubReceiver extends LinkEntity {
    /**
     * Instantiates a receiver that can be used to receive events over an AMQP receiver link in
     * either batching or streaming mode.
     * @param context -        The connection context corresponding to the EventHubClient instance
     * @param consumerGroup -  The consumer group from which the receiver should receive events from.
     * @param partitionId -    The Partition ID from which to receive.
     * @param eventPosition -  The position in the stream from where to start receiving events.
     * @param options -      Receiver options.
     */
    constructor(context, consumerGroup, partitionId, eventPosition, options = {}) {
        super(context, {
            partitionId: partitionId,
            name: context.config.getReceiverAddress(partitionId, consumerGroup),
        });
        /**
         * The sequence number of the most recently received AMQP message.
         */
        this._checkpoint = -1;
        /**
         * A queue of events that were received from the AMQP link but not consumed externally by `EventHubConsumer`
         */
        this._internalQueue = [];
        /**
         * Indicates that events in the internal queue are being processed to be consumed by `EventHubConsumer`
         */
        this._usingInternalQueue = false;
        /**
         * Indicates if messages are being received from this receiver.
         */
        this._isReceivingMessages = false;
        /**
         * Indicated if messages are being received in streaming mode.
         */
        this._isStreaming = false;
        /**
         * Denotes if close() was called on this receiver
         */
        this._isClosed = false;
        this.consumerGroup = consumerGroup;
        this.address = context.config.getReceiverAddress(partitionId, this.consumerGroup);
        this.audience = context.config.getReceiverAudience(partitionId, this.consumerGroup);
        this.ownerLevel = options.ownerLevel;
        this.eventPosition = eventPosition;
        this.options = options;
        this.runtimeInfo = {};
    }
    /**
     * Returns sequenceNumber of the last event received from the service. This will not match the
     * last event received by `EventHubConsumer` when the `_internalQueue` is not empty
     * @readonly
     */
    get checkpoint() {
        return this._checkpoint;
    }
    /**
     * Indicates if messages are being received from this receiver.
     * @readonly
     */
    get isReceivingMessages() {
        return this._isReceivingMessages;
    }
    /**
     * Indicates if the receiver has been closed.
     */
    get isClosed() {
        return this._isClosed;
    }
    /**
     * The last enqueued event information. This property will only
     * be enabled when `trackLastEnqueuedEventProperties` option is set to true
     * @readonly
     */
    get lastEnqueuedEventProperties() {
        return this.runtimeInfo;
    }
    _onAmqpMessage(context) {
        if (!context.message) {
            return;
        }
        const data = fromRheaMessage(context.message, !!this.options.skipParsingBodyAsJson);
        const rawMessage = data.getRawAmqpMessage();
        const receivedEventData = {
            body: data.body,
            properties: data.properties,
            offset: data.offset,
            sequenceNumber: data.sequenceNumber,
            enqueuedTimeUtc: data.enqueuedTimeUtc,
            partitionKey: data.partitionKey,
            systemProperties: data.systemProperties,
            getRawAmqpMessage() {
                return rawMessage;
            },
        };
        if (data.correlationId != null) {
            receivedEventData.correlationId = data.correlationId;
        }
        if (data.contentType != null) {
            receivedEventData.contentType = data.contentType;
        }
        if (data.messageId != null) {
            receivedEventData.messageId = data.messageId;
        }
        this._checkpoint = receivedEventData.sequenceNumber;
        if (this.options.trackLastEnqueuedEventProperties && data) {
            this.runtimeInfo.sequenceNumber = data.lastSequenceNumber;
            this.runtimeInfo.enqueuedOn = data.lastEnqueuedTime;
            this.runtimeInfo.offset = data.lastEnqueuedOffset;
            this.runtimeInfo.retrievedOn = data.retrievalTime;
        }
        // Add to internal queue if
        // - There are no listeners, we are probably getting events due to pending credits
        // - Or Events from internal queue are being processed, so add to it to ensure order of processing is retained
        if (!this._onMessage || this._usingInternalQueue) {
            this._internalQueue.push(receivedEventData);
        }
        else {
            if (this._isStreaming) {
                this._addCredit(1);
            }
            this._onMessage(receivedEventData);
        }
    }
    _onAmqpError(context) {
        const rheaReceiver = this._receiver || context.receiver;
        const amqpError = rheaReceiver && rheaReceiver.error;
        logger.verbose("[%s] 'receiver_error' event occurred on the receiver '%s' with address '%s'. " +
            "The associated error is: %O", this._context.connectionId, this.name, this.address, amqpError);
        if (this._onError && amqpError) {
            const error = coreAmqp.translate(amqpError);
            logErrorStackTrace(error);
            this._onError(error);
        }
    }
    _onAmqpSessionError(context) {
        const sessionError = context.session && context.session.error;
        logger.verbose("[%s] 'session_error' event occurred on the session of receiver '%s' with address '%s'. " +
            "The associated error is: %O", this._context.connectionId, this.name, this.address, sessionError);
        if (this._onError && sessionError) {
            const error = coreAmqp.translate(sessionError);
            logErrorStackTrace(error);
            this._onError(error);
        }
    }
    async _onAmqpClose(context) {
        const rheaReceiver = this._receiver || context.receiver;
        logger.verbose("[%s] 'receiver_close' event occurred on the receiver '%s' with address '%s'. " +
            "Value for isItselfClosed on the receiver is: '%s' " +
            "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, rheaReceiver ? rheaReceiver.isItselfClosed().toString() : undefined, this.isConnecting);
        if (rheaReceiver && !this.isConnecting) {
            // Call close to clean up timers & other resources
            await rheaReceiver.close().catch((err) => {
                logger.verbose("[%s] Error when closing receiver [%s] after 'receiver_close' event: %O", this._context.connectionId, this.name, err);
            });
        }
    }
    async _onAmqpSessionClose(context) {
        const rheaReceiver = this._receiver || context.receiver;
        logger.verbose("[%s] 'session_close' event occurred on the session of receiver '%s' with address '%s'. " +
            "Value for isSessionItselfClosed on the session is: '%s' " +
            "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, rheaReceiver ? rheaReceiver.isSessionItselfClosed().toString() : undefined, this.isConnecting);
        if (rheaReceiver && !this.isConnecting) {
            // Call close to clean up timers & other resources
            await rheaReceiver.close().catch((err) => {
                logger.verbose("[%s] Error when closing receiver [%s] after 'session_close' event: %O", this._context.connectionId, this.name, err);
            });
        }
    }
    async abort() {
        const desc = `[${this._context.connectionId}] The receive operation on the Receiver "${this.name}" with ` +
            `address "${this.address}" has been cancelled by the user.`;
        // Cancellation is user-intended, so log to info instead of warning.
        logger.info(desc);
        if (this._onError) {
            const error = new abortController.AbortError(coreAmqp.StandardAbortMessage);
            this._onError(error);
        }
        this.clearHandlers();
        await this.close();
    }
    /**
     * Clears the user-provided handlers and updates the receiving messages flag.
     */
    clearHandlers() {
        if (this._abortSignal && this._onAbort) {
            this._abortSignal.removeEventListener("abort", this._onAbort);
        }
        this._abortSignal = undefined;
        this._onAbort = undefined;
        this._onError = undefined;
        this._onMessage = undefined;
        this._isReceivingMessages = false;
        this._isStreaming = false;
    }
    /**
     * Closes the underlying AMQP receiver.
     */
    async close() {
        try {
            this.clearHandlers();
            if (!this._receiver) {
                return;
            }
            const receiverLink = this._receiver;
            this._deleteFromCache();
            await this._closeLink(receiverLink);
        }
        catch (err) {
            const msg = `[${this._context.connectionId}] An error occurred while closing receiver ${this.name}: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`;
            logger.warning(msg);
            logErrorStackTrace(err);
            throw err;
        }
        finally {
            this._isClosed = true;
        }
    }
    /**
     * Determines whether the AMQP receiver link is open. If open then returns true else returns false.
     * @returns boolean
     */
    isOpen() {
        const result = Boolean(this._receiver && this._receiver.isOpen());
        logger.verbose("[%s] Receiver '%s' with address '%s' is open? -> %s", this._context.connectionId, this.name, this.address, result);
        return result;
    }
    /**
     * Registers the user's onMessage and onError handlers.
     * Sends buffered events from the queue before adding additional credits to the AMQP link.
     */
    registerHandlers(onMessage, onError, maximumCreditCount, isStreaming, abortSignal, onAbort) {
        this._abortSignal = abortSignal;
        this._onAbort = onAbort;
        this._onError = onError;
        this._onMessage = onMessage;
        this._isStreaming = isStreaming;
        // indicate that messages are being received.
        this._isReceivingMessages = true;
        this._useInternalQueue(onMessage, abortSignal)
            .then(async (processedEventCount) => {
            if (this._onMessage !== onMessage) {
                // the original handler has been removed, so no further action required.
                return;
            }
            // check if more messages are required
            if (!isStreaming && maximumCreditCount - processedEventCount <= 0) {
                return;
            }
            if (!this.isOpen()) {
                try {
                    await this.initialize({
                        abortSignal,
                        timeoutInMs: getRetryAttemptTimeoutInMs(this.options.retryOptions),
                    });
                }
                catch (err) {
                    if (this._onError === onError) {
                        onError(err);
                    }
                    return;
                }
            }
            else {
                logger.verbose("[%s] Receiver link already present, hence reusing it.", this._context.connectionId);
            }
            // add credits
            const existingCredits = this._receiver ? this._receiver.credit : 0;
            const prcoessedEventCountToExclude = isStreaming ? 0 : processedEventCount;
            const creditsToAdd = Math.max(maximumCreditCount - (existingCredits + prcoessedEventCountToExclude), 0);
            this._addCredit(creditsToAdd);
            return;
        })
            .catch((err) => {
            // something really unexpected happened, so attempt to call user's error handler
            if (this._onError === onError) {
                onError(err);
            }
        });
    }
    _addCredit(credit) {
        if (this._receiver) {
            this._receiver.addCredit(credit);
        }
    }
    _deleteFromCache() {
        this._receiver = undefined;
        delete this._context.receivers[this.name];
        logger.verbose("[%s] Deleted the receiver '%s' from the client cache.", this._context.connectionId, this.name);
    }
    async _useInternalQueue(onMessage, abortSignal) {
        let processedMessagesCount = 0;
        // allow the event loop to process any blocking code outside
        // this code path before sending any events.
        await coreAmqp.delay(0);
        this._usingInternalQueue = true;
        while (this._internalQueue.length) {
            if (!this._onMessage) {
                break;
            }
            if (abortSignal && abortSignal.aborted) {
                break;
            }
            // These will not be equal if clearHandlers and registerHandlers were called
            // in the same tick of the event loop. If onMessage isn't the currently active
            // handler, it should stop getting messages from the queue.
            if (this._onMessage !== onMessage) {
                break;
            }
            const eventData = this._internalQueue.splice(0, 1)[0];
            processedMessagesCount++;
            onMessage(eventData);
            // allow the event loop to process any blocking code outside
            // this code path before sending the next event.
            await coreAmqp.delay(0);
        }
        this._usingInternalQueue = false;
        return processedMessagesCount;
    }
    /**
     * Creates a new AMQP receiver under a new AMQP session.
     */
    async initialize({ abortSignal, timeoutInMs, }) {
        try {
            if (!this.isOpen() && !this.isConnecting) {
                this.isConnecting = true;
                // Wait for the connectionContext to be ready to open the link.
                await this._context.readyToOpenLink();
                await this._negotiateClaim({ setTokenRenewal: false, abortSignal, timeoutInMs });
                const receiverOptions = {
                    onClose: (context) => this._onAmqpClose(context),
                    onError: (context) => this._onAmqpError(context),
                    onMessage: (context) => this._onAmqpMessage(context),
                    onSessionClose: (context) => this._onAmqpSessionClose(context),
                    onSessionError: (context) => this._onAmqpSessionError(context),
                };
                if (this.checkpoint > -1) {
                    receiverOptions.eventPosition = { sequenceNumber: this.checkpoint };
                }
                const options = this._createReceiverOptions(receiverOptions);
                logger.verbose("[%s] Trying to create receiver '%s' with options %O", this._context.connectionId, this.name, options);
                this._receiver = await this._context.connection.createReceiver(Object.assign(Object.assign({}, options), { abortSignal }));
                this.isConnecting = false;
                logger.verbose("[%s] Receiver '%s' created with receiver options: %O", this._context.connectionId, this.name, options);
                // store the underlying link in a cache
                this._context.receivers[this.name] = this;
                this._ensureTokenRenewal();
            }
            else {
                logger.verbose("[%s] The receiver '%s' with address '%s' is open -> %s and is connecting " +
                    "-> %s. Hence not reconnecting.", this._context.connectionId, this.name, this.address, this.isOpen(), this.isConnecting);
            }
        }
        catch (err) {
            this.isConnecting = false;
            const error = coreAmqp.translate(err);
            logger.warning("[%s] An error occured while creating the receiver '%s': %s", this._context.connectionId, this.name, `${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
            logErrorStackTrace(err);
            throw error;
        }
    }
    /**
     * Creates the options that need to be specified while creating an AMQP receiver link.
     */
    _createReceiverOptions(options) {
        const rcvrOptions = {
            name: this.name,
            autoaccept: true,
            source: {
                address: this.address,
            },
            credit_window: 0,
            onMessage: options.onMessage,
            onError: options.onError,
            onClose: options.onClose,
            onSessionError: options.onSessionError,
            onSessionClose: options.onSessionClose,
        };
        if (typeof this.ownerLevel === "number") {
            rcvrOptions.properties = {
                [coreAmqp.Constants.attachEpoch]: rheaPromise.types.wrap_long(this.ownerLevel),
            };
        }
        if (this.options.trackLastEnqueuedEventProperties) {
            rcvrOptions.desired_capabilities = coreAmqp.Constants.enableReceiverRuntimeMetricName;
        }
        const eventPosition = options.eventPosition || this.eventPosition;
        if (eventPosition) {
            // Set filter on the receiver if event position is specified.
            const filterClause = getEventPositionFilter(eventPosition);
            if (filterClause) {
                rcvrOptions.source.filter = {
                    "apache.org:selector-filter:string": rheaPromise.types.wrap_described(filterClause, 0x468c00000004),
                };
            }
        }
        return rcvrOptions;
    }
    /**
     * Returns a promise that resolves to an array of events received from the service.
     *
     * @param maxMessageCount - The maximum number of messages to receive.
     * @param maxWaitTimeInSeconds - The maximum amount of time to wait to build up the requested message count;
     * If not provided, it defaults to 60 seconds.
     * @param abortSignal - An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     *
     * @throws AbortError if the operation is cancelled via the abortSignal.
     * @throws MessagingError if an error is encountered while receiving a message.
     * @throws Error if the underlying connection or receiver has been closed.
     * Create a new EventHubConsumer using the EventHubClient createConsumer method.
     * @throws Error if the receiver is already receiving messages.
     */
    async receiveBatch(maxMessageCount, maxWaitTimeInSeconds = 60, abortSignal) {
        // store events across multiple retries
        const receivedEvents = [];
        const retrieveEvents = () => {
            return new Promise((resolve, reject) => {
                // if this consumer was closed,
                // resolve the operation's promise with the events collected thus far in case
                // the promise hasn't already been resolved.
                if (this._isClosed || this._context.wasConnectionCloseCalled) {
                    return resolve(receivedEvents);
                }
                let timer;
                const logOnAbort = () => {
                    const name = this.name;
                    const address = this.address;
                    const desc = `[${this._context.connectionId}] The request operation on the Receiver "${name}" with ` +
                        `address "${address}" has been cancelled by the user.`;
                    // Cancellation is intentional so logging to 'info'.
                    logger.info(desc);
                };
                const rejectOnAbort = async () => {
                    logOnAbort();
                    try {
                        await this.close();
                    }
                    finally {
                        reject(new abortController.AbortError(coreAmqp.StandardAbortMessage));
                    }
                };
                // operation has been cancelled, so exit immediately
                if (abortSignal && abortSignal.aborted) {
                    return rejectOnAbort();
                }
                // updates the prefetch count so that the baseConsumer adds
                // the correct number of credits to receive the same number of events.
                const prefetchCount = Math.max(maxMessageCount - receivedEvents.length, 0);
                if (prefetchCount === 0) {
                    return resolve(receivedEvents);
                }
                logger.verbose("[%s] Receiver '%s', setting the prefetch count to %d.", this._context.connectionId, this.name, prefetchCount);
                const cleanUpBeforeReturn = () => {
                    this.clearHandlers();
                    clearTimeout(timer);
                };
                const onAbort = () => {
                    clearTimeout(timer);
                    rejectOnAbort();
                };
                this.registerHandlers((eventData) => {
                    receivedEvents.push(eventData);
                    // resolve the operation's promise after the requested
                    // number of events are received.
                    if (receivedEvents.length === maxMessageCount) {
                        logger.info("[%s] Batching Receiver '%s', %d messages received within %d seconds.", this._context.connectionId, this.name, receivedEvents.length, maxWaitTimeInSeconds);
                        cleanUpBeforeReturn();
                        resolve(receivedEvents);
                    }
                }, (err) => {
                    // restore events to the front of the internal queue.
                    while (receivedEvents.length) {
                        this._internalQueue.unshift(receivedEvents.pop());
                    }
                    cleanUpBeforeReturn();
                    if (err.name === "AbortError") {
                        rejectOnAbort();
                    }
                    else {
                        reject(err);
                    }
                }, maxMessageCount - receivedEvents.length, false, abortSignal, onAbort);
                const addTimeout = () => {
                    const msg = "[%s] Setting the wait timer for %d seconds for receiver '%s'.";
                    logger.verbose(msg, this._context.connectionId, maxWaitTimeInSeconds, this.name);
                    // resolve the operation's promise after the requested
                    // max number of seconds have passed.
                    timer = setTimeout(() => {
                        logger.info("[%s] Batching Receiver '%s', %d messages received when max wait time in seconds %d is over.", this._context.connectionId, this.name, receivedEvents.length, maxWaitTimeInSeconds);
                        cleanUpBeforeReturn();
                        resolve(receivedEvents);
                    }, maxWaitTimeInSeconds * 1000);
                };
                addTimeout();
                if (abortSignal && !abortSignal.aborted) {
                    abortSignal.addEventListener("abort", onAbort);
                }
            });
        };
        const retryOptions = this.options.retryOptions || {};
        const config = Object.defineProperties({
            operation: retrieveEvents,
            operationType: coreAmqp.RetryOperationType.receiveMessage,
            abortSignal: abortSignal,
            retryOptions: retryOptions,
        }, {
            connectionId: {
                enumerable: true,
                get: () => {
                    return this._context.connectionId;
                },
            },
            connectionHost: {
                enumerable: true,
                get: () => {
                    return this._context.config.host;
                },
            },
        });
        return coreAmqp.retry(config);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 */
class PartitionPump {
    constructor(_context, partitionProcessor, _startPosition, options) {
        this._context = _context;
        this._startPosition = _startPosition;
        this._isReceiving = false;
        this._isStopped = false;
        this._partitionProcessor = partitionProcessor;
        this._processorOptions = options;
        this._abortController = new abortController.AbortController();
    }
    get isReceiving() {
        return this._isReceiving;
    }
    async start() {
        this._isReceiving = true;
        try {
            await this._partitionProcessor.initialize();
        }
        catch (err) {
            // swallow the error from the user-defined code
            this._partitionProcessor.processError(err);
        }
        // this is intentionally not await'd - the _receiveEvents loop will continue to
        // execute and can be stopped by calling .stop()
        this._receiveEvents(this._partitionProcessor.partitionId);
        logger.info(`Successfully started the receiver for partition "${this._partitionProcessor.partitionId}".`);
    }
    /**
     * Creates a new `EventHubReceiver` and replaces any existing receiver.
     * @param partitionId - The partition the receiver should read messages from.
     * @param lastSeenSequenceNumber - The sequence number to begin receiving messages from (exclusive).
     * If `-1`, then the PartitionPump's startPosition will be used instead.
     */
    _setOrReplaceReceiver(partitionId, lastSeenSequenceNumber) {
        // Determine what the new EventPosition should be.
        // If this PartitionPump has received events, we'll start from the last
        // seen sequenceNumber (exclusive).
        // Otherwise, use the `_startPosition`.
        const currentEventPosition = lastSeenSequenceNumber >= 0
            ? {
                sequenceNumber: lastSeenSequenceNumber,
                isInclusive: false,
            }
            : this._startPosition;
        // Set or replace the PartitionPump's receiver.
        this._receiver = new EventHubReceiver(this._context, this._partitionProcessor.consumerGroup, partitionId, currentEventPosition, {
            ownerLevel: this._processorOptions.ownerLevel,
            trackLastEnqueuedEventProperties: this._processorOptions.trackLastEnqueuedEventProperties,
            retryOptions: this._processorOptions.retryOptions,
            skipParsingBodyAsJson: this._processorOptions.skipParsingBodyAsJson,
        });
        return this._receiver;
    }
    async _receiveEvents(partitionId) {
        let lastSeenSequenceNumber = -1;
        let receiver = this._setOrReplaceReceiver(partitionId, lastSeenSequenceNumber);
        while (this._isReceiving) {
            try {
                // Check if the receiver was closed so we can recreate it.
                if (receiver.isClosed) {
                    receiver = this._setOrReplaceReceiver(partitionId, lastSeenSequenceNumber);
                }
                const receivedEvents = await receiver.receiveBatch(this._processorOptions.maxBatchSize, this._processorOptions.maxWaitTimeInSeconds, this._abortController.signal);
                if (this._processorOptions.trackLastEnqueuedEventProperties &&
                    receiver.lastEnqueuedEventProperties) {
                    this._partitionProcessor.lastEnqueuedEventProperties =
                        receiver.lastEnqueuedEventProperties;
                }
                // avoid calling user's processEvents handler if the pump was stopped while receiving events
                if (!this._isReceiving) {
                    return;
                }
                if (receivedEvents.length) {
                    lastSeenSequenceNumber = receivedEvents[receivedEvents.length - 1].sequenceNumber;
                }
                await tracingClient.withSpan("PartitionPump.process", {}, () => this._partitionProcessor.processEvents(receivedEvents), toProcessingSpanOptions(receivedEvents, this._context.config));
            }
            catch (err) {
                // check if this pump is still receiving
                // it may not be if the EventProcessor was stopped during processEvents
                if (!this._isReceiving) {
                    // no longer receiving, so close was called from somewhere else
                    return;
                }
                logger.warning(`An error was thrown while receiving or processing events on partition "${this._partitionProcessor.partitionId}"`);
                logErrorStackTrace(err);
                // forward error to user's processError and swallow errors they may throw
                try {
                    await this._partitionProcessor.processError(err);
                }
                catch (errorFromUser) {
                    // Using verbose over warning because this error is swallowed.
                    logger.verbose("An error was thrown by user's processError method: ", errorFromUser);
                }
                // close the partition processor if a non-retryable error was encountered
                if (typeof err !== "object" || !err.retryable) {
                    try {
                        // If the exception indicates that the partition was stolen (i.e some other consumer with same ownerlevel
                        // started consuming the partition), update the closeReason
                        if (err.code === "ReceiverDisconnectedError") {
                            return await this.stop(exports.CloseReason.OwnershipLost);
                        }
                        // this will close the pump and will break us out of the while loop
                        return await this.stop(exports.CloseReason.Shutdown);
                    }
                    catch (errorFromStop) {
                        // Using verbose over warning because this error is swallowed.
                        logger.verbose(`An error occurred while closing the receiver with reason ${exports.CloseReason.Shutdown}: `, errorFromStop);
                    }
                }
            }
        }
    }
    async stop(reason) {
        if (this._isStopped) {
            return;
        }
        this._isStopped = true;
        this._isReceiving = false;
        try {
            // Trigger the cancellation before closing the receiver,
            // otherwise the receiver will remove the listener on the abortSignal
            // before it has a chance to be emitted.
            this._abortController.abort();
            if (this._receiver) {
                await this._receiver.close();
            }
            await this._partitionProcessor.close(reason);
        }
        catch (err) {
            logger.warning(`An error occurred while closing the receiver: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
            logErrorStackTrace(err);
            this._partitionProcessor.processError(err);
            throw err;
        }
    }
}
/**
 * @internal
 */
function toProcessingSpanOptions(receivedEvents, eventHubProperties) {
    const spanLinks = [];
    for (const receivedEvent of receivedEvents) {
        const tracingContext = extractSpanContextFromEventData(receivedEvent);
        if (tracingContext) {
            spanLinks.push({
                tracingContext,
                attributes: {
                    enqueuedTime: receivedEvent.enqueuedTimeUtc.getTime(),
                },
            });
        }
    }
    return Object.assign({ spanLinks, spanKind: "consumer" }, toSpanOptions(eventHubProperties));
}

// Copyright (c) Microsoft Corporation.
/**
 * The PumpManager handles the creation and removal of PartitionPumps.
 * It also starts a PartitionPump when it is created, and stops a
 * PartitionPump when it is removed.
 * @internal
 */
class PumpManagerImpl {
    constructor(eventProcessorName, eventProcessorOptions) {
        this._partitionIdToPumps = {};
        this._eventProcessorName = eventProcessorName;
        this._options = eventProcessorOptions;
    }
    /**
     * Returns a list of partitionIds that are actively receiving messages.
     */
    receivingFromPartitions() {
        return Object.keys(this._partitionIdToPumps).filter((id) => {
            const pump = this._partitionIdToPumps[id];
            return Boolean(pump && pump.isReceiving);
        });
    }
    /**
     * Indicates whether the pump manager is actively receiving events from a given partition.
     * @internal
     */
    isReceivingFromPartition(partitionId) {
        const pump = this._partitionIdToPumps[partitionId];
        return Boolean(pump && pump.isReceiving);
    }
    /**
     * Creates and starts a PartitionPump.
     * @param startPosition - The position in the partition to start reading from.
     * @param connectionContext - The ConnectionContext to forward to the PartitionPump.
     * @param partitionProcessor - The PartitionProcessor to forward to the PartitionPump.
     */
    async createPump(startPosition, connectionContext, partitionProcessor, abortSignal) {
        const partitionId = partitionProcessor.partitionId;
        if (abortSignal.aborted) {
            logger.verbose(`${this._eventProcessorName}] The subscription was closed before creating the pump for partition ${partitionId}.`);
            return;
        }
        // attempt to get an existing pump
        const existingPump = this._partitionIdToPumps[partitionId];
        if (existingPump) {
            if (existingPump.isReceiving) {
                logger.verbose(`[${this._eventProcessorName}] [${partitionId}] The existing pump is running.`);
                return;
            }
            logger.verbose(`[${this._eventProcessorName}] [${partitionId}] The existing pump is not running.`);
            await this.removePump(partitionId, exports.CloseReason.OwnershipLost);
        }
        logger.verbose(`[${this._eventProcessorName}] [${partitionId}] Creating a new pump.`);
        const pump = new PartitionPump(connectionContext, partitionProcessor, startPosition, this._options);
        try {
            // Set the pump before starting it in case the user
            // closes the subscription while `start()` is in progress.
            this._partitionIdToPumps[partitionId] = pump;
            await pump.start();
        }
        catch (err) {
            logger.verbose(`[${this._eventProcessorName}] [${partitionId}] An error occured while adding/updating a pump: ${err}`);
            logErrorStackTrace(err);
        }
    }
    /**
     * Stop a PartitionPump and removes it from the internal map.
     * @param partitionId - The partitionId to remove the associated PartitionPump from.
     * @param reason - The reason for removing the pump.
     */
    async removePump(partitionId, reason) {
        try {
            const pump = this._partitionIdToPumps[partitionId];
            if (pump) {
                delete this._partitionIdToPumps[partitionId];
                logger.verbose(`[${this._eventProcessorName}] [${partitionId}] Stopping the pump.`);
                await pump.stop(reason);
            }
            else {
                logger.verbose(`[${this._eventProcessorName}] [${partitionId}] No pump was found to remove.`);
            }
        }
        catch (err) {
            logger.verbose(`[${this._eventProcessorName}] [${partitionId}] An error occured while removing a pump: ${err}`);
            logErrorStackTrace(err);
        }
    }
    /**
     * Stops all PartitionPumps and removes them from the internal map.
     * @param reason - The reason for removing the pump.
     */
    async removeAllPumps(reason) {
        const partitionIds = Object.keys(this._partitionIdToPumps);
        logger.verbose(`[${this._eventProcessorName}] Removing all pumps due to reason ${reason}.`);
        const tasks = [];
        for (const partitionId of partitionIds) {
            const pump = this._partitionIdToPumps[partitionId];
            if (pump) {
                tasks.push(pump.stop(reason));
            }
        }
        try {
            await Promise.all(tasks);
        }
        catch (err) {
            logger.verbose(`[${this._eventProcessorName}] An error occured while removing all pumps: ${err}`);
            logErrorStackTrace(err);
        }
        finally {
            this._partitionIdToPumps = {};
        }
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @param delayInMs - The number of milliseconds to be delayed.
 * @param abortSignal - The abortSignal associated with the containing operation.
 * @internal
 */
async function delayWithoutThrow(delayInMs, abortSignal) {
    try {
        await coreAmqp.delay(delayInMs, abortSignal);
    }
    catch (_a) {
        /* no-op to swallow AbortError */
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * Event Processor based applications consist of one or more instances of EventProcessor which have been
 * configured to consume events from the same Event Hub and consumer group. They balance the
 * workload across different instances by distributing the partitions to be processed among themselves.
 * They also allow the user to track progress when events are processed using checkpoints.
 *
 * A checkpoint is meant to represent the last successfully processed event by the user from a particular
 * partition of a consumer group in an Event Hub instance.
 *
 * You need the below to create an instance of `EventProcessor`
 * - The name of the consumer group from which you want to process events
 * - An instance of `EventHubClient` class that was created for the Event Hub instance.
 * - A user implemented class that extends the `PartitionProcessor` class. To get started, you can use the
 * base class `PartitionProcessor` which simply logs the incoming events. To provide your code to process incoming
 * events, extend this class and override the `processEvents()` method. For example:
 * ```js
 * class SamplePartitionProcessor extends PartitionProcessor {
 *     async processEvents(events) {
 *        // user code to process events here
 *        // Information on the partition being processed is available as properties on the `SamplePartitionProcessor` class
 *        // use `this.updateCheckpoint()` method to update checkpoints as needed
 *     }
 * }
 * ```
 * - An instance of `CheckpointStore`. See &commat;azure/eventhubs-checkpointstore-blob for an implementation.
 * For production, choose an implementation that will store checkpoints and partition ownership details to a durable store.
 * Implementations of `CheckpointStore` can be found on npm by searching for packages with the prefix &commat;azure/eventhub-checkpointstore-.
 *
 * @internal
 */
class EventProcessor {
    /**
     * @param consumerGroup - The name of the consumer group from which you want to process events.
     * @param eventHubClient - An instance of `EventHubClient` that was created for the Event Hub instance.
     * @param PartitionProcessorClass - A user-provided class that extends the `PartitionProcessor` class.
     * This class will be responsible for processing and checkpointing events.
     * @param checkpointStore - An instance of `CheckpointStore`. See &commat;azure/eventhubs-checkpointstore-blob for an implementation.
     * For production, choose an implementation that will store checkpoints and partition ownership details to a durable store.
     * @param options - A set of options to configure the Event Processor
     * - `maxBatchSize`         : The max size of the batch of events passed each time to user code for processing.
     * - `maxWaitTimeInSeconds` : The maximum amount of time to wait to build up the requested message count before
     * passing the data to user code for processing. If not provided, it defaults to 60 seconds.
     */
    constructor(_consumerGroup, _context, _subscriptionEventHandlers, _checkpointStore, options) {
        this._consumerGroup = _consumerGroup;
        this._context = _context;
        this._subscriptionEventHandlers = _subscriptionEventHandlers;
        this._checkpointStore = _checkpointStore;
        this._isRunning = false;
        if (options.ownerId) {
            this._id = options.ownerId;
            logger.verbose(`Starting event processor with ID ${this._id}`);
        }
        else {
            this._id = uuid.v4();
            logger.verbose(`Starting event processor with autogenerated ID ${this._id}`);
        }
        this._eventHubName = this._context.config.entityPath;
        this._fullyQualifiedNamespace = this._context.config.host;
        this._processorOptions = options;
        this._pumpManager =
            options.pumpManager || new PumpManagerImpl(this._id, this._processorOptions);
        this._processingTarget = options.processingTarget;
        this._loopIntervalInMs = options.loopIntervalInMs;
        this._loadBalancingStrategy = options.loadBalancingStrategy;
    }
    /**
     * The unique identifier for the EventProcessor.
     */
    get id() {
        return this._id;
    }
    _createPartitionOwnershipRequest(partitionOwnershipMap, partitionIdToClaim) {
        const previousPartitionOwnership = partitionOwnershipMap.get(partitionIdToClaim);
        const partitionOwnership = {
            ownerId: this._id,
            partitionId: partitionIdToClaim,
            fullyQualifiedNamespace: this._fullyQualifiedNamespace,
            consumerGroup: this._consumerGroup,
            eventHubName: this._eventHubName,
            etag: previousPartitionOwnership ? previousPartitionOwnership.etag : undefined,
        };
        return partitionOwnership;
    }
    /*
     * Claim ownership of the given partition if it's available
     */
    async _claimOwnership(ownershipRequest, abortSignal) {
        if (abortSignal.aborted) {
            logger.verbose(`[${this._id}] Subscription was closed before claiming ownership of ${ownershipRequest.partitionId}.`);
            return;
        }
        logger.info(`[${this._id}] Attempting to claim ownership of partition ${ownershipRequest.partitionId}.`);
        try {
            const claimedOwnerships = await this._checkpointStore.claimOwnership([ownershipRequest]);
            // can happen if the partition was claimed out from underneath us - we shouldn't
            // attempt to spin up a processor.
            if (!claimedOwnerships.length) {
                return;
            }
            logger.info(`[${this._id}] Successfully claimed ownership of partition ${ownershipRequest.partitionId}.`);
            await this._startPump(ownershipRequest.partitionId, abortSignal);
        }
        catch (err) {
            logger.warning(`[${this.id}] Failed to claim ownership of partition ${ownershipRequest.partitionId}`);
            logErrorStackTrace(err);
            await this._handleSubscriptionError(err);
        }
    }
    async _startPump(partitionId, abortSignal) {
        if (abortSignal.aborted) {
            logger.verbose(`[${this._id}] The subscription was closed before starting to read from ${partitionId}.`);
            return;
        }
        if (this._pumpManager.isReceivingFromPartition(partitionId)) {
            logger.verbose(`[${this._id}] There is already an active partitionPump for partition "${partitionId}", skipping pump creation.`);
            return;
        }
        logger.verbose(`[${this._id}] [${partitionId}] Calling user-provided PartitionProcessorFactory.`);
        const partitionProcessor = new PartitionProcessor(this._subscriptionEventHandlers, this._checkpointStore, {
            fullyQualifiedNamespace: this._fullyQualifiedNamespace,
            eventHubName: this._eventHubName,
            consumerGroup: this._consumerGroup,
            partitionId: partitionId,
            eventProcessorId: this.id,
        });
        const eventPosition = await this._getStartingPosition(partitionId);
        await this._pumpManager.createPump(eventPosition, this._context, partitionProcessor, abortSignal);
        logger.verbose(`[${this._id}] PartitionPump created successfully.`);
    }
    async _getStartingPosition(partitionIdToClaim) {
        const availableCheckpoints = await this._checkpointStore.listCheckpoints(this._fullyQualifiedNamespace, this._eventHubName, this._consumerGroup);
        const validCheckpoints = availableCheckpoints.filter((chk) => chk.partitionId === partitionIdToClaim);
        if (validCheckpoints.length > 0) {
            return { offset: validCheckpoints[0].offset };
        }
        logger.verbose(`No checkpoint found for partition ${partitionIdToClaim}. Looking for fallback.`);
        return getStartPosition(partitionIdToClaim, this._processorOptions.startPosition);
    }
    async _runLoopForSinglePartition(partitionId, abortSignal) {
        while (!abortSignal.aborted) {
            try {
                await this._startPump(partitionId, abortSignal);
            }
            catch (err) {
                logger.warning(`[${this._id}] An error occured within the EventProcessor loop: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
                logErrorStackTrace(err);
                await this._handleSubscriptionError(err);
            }
            finally {
                // sleep for some time after which we can attempt to create a pump again.
                logger.verbose(`[${this._id}] Pausing the EventProcessor loop for ${this._loopIntervalInMs} ms.`);
                // swallow errors from delay since it's fine for delay to exit early
                await delayWithoutThrow(this._loopIntervalInMs, abortSignal);
            }
        }
        this._isRunning = false;
    }
    /**
     * Every loop to this method will result in this EventProcessor owning at most one new partition.
     *
     * The load is considered balanced when no active EventProcessor owns 2 partitions more than any other active
     * EventProcessor. Given that each invocation to this method results in ownership claim of at most one partition,
     * this algorithm converges gradually towards a steady state.
     *
     * When a new partition is claimed, this method is also responsible for starting a partition pump that creates an
     * EventHubConsumer for processing events from that partition.
     */
    async _runLoopWithLoadBalancing(loadBalancingStrategy, abortSignal) {
        let cancelLoopResolver;
        // This provides a mechanism for exiting the loop early
        // if the subscription has had `close` called.
        const cancelLoopPromise = new Promise((resolve) => {
            cancelLoopResolver = resolve;
            if (abortSignal.aborted) {
                resolve();
                return;
            }
            abortSignal.addEventListener("abort", resolve);
        });
        // Periodically check if any partitions need to be claimed and claim them.
        while (!abortSignal.aborted) {
            const iterationStartTimeInMs = Date.now();
            try {
                const { partitionIds } = await this._context.managementSession.getEventHubProperties({
                    abortSignal: abortSignal,
                });
                await this._performLoadBalancing(loadBalancingStrategy, partitionIds, abortSignal);
            }
            catch (err) {
                logger.warning(`[${this._id}] An error occured within the EventProcessor loop: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
                logErrorStackTrace(err);
                // Protect against the scenario where the user awaits on subscription.close() from inside processError.
                await Promise.race([this._handleSubscriptionError(err), cancelLoopPromise]);
            }
            finally {
                // Sleep for some time, then continue the loop.
                const iterationDeltaInMs = Date.now() - iterationStartTimeInMs;
                const delayDurationInMs = Math.max(this._loopIntervalInMs - iterationDeltaInMs, 0);
                logger.verbose(`[${this._id}] Pausing the EventProcessor loop for ${delayDurationInMs} ms.`);
                // Swallow the error since it's fine to exit early from the delay.
                await delayWithoutThrow(delayDurationInMs, abortSignal);
            }
        }
        if (cancelLoopResolver) {
            abortSignal.removeEventListener("abort", cancelLoopResolver);
        }
        this._isRunning = false;
    }
    async _performLoadBalancing(loadBalancingStrategy, partitionIds, abortSignal) {
        if (abortSignal.aborted)
            throw new abortController.AbortError("The operation was aborted.");
        // Retrieve current partition ownership details from the datastore.
        const partitionOwnership = await this._checkpointStore.listOwnership(this._fullyQualifiedNamespace, this._eventHubName, this._consumerGroup);
        if (abortSignal.aborted)
            throw new abortController.AbortError("The operation was aborted.");
        const partitionOwnershipMap = new Map();
        const nonAbandonedPartitionOwnershipMap = new Map();
        const partitionsToRenew = [];
        // Separate abandoned ownerships from claimed ownerships.
        // We only want to pass active partition ownerships to the
        // load balancer, but we need to hold onto the abandoned
        // partition ownerships because we need the etag to claim them.
        for (const ownership of partitionOwnership) {
            partitionOwnershipMap.set(ownership.partitionId, ownership);
            if (!isAbandoned(ownership)) {
                nonAbandonedPartitionOwnershipMap.set(ownership.partitionId, ownership);
            }
            if (ownership.ownerId === this._id &&
                this._pumpManager.isReceivingFromPartition(ownership.partitionId)) {
                partitionsToRenew.push(ownership.partitionId);
            }
        }
        // Pass the list of all the partition ids and the collection of claimed partition ownerships
        // to the load balance strategy.
        // The load balancing strategy only needs to know the full list of partitions,
        // and which of those are currently claimed.
        // Since abandoned partitions are no longer claimed, we exclude them.
        const partitionsToClaim = loadBalancingStrategy.getPartitionsToCliam(this._id, nonAbandonedPartitionOwnershipMap, partitionIds);
        partitionsToClaim.push(...partitionsToRenew);
        const uniquePartitionsToClaim = new Set(partitionsToClaim);
        for (const partitionToClaim of uniquePartitionsToClaim) {
            const partitionOwnershipRequest = this._createPartitionOwnershipRequest(partitionOwnershipMap, partitionToClaim);
            await this._claimOwnership(partitionOwnershipRequest, abortSignal);
        }
    }
    /**
     * This is called when there are errors that are not specific to a partition (ex: load balancing)
     */
    async _handleSubscriptionError(err) {
        // filter out any internal "expected" errors
        if (err.name === "AbortError") {
            return;
        }
        if (this._subscriptionEventHandlers.processError) {
            try {
                await this._subscriptionEventHandlers.processError(err, {
                    fullyQualifiedNamespace: this._fullyQualifiedNamespace,
                    eventHubName: this._eventHubName,
                    consumerGroup: this._consumerGroup,
                    partitionId: "",
                    updateCheckpoint: async () => {
                        /* no-op */
                    },
                });
            }
            catch (errorFromUser) {
                logger.verbose(`[${this._id}] An error was thrown from the user's processError handler: ${errorFromUser}`);
            }
        }
    }
    /**
     * Starts the `EventProcessor`. Based on the number of instances of `EventProcessor` that are running for the
     * same consumer group, the partitions are distributed among these instances to process events.
     *
     * For each partition, the user provided `PartitionProcessor` is instantiated.
     *
     * Subsequent calls to start will be ignored if this event processor is already running.
     * Calling `start()` after `stop()` is called will restart this event processor.
     *
     */
    start() {
        if (this._isRunning) {
            logger.verbose(`[${this._id}] Attempted to start an already running EventProcessor.`);
            return;
        }
        this._isRunning = true;
        this._abortController = new abortController.AbortController();
        logger.verbose(`[${this._id}] Starting an EventProcessor.`);
        if (this._processingTarget) {
            logger.verbose(`[${this._id}] Single partition target: ${this._processingTarget}`);
            this._loopTask = this._runLoopForSinglePartition(this._processingTarget, this._abortController.signal);
        }
        else {
            logger.verbose(`[${this._id}] Multiple partitions, using load balancer`);
            this._loopTask = this._runLoopWithLoadBalancing(this._loadBalancingStrategy, this._abortController.signal);
        }
    }
    isRunning() {
        return this._isRunning;
    }
    /**
     * Stops processing events for all partitions owned by this event processor.
     * All `PartitionProcessor` will be shutdown and any open resources will be closed.
     *
     * Subsequent calls to stop will be ignored if the event processor is not running.
     *
     */
    async stop() {
        logger.verbose(`[${this._id}] Stopping an EventProcessor.`);
        if (this._abortController) {
            // cancel the event processor loop
            this._abortController.abort();
        }
        try {
            // remove all existing pumps
            await this._pumpManager.removeAllPumps(exports.CloseReason.Shutdown);
            // waits for the event processor loop to complete
            // will complete immediately if _loopTask is undefined
            if (this._loopTask) {
                await this._loopTask;
            }
        }
        catch (err) {
            logger.verbose(`[${this._id}] An error occured while stopping the EventProcessor: ${err}`);
        }
        finally {
            logger.verbose(`[${this._id}] EventProcessor stopped.`);
        }
        if (this._processingTarget) {
            logger.verbose(`[${this._id}] No partitions owned, skipping abandoning.`);
        }
        else {
            await this.abandonPartitionOwnerships();
        }
    }
    async abandonPartitionOwnerships() {
        logger.verbose(`[${this._id}] Abandoning owned partitions`);
        const allOwnerships = await this._checkpointStore.listOwnership(this._fullyQualifiedNamespace, this._eventHubName, this._consumerGroup);
        const ourOwnerships = allOwnerships.filter((ownership) => ownership.ownerId === this._id);
        // unclaim any partitions that we currently own
        for (const ownership of ourOwnerships) {
            ownership.ownerId = "";
        }
        return this._checkpointStore.claimOwnership(ourOwnerships);
    }
}
function isAbandoned(ownership) {
    return ownership.ownerId === "";
}
function getStartPosition(partitionIdToClaim, startPositions) {
    if (startPositions == null) {
        return latestEventPosition;
    }
    if (isEventPosition(startPositions)) {
        return startPositions;
    }
    const startPosition = startPositions[partitionIdToClaim];
    if (startPosition == null) {
        return latestEventPosition;
    }
    return startPosition;
}

// Copyright (c) Microsoft Corporation.
/**
 * Parses given connection string into the different properties applicable to Azure Event Hubs.
 * The properties are useful to then construct an EventHubProducerClient or an EventHubConsumerClient.
 * @param connectionString - The connection string associated with the Shared Access Policy created
 * for the Event Hubs namespace.
 */
function parseEventHubConnectionString(connectionString) {
    const parsedResult = coreAmqp.parseConnectionString(connectionString);
    validateProperties(parsedResult.Endpoint, parsedResult.SharedAccessSignature, parsedResult.SharedAccessKey, parsedResult.SharedAccessKeyName);
    const output = {
        fullyQualifiedNamespace: (parsedResult.Endpoint.match(".*://([^/]*)") || [])[1],
        endpoint: parsedResult.Endpoint,
    };
    if (parsedResult.EntityPath) {
        output.eventHubName = parsedResult.EntityPath;
    }
    if (parsedResult.SharedAccessSignature) {
        output.sharedAccessSignature = parsedResult.SharedAccessSignature;
    }
    if (parsedResult.SharedAccessKey && parsedResult.SharedAccessKeyName) {
        output.sharedAccessKey = parsedResult.SharedAccessKey;
        output.sharedAccessKeyName = parsedResult.SharedAccessKeyName;
    }
    return output;
}
/**
 * @internal
 */
function validateProperties(endpoint, sharedAccessSignature, sharedAccessKey, sharedAccessKeyName) {
    if (!endpoint) {
        throw new Error("Connection string should have an Endpoint key.");
    }
    if (sharedAccessSignature) {
        if (sharedAccessKey || sharedAccessKeyName) {
            throw new Error("Connection string cannot have both SharedAccessSignature and SharedAccessKey keys.");
        }
    }
    else if (sharedAccessKey && !sharedAccessKeyName) {
        throw new Error("Connection string with SharedAccessKey should have SharedAccessKeyName.");
    }
    else if (!sharedAccessKey && sharedAccessKeyName) {
        throw new Error("Connection string with SharedAccessKeyName should have SharedAccessKey as well.");
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 * Descibes the EventHubs Management Client that talks
 * to the $management endpoint over AMQP connection.
 */
class ManagementClient extends LinkEntity {
    /**
     * Instantiates the management client.
     * @param context - The connection context.
     * @param address - The address for the management endpoint. For IotHub it will be
     * `/messages/events/$management`.
     */
    constructor(context, options) {
        super(context, {
            address: options && options.address ? options.address : coreAmqp.Constants.management,
            audience: options && options.audience ? options.audience : context.config.getManagementAudience(),
        });
        this.managementLock = `${coreAmqp.Constants.managementRequestKey}-${uuid.v4()}`;
        /**
         * The reply to Guid for the management client.
         */
        this.replyTo = uuid.v4();
        this._context = context;
        this.entityPath = context.config.entityPath;
    }
    /**
     * Gets the security token for the management application properties.
     * @internal
     */
    async getSecurityToken() {
        if (coreAmqp.isSasTokenProvider(this._context.tokenCredential)) {
            // the security_token has the $management address removed from the end of the audience
            // expected audience: sb://fully.qualified.namespace/event-hub-name/$management
            const audienceParts = this.audience.split("/");
            // for management links, address should be '$management'
            if (audienceParts[audienceParts.length - 1] === this.address) {
                audienceParts.pop();
            }
            const audience = audienceParts.join("/");
            return this._context.tokenCredential.getToken(audience);
        }
        // aad credentials use the aad scope
        return this._context.tokenCredential.getToken(coreAmqp.Constants.aadEventHubsScope);
    }
    /**
     * Provides the eventhub runtime information.
     */
    async getEventHubProperties(options = {}) {
        throwErrorIfConnectionClosed(this._context);
        return tracingClient.withSpan("ManagementClient.getEventHubProperties", options, async (updatedOptions) => {
            try {
                const securityToken = await this.getSecurityToken();
                const request = {
                    body: Buffer.from(JSON.stringify([])),
                    message_id: uuid.v4(),
                    reply_to: this.replyTo,
                    application_properties: {
                        operation: coreAmqp.Constants.readOperation,
                        name: this.entityPath,
                        type: `${coreAmqp.Constants.vendorString}:${coreAmqp.Constants.eventHub}`,
                        security_token: securityToken === null || securityToken === void 0 ? void 0 : securityToken.token,
                    },
                };
                const info = await this._makeManagementRequest(request, Object.assign(Object.assign({}, updatedOptions), { requestName: "getHubRuntimeInformation" }));
                const runtimeInfo = {
                    name: info.name,
                    createdOn: new Date(info.created_at),
                    partitionIds: info.partition_ids,
                };
                logger.verbose("[%s] The hub runtime info is: %O", this._context.connectionId, runtimeInfo);
                return runtimeInfo;
            }
            catch (error) {
                logger.warning(`An error occurred while getting the hub runtime information: ${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
                logErrorStackTrace(error);
                throw error;
            }
        }, toSpanOptions(this._context.config));
    }
    /**
     * Provides information about the specified partition.
     * @param partitionId - Partition ID for which partition information is required.
     */
    async getPartitionProperties(partitionId, options = {}) {
        throwErrorIfConnectionClosed(this._context);
        throwTypeErrorIfParameterMissing(this._context.connectionId, "getPartitionProperties", "partitionId", partitionId);
        partitionId = String(partitionId);
        return tracingClient.withSpan("ManagementClient.getPartitionProperties", options, async (updatedOptions) => {
            try {
                const securityToken = await this.getSecurityToken();
                const request = {
                    body: Buffer.from(JSON.stringify([])),
                    message_id: uuid.v4(),
                    reply_to: this.replyTo,
                    application_properties: {
                        operation: coreAmqp.Constants.readOperation,
                        name: this.entityPath,
                        type: `${coreAmqp.Constants.vendorString}:${coreAmqp.Constants.partition}`,
                        partition: `${partitionId}`,
                        security_token: securityToken === null || securityToken === void 0 ? void 0 : securityToken.token,
                    },
                };
                const info = await this._makeManagementRequest(request, Object.assign(Object.assign({}, updatedOptions), { requestName: "getPartitionInformation" }));
                const partitionInfo = {
                    beginningSequenceNumber: info.begin_sequence_number,
                    eventHubName: info.name,
                    lastEnqueuedOffset: info.last_enqueued_offset,
                    lastEnqueuedOnUtc: new Date(info.last_enqueued_time_utc),
                    lastEnqueuedSequenceNumber: info.last_enqueued_sequence_number,
                    partitionId: info.partition,
                    isEmpty: info.is_partition_empty,
                };
                logger.verbose("[%s] The partition info is: %O.", this._context.connectionId, partitionInfo);
                return partitionInfo;
            }
            catch (error) {
                logger.warning(`An error occurred while getting the partition information: ${error === null || error === void 0 ? void 0 : error.name}: ${error === null || error === void 0 ? void 0 : error.message}`);
                logErrorStackTrace(error);
                throw error;
            }
        }, toSpanOptions(this._context.config));
    }
    /**
     * Closes the AMQP management session to the Event Hub for this client,
     * returning a promise that will be resolved when disconnection is completed.
     */
    async close() {
        try {
            // Always clear the timeout, as the isOpen check may report
            // false without ever having cleared the timeout otherwise.
            clearTimeout(this._tokenRenewalTimer);
            if (this._isMgmtRequestResponseLinkOpen()) {
                const mgmtLink = this._mgmtReqResLink;
                this._mgmtReqResLink = undefined;
                await mgmtLink.close();
                logger.info("Successfully closed the management session.");
            }
        }
        catch (err) {
            const msg = `An error occurred while closing the management session: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`;
            logger.warning(msg);
            logErrorStackTrace(err);
            throw new Error(msg);
        }
    }
    async _init({ abortSignal, timeoutInMs, }) {
        try {
            if (!this._isMgmtRequestResponseLinkOpen()) {
                // Wait for the connectionContext to be ready to open the link.
                await this._context.readyToOpenLink();
                await this._negotiateClaim({ setTokenRenewal: false, abortSignal, timeoutInMs });
                const rxopt = {
                    source: { address: this.address },
                    name: this.replyTo,
                    target: { address: this.replyTo },
                    onSessionError: (context) => {
                        const id = context.connection.options.id;
                        const ehError = coreAmqp.translate(context.session.error);
                        logger.verbose("[%s] An error occurred on the session for request/response links for " +
                            "$management: %O", id, ehError);
                    },
                };
                const sropt = {
                    target: { address: this.address },
                };
                logger.verbose("[%s] Creating sender/receiver links on a session for $management endpoint with " +
                    "srOpts: %o, receiverOpts: %O.", this._context.connectionId, sropt, rxopt);
                this._mgmtReqResLink = await coreAmqp.RequestResponseLink.create(this._context.connection, sropt, rxopt, { abortSignal });
                this._mgmtReqResLink.sender.on(rheaPromise.SenderEvents.senderError, (context) => {
                    const id = context.connection.options.id;
                    const ehError = coreAmqp.translate(context.sender.error);
                    logger.verbose("[%s] An error occurred on the $management sender link.. %O", id, ehError);
                });
                this._mgmtReqResLink.receiver.on(rheaPromise.ReceiverEvents.receiverError, (context) => {
                    const id = context.connection.options.id;
                    const ehError = coreAmqp.translate(context.receiver.error);
                    logger.verbose("[%s] An error occurred on the $management receiver link.. %O", id, ehError);
                });
                logger.verbose("[%s] Created sender '%s' and receiver '%s' links for $management endpoint.", this._context.connectionId, this._mgmtReqResLink.sender.name, this._mgmtReqResLink.receiver.name);
                await this._ensureTokenRenewal();
            }
        }
        catch (err) {
            const translatedError = coreAmqp.translate(err);
            logger.warning(`[${this._context.connectionId}] An error occured while establishing the $management links: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Helper method to make the management request
     * @param request - The AMQP message to send
     * @param options - The options to use when sending a request over a $management link
     */
    async _makeManagementRequest(request, options = {}) {
        const retryOptions = options.retryOptions || {};
        try {
            const abortSignal = options && options.abortSignal;
            const sendOperationPromise = async () => {
                let count = 0;
                const retryTimeoutInMs = getRetryAttemptTimeoutInMs(options.retryOptions);
                let timeTakenByInit = 0;
                if (!this._isMgmtRequestResponseLinkOpen()) {
                    logger.verbose("[%s] Acquiring lock to get the management req res link.", this._context.connectionId);
                    const initOperationStartTime = Date.now();
                    try {
                        await coreAmqp.defaultCancellableLock.acquire(this.managementLock, () => {
                            const acquireLockEndTime = Date.now();
                            const timeoutInMs = retryTimeoutInMs - (acquireLockEndTime - initOperationStartTime);
                            return this._init({ abortSignal, timeoutInMs });
                        }, { abortSignal, timeoutInMs: retryTimeoutInMs });
                    }
                    catch (err) {
                        const translatedError = coreAmqp.translate(err);
                        logger.warning("[%s] An error occurred while creating the management link %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
                        logErrorStackTrace(translatedError);
                        throw translatedError;
                    }
                    timeTakenByInit = Date.now() - initOperationStartTime;
                }
                const remainingOperationTimeoutInMs = retryTimeoutInMs - timeTakenByInit;
                const sendRequestOptions = {
                    abortSignal: options.abortSignal,
                    requestName: options.requestName,
                    timeoutInMs: remainingOperationTimeoutInMs,
                };
                count++;
                if (count !== 1) {
                    // Generate a new message_id every time after the first attempt
                    request.message_id = rheaPromise.generate_uuid();
                }
                else if (!request.message_id) {
                    // Set the message_id in the first attempt only if it is not set
                    request.message_id = rheaPromise.generate_uuid();
                }
                return this._mgmtReqResLink.sendRequest(request, sendRequestOptions);
            };
            const config = Object.defineProperties({
                operation: sendOperationPromise,
                operationType: coreAmqp.RetryOperationType.management,
                abortSignal: abortSignal,
                retryOptions: retryOptions,
            }, {
                connectionId: {
                    enumerable: true,
                    get: () => {
                        return this._context.connectionId;
                    },
                },
            });
            return (await coreAmqp.retry(config)).body;
        }
        catch (err) {
            const translatedError = coreAmqp.translate(err);
            logger.warning("[%s] An error occurred during send on management request-response link with address '%s': %s", this._context.connectionId, this.address, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    _isMgmtRequestResponseLinkOpen() {
        return this._mgmtReqResLink && this._mgmtReqResLink.isOpen();
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * Parses the host, hostname, and port from an endpoint.
 * @param endpoint - And endpoint to parse.
 * @internal
 */
function parseEndpoint(endpoint) {
    const hostMatch = endpoint.match(/.*:\/\/([^/]*)/);
    if (!hostMatch) {
        throw new TypeError(`Invalid endpoint missing host: ${endpoint}`);
    }
    const [, host] = hostMatch;
    const [hostname, port] = host.split(":");
    return { host, hostname, port };
}

// Copyright (c) Microsoft Corporation.
/**
 * Describes the connection config object that is created after parsing an EventHub connection
 * string. It also provides some convenience methods for getting the address and audience for
 * different entities.
 * @internal
 */
// eslint-disable-next-line @typescript-eslint/no-redeclare -- renaming constant would be a breaking change.
const EventHubConnectionConfig = {
    /**
     * Creates the connection config.
     * @param connectionString - The connection string for a given service like
     * EventHub/ServiceBus.
     * @param path - The name/path of the entity (hub name) to which the
     * connection needs to happen. This will override the EntityPath in the connectionString
     * if present.
     * @returns EventHubConnectionConfig
     */
    create(connectionString, path) {
        const config = coreAmqp.ConnectionConfig.create(connectionString, path);
        if (!config.entityPath) {
            throw new TypeError(`Either provide "path" or the "connectionString": "${connectionString}", ` +
                `must contain EntityPath="<path-to-the-entity>".`);
        }
        return EventHubConnectionConfig.createFromConnectionConfig(config);
    },
    /**
     * Creates an EventHubConnectionConfig from the provided base ConnectionConfig.
     * @param config - The base connection config from which the EventHubConnectionConfig needs to be
     * created.
     * @returns EventHubConnectionConfig
     */
    createFromConnectionConfig(config) {
        coreAmqp.ConnectionConfig.validate(config, { isEntityPathRequired: true });
        config.getManagementAudience = () => {
            return `${config.endpoint}${config.entityPath}/$management`;
        };
        config.getManagementAddress = () => {
            return `${config.entityPath}/$management`;
        };
        config.getSenderAudience = (partitionId) => {
            if (partitionId != undefined) {
                return `${config.endpoint}${config.entityPath}/Partitions/${partitionId}`;
            }
            else {
                return `${config.endpoint}${config.entityPath}`;
            }
        };
        config.getSenderAddress = (partitionId) => {
            if (partitionId != undefined) {
                return `${config.entityPath}/Partitions/${partitionId}`;
            }
            else {
                return `${config.entityPath}`;
            }
        };
        config.getReceiverAudience = (partitionId, consumergroup) => {
            if (!consumergroup)
                consumergroup = "$default";
            return (`${config.endpoint}${config.entityPath}/ConsumerGroups/${consumergroup}/` +
                `Partitions/${partitionId}`);
        };
        config.getReceiverAddress = (partitionId, consumergroup) => {
            if (!consumergroup)
                consumergroup = "$default";
            return `${config.entityPath}/ConsumerGroups/${consumergroup}/Partitions/${partitionId}`;
        };
        return config;
    },
    /**
     * Updates the provided EventHubConnectionConfig to use the custom endpoint address.
     * @param config - An existing connection configuration to be updated.
     * @param customEndpointAddress - The custom endpoint address to use.
     */
    setCustomEndpointAddress(config, customEndpointAddress) {
        // The amqpHostname should match the host prior to using the custom endpoint.
        config.amqpHostname = config.host;
        const { hostname, port } = parseEndpoint(customEndpointAddress);
        // Since we specify the port separately, set host to the customEndpointAddress hostname.
        config.host = hostname;
        if (port) {
            config.port = parseInt(port, 10);
        }
    },
    /**
     * Validates the properties of connection config.
     * @param config - The connection config to be validated.
     * @returns void
     */
    validate(config) {
        return coreAmqp.ConnectionConfig.validate(config, { isEntityPathRequired: true });
    },
};

// Copyright (c) Microsoft Corporation.
/**
 * Returns information about the platform this function is being run on.
 * @internal
 */
function getRuntimeInfo() {
    return `NODE-VERSION ${process.version}; ${os__namespace.type()} ${os__namespace.release()}`;
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 */
var ConnectionContext;
(function (ConnectionContext) {
    /**
     * The user agent string for the EventHubs client.
     * See guideline at https://github.com/Azure/azure-sdk/blob/main/docs/design/Telemetry.mdk
     */
    const userAgent = `azsdk-js-azureeventhubs/${packageJsonInfo.version} (${getRuntimeInfo()})`;
    function getUserAgent(options) {
        const finalUserAgent = options.userAgent ? `${userAgent},${options.userAgent}` : userAgent;
        if (finalUserAgent.length > coreAmqp.Constants.maxUserAgentLength) {
            throw new Error(`The user-agent string cannot be more than ${coreAmqp.Constants.maxUserAgentLength} characters in length.` +
                `The given user-agent string is: ${finalUserAgent} with length: ${finalUserAgent.length}`);
        }
        return finalUserAgent;
    }
    ConnectionContext.getUserAgent = getUserAgent;
    function create(config, tokenCredential, options) {
        if (!options)
            options = {};
        config.webSocket = options.webSocketOptions && options.webSocketOptions.webSocket;
        config.webSocketEndpointPath = "$servicebus/websocket";
        config.webSocketConstructorOptions =
            options.webSocketOptions && options.webSocketOptions.webSocketConstructorOptions;
        const parameters = {
            config: config,
            // re-enabling this will be a post-GA discussion.
            // dataTransformer: options.dataTransformer,
            isEntityPathRequired: true,
            connectionProperties: {
                product: "MSJSClient",
                userAgent: getUserAgent(options),
                version: packageJsonInfo.version,
            },
        };
        // Let us create the base context and then add EventHub specific ConnectionContext properties.
        const connectionContext = coreAmqp.ConnectionContextBase.create(parameters);
        connectionContext.tokenCredential = tokenCredential;
        connectionContext.wasConnectionCloseCalled = false;
        connectionContext.senders = {};
        connectionContext.receivers = {};
        const mOptions = {
            address: options.managementSessionAddress,
            audience: options.managementSessionAudience,
        };
        connectionContext.managementSession = new ManagementClient(connectionContext, mOptions);
        let waitForConnectionRefreshResolve;
        let waitForConnectionRefreshPromise;
        Object.assign(connectionContext, {
            isConnectionClosing() {
                // When the connection is not open, but the remote end is open,
                // then the rhea connection is in the process of terminating.
                return Boolean(!this.connection.isOpen() && this.connection.isRemoteOpen());
            },
            async readyToOpenLink() {
                // Check that the connection isn't in the process of closing.
                // This can happen when the idle timeout has been reached but
                // the underlying socket is waiting to be destroyed.
                if (this.isConnectionClosing()) {
                    // Wait for the disconnected event that indicates the underlying socket has closed.
                    await this.waitForDisconnectedEvent();
                }
                // Wait for the connection to be reset.
                await this.waitForConnectionReset();
            },
            waitForDisconnectedEvent() {
                return new Promise((resolve) => {
                    logger.verbose(`[${this.connectionId}] Attempting to reinitialize connection` +
                        ` but the connection is in the process of closing.` +
                        ` Waiting for the disconnect event before continuing.`);
                    this.connection.once(rheaPromise.ConnectionEvents.disconnected, resolve);
                });
            },
            waitForConnectionReset() {
                // Check if the connection is currently in the process of disconnecting.
                if (waitForConnectionRefreshPromise) {
                    return waitForConnectionRefreshPromise;
                }
                return Promise.resolve();
            },
            async close() {
                var _a;
                try {
                    if (this.connection.isOpen()) {
                        // Close all the senders.
                        await Promise.all(Object.keys(connectionContext.senders).map((name) => {
                            var _a;
                            return (_a = connectionContext.senders[name]) === null || _a === void 0 ? void 0 : _a.close().catch(() => {
                                /* error already logged, swallow it here */
                            });
                        }));
                        // Close all the receivers.
                        await Promise.all(Object.keys(connectionContext.receivers).map((name) => {
                            var _a;
                            return (_a = connectionContext.receivers[name]) === null || _a === void 0 ? void 0 : _a.close().catch(() => {
                                /* error already logged, swallow it here */
                            });
                        }));
                        // Close the cbs session;
                        await this.cbsSession.close();
                        // Close the management session
                        await ((_a = this.managementSession) === null || _a === void 0 ? void 0 : _a.close());
                        await this.connection.close();
                        this.wasConnectionCloseCalled = true;
                        logger.info("Closed the amqp connection '%s' on the client.", this.connectionId);
                    }
                }
                catch (err) {
                    const errorDescription = err instanceof Error ? `${err.name}: ${err.message}` : JSON.stringify(err);
                    logger.warning(`An error occurred while closing the connection "${this.connectionId}":\n${errorDescription}`);
                    logErrorStackTrace(err);
                    throw err;
                }
            },
        });
        // Define listeners to be added to the connection object for
        // "connection_open" and "connection_error" events.
        const onConnectionOpen = () => {
            connectionContext.wasConnectionCloseCalled = false;
            logger.verbose("[%s] setting 'wasConnectionCloseCalled' property of connection context to %s.", connectionContext.connection.id, connectionContext.wasConnectionCloseCalled);
        };
        const onDisconnected = async (context) => {
            var _a, _b;
            if (waitForConnectionRefreshPromise) {
                return;
            }
            waitForConnectionRefreshPromise = new Promise((resolve) => {
                waitForConnectionRefreshResolve = resolve;
            });
            try {
                logger.verbose("[%s] 'disconnected' event occurred on the amqp connection.", connectionContext.connection.id);
                if (context.connection && context.connection.error) {
                    logger.verbose("[%s] Accompanying error on the context.connection: %O", connectionContext.connection.id, context.connection && context.connection.error);
                }
                if (context.error) {
                    logger.verbose("[%s] Accompanying error on the context: %O", connectionContext.connection.id, context.error);
                }
                const state = {
                    wasConnectionCloseCalled: connectionContext.wasConnectionCloseCalled,
                    numSenders: Object.keys(connectionContext.senders).length,
                    numReceivers: Object.keys(connectionContext.receivers).length,
                };
                logger.verbose("[%s] Closing all open senders and receivers in the state: %O", connectionContext.connection.id, state);
                // Clear internal map maintained by rhea to avoid reconnecting of old links once the
                // connection is back up.
                connectionContext.connection.removeAllSessions();
                // Close the cbs session to ensure all the event handlers are released.
                await ((_a = connectionContext.cbsSession) === null || _a === void 0 ? void 0 : _a.close().catch(() => {
                    /* error already logged, swallow it here */
                }));
                // Close the management session to ensure all the event handlers are released.
                await ((_b = connectionContext.managementSession) === null || _b === void 0 ? void 0 : _b.close().catch(() => {
                    /* error already logged, swallow it here */
                }));
                // Close all senders and receivers to ensure clean up of timers & other resources.
                if (state.numSenders || state.numReceivers) {
                    await Promise.all(Object.keys(connectionContext.senders).map((name) => {
                        var _a;
                        return (_a = connectionContext.senders[name]) === null || _a === void 0 ? void 0 : _a.close().catch(() => {
                            /* error already logged, swallow it here */
                        });
                    }));
                    await Promise.all(Object.keys(connectionContext.receivers).map((name) => {
                        var _a;
                        return (_a = connectionContext.receivers[name]) === null || _a === void 0 ? void 0 : _a.close().catch(() => {
                            /* error already logged, swallow it here */
                        });
                    }));
                }
            }
            catch (err) {
                logger.verbose(`[${connectionContext.connectionId}] An error occurred while closing the connection in 'disconnected'. %O`, err);
            }
            try {
                await refreshConnection(connectionContext);
            }
            catch (err) {
                logger.verbose(`[${connectionContext.connectionId}] An error occurred while refreshing the connection in 'disconnected'. %O`, err);
            }
            finally {
                waitForConnectionRefreshResolve();
                waitForConnectionRefreshPromise = undefined;
            }
        };
        const protocolError = async (context) => {
            logger.verbose("[%s] 'protocol_error' event occurred on the amqp connection.", connectionContext.connection.id);
            if (context.connection && context.connection.error) {
                logger.verbose("[%s] Accompanying error on the context.connection: %O", connectionContext.connection.id, context.connection && context.connection.error);
            }
            if (context.error) {
                logger.verbose("[%s] Accompanying error on the context: %O", connectionContext.connection.id, context.error);
            }
        };
        const error = async (context) => {
            logger.verbose("[%s] 'error' event occurred on the amqp connection.", connectionContext.connection.id);
            if (context.connection && context.connection.error) {
                logger.verbose("[%s] Accompanying error on the context.connection: %O", connectionContext.connection.id, context.connection && context.connection.error);
            }
            if (context.error) {
                logger.verbose("[%s] Accompanying error on the context: %O", connectionContext.connection.id, context.error);
            }
        };
        function addConnectionListeners(connection) {
            // Add listeners on the connection object.
            connection.on(rheaPromise.ConnectionEvents.connectionOpen, onConnectionOpen);
            connection.on(rheaPromise.ConnectionEvents.disconnected, onDisconnected);
            connection.on(rheaPromise.ConnectionEvents.protocolError, protocolError);
            connection.on(rheaPromise.ConnectionEvents.error, error);
        }
        function cleanConnectionContext(context) {
            // Remove listeners from the connection object.
            context.connection.removeListener(rheaPromise.ConnectionEvents.connectionOpen, onConnectionOpen);
            context.connection.removeListener(rheaPromise.ConnectionEvents.disconnected, onDisconnected);
            context.connection.removeListener(rheaPromise.ConnectionEvents.protocolError, protocolError);
            context.connection.removeListener(rheaPromise.ConnectionEvents.error, error);
            // Close the connection
            return context.connection.close();
        }
        async function refreshConnection(context) {
            const originalConnectionId = context.connectionId;
            try {
                await cleanConnectionContext(context);
            }
            catch (err) {
                logger.verbose(`[${context.connectionId}] There was an error closing the connection before reconnecting: %O`, err);
            }
            // Create a new connection, id, locks, and cbs client.
            context.refreshConnection();
            addConnectionListeners(context.connection);
            logger.verbose(`The connection "${originalConnectionId}" has been updated to "${context.connectionId}".`);
        }
        addConnectionListeners(connectionContext.connection);
        logger.verbose("[%s] Created connection context successfully.", connectionContext.connectionId);
        return connectionContext;
    }
    ConnectionContext.create = create;
})(ConnectionContext || (ConnectionContext = {}));
/**
 * Helper method to create a ConnectionContext from the input passed to either
 * EventHubProducerClient or EventHubConsumerClient constructors
 *
 * @internal
 */
function createConnectionContext(hostOrConnectionString, eventHubNameOrOptions, credentialOrOptions, options) {
    let connectionString;
    let config;
    let credential;
    hostOrConnectionString = String(hostOrConnectionString);
    if (!isCredential(credentialOrOptions)) {
        const parsedCS = parseEventHubConnectionString(hostOrConnectionString);
        if (!(parsedCS.eventHubName ||
            (typeof eventHubNameOrOptions === "string" && eventHubNameOrOptions))) {
            throw new TypeError(`Either provide "eventHubName" or the "connectionString": "${hostOrConnectionString}", ` +
                `must contain "EntityPath=<your-event-hub-name>".`);
        }
        if (parsedCS.eventHubName &&
            typeof eventHubNameOrOptions === "string" &&
            eventHubNameOrOptions &&
            parsedCS.eventHubName !== eventHubNameOrOptions) {
            throw new TypeError(`The entity path "${parsedCS.eventHubName}" in connectionString: "${hostOrConnectionString}" ` +
                `doesn't match with eventHubName: "${eventHubNameOrOptions}".`);
        }
        connectionString = hostOrConnectionString;
        if (typeof eventHubNameOrOptions !== "string") {
            // connectionstring and/or options were passed to constructor
            config = EventHubConnectionConfig.create(connectionString);
            options = eventHubNameOrOptions;
        }
        else {
            // connectionstring, eventHubName and/or options were passed to constructor
            const eventHubName = eventHubNameOrOptions;
            config = EventHubConnectionConfig.create(connectionString, eventHubName);
            options = credentialOrOptions;
        }
        const parsed = parseEventHubConnectionString(connectionString);
        // Since connectionString was passed, create a TokenProvider.
        credential = coreAmqp.createSasTokenProvider(parsed);
    }
    else {
        // host, eventHubName, a TokenCredential and/or options were passed to constructor
        const eventHubName = eventHubNameOrOptions;
        let host = hostOrConnectionString;
        if (coreAuth.isNamedKeyCredential(credentialOrOptions) || coreAuth.isSASCredential(credentialOrOptions)) {
            credential = coreAmqp.createSasTokenProvider(credentialOrOptions);
        }
        else {
            credential = credentialOrOptions;
        }
        if (!eventHubName) {
            throw new TypeError(`"eventHubName" is missing`);
        }
        if (!host.endsWith("/"))
            host += "/";
        connectionString = `Endpoint=sb://${host};SharedAccessKeyName=defaultKeyName;SharedAccessKey=defaultKeyValue;EntityPath=${eventHubName}`;
        config = EventHubConnectionConfig.create(connectionString);
    }
    if (options === null || options === void 0 ? void 0 : options.customEndpointAddress) {
        EventHubConnectionConfig.setCustomEndpointAddress(config, options.customEndpointAddress);
    }
    coreAmqp.ConnectionConfig.validate(config);
    return ConnectionContext.create(config, credential, options);
}

// Copyright (c) Microsoft Corporation.
/**
 * This method will create a new map of partition id and PartitionOwnership containing only those partitions
 * that are actively owned.
 * All entries in the original map that haven't been modified for a duration of time greater than the allowed
 * inactivity time limit are assumed to be owned by dead event processors.
 * These will not be included in the map returned by this method.
 *
 * @param partitionOwnershipMap - The existing PartitionOwnerships mapped by partition.
 * @param expirationIntervalInMs - The length of time a PartitionOwnership claim is valid.
 */
function getActivePartitionOwnerships(partitionOwnershipMap, expirationIntervalInMs) {
    const activePartitionOwnershipMap = new Map();
    partitionOwnershipMap.forEach((partitionOwnership, partitionId) => {
        // If lastModifiedtimeInMs is missing, assume it is inactive.
        if (typeof partitionOwnership.lastModifiedTimeInMs === "undefined" ||
            partitionOwnership.lastModifiedTimeInMs === null) {
            return;
        }
        const timeSincePartitionClaimed = Date.now() - partitionOwnership.lastModifiedTimeInMs;
        if (timeSincePartitionClaimed < expirationIntervalInMs && partitionOwnership.ownerId) {
            activePartitionOwnershipMap.set(partitionId, partitionOwnership);
        }
    });
    return activePartitionOwnershipMap;
}
/**
 * Calculates the minimum number of partitions each EventProcessor should own,
 * and the number of EventProcessors that should have an extra partition assigned.
 * @param ownerToOwnershipMap - The current ownerships for partitions.
 * @param partitionIds - The full list of the Event Hub's partition ids.
 * @internal
 */
function calculateBalancedLoadCounts(ownerToOwnershipMap, partitionIds) {
    // Calculate the minimum number of partitions every EventProcessor should own when the load
    // is evenly distributed.
    const minPartitionsPerOwner = Math.floor(partitionIds.length / ownerToOwnershipMap.size);
    // If the number of partitions in the Event Hub is not evenly divisible by the number of active
    // EventProcesrrors, some EventProcessors may own 1 partition in addition to the minimum when the
    // load is balanced.
    // Calculate the number of EventProcessors that can own an additional partition.
    const requiredNumberOfOwnersWithExtraPartition = partitionIds.length % ownerToOwnershipMap.size;
    return {
        minPartitionsPerOwner,
        requiredNumberOfOwnersWithExtraPartition,
    };
}
/**
 * Counts the EventProcessors and tallies them by type.
 *
 * To be in balance we need to make sure that each EventProcessor is only consuming
 * their fair share.
 *
 * When the partitions are divvied up we will sometimes end up with some EventProcessors
 * that will have 1 more partition than others.
 * This can happen if the number of partitions is not evenly divisible by the number of EventProcessors.
 *
 * So this function largely exists to support isLoadBalanced() and
 * shouldOwnMorePartitions(), both of which depend on knowing if our current list
 * of EventProcessors is actually in the proper state.
 *
 * @param minPartitionsPerOwner - The number of required partitions per EventProcessor.
 * @param ownerToOwnershipMap - The current ownerships for partitions.
 * @internal
 */
function getEventProcessorCounts(minPartitionsPerOwner, ownerToOwnershipMap) {
    const counts = {
        haveRequiredPartitions: 0,
        haveAdditionalPartition: 0,
        haveTooManyPartitions: 0,
    };
    for (const ownershipList of ownerToOwnershipMap.values()) {
        const numberOfPartitions = ownershipList.length;
        // there are basically three kinds of partition counts
        // for a processor:
        if (numberOfPartitions === minPartitionsPerOwner) {
            // 1. Has _exactly_ the required number of partitions
            counts.haveRequiredPartitions++;
        }
        else if (numberOfPartitions === minPartitionsPerOwner + 1) {
            // 2. Has the required number plus one extra (correct in cases)
            // where the # of partitions is not evenly divisible by the
            // number of processors.
            counts.haveAdditionalPartition++;
        }
        else if (numberOfPartitions > minPartitionsPerOwner + 1) {
            // 3. has more than the possible # of partitions required
            counts.haveTooManyPartitions++;
        }
    }
    return counts;
}
/**
 * Validates that we are currently in a balanced state - all EventProcessors own the
 * minimum required number of partitions (and additional partitions, if the # of partitions
 * is not evenly divisible by the # of EventProcessors).
 *
 * @param requiredNumberOfOwnersWithExtraPartition - The # of EventProcessors that process an additional partition, in addition to the required minimum.
 * @param totalExpectedProcessors - The total # of EventProcessors we expect.
 * @param eventProcessorCounts - EventProcessor counts, grouped by criteria.
 * @internal
 */
function isLoadBalanced(requiredNumberOfOwnersWithExtraPartition, totalExpectedEventProcessors, { haveAdditionalPartition, haveRequiredPartitions }) {
    return (haveAdditionalPartition === requiredNumberOfOwnersWithExtraPartition &&
        haveRequiredPartitions + haveAdditionalPartition === totalExpectedEventProcessors);
}
/**
 * Determines the number of new partitions to claim for this particular processor.
 *
 * @param minRequired - The minimum required number of partitions.
 * @param requiredNumberOfOwnersWithExtraPartition - The current number of processors that should have an additional partition.
 * @param numPartitionsOwnedByUs - The number of partitions we currently own.
 * @param eventProcessorCounts - Processors, grouped by criteria.
 * @internal
 */
function getNumberOfPartitionsToClaim(minRequiredPartitionCount, requiredNumberOfOwnersWithExtraPartition, numPartitionsOwnedByUs, { haveAdditionalPartition, haveTooManyPartitions }) {
    let actualRequiredPartitionCount = minRequiredPartitionCount;
    if (requiredNumberOfOwnersWithExtraPartition > 0 &&
        // Eventually the `haveTooManyPartitions` will decay into `haveAdditionalPartition`
        // EventProcessors as partitions are balanced to consumers that aren't at par.
        // We can consider them to be `haveAdditionalPartition` EventProcessors for our purposes.
        haveAdditionalPartition + haveTooManyPartitions < requiredNumberOfOwnersWithExtraPartition) {
        // Overall we don't have enough EventProcessors that are taking on an additional partition
        // so we should attempt to.
        actualRequiredPartitionCount = minRequiredPartitionCount + 1;
    }
    return actualRequiredPartitionCount - numPartitionsOwnedByUs;
}
/**
 * Determines which partitions can be stolen from other owners while maintaining
 * a balanced state.
 * @param numberOfPartitionsToClaim - The number of partitions the owner needs to claim to reach a balanced state.
 * @param minPartitionsPerOwner - The minimum number of partitions each owner needs for the partition load to be balanced.
 * @param requiredNumberOfOwnersWithExtraPartition - The number of owners that should have 1 extra partition.
 * @param ourOwnerId - The id of _our_ owner.
 * @param ownerToOwnershipMap - The current ownerships for partitions.
 * @internal
 */
function findPartitionsToSteal(numberOfPartitionsToClaim, minPartitionsPerOwner, requiredNumberOfOwnersWithExtraPartition, ourOwnerId, ownerToOwnershipMap) {
    const partitionsToSteal = [];
    // Create a list of PartitionOwnership lists that we can steal from.
    const listOfPartitionOwnerships = [];
    ownerToOwnershipMap.forEach((partitionOwnerships, ownerId) => {
        if (ownerId === ourOwnerId || partitionOwnerships.length <= minPartitionsPerOwner)
            return;
        listOfPartitionOwnerships.push(partitionOwnerships);
    });
    // Sort the list in descending order based on the length of each element.
    listOfPartitionOwnerships.sort((a, b) => {
        if (a.length > b.length)
            return -1;
        if (a.length < b.length)
            return 1;
        return 0;
    });
    // Attempt to steal partitions from EventProcessors that have the most partitions 1st,
    // then work our way down.
    let ownersEncounteredWithExtraPartitions = 0;
    let currentPartitionOwnershipList = listOfPartitionOwnerships.shift();
    while (numberOfPartitionsToClaim > 0 && currentPartitionOwnershipList) {
        let ownersExpectedPartitionCount = minPartitionsPerOwner;
        // Determine if the current owner should be allowed to have an extra partition.
        if (ownersEncounteredWithExtraPartitions < requiredNumberOfOwnersWithExtraPartition) {
            ownersExpectedPartitionCount++;
        }
        ownersEncounteredWithExtraPartitions++;
        let numberAvailableToSteal = currentPartitionOwnershipList.length - ownersExpectedPartitionCount;
        // Claim as many random partitions as possible.
        while (Math.min(numberOfPartitionsToClaim, numberAvailableToSteal)) {
            const indexToClaim = Math.floor(Math.random() * currentPartitionOwnershipList.length);
            partitionsToSteal.push(currentPartitionOwnershipList.splice(indexToClaim, 1)[0].partitionId);
            numberOfPartitionsToClaim--;
            numberAvailableToSteal--;
        }
        // Move on to the next list of PartitionOwnership.
        currentPartitionOwnershipList = listOfPartitionOwnerships.shift();
    }
    return partitionsToSteal;
}
/**
 * Identifies all of the partitions that can be claimed by the specified owner for
 * that owner to reach a balanced state.
 * @param OwnerId - The id we should assume is _our_ id when checking for ownership.
 * @param claimedPartitionOwnershipMap - The current claimed ownerships for partitions.
 * @param partitionIds - Partitions to assign owners to.
 * @param expirationIntervalInMs - The length of time a partition claim is valid.
 * @returns Partition ids that may be claimed.
 * @internal
 */
function listAvailablePartitions(ownerId, claimedPartitionOwnershipMap, partitionIds, expirationIntervalInMs) {
    if (!partitionIds.length) {
        return [];
    }
    // Collect only the PartitionOwnership that have been updated within the expiration interval.
    // Any PartitionOwnership that has been updated outside the expiration interval can be claimed.
    const activePartitionOwnershipMap = getActivePartitionOwnerships(claimedPartitionOwnershipMap, expirationIntervalInMs);
    logger.verbose(`[${ownerId}] Number of active ownership records: ${activePartitionOwnershipMap.size}.`);
    if (activePartitionOwnershipMap.size === 0) {
        // All partitions in this Event Hub are available to claim.
        return partitionIds;
    }
    // Map ownerIds to the partitions they own so that we can determine how many each owner has.
    const ownerToOwnershipMap = new Map();
    for (const activeOwnership of activePartitionOwnershipMap.values()) {
        const partitionOwnershipList = ownerToOwnershipMap.get(activeOwnership.ownerId) || [];
        partitionOwnershipList.push(activeOwnership);
        ownerToOwnershipMap.set(activeOwnership.ownerId, partitionOwnershipList);
    }
    // Add the current EventProcessor to the map of owners to ownerships if it doesn't exist.
    if (!ownerToOwnershipMap.has(ownerId)) {
        ownerToOwnershipMap.set(ownerId, []);
    }
    logger.info(`[${ownerId}] Number of active event processors: ${ownerToOwnershipMap.size}.`);
    const { minPartitionsPerOwner, requiredNumberOfOwnersWithExtraPartition } = calculateBalancedLoadCounts(ownerToOwnershipMap, partitionIds);
    logger.verbose(`[${ownerId}] Expected minimum number of partitions per event processor: ${minPartitionsPerOwner},` +
        `expected number of event processors with additional partition: ${requiredNumberOfOwnersWithExtraPartition}.`);
    // Get some stats representing the current state the world with regards to how balanced the
    // partitions are across EventProcessors.
    const eventProcessorCounts = getEventProcessorCounts(minPartitionsPerOwner, ownerToOwnershipMap);
    if (isLoadBalanced(requiredNumberOfOwnersWithExtraPartition, ownerToOwnershipMap.size, eventProcessorCounts)) {
        // When the partitions are evenly distributed, no change required.
        return [];
    }
    let numberOfPartitionsToClaim = getNumberOfPartitionsToClaim(minPartitionsPerOwner, requiredNumberOfOwnersWithExtraPartition, ownerToOwnershipMap.get(ownerId).length, eventProcessorCounts);
    if (numberOfPartitionsToClaim <= 0) {
        return [];
    }
    const partitionsToClaim = [];
    const unclaimedPartitionIds = partitionIds.filter((id) => !activePartitionOwnershipMap.has(id));
    // Prioritize getting unclaimed partitions first.
    while (Math.min(numberOfPartitionsToClaim, unclaimedPartitionIds.length)) {
        const indexToClaim = Math.floor(Math.random() * unclaimedPartitionIds.length);
        partitionsToClaim.push(unclaimedPartitionIds.splice(indexToClaim, 1)[0]);
        numberOfPartitionsToClaim--;
    }
    if (numberOfPartitionsToClaim === 0) {
        return partitionsToClaim;
    }
    // Find partitions that can be stolen from other EventProcessors.
    const partitionsToSteal = findPartitionsToSteal(numberOfPartitionsToClaim, minPartitionsPerOwner, requiredNumberOfOwnersWithExtraPartition, ownerId, ownerToOwnershipMap);
    return partitionsToClaim.concat(partitionsToSteal);
}

// Copyright (c) Microsoft Corporation.
/**
 * The BalancedLoadBalancerStrategy is meant to be used when the user
 * wants to reach a load balanced state with less partition 'thrashing'.
 *
 * Partition thrashing - where a partition changes owners - is minimized
 * by only returning a single partition to claim at a time.
 * This minimizes the number of times a partition should need to be stolen.
 * @internal
 */
class BalancedLoadBalancingStrategy {
    /**
     * Creates an instance of BalancedLoadBalancingStrategy.
     *
     * @param _partitionOwnershipExpirationIntervalInMs - The length of time a partition claim is valid.
     */
    constructor(_partitionOwnershipExpirationIntervalInMs) {
        this._partitionOwnershipExpirationIntervalInMs = _partitionOwnershipExpirationIntervalInMs;
    }
    /**
     * Implements load balancing by taking into account current ownership and
     * the full set of partitions in the Event Hub.
     * @param ourOwnerId - The id we should assume is _our_ id when checking for ownership.
     * @param claimedPartitionOwnershipMap - The current claimed ownerships for partitions.
     * @param partitionIds - Partitions to assign owners to.
     * @returns Partition ids to claim.
     */
    getPartitionsToCliam(ourOwnerId, claimedPartitionOwnershipMap, partitionIds) {
        const claimablePartitions = listAvailablePartitions(ourOwnerId, claimedPartitionOwnershipMap, partitionIds, this._partitionOwnershipExpirationIntervalInMs);
        if (!claimablePartitions.length) {
            return [];
        }
        const randomIndex = Math.floor(Math.random() * claimablePartitions.length);
        return [claimablePartitions[randomIndex]];
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 */
class GreedyLoadBalancingStrategy {
    /**
     * Creates an instance of GreedyLoadBalancingStrategy.
     *
     * @param _partitionOwnershipExpirationIntervalInMs - The length of time a partition claim is valid.
     */
    constructor(_partitionOwnershipExpirationIntervalInMs) {
        this._partitionOwnershipExpirationIntervalInMs = _partitionOwnershipExpirationIntervalInMs;
    }
    /**
     * Implements load balancing by taking into account current ownership and
     * the new set of partitions to add.
     * @param ourOwnerId - The id we should assume is _our_ id when checking for ownership.
     * @param claimedPartitionOwnershipMap - The current claimed ownerships for partitions.
     * @param partitionIds - Partitions to assign owners to.
     * @returns Partition ids to claim.
     */
    getPartitionsToCliam(ourOwnerId, claimedPartitionOwnershipMap, partitionIds) {
        return listAvailablePartitions(ourOwnerId, claimedPartitionOwnershipMap, partitionIds, this._partitionOwnershipExpirationIntervalInMs);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The `EventProcessor` relies on a `CheckpointStore` to store checkpoints and handle partition
 * ownerships. `InMemoryCheckpointStore` is simple partition manager that stores checkpoints and
 * partition ownerships in memory of your program.
 *
 * You can use the `InMemoryCheckpointStore` to get started with using the `EventProcessor`.
 * But in production, you should choose an implementation of the `CheckpointStore` interface that will
 * store the checkpoints and partition ownerships to a durable store instead.
 *
 * @internal
 */
class InMemoryCheckpointStore {
    constructor() {
        this._partitionOwnershipMap = new Map();
        this._committedCheckpoints = new Map();
    }
    /**
     * Get the list of all existing partition ownership from the underlying data store. Could return empty
     * results if there are is no existing ownership information.
     *
     * @param fullyQualifiedNamespace - The fully qualified Event Hubs namespace. This is likely to be similar to
     * <yournamespace>.servicebus.windows.net.
     * @param eventHubName - The event hub name.
     * @param consumerGroup - The consumer group name.
     * @returns Partition ownership details of all the partitions that have/had an owner..
     */
    async listOwnership(_fullyQualifiedNamespace, _eventHubName, _consumerGroup) {
        const ownerships = [];
        for (const value of this._partitionOwnershipMap.values()) {
            ownerships.push(Object.assign({}, value));
        }
        return ownerships;
    }
    /**
     * Claim ownership of a list of partitions. This will return the list of partitions that were owned
     * successfully.
     *
     * @param partitionOwnership - The list of partition ownership this instance is claiming to own.
     * @returns A list partitions this instance successfully claimed ownership.
     */
    async claimOwnership(partitionOwnership) {
        const claimedOwnerships = [];
        for (const ownership of partitionOwnership) {
            if (!this._partitionOwnershipMap.has(ownership.partitionId) ||
                this._partitionOwnershipMap.get(ownership.partitionId).etag === ownership.etag) {
                const date = new Date();
                const newOwnership = Object.assign(Object.assign({}, ownership), { etag: rheaPromise.generate_uuid(), lastModifiedTimeInMs: date.getTime() });
                this._partitionOwnershipMap.set(newOwnership.partitionId, newOwnership);
                claimedOwnerships.push(newOwnership);
            }
        }
        return claimedOwnerships;
    }
    /**
     * Updates the checkpoint in the data store for a partition.
     *
     * @param checkpoint - The checkpoint.
     */
    async updateCheckpoint(checkpoint) {
        throwTypeErrorIfParameterMissing("", "updateCheckpoint", "sequenceNumber", checkpoint.sequenceNumber);
        throwTypeErrorIfParameterMissing("", "updateCheckpoint", "offset", checkpoint.offset);
        checkpoint = Object.assign({}, checkpoint);
        const partitionOwnership = this._partitionOwnershipMap.get(checkpoint.partitionId);
        if (partitionOwnership) {
            partitionOwnership.etag = rheaPromise.generate_uuid();
            const key = `${checkpoint.fullyQualifiedNamespace}:${checkpoint.eventHubName}:${checkpoint.consumerGroup}`;
            let partitionMap = this._committedCheckpoints.get(key);
            if (partitionMap == null) {
                partitionMap = new Map();
                this._committedCheckpoints.set(key, partitionMap);
            }
            partitionMap.set(checkpoint.partitionId, checkpoint);
        }
    }
    async listCheckpoints(fullyQualifiedNamespace, eventHubName, consumerGroup) {
        const key = `${fullyQualifiedNamespace}:${eventHubName}:${consumerGroup}`;
        const partitionMap = this._committedCheckpoints.get(key);
        if (partitionMap == null) {
            return [];
        }
        const checkpoints = [];
        for (const value of partitionMap.values()) {
            checkpoints.push(Object.assign({}, value));
        }
        return checkpoints;
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * Used by EventHubConsumerClient to prevent accidentally spinning up multiple
 * subscriptions against the same set of partitions.
 *
 * This is needed now that EventHubConsumerClient only uses a single CheckpointStore
 * instance - otherwise users will see unpredictable results as their event processor
 * continually steals/overwrites checkpointing and ownership with itself.
 *
 * @internal
 */
class PartitionGate {
    constructor() {
        this._partitions = new Set();
    }
    /**
     * Adds a partition, throwing an Error if there is a conflict with partitions (including "all")
     * that are already added.
     *
     * @param partitionId - A partition ID or the constant "all"
     */
    add(partitionId) {
        if ((partitionId === "all" && this._partitions.size > 0) ||
            this._partitions.has(partitionId) ||
            this._partitions.has("all")) {
            throw new Error(`Partition already has a subscriber.`);
        }
        this._partitions.add(partitionId);
    }
    /**
     * Removes a partition
     *
     * @param partitionId - A partition ID or the constant "all"
     */
    remove(partitionId) {
        this._partitions.delete(partitionId);
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * The UnbalancedLoadBalancingStrategy does no actual load balancing.
 * It is intended to be used when you want to avoid load balancing
 * and consume a set of partitions.
 * @internal
 */
class UnbalancedLoadBalancingStrategy {
    /**
     * Implements load balancing by taking into account current ownership and
     * the full set of partitions in the Event Hub.
     * @param _ourOwnerId - The id we should assume is _our_ id when checking for ownership.
     * @param _claimedPartitionOwnershipMap - The current claimed ownerships for partitions.
     * @param partitionIds - Partitions to assign owners to.
     * @returns Partition ids to claim.
     */
    getPartitionsToCliam(_ourOwnerId, _claimedPartitionOwnershipMap, partitionIds) {
        return partitionIds;
    }
}

// Copyright (c) Microsoft Corporation.
const defaultConsumerClientOptions = {
    // to support our current "process single event only" workflow we'll also purposefully
    // only request a single event at a time.
    maxBatchSize: 1,
    maxWaitTimeInSeconds: 60,
};
/**
 * The `EventHubConsumerClient` class is used to consume events from an Event Hub.
 *
 * There are multiple ways to create an `EventHubConsumerClient`
 * - Use the connection string from the SAS policy created for your Event Hub instance.
 * - Use the connection string from the SAS policy created for your Event Hub namespace,
 * and the name of the Event Hub instance
 * - Use the full namespace like `<yournamespace>.servicebus.windows.net`, and a credentials object.
 *
 * Optionally, you can also pass:
 * - An options bag to configure the retry policy or proxy settings.
 * - A checkpoint store that is used by the client to read checkpoints to determine the position from where it should
 * resume receiving events when your application gets restarted. The checkpoint store is also used by the client
 * to load balance multiple instances of your application.
 */
class EventHubConsumerClient {
    constructor(_consumerGroup, connectionStringOrFullyQualifiedNamespace2, checkpointStoreOrEventHubNameOrOptions3, checkpointStoreOrCredentialOrOptions4, checkpointStoreOrOptions5, options6) {
        var _a;
        this._consumerGroup = _consumerGroup;
        this._partitionGate = new PartitionGate();
        this._id = uuid.v4();
        /**
         * The Subscriptions that were spawned by calling `subscribe()`.
         * Subscriptions that have been stopped by the user will not
         * be present in this set.
         */
        this._subscriptions = new Set();
        if (isCredential(checkpointStoreOrCredentialOrOptions4)) {
            // #3 or 3.1
            logger.info("Creating EventHubConsumerClient with TokenCredential.");
            if (isCheckpointStore(checkpointStoreOrOptions5)) {
                // 3.1
                this._checkpointStore = checkpointStoreOrOptions5;
                this._userChoseCheckpointStore = true;
                this._clientOptions = options6 || {};
            }
            else {
                this._checkpointStore = new InMemoryCheckpointStore();
                this._userChoseCheckpointStore = false;
                this._clientOptions = checkpointStoreOrOptions5 || {};
            }
            this._context = createConnectionContext(connectionStringOrFullyQualifiedNamespace2, checkpointStoreOrEventHubNameOrOptions3, checkpointStoreOrCredentialOrOptions4, this._clientOptions);
        }
        else if (typeof checkpointStoreOrEventHubNameOrOptions3 === "string") {
            // #2 or 2.1
            logger.info("Creating EventHubConsumerClient with connection string and event hub name.");
            if (isCheckpointStore(checkpointStoreOrCredentialOrOptions4)) {
                // 2.1
                this._checkpointStore = checkpointStoreOrCredentialOrOptions4;
                this._userChoseCheckpointStore = true;
                this._clientOptions = checkpointStoreOrOptions5 || {};
            }
            else {
                // 2
                this._checkpointStore = new InMemoryCheckpointStore();
                this._userChoseCheckpointStore = false;
                this._clientOptions = checkpointStoreOrCredentialOrOptions4 || {};
            }
            this._context = createConnectionContext(connectionStringOrFullyQualifiedNamespace2, checkpointStoreOrEventHubNameOrOptions3, this._clientOptions);
        }
        else {
            // #1 or 1.1
            logger.info("Creating EventHubConsumerClient with connection string.");
            if (isCheckpointStore(checkpointStoreOrEventHubNameOrOptions3)) {
                // 1.1
                this._checkpointStore = checkpointStoreOrEventHubNameOrOptions3;
                this._userChoseCheckpointStore = true;
                this._clientOptions =
                    checkpointStoreOrCredentialOrOptions4 || {};
            }
            else {
                // 1
                this._checkpointStore = new InMemoryCheckpointStore();
                this._userChoseCheckpointStore = false;
                this._clientOptions =
                    checkpointStoreOrEventHubNameOrOptions3 || {};
            }
            this._context = createConnectionContext(connectionStringOrFullyQualifiedNamespace2, this._clientOptions);
        }
        this._loadBalancingOptions = Object.assign({ 
            // default options
            strategy: "balanced", updateIntervalInMs: 10000, partitionOwnershipExpirationIntervalInMs: 60000 }, (_a = this._clientOptions) === null || _a === void 0 ? void 0 : _a.loadBalancingOptions);
    }
    /**
     * @readonly
     * The name of the Event Hub instance for which this client is created.
     */
    get eventHubName() {
        return this._context.config.entityPath;
    }
    /**
     * @readonly
     * The fully qualified namespace of the Event Hub instance for which this client is created.
     * This is likely to be similar to <yournamespace>.servicebus.windows.net.
     */
    get fullyQualifiedNamespace() {
        return this._context.config.host;
    }
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    async close() {
        // Stop all the actively running subscriptions.
        const activeSubscriptions = Array.from(this._subscriptions);
        await Promise.all(activeSubscriptions.map((subscription) => {
            return subscription.close();
        }));
        // Close the connection via the connection context.
        return this._context.close();
    }
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options = {}) {
        return this._context
            .managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }))
            .then((eventHubProperties) => {
            return eventHubProperties.partitionIds;
        });
    }
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId, options = {}) {
        return this._context.managementSession.getPartitionProperties(partitionId, Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubConsumerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options = {}) {
        return this._context.managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
    subscribe(handlersOrPartitionId1, optionsOrHandlers2, possibleOptions3) {
        let eventProcessor;
        let targetedPartitionId;
        if (isSubscriptionEventHandlers(handlersOrPartitionId1)) {
            // #1: subscribe overload - read from all partitions
            const options = optionsOrHandlers2;
            if (options && options.startPosition) {
                validateEventPositions(options.startPosition);
            }
            ({ targetedPartitionId, eventProcessor } = this.createEventProcessorForAllPartitions(handlersOrPartitionId1, options));
        }
        else if (isSubscriptionEventHandlers(optionsOrHandlers2)) {
            // #2: subscribe overload (read from specific partition IDs), don't coordinate
            const options = possibleOptions3;
            if (options && options.startPosition) {
                validateEventPositions(options.startPosition);
            }
            ({ targetedPartitionId, eventProcessor } = this.createEventProcessorForSinglePartition(
            // cast to string as downstream code expects partitionId to be string, but JS users could have given us anything.
            // we don't validate the user input and instead rely on service throwing errors if any
            String(handlersOrPartitionId1), optionsOrHandlers2, possibleOptions3));
        }
        else {
            throw new TypeError("Unhandled subscribe() overload");
        }
        eventProcessor.start();
        const subscription = {
            get isRunning() {
                return eventProcessor.isRunning();
            },
            close: () => {
                this._partitionGate.remove(targetedPartitionId);
                this._subscriptions.delete(subscription);
                return eventProcessor.stop();
            },
        };
        this._subscriptions.add(subscription);
        return subscription;
    }
    /**
     * Gets the LoadBalancing strategy that should be used based on what the user provided.
     */
    _getLoadBalancingStrategy() {
        var _a;
        if (!this._userChoseCheckpointStore) {
            // The default behavior when a checkpointstore isn't provided
            // is to always grab all the partitions.
            return new UnbalancedLoadBalancingStrategy();
        }
        const partitionOwnershipExpirationIntervalInMs = this._loadBalancingOptions.partitionOwnershipExpirationIntervalInMs;
        if (((_a = this._loadBalancingOptions) === null || _a === void 0 ? void 0 : _a.strategy) === "greedy") {
            return new GreedyLoadBalancingStrategy(partitionOwnershipExpirationIntervalInMs);
        }
        // The default behavior when a checkpointstore is provided is
        // to grab one partition at a time.
        return new BalancedLoadBalancingStrategy(partitionOwnershipExpirationIntervalInMs);
    }
    createEventProcessorForAllPartitions(subscriptionEventHandlers, options) {
        this._partitionGate.add("all");
        if (this._userChoseCheckpointStore) {
            logger.verbose("EventHubConsumerClient subscribing to all partitions, using a checkpoint store.");
        }
        else {
            logger.verbose("EventHubConsumerClient subscribing to all partitions, no checkpoint store.");
        }
        const loadBalancingStrategy = this._getLoadBalancingStrategy();
        const eventProcessor = this._createEventProcessor(this._context, subscriptionEventHandlers, this._checkpointStore, Object.assign(Object.assign(Object.assign({}, defaultConsumerClientOptions), options), { ownerLevel: getOwnerLevel(options, this._userChoseCheckpointStore), 
            // make it so all the event processors process work with the same overarching owner ID
            // this allows the EventHubConsumer to unify all the work for any processors that it spawns
            ownerId: this._id, retryOptions: this._clientOptions.retryOptions, loadBalancingStrategy, loopIntervalInMs: this._loadBalancingOptions.updateIntervalInMs }));
        return { targetedPartitionId: "all", eventProcessor };
    }
    createEventProcessorForSinglePartition(partitionId, eventHandlers, options) {
        var _a;
        this._partitionGate.add(partitionId);
        const subscribeOptions = options;
        if (this._userChoseCheckpointStore) {
            logger.verbose(`EventHubConsumerClient subscribing to specific partition (${partitionId}), using a checkpoint store.`);
        }
        else {
            logger.verbose(`EventHubConsumerClient subscribing to specific partition (${partitionId}), no checkpoint store.`);
        }
        const eventProcessor = this._createEventProcessor(this._context, eventHandlers, this._checkpointStore, Object.assign(Object.assign(Object.assign({}, defaultConsumerClientOptions), options), { processingTarget: partitionId, ownerLevel: getOwnerLevel(subscribeOptions, this._userChoseCheckpointStore), retryOptions: this._clientOptions.retryOptions, loadBalancingStrategy: new UnbalancedLoadBalancingStrategy(), loopIntervalInMs: (_a = this._loadBalancingOptions.updateIntervalInMs) !== null && _a !== void 0 ? _a : 10000 }));
        return { targetedPartitionId: partitionId, eventProcessor };
    }
    _createEventProcessor(connectionContext, subscriptionEventHandlers, checkpointStore, options) {
        return new EventProcessor(this._consumerGroup, connectionContext, subscriptionEventHandlers, checkpointStore, options);
    }
}
/**
 * The name of the default consumer group in the Event Hubs service.
 */
EventHubConsumerClient.defaultConsumerGroupName = coreAmqp.Constants.defaultConsumerGroup;
/**
 * @internal
 */
function isCheckpointStore(possible) {
    if (!possible) {
        return false;
    }
    const checkpointStore = possible;
    return (typeof checkpointStore.claimOwnership === "function" &&
        typeof checkpointStore.listCheckpoints === "function" &&
        typeof checkpointStore.listOwnership === "function" &&
        typeof checkpointStore.updateCheckpoint === "function");
}
/**
 * @internal
 */
function isSubscriptionEventHandlers(possible) {
    return typeof possible.processEvents === "function";
}
function getOwnerLevel(options, userChoseCheckpointStore) {
    if (options && options.ownerLevel) {
        return options.ownerLevel;
    }
    if (userChoseCheckpointStore) {
        return 0;
    }
    else {
        return undefined;
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * Describes the EventHubSender that will send event data to EventHub.
 * @internal
 */
class EventHubSender extends LinkEntity {
    /**
     * Creates a new EventHubSender instance.
     * @param context - The connection context.
     * @param options - Options used to configure the EventHubSender.
     */
    constructor(context, { partitionId, enableIdempotentProducer, partitionPublishingOptions }) {
        super(context, {
            name: context.config.getSenderAddress(partitionId),
            partitionId: partitionId,
        });
        /**
         * The unique lock name per connection that is used to acquire the
         * lock for establishing a sender link by an entity on that connection.
         * @readonly
         */
        this.senderLock = `sender-${uuid.v4()}`;
        this.address = context.config.getSenderAddress(partitionId);
        this.audience = context.config.getSenderAudience(partitionId);
        this._isIdempotentProducer = enableIdempotentProducer;
        this._userProvidedPublishingOptions = partitionPublishingOptions;
        this._onAmqpError = (eventContext) => {
            const senderError = eventContext.sender && eventContext.sender.error;
            logger.verbose("[%s] 'sender_error' event occurred on the sender '%s' with address '%s'. " +
                "The associated error is: %O", this._context.connectionId, this.name, this.address, senderError);
            // TODO: Consider rejecting promise in trySendBatch() or createBatch()
        };
        this._onSessionError = (eventContext) => {
            const sessionError = eventContext.session && eventContext.session.error;
            logger.verbose("[%s] 'session_error' event occurred on the session of sender '%s' with address '%s'. " +
                "The associated error is: %O", this._context.connectionId, this.name, this.address, sessionError);
            // TODO: Consider rejecting promise in trySendBatch() or createBatch()
        };
        this._onAmqpClose = async (eventContext) => {
            const sender = this._sender || eventContext.sender;
            logger.verbose("[%s] 'sender_close' event occurred on the sender '%s' with address '%s'. " +
                "Value for isItselfClosed on the receiver is: '%s' " +
                "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, sender ? sender.isItselfClosed().toString() : undefined, this.isConnecting);
            if (sender && !this.isConnecting) {
                // Call close to clean up timers & other resources
                await sender.close().catch((err) => {
                    logger.verbose("[%s] Error when closing sender [%s] after 'sender_close' event: %O", this._context.connectionId, this.name, err);
                });
            }
        };
        this._onSessionClose = async (eventContext) => {
            const sender = this._sender || eventContext.sender;
            logger.verbose("[%s] 'session_close' event occurred on the session of sender '%s' with address '%s'. " +
                "Value for isSessionItselfClosed on the session is: '%s' " +
                "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, sender ? sender.isSessionItselfClosed().toString() : undefined, this.isConnecting);
            if (sender && !this.isConnecting) {
                // Call close to clean up timers & other resources
                await sender.close().catch((err) => {
                    logger.verbose("[%s] Error when closing sender [%s] after 'session_close' event: %O", this._context.connectionId, this.name, err);
                });
            }
        };
    }
    /**
     * Deletes the sender from the context. Clears the token renewal timer. Closes the sender link.
     * @returns Promise<void>
     */
    async close() {
        try {
            if (this._sender) {
                logger.info("[%s] Closing the Sender for the entity '%s'.", this._context.connectionId, this._context.config.entityPath);
                const senderLink = this._sender;
                this._deleteFromCache();
                await this._closeLink(senderLink);
            }
        }
        catch (err) {
            const msg = `[${this._context.connectionId}] An error occurred while closing sender ${this.name}: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`;
            logger.warning(msg);
            logErrorStackTrace(err);
            throw err;
        }
    }
    /**
     * Determines whether the AMQP sender link is open. If open then returns true else returns false.
     * @returns boolean
     */
    isOpen() {
        const result = Boolean(this._sender && this._sender.isOpen());
        logger.verbose("[%s] Sender '%s' with address '%s' is open? -> %s", this._context.connectionId, this.name, this.address, result);
        return result;
    }
    /**
     * Returns maximum message size on the AMQP sender link.
     * @param abortSignal - An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     * @returns Promise<number>
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    async getMaxMessageSize(options = {}) {
        const sender = await this._getLink(options);
        return sender.maxMessageSize;
    }
    /**
     * Get the information about the state of publishing for a partition as observed by the `EventHubSender`.
     * This data can always be read, but will only be populated with information relevant to the active features
     * for the producer client.
     */
    async getPartitionPublishingProperties(options = {}) {
        var _a, _b;
        if (this._localPublishingProperties) {
            // Send a copy of the properties so it can't be mutated downstream.
            return Object.assign({}, this._localPublishingProperties);
        }
        const properties = {
            isIdempotentPublishingEnabled: this._isIdempotentProducer,
            partitionId: (_a = this.partitionId) !== null && _a !== void 0 ? _a : "",
        };
        if (this._isIdempotentProducer) {
            this._sender = await this._getLink(options);
            // await this._createLinkIfNotOpen(options);
            if (!this._sender) {
                // createLinkIfNotOpen should throw if `this._sender` can't be created, but just in case it gets
                // deleted while setting up token refreshing, make sure it exists.
                throw new Error(`Failed to retrieve partition publishing properties for partition "${this.partitionId}".`);
            }
            const { [idempotentProducerAmqpPropertyNames.epoch]: ownerLevel, [idempotentProducerAmqpPropertyNames.producerId]: producerGroupId, [idempotentProducerAmqpPropertyNames.producerSequenceNumber]: lastPublishedSequenceNumber, } = (_b = this._sender.properties) !== null && _b !== void 0 ? _b : {};
            properties.ownerLevel = parseInt(ownerLevel, 10);
            properties.producerGroupId = parseInt(producerGroupId, 10);
            properties.lastPublishedSequenceNumber = parseInt(lastPublishedSequenceNumber, 10);
        }
        this._localPublishingProperties = properties;
        // Send a copy of the properties so it can't be mutated downstream.
        return Object.assign({}, properties);
    }
    /**
     * Send a batch of EventData to the EventHub. The "message_annotations",
     * "application_properties" and "properties" of the first message will be set as that
     * of the envelope (batch message).
     * @param events -  An array of EventData objects to be sent in a Batch message.
     * @param options - Options to control the way the events are batched along with request options
     */
    async send(events, options) {
        try {
            logger.info("[%s] Sender '%s', trying to send EventData[].", this._context.connectionId, this.name);
            if (this._isIdempotentProducer && this._hasPendingSend) {
                throw new Error(`There can only be 1 "sendBatch" call in-flight per partition while "enableIdempotentRetries" is set to true.`);
            }
            const eventCount = isEventDataBatch(events) ? events.count : events.length;
            if (eventCount === 0) {
                logger.info(`[${this._context.connectionId}] No events were passed to sendBatch.`);
                return;
            }
            if (this._isIdempotentProducer) {
                this._hasPendingSend = true;
            }
            logger.info("[%s] Sender '%s', sending encoded batch message.", this._context.connectionId, this.name);
            await this._trySendBatch(events, options);
            if (this._isIdempotentProducer) {
                commitIdempotentSequenceNumbers(events);
                if (this._localPublishingProperties) {
                    const { lastPublishedSequenceNumber = 0 } = this._localPublishingProperties;
                    // Increment the lastPublishedSequenceNumber based on the number of events published.
                    this._localPublishingProperties.lastPublishedSequenceNumber =
                        lastPublishedSequenceNumber + eventCount;
                }
            }
            return;
        }
        catch (err) {
            rollbackIdempotentSequenceNumbers(events);
            logger.warning(`An error occurred while sending the batch message ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
            logErrorStackTrace(err);
            throw err;
        }
        finally {
            if (this._isIdempotentProducer) {
                this._hasPendingSend = false;
            }
        }
    }
    /**
     * @param sender - The rhea sender that contains the idempotent producer properties.
     */
    _populateLocalPublishingProperties(sender) {
        var _a, _b;
        const { [idempotentProducerAmqpPropertyNames.epoch]: ownerLevel, [idempotentProducerAmqpPropertyNames.producerId]: producerGroupId, [idempotentProducerAmqpPropertyNames.producerSequenceNumber]: lastPublishedSequenceNumber, } = (_a = sender.properties) !== null && _a !== void 0 ? _a : {};
        this._localPublishingProperties = {
            isIdempotentPublishingEnabled: this._isIdempotentProducer,
            partitionId: (_b = this.partitionId) !== null && _b !== void 0 ? _b : "",
            lastPublishedSequenceNumber,
            ownerLevel,
            producerGroupId,
        };
    }
    _deleteFromCache() {
        this._sender = undefined;
        delete this._context.senders[this.name];
        logger.verbose("[%s] Deleted the sender '%s' with address '%s' from the client cache.", this._context.connectionId, this.name, this.address);
    }
    _createSenderOptions(newName) {
        if (newName)
            this.name = `${uuid.v4()}`;
        const srOptions = {
            name: this.name,
            target: {
                address: this.address,
            },
            onError: this._onAmqpError,
            onClose: this._onAmqpClose,
            onSessionError: this._onSessionError,
            onSessionClose: this._onSessionClose,
        };
        if (this._isIdempotentProducer) {
            srOptions.desired_capabilities = idempotentProducerAmqpPropertyNames.capability;
            const idempotentProperties = generateIdempotentLinkProperties(this._userProvidedPublishingOptions, this._localPublishingProperties);
            srOptions.properties = idempotentProperties;
        }
        logger.verbose("Creating sender with options: %O", srOptions);
        return srOptions;
    }
    /**
     * Tries to send the message to EventHub if there is enough credit to send them
     * and the circular buffer has available space to settle the message after sending them.
     *
     * We have implemented a synchronous send over here in the sense that we shall be waiting
     * for the message to be accepted or rejected and accordingly resolve or reject the promise.
     * @param rheaMessage - The message to be sent to EventHub.
     * @returns Promise<void>
     */
    async _trySendBatch(events, options = {}) {
        const abortSignal = options.abortSignal;
        const retryOptions = options.retryOptions || {};
        const timeoutInMs = getRetryAttemptTimeoutInMs(retryOptions);
        retryOptions.timeoutInMs = timeoutInMs;
        const sendEventPromise = async () => {
            var _a, _b;
            const initStartTime = Date.now();
            // TODO: (jeremymeng) A or B
            // variant A:
            const sender = await this._getLink(options);
            // variant B
            // await this._createLinkIfNotOpen(options);
            const publishingProps = await this.getPartitionPublishingProperties(options);
            const timeTakenByInit = Date.now() - initStartTime;
            logger.verbose("[%s] Sender '%s', credit: %d available: %d", this._context.connectionId, this.name, sender.credit, sender.session.outgoing.available());
            let waitTimeForSendable = 1000;
            if (!sender.sendable() && timeoutInMs - timeTakenByInit > waitTimeForSendable) {
                logger.verbose("%s Sender '%s', waiting for 1 second for sender to become sendable", this._context.connectionId, this.name);
                await coreAmqp.delay(waitTimeForSendable);
                logger.verbose("%s Sender '%s' after waiting for a second, credit: %d available: %d", this._context.connectionId, this.name, sender.credit, (_b = (_a = sender.session) === null || _a === void 0 ? void 0 : _a.outgoing) === null || _b === void 0 ? void 0 : _b.available());
            }
            else {
                waitTimeForSendable = 0;
            }
            if (!sender.sendable()) {
                // let us retry to send the message after some time.
                const msg = `[${this._context.connectionId}] Sender "${this.name}", ` +
                    `cannot send the message right now. Please try later.`;
                logger.warning(msg);
                const amqpError = {
                    condition: coreAmqp.ErrorNameConditionMapper.SenderBusyError,
                    description: msg,
                };
                throw coreAmqp.translate(amqpError);
            }
            logger.verbose("[%s] Sender '%s', sending message with id '%s'.", this._context.connectionId, this.name);
            if (timeoutInMs <= timeTakenByInit + waitTimeForSendable) {
                const desc = `${this._context.connectionId} Sender "${this.name}" ` +
                    `with address "${this.address}", was not able to send the message right now, due ` +
                    `to operation timeout.`;
                logger.warning(desc);
                const e = {
                    condition: coreAmqp.ErrorNameConditionMapper.ServiceUnavailableError,
                    description: desc,
                };
                throw coreAmqp.translate(e);
            }
            try {
                const encodedMessage = transformEventsForSend(events, publishingProps, options);
                const delivery = await sender.send(encodedMessage, {
                    format: 0x80013700,
                    timeoutInSeconds: (timeoutInMs - timeTakenByInit - waitTimeForSendable) / 1000,
                    abortSignal,
                });
                logger.info("[%s] Sender '%s', sent message with delivery id: %d", this._context.connectionId, this.name, delivery.id);
            }
            catch (err) {
                const error = err.innerError || err;
                const translatedError = translateError(error);
                throw translatedError;
            }
        };
        const config = {
            operation: sendEventPromise,
            connectionId: this._context.connectionId,
            operationType: coreAmqp.RetryOperationType.sendMessage,
            abortSignal: abortSignal,
            retryOptions: retryOptions,
        };
        try {
            await coreAmqp.retry(config);
        }
        catch (err) {
            const translatedError = coreAmqp.translate(err);
            logger.warning("[%s] Sender '%s', An error occurred while sending the message %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    async _getLink(options = {}) {
        if (this.isOpen() && this._sender) {
            return this._sender;
        }
        const retryOptions = options.retryOptions || {};
        const timeoutInMs = getRetryAttemptTimeoutInMs(retryOptions);
        retryOptions.timeoutInMs = timeoutInMs;
        const senderOptions = this._createSenderOptions();
        const startTime = Date.now();
        const createLinkPromise = async () => {
            return coreAmqp.defaultCancellableLock.acquire(this.senderLock, () => {
                const taskStartTime = Date.now();
                const taskTimeoutInMs = timeoutInMs - (taskStartTime - startTime);
                return this._init(Object.assign(Object.assign({}, senderOptions), { abortSignal: options.abortSignal, timeoutInMs: taskTimeoutInMs }));
            }, { abortSignal: options.abortSignal, timeoutInMs: timeoutInMs });
        };
        const config = {
            operation: createLinkPromise,
            connectionId: this._context.connectionId,
            operationType: coreAmqp.RetryOperationType.senderLink,
            abortSignal: options.abortSignal,
            retryOptions: retryOptions,
        };
        try {
            return await coreAmqp.retry(config);
        }
        catch (err) {
            const translatedError = coreAmqp.translate(err);
            logger.warning("[%s] An error occurred while creating the sender %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Initializes the sender session on the connection.
     * Should only be called from _createLinkIfNotOpen
     */
    async _init(options) {
        try {
            if (!this.isOpen() || !this._sender) {
                // Wait for the connectionContext to be ready to open the link.
                await this._context.readyToOpenLink();
                await this._negotiateClaim({
                    setTokenRenewal: false,
                    abortSignal: options.abortSignal,
                    timeoutInMs: options.timeoutInMs,
                });
                logger.verbose("[%s] Trying to create sender '%s'...", this._context.connectionId, this.name);
                const sender = await this._context.connection.createAwaitableSender(options);
                sender.setMaxListeners(1000);
                this._sender = sender;
                this._populateLocalPublishingProperties(this._sender);
                this.isConnecting = false;
                logger.verbose("[%s] Sender '%s' created with sender options: %O", this._context.connectionId, this.name, options);
                // It is possible for someone to close the sender and then start it again.
                // Thus make sure that the sender is present in the client cache.
                if (!this._context.senders[this.name])
                    this._context.senders[this.name] = this;
                this._ensureTokenRenewal();
                return sender;
            }
            else {
                logger.verbose("[%s] The sender '%s' with address '%s' is open -> %s. Hence not reconnecting.", this._context.connectionId, this.name, this.address, this.isOpen());
                return this._sender;
            }
        }
        catch (err) {
            const translatedError = coreAmqp.translate(err);
            logger.warning("[%s] An error occurred while creating the sender %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Creates a new sender to the given event hub, and optionally to a given partition if it is
     * not present in the context or returns the one present in the context.
     * @hidden
     * @param options - Options used to configure the EventHubSender.
     */
    static create(context, options) {
        const ehSender = new EventHubSender(context, options);
        if (!context.senders[ehSender.name]) {
            context.senders[ehSender.name] = ehSender;
        }
        return context.senders[ehSender.name];
    }
}
/**
 * Generates the link properties for an indemopotent sender given
 * based on the user-provided and locally-cached publishing options.
 *
 * Note: The set of idempotent properties a user specifies at EventHubProducerClient instantiation-time
 * is slightly different than what the service returns and the EventHubSender keeps track of locally.
 *
 * The difference is that the user specifies the `startingSequenceNumber`, whereas the local options
 * (those returned by getPartitionPublishingProperties) specifies `lastPublishedSequenceNumber`.
 *
 * These _can_ be the same, but the user is technically free to set any `startingSequenceNumber` they want.
 * @internal
 */
function generateIdempotentLinkProperties(userProvidedPublishingOptions, localPublishingOptions) {
    let ownerLevel;
    let producerGroupId;
    let sequenceNumber;
    // Prefer local publishing options since this is the up-to-date state of the sender.
    // Only use user-provided publishing options the first time we create the link.
    if (localPublishingOptions) {
        ownerLevel = localPublishingOptions.ownerLevel;
        producerGroupId = localPublishingOptions.producerGroupId;
        sequenceNumber = localPublishingOptions.lastPublishedSequenceNumber;
    }
    else if (userProvidedPublishingOptions) {
        ownerLevel = userProvidedPublishingOptions.ownerLevel;
        producerGroupId = userProvidedPublishingOptions.producerGroupId;
        sequenceNumber = userProvidedPublishingOptions.startingSequenceNumber;
    }
    else {
        // If we don't have any properties at all, send an empty object.
        // This will cause the service to generate a new producer-id for our client.
        return {};
    }
    // The service requires that if ANY_ of these properties are defined,
    // they _ALL_ have to be defined.
    // If we don't have one of the required values, use `null` and the
    // service will provide it.
    const idempotentLinkProperties = {
        [idempotentProducerAmqpPropertyNames.epoch]: isDefined(ownerLevel)
            ? rheaPromise.types.wrap_short(ownerLevel)
            : null,
        [idempotentProducerAmqpPropertyNames.producerId]: isDefined(producerGroupId)
            ? rheaPromise.types.wrap_long(producerGroupId)
            : null,
        [idempotentProducerAmqpPropertyNames.producerSequenceNumber]: isDefined(sequenceNumber)
            ? rheaPromise.types.wrap_int(sequenceNumber)
            : null,
    };
    return idempotentLinkProperties;
}
/**
 * Encodes a list or batch of events into a single binary message that can be sent to the service.
 *
 * Prior to encoding, any special properties not specified by the user, such as tracing or idempotent
 * properties, are assigned to the list or batch of events as needed.
 *
 * @internal
 * @param events - Events to transform for sending to the service.
 * @param publishingProps - Describes the current publishing state for the partition.
 * @param options - Options used to configure this function.
 */
function transformEventsForSend(events, publishingProps, options = {}) {
    var _a;
    if (isEventDataBatch(events)) {
        return events._generateMessage(publishingProps);
    }
    else {
        const eventCount = events.length;
        // convert events to rhea messages
        const rheaMessages = [];
        const tracingProperties = (_a = options.tracingProperties) !== null && _a !== void 0 ? _a : [];
        for (let i = 0; i < eventCount; i++) {
            const originalEvent = events[i];
            const tracingProperty = tracingProperties[i];
            // Create a copy of the user's event so we can add the tracing property.
            const event = Object.assign(Object.assign({}, originalEvent), { properties: Object.assign(Object.assign({}, originalEvent.properties), tracingProperty) });
            const rheaMessage = toRheaMessage(event, options.partitionKey);
            // populate idempotent message annotations
            const { lastPublishedSequenceNumber = 0 } = publishingProps;
            const startingSequenceNumber = lastPublishedSequenceNumber + 1;
            const pendingPublishSequenceNumber = startingSequenceNumber + i;
            populateIdempotentMessageAnnotations(rheaMessage, Object.assign(Object.assign({}, publishingProps), { publishSequenceNumber: pendingPublishSequenceNumber }));
            if (publishingProps.isIdempotentPublishingEnabled) {
                // Set pending seq number on user's event.
                originalEvent[PENDING_PUBLISH_SEQ_NUM_SYMBOL] =
                    pendingPublishSequenceNumber;
            }
            rheaMessages.push(rheaMessage);
        }
        // Encode every amqp message and then convert every encoded message to amqp data section
        const batchMessage = {
            body: rheaPromise.message.data_sections(rheaMessages.map(rheaPromise.message.encode)),
        };
        // Set message_annotations of the first message as
        // that of the envelope (batch message).
        if (rheaMessages[0].message_annotations) {
            batchMessage.message_annotations = Object.assign({}, rheaMessages[0].message_annotations);
        }
        // Finally encode the envelope (batch message).
        return rheaPromise.message.encode(batchMessage);
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The `EventHubProducerClient` class is used to send events to an Event Hub.
 *
 * There are multiple ways to create an `EventHubProducerClient`
 * - Use the connection string from the SAS policy created for your Event Hub instance.
 * - Use the connection string from the SAS policy created for your Event Hub namespace,
 * and the name of the Event Hub instance
 * - Use the full namespace like `<yournamespace>.servicebus.windows.net`, and a credentials object.
 *
 * Optionally, you can also pass an options bag to configure the retry policy or proxy settings.
 *
 */
class EventHubProducerClient {
    constructor(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4 // eslint-disable-line @azure/azure-sdk/ts-naming-options
    ) {
        this._context = createConnectionContext(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4);
        if (typeof eventHubNameOrOptions2 !== "string") {
            this._clientOptions = eventHubNameOrOptions2 || {};
        }
        else if (!isCredential(credentialOrOptions3)) {
            this._clientOptions = credentialOrOptions3 || {};
        }
        else {
            this._clientOptions = options4 || {};
        }
        this._sendersMap = new Map();
    }
    /**
     * @readonly
     * The name of the Event Hub instance for which this client is created.
     */
    get eventHubName() {
        return this._context.config.entityPath;
    }
    /**
     * @readonly
     * The fully qualified namespace of the Event Hub instance for which this client is created.
     * This is likely to be similar to <yournamespace>.servicebus.windows.net.
     */
    get fullyQualifiedNamespace() {
        return this._context.config.host;
    }
    /**
     * Creates an instance of `EventDataBatch` to which one can add events until the maximum supported size is reached.
     * The batch can be passed to the {@link sendBatch} method of the `EventHubProducerClient` to be sent to Azure Event Hubs.
     *
     * Events with different values for partitionKey or partitionId will need to be put into different batches.
     * To simplify such batch management across partitions or to have the client automatically batch events
     * and send them in specific intervals, use `EventHubBufferedProducerClient` instead.
     *
     * The below example assumes you have an array of events at hand to be batched safely.
     * If you have events coming in one by one, `EventHubBufferedProducerClient` is recommended instead
     * for effecient management of batches.
     *
     * Example usage:
     * ```ts
     * const client = new EventHubProducerClient(connectionString);
     * let batch = await client.createBatch();
     * for (let i = 0; i < messages.length; i++) {
     *  if (!batch.tryAdd(messages[i])) {
     *    await client.sendBatch(batch);
     *    batch = await client.createBatch();
     *    if (!batch.tryAdd(messages[i])) {
     *      throw new Error("Message too big to fit")
     *    }
     *    if (i === messages.length - 1) {
     *      await client.sendBatch(batch);
     *    }
     *   }
     * }
     * ```
     *
     * @param options -  Configures the behavior of the batch.
     * - `partitionKey`  : A value that is hashed and used by the Azure Event Hubs service to determine the partition to which
     * the events need to be sent.
     * - `partitionId`   : Id of the partition to which the batch of events need to be sent.
     * - `maxSizeInBytes`: The upper limit for the size of batch. The `tryAdd` function will return `false` after this limit is reached.
     * - `abortSignal`   : A signal the request to cancel the operation.
     * @returns Promise<EventDataBatch>
     * @throws Error if both `partitionId` and `partitionKey` are set in the options.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal in the options.
     */
    async createBatch(options = {}) {
        var _a;
        throwErrorIfConnectionClosed(this._context);
        const partitionId = isDefined(options.partitionId) ? String(options.partitionId) : undefined;
        validateProducerPartitionSettings({
            enableIdempotentRetries: this._enableIdempotentRetries,
            partitionId,
            partitionKey: options.partitionKey,
        });
        let sender = this._sendersMap.get(partitionId || "");
        if (!sender) {
            const partitionPublishingOptions = isDefined(partitionId)
                ? (_a = this._partitionOptions) === null || _a === void 0 ? void 0 : _a[partitionId]
                : undefined;
            sender = EventHubSender.create(this._context, {
                enableIdempotentProducer: Boolean(this._enableIdempotentRetries),
                partitionId,
                partitionPublishingOptions,
            });
            this._sendersMap.set(partitionId || "", sender);
        }
        let maxMessageSize = await sender.getMaxMessageSize({
            retryOptions: this._clientOptions.retryOptions,
            abortSignal: options.abortSignal,
        });
        if (options.maxSizeInBytes) {
            if (options.maxSizeInBytes > maxMessageSize) {
                const error = new Error(`Max message size (${options.maxSizeInBytes} bytes) is greater than maximum message size (${maxMessageSize} bytes) on the AMQP sender link.`);
                logger.warning(`[${this._context.connectionId}] ${error.message}`);
                logErrorStackTrace(error);
                throw error;
            }
            maxMessageSize = options.maxSizeInBytes;
        }
        return new EventDataBatchImpl(this._context, maxMessageSize, Boolean(this._enableIdempotentRetries), options.partitionKey, partitionId);
    }
    /**
     * Get the information about the state of publishing for a partition as observed by the `EventHubProducerClient`.
     * This data can always be read, but will only be populated with information relevant to the active features
     * for the producer client.
     *
     * @param partitionId - Id of the partition from which to retrieve publishing properties.
     * @param options - The set of options to apply to the operation call.
     * - `abortSignal`  : A signal the request to cancel the send operation.
     * @returns Promise<void>
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    // eslint-disable-next-line @typescript-eslint/ban-ts-comment
    // @ts-ignore this is called in EventHubBufferedProducerClient via cast-to-any workaround
    async getPartitionPublishingProperties(partitionId, options = {}) {
        var _a;
        if (!isDefined(partitionId)) {
            throw new TypeError(`getPartitionPublishingProperties called without required argument "partitionId"`);
        }
        if (typeof partitionId === "number") {
            partitionId = String(partitionId);
        }
        let sender = this._sendersMap.get(partitionId);
        if (!sender) {
            sender = EventHubSender.create(this._context, {
                enableIdempotentProducer: Boolean(this._enableIdempotentRetries),
                partitionId,
                partitionPublishingOptions: (_a = this._partitionOptions) === null || _a === void 0 ? void 0 : _a[partitionId],
            });
            this._sendersMap.set(partitionId, sender);
        }
        return sender.getPartitionPublishingProperties(options);
    }
    async sendBatch(batch, options = {}) {
        throwErrorIfConnectionClosed(this._context);
        throwTypeErrorIfParameterMissing(this._context.connectionId, "sendBatch", "batch", batch);
        let partitionId;
        let partitionKey;
        // Holds an EventData properties object containing tracing properties.
        // This lets us avoid cloning batch when it is EventData[], which is
        // important as the idempotent EventHubSender needs to decorate the
        // original EventData passed through.
        const eventDataTracingProperties = [];
        // link message span contexts
        let spanContextsToLink = [];
        if (isEventDataBatch(batch)) {
            if (this._enableIdempotentRetries &&
                isDefined(batch.startingPublishedSequenceNumber)) {
                throw new Error(idempotentAlreadyPublished);
            }
            const partitionAssignment = extractPartitionAssignmentFromBatch(batch, options);
            partitionId = partitionAssignment.partitionId;
            partitionKey = partitionAssignment.partitionKey;
            spanContextsToLink = batch._messageSpanContexts;
        }
        else {
            if (!Array.isArray(batch)) {
                batch = [batch];
            }
            if (batch.some((event) => isDefined(event._publishedSequenceNumber))) {
                throw new Error(idempotentSomeAlreadyPublished);
            }
            const partitionAssignment = extractPartitionAssignmentFromOptions(options);
            partitionId = partitionAssignment.partitionId;
            partitionKey = partitionAssignment.partitionKey;
            for (let i = 0; i < batch.length; i++) {
                batch[i] = instrumentEventData(batch[i], options, this._context.config.entityPath, this._context.config.host).event;
                eventDataTracingProperties[i] = batch[i].properties;
            }
        }
        validateProducerPartitionSettings({
            enableIdempotentRetries: this._enableIdempotentRetries,
            partitionId,
            partitionKey,
        });
        return tracingClient.withSpan(`${EventHubProducerClient.name}.${this.sendBatch.name}`, options, (updatedOptions) => {
            var _a;
            let sender = this._sendersMap.get(partitionId || "");
            if (!sender) {
                const partitionPublishingOptions = isDefined(partitionId)
                    ? (_a = this._partitionOptions) === null || _a === void 0 ? void 0 : _a[partitionId]
                    : undefined;
                sender = EventHubSender.create(this._context, {
                    enableIdempotentProducer: Boolean(this._enableIdempotentRetries),
                    partitionId,
                    partitionPublishingOptions,
                });
                this._sendersMap.set(partitionId || "", sender);
            }
            return sender.send(batch, Object.assign(Object.assign({}, updatedOptions), { partitionId,
                partitionKey, retryOptions: this._clientOptions.retryOptions }));
        }, Object.assign({ spanLinks: spanContextsToLink.map((tracingContext) => {
                return { tracingContext };
            }) }, toSpanOptions(this._context.config, "client")));
    }
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    async close() {
        await this._context.close();
        for (const pair of this._sendersMap) {
            await pair[1].close();
        }
        this._sendersMap.clear();
    }
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options = {}) {
        return this._context.managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options = {}) {
        return this._context
            .managementSession.getEventHubProperties(Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }))
            .then((eventHubProperties) => {
            return eventHubProperties.partitionIds;
        });
    }
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId, options = {}) {
        return this._context.managementSession.getPartitionProperties(partitionId, Object.assign(Object.assign({}, options), { retryOptions: this._clientOptions.retryOptions }));
    }
}
/**
 * @internal
 */
function extractPartitionAssignmentFromOptions(options = {}) {
    const result = {};
    const { partitionId, partitionKey } = options;
    if (isDefined(partitionId)) {
        result.partitionId = String(partitionId);
    }
    if (isDefined(partitionKey)) {
        result.partitionKey = String(partitionKey);
    }
    return result;
}
/**
 * @internal
 */
function extractPartitionAssignmentFromBatch(batch, options) {
    const result = {};
    const partitionId = batch.partitionId;
    const partitionKey = batch.partitionKey;
    const { partitionId: unexpectedPartitionId, partitionKey: unexpectedPartitionKey } = extractPartitionAssignmentFromOptions(options);
    if (unexpectedPartitionKey && partitionKey !== unexpectedPartitionKey) {
        throw new Error(`The partitionKey (${unexpectedPartitionKey}) set on sendBatch does not match the partitionKey (${partitionKey}) set when creating the batch.`);
    }
    if (unexpectedPartitionId && unexpectedPartitionId !== partitionId) {
        throw new Error(`The partitionId (${unexpectedPartitionId}) set on sendBatch does not match the partitionId (${partitionId}) set when creating the batch.`);
    }
    if (isDefined(partitionId)) {
        result.partitionId = String(partitionId);
    }
    if (isDefined(partitionKey)) {
        result.partitionKey = String(partitionKey);
    }
    return result;
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * `AwaitableQueue` stores items in the order that they are received.
 *
 * This differs from ordinary Queues in that `shift` returns a Promise for a value.
 * This allows a consumer of the queue to request an item that the queue does not yet have.
 *
 * @internal
 */
class AwaitableQueue {
    constructor() {
        this._resolvers = [];
        this._items = [];
    }
    size() {
        return this._items.length;
    }
    /**
     * Returns a Promise that will resolve with the next item in the queue.
     */
    shift() {
        const item = this._items.shift();
        if (typeof item !== "undefined") {
            return Promise.resolve(item);
        }
        return new Promise((resolve) => this._resolvers.push(resolve));
    }
    /**
     * Appends new item to the queue.
     */
    push(item) {
        if (!this._resolveNextItem(item)) {
            this._items.push(item);
        }
    }
    _resolveNextItem(item) {
        const resolver = this._resolvers.shift();
        if (!resolver) {
            return false;
        }
        resolver(item);
        return true;
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * @internal
 * Returns a promise and the promise's resolve and reject methods.
 */
function getPromiseParts() {
    let resolver;
    let rejector;
    const promise = new Promise((resolve, reject) => {
        resolver = resolve;
        rejector = reject;
    });
    return {
        promise,
        resolve: resolver,
        reject: rejector,
    };
}

// Copyright (c) Microsoft Corporation.
/**
 * The `BatchingPartitionChannel` is responsible for accepting enqueued events
 * and optimally batching and sending them to an Event Hub.
 * @internal
 */
class BatchingPartitionChannel {
    constructor({ loopAbortSignal, maxBufferSize, maxWaitTimeInMs, onSendEventsErrorHandler, onSendEventsSuccessHandler, partitionId, producer, }) {
        this._eventQueue = new AwaitableQueue();
        this._batchedEvents = [];
        this._bufferCount = 0;
        this._readyQueue = [];
        this._flushState = {
            isFlushing: false,
        };
        this._isRunning = false;
        this._lastBatchCreationTime = 0;
        this._loopAbortSignal = loopAbortSignal;
        this._maxBufferSize = maxBufferSize;
        this._maxWaitTimeInMs = maxWaitTimeInMs;
        this._onSendEventsErrorHandler = onSendEventsErrorHandler;
        this._onSendEventsSuccessHandler = onSendEventsSuccessHandler;
        this._partitionId = partitionId;
        this._producer = producer;
    }
    getCurrentBufferedCount() {
        return this._bufferCount;
    }
    async enqueueEvent(event) {
        await this._ready();
        this._eventQueue.push(event);
        this._bufferCount++;
        if (!this._isRunning) {
            this._isRunning = true;
            this._startPublishLoop().catch((e) => {
                logger.error(`The following error occured during batch creation or sending: ${JSON.stringify(e, undefined, "  ")}`);
            });
        }
    }
    /**
     * Sets the flush state so that no new events can be enqueued until
     * all the currently buffered events are sent to the Event Hub.
     *
     * Returns a promise that resolves once flushing is complete.
     */
    async flush(_options = {}) {
        const state = this._flushState;
        if (state.isFlushing) {
            return state.currentPromise;
        }
        if (this.getCurrentBufferedCount() === 0) {
            return Promise.resolve();
        }
        const { promise, resolve } = getPromiseParts();
        this._flushState = { isFlushing: true, currentPromise: promise, resolve };
        return promise;
    }
    /**
     * Returns a promise that resolves once there is room for events to be added
     * to the buffer.
     */
    _ready() {
        const currentBufferedCount = this.getCurrentBufferedCount();
        // If the buffer isn't full and we don't have any pending `ready()` calls,
        // then it's safe to return right away.
        if (currentBufferedCount < this._maxBufferSize &&
            !this._readyQueue.length &&
            !this._flushState.isFlushing) {
            return Promise.resolve();
        }
        const { promise: readyPromise, reject, resolve } = getPromiseParts();
        this._readyQueue.push({ resolve, reject });
        return readyPromise;
    }
    /**
     * Starts the loop that creates batches and sends them to the Event Hub.
     *
     * The loop will run until the `_loopAbortSignal` is aborted.
     */
    async _startPublishLoop() {
        let batch;
        let futureEvent = this._eventQueue.shift();
        // `eventToAddToBatch` is used to keep track of an event that has been removed
        // from the queue, but has not yet been added to a batch.
        // This prevents losing an event if a `sendBatch` or `createBatch` call fails
        // before the event is added to a batch.
        let eventToAddToBatch;
        while (!this._loopAbortSignal.aborted) {
            try {
                if (!isDefined(batch)) {
                    batch = await this._createBatch();
                }
                const timeSinceLastBatchCreation = Date.now() - this._lastBatchCreationTime;
                const maximumTimeToWaitForEvent = batch.count
                    ? Math.max(this._maxWaitTimeInMs - timeSinceLastBatchCreation, 0)
                    : this._maxWaitTimeInMs;
                const event = eventToAddToBatch !== null && eventToAddToBatch !== void 0 ? eventToAddToBatch : (await Promise.race([futureEvent, coreAmqp.delay(maximumTimeToWaitForEvent)]));
                if (!event) {
                    // We didn't receive an event within the allotted time.
                    // Send the existing batch if it has events in it.
                    if (batch.count) {
                        await this._producer.sendBatch(batch);
                        this._reportSuccess();
                        batch = await this._createBatch();
                    }
                    continue;
                }
                else if (!eventToAddToBatch) {
                    eventToAddToBatch = event;
                    // We received an event, so get a promise for the next one.
                    futureEvent = this._eventQueue.shift();
                }
                const didAdd = batch.tryAdd(event);
                if (didAdd) {
                    // This event will definitely make it to one of the user-provided handlers
                    // since it was added to a batch.
                    // Store it so we can return it in a handler later.
                    this._batchedEvents.push(event);
                    // Clear reference to existing event since it has been added to the batch.
                    eventToAddToBatch = undefined;
                }
                if (didAdd && batch.count >= this._maxBufferSize) {
                    // Whenever batch.count exceeds the max count of buffered events, send the batch.
                    await this._producer.sendBatch(batch);
                    this._reportSuccess();
                    batch = await this._createBatch();
                }
                else if (!didAdd && batch.count) {
                    // If the event wasn't able to be added and the current batch isn't empty,
                    // attempt to send the current batch and add the event to a new batch.
                    await this._producer.sendBatch(batch);
                    this._reportSuccess();
                    batch = await this._createBatch();
                }
                if (!didAdd && !batch.tryAdd(event)) {
                    // TODO: Report MaxMesageSizeExceeded error. Mimic service's error.
                    this._reportFailure(new Error("Placeholder for max message size exceeded"), event);
                }
                else if (!didAdd) {
                    // Handles the case where the event _was_ successfull added to the new batch.
                    this._batchedEvents.push(event);
                }
                // Clear reference to existing event since it has been added to the batch.
                eventToAddToBatch = undefined;
            }
            catch (err) {
                if (!isObjectWithProperties(err, ["name"]) || err.name !== "AbortError") {
                    this._reportFailure(err);
                    batch = undefined;
                    this._batchedEvents = [];
                }
            }
        }
    }
    /**
     * Helper method that returns an `EventDataBatch`.
     * This also has the side effects of
     *  - keeping track of batch creation time: needed for maxWaitTime calculations.
     *  - clearing reference to batched events.
     *  - incrementing the readiness: creating a new batch indicates the buffer
     *    should have room, so we can resolve some pending `ready()` calls.
     */
    async _createBatch() {
        this._lastBatchCreationTime = Date.now();
        this._batchedEvents = [];
        const batch = await this._producer.createBatch({
            partitionId: this._partitionId,
        });
        this._incrementReadiness();
        return batch;
    }
    /**
     * This method will resolve as many pending `ready()` calls as it can
     * based on how much space remains in the buffer.
     *
     * If the channel is currently flushing, this is a no-op. This prevents
     * `enqueueEvent` calls from adding the event to the buffer until flushing
     * completes.
     */
    _incrementReadiness() {
        var _a;
        if (this._flushState.isFlushing) {
            return;
        }
        const currentBufferedCount = this.getCurrentBufferedCount();
        const num = Math.min(this._maxBufferSize - currentBufferedCount, this._readyQueue.length);
        for (let i = 0; i < num; i++) {
            (_a = this._readyQueue.shift()) === null || _a === void 0 ? void 0 : _a.resolve();
        }
    }
    /**
     * Calls the user-provided `onSendEventsSuccessHandler` with the events
     * that were successfully sent.
     */
    _reportSuccess() {
        var _a;
        this._bufferCount = this._bufferCount - this._batchedEvents.length;
        this._updateFlushState();
        try {
            (_a = this._onSendEventsSuccessHandler) === null || _a === void 0 ? void 0 : _a.call(this, {
                events: this._batchedEvents,
                partitionId: this._partitionId,
            });
        }
        catch (e) {
            logger.error(`The following error occurred in the onSendEventsSuccessHandler: ${JSON.stringify(e, undefined, "  ")}`);
        }
    }
    /**
     * Calls the user-provided `onSendEventsErrorHandler` with an error and the events
     * that were not successfully sent.
     */
    _reportFailure(err, event) {
        this._bufferCount = this._bufferCount - (event ? 1 : this._batchedEvents.length);
        this._updateFlushState();
        try {
            this._onSendEventsErrorHandler({
                error: err,
                events: event ? [event] : this._batchedEvents,
                partitionId: this._partitionId,
            });
        }
        catch (e) {
            logger.error(`The following error occurred in the onSendEventsErrorHandler: ${JSON.stringify(e, undefined, "  ")}`);
        }
    }
    /**
     * Updates the channel's flush state once the size of the
     * event buffer has decreased to 0.
     */
    _updateFlushState() {
        const state = this._flushState;
        if (!state.isFlushing || this.getCurrentBufferedCount() !== 0) {
            return;
        }
        state.resolve();
        this._flushState = { isFlushing: false };
        this._incrementReadiness();
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/* eslint-disable no-fallthrough */
function mapPartitionKeyToId(partitionKey, partitionCount) {
    const hashedParitionKey = hashPartitionKey(partitionKey);
    return Math.abs(hashedParitionKey % partitionCount);
}
/**
 * @internal
 */
function hashPartitionKey(partitionKey) {
    const hash = computeHash(Buffer.from(partitionKey, "utf8"));
    return castToInt16(hash.c ^ hash.b);
}
function readUInt32(data, offset) {
    return data.readUInt32LE(offset);
}
function castToInt16(n) {
    return new Int16Array([n])[0];
}
function computeHash(data, seed1 = 0, seed2 = 0) {
    let a, b, c;
    a = b = c = 0xdeadbeef + data.length + seed1;
    c += seed2;
    let index = 0, size = data.length;
    while (size > 12) {
        a += readUInt32(data, index);
        b += readUInt32(data, index + 4);
        c += readUInt32(data, index + 8);
        a -= c;
        a ^= (c << 4) | (c >>> 28);
        c += b;
        b -= a;
        b ^= (a << 6) | (a >>> 26);
        a += c;
        c -= b;
        c ^= (b << 8) | (b >>> 24);
        b += a;
        a -= c;
        a ^= (c << 16) | (c >>> 16);
        c += b;
        b -= a;
        b ^= (a << 19) | (a >>> 13);
        a += c;
        c -= b;
        c ^= (b << 4) | (b >>> 28);
        b += a;
        index += 12;
        size -= 12;
    }
    let curr = size;
    switch (curr) {
        case 12:
            a += readUInt32(data, index);
            b += readUInt32(data, index + 4);
            c += readUInt32(data, index + 8);
            break;
        case 11:
            c += data[index + 10] << 16;
            curr = 10;
        case 10:
            c += data[index + 9] << 8;
            curr = 9;
        case 9:
            c += data[index + 8];
            curr = 8;
        case 8:
            b += readUInt32(data, index + 4);
            a += readUInt32(data, index);
            break;
        case 7:
            b += data[index + 6] << 16;
            curr = 6;
        case 6:
            b += data[index + 5] << 8;
            curr = 5;
        case 5:
            b += data[index + 4];
            curr = 4;
        case 4:
            a += readUInt32(data, index);
            break;
        case 3:
            a += data[index + 2] << 16;
            curr = 2;
        case 2:
            a += data[index + 1] << 8;
            curr = 1;
        case 1:
            a += data[index];
            break;
        case 0:
            return { b: b >>> 0, c: c >>> 0 };
    }
    c ^= b;
    c -= (b << 14) | (b >>> 18);
    a ^= c;
    a -= (c << 11) | (c >>> 21);
    b ^= a;
    b -= (a << 25) | (a >>> 7);
    c ^= b;
    c -= (b << 16) | (b >>> 16);
    a ^= c;
    a -= (c << 4) | (c >>> 28);
    b ^= a;
    b -= (a << 14) | (a >>> 18);
    c ^= b;
    c -= (b << 24) | (b >>> 8);
    return { b: b >>> 0, c: c >>> 0 };
}

// Copyright (c) Microsoft Corporation.
/**
 * @internal
 * Assigns a partition based on the partition ids it knows about and an optional partition id or partition key.
 */
class PartitionAssigner {
    constructor() {
        this._partitions = [];
        this._lastRoundRobinPartitionIndex = -1;
    }
    /**
     * Set the partition ids that can be used when assigning a partition.
     * @param partitionIds - All valid partition ids.
     */
    setPartitionIds(partitionIds) {
        this._partitions = partitionIds;
    }
    /**
     * Returns a partitionId from the list of partition ids set via `setPartitionIds`.
     *
     * If a partitionId is specified, then that will be returned directly.
     * If a partitionKey is specified, then a partitionId will be calculated based on the partitionKey.
     * Specifying both partitionId and partitionKey results in an error.
     *
     * If neither partitionId nor partitionKey are specified, then a partitionId will be selected
     * based on a round-robin approach.
     */
    assignPartition({ partitionId, partitionKey, }) {
        if (isDefined(partitionId) && isDefined(partitionKey)) {
            throw new Error(`The partitionId (${partitionId}) and partitionKey (${partitionKey}) cannot both be specified.`);
        }
        if (!this._partitions.length) {
            throw new Error(`Unable to determine partitionIds, can't assign partitionId.`);
        }
        if (isDefined(partitionId) && this._partitions.includes(partitionId)) {
            return partitionId;
        }
        if (isDefined(partitionKey)) {
            return mapPartitionKeyToId(partitionKey, this._partitions.length).toString();
        }
        return this._assignRoundRobinPartition();
    }
    _assignRoundRobinPartition() {
        const maxPartitionIndex = this._partitions.length - 1;
        const proposedPartitionIndex = this._lastRoundRobinPartitionIndex + 1;
        const nextPartitionIndex = proposedPartitionIndex > maxPartitionIndex ? 0 : proposedPartitionIndex;
        this._lastRoundRobinPartitionIndex = nextPartitionIndex;
        return this._partitions[nextPartitionIndex];
    }
}

// Copyright (c) Microsoft Corporation.
/**
 * The `EventHubBufferedProducerClient`is used to publish events to a specific Event Hub.
 *
 * The `EventHubBufferedProducerClient` does not publish events immediately.
 * Instead, events are buffered so they can be efficiently batched and published
 * when the batch is full or the `maxWaitTimeInMs` has elapsed with no new events
 * enqueued.
 *
 * Depending on the options specified when events are enqueued, they may be
 * automatically assigned to a partition, grouped according to the specified partition key,
 * or assigned a specifically requested partition.
 *
 * This model is intended to shift the burden of batch management from callers, at the cost of
 * non-deterministic timing, for when events will be published. There are additional trade-offs
 * to consider, as well:
 * - If the application crashes, events in the buffer will not have been published. To prevent
 *   data loss, callers are encouraged to track publishing progress using the
 *   `onSendEventsSuccessHandler` and `onSendEventsErrorHandler` handlers.
 * - Events specifying a partition key may be assigned a different partition than those using
 *   the same key with other producers.
 * - In the unlikely event that a partition becomes temporarily unavailable, the
 *   `EventHubBufferedProducerClient` may take longer to recover than other producers.
 *
 * In scenarios where it is important to have events published immediately with a deterministic
 * outcome, ensure that partition keys are assigned to a partition consistent with other
 * publishers, or where maximizing availability is a requirement, using the
 * `EventHubProducerClient` is recommended.
 */
class EventHubBufferedProducerClient {
    constructor(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4) {
        /**
         * Controls the `abortSignal` passed to each `BatchingPartitionChannel`.
         * Used to signal when a channel should stop waiting for events.
         */
        this._abortController = new abortController.AbortController();
        /**
         * Indicates whether the client has been explicitly closed.
         */
        this._isClosed = false;
        /**
         * Handles assigning partitions.
         */
        this._partitionAssigner = new PartitionAssigner();
        /**
         * The known partitionIds that will be used when assigning events to partitions.
         */
        this._partitionIds = [];
        /**
         * Mapping of partitionIds to `BatchingPartitionChannels`.
         * Each `BatchingPartitionChannel` handles buffering events and backpressure independently.
         */
        this._partitionChannels = new Map();
        /**
         * The interval at which the background management operation should run.
         */
        this._backgroundManagementInterval = 10000; // 10 seconds
        /**
         * Indicates whether the background management loop is running.
         */
        this._isBackgroundManagementRunning = false;
        if (typeof eventHubNameOrOptions2 !== "string") {
            this._producer = new EventHubProducerClient(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2);
            this._clientOptions = Object.assign({}, eventHubNameOrOptions2);
        }
        else if (!isCredential(credentialOrOptions3)) {
            this._producer = new EventHubProducerClient(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3);
            this._clientOptions = Object.assign({}, credentialOrOptions3);
        }
        else {
            this._producer = new EventHubProducerClient(fullyQualifiedNamespaceOrConnectionString1, eventHubNameOrOptions2, credentialOrOptions3, options4);
            this._clientOptions = Object.assign({}, options4);
        }
        // setting internal idempotent publishing option on the standard producer.
        this._producer._enableIdempotentRetries = this._clientOptions.enableIdempotentRetries;
    }
    /**
     * @readonly
     * The name of the Event Hub instance for which this client is created.
     */
    get eventHubName() {
        return this._producer.eventHubName;
    }
    /**
     * @readonly
     * The fully qualified namespace of the Event Hub instance for which this client is created.
     * This is likely to be similar to <yournamespace>.servicebus.windows.net.
     */
    get fullyQualifiedNamespace() {
        return this._producer.fullyQualifiedNamespace;
    }
    /**
     * Closes the AMQP connection to the Event Hub instance,
     * returning a promise that will be resolved when disconnection is completed.
     *
     * This will wait for enqueued events to be flushed to the service before closing
     * the connection.
     * To close without flushing, set the `flush` option to `false`.
     *
     * @param options - The set of options to apply to the operation call.
     * @returns Promise<void>
     * @throws Error if the underlying connection encounters an error while closing.
     */
    async close(options = {}) {
        logger.verbose("closing buffered producer client...");
        if (!isDefined(options.flush) || options.flush === true) {
            await this.flush(options);
        }
        // Calling abort signals to the BatchingPartitionChannels that they
        // should stop reading/sending events, and to the background management
        // loop that it should stop periodic partition id updates.
        this._abortController.abort();
        await this._producer.close();
        this._isClosed = true;
    }
    /**
     * Enqueues an event into the buffer to be published to the Event Hub.
     * If there is no capacity in the buffer when this method is invoked,
     * it will wait for space to become available and ensure that the event
     * has been enqueued.
     *
     * When this call returns, the event has been accepted into the buffer,
     * but it may not have been published yet.
     * Publishing will take place at a nondeterministic point in the future as the buffer is processed.
     *
     * @param events - An {@link EventData} or `AmqpAnnotatedMessage`.
     * @param options - A set of options that can be specified to influence the way in which
     * the event is sent to the associated Event Hub.
     * - `abortSignal`  : A signal used to cancel the enqueueEvent operation.
     * - `partitionId`  : The partition this set of events will be sent to. If set, `partitionKey` can not be set.
     * - `partitionKey` : A value that is hashed to produce a partition assignment. If set, `partitionId` can not be set.
     * @returns The total number of events that are currently buffered and waiting to be published, across all partitions.
     */
    async enqueueEvent(event, options = {}) {
        if (this._isClosed) {
            throw new Error(`This EventHubBufferedProducerClient has already been closed. Create a new client to enqueue events.`);
        }
        if (!this._partitionIds.length) {
            await this._updatePartitionIds();
        }
        if (!this._isBackgroundManagementRunning) {
            this._startPartitionIdsUpdateLoop().catch((e) => {
                logger.error(`The following error occured during batch creation or sending: ${JSON.stringify(e, undefined, "  ")}`);
            });
            this._isBackgroundManagementRunning = true;
        }
        const partitionId = this._partitionAssigner.assignPartition({
            partitionId: options.partitionId,
            partitionKey: options.partitionKey,
        });
        const partitionChannel = this._getPartitionChannel(partitionId);
        await partitionChannel.enqueueEvent(event);
        return this._getTotalBufferedEventsCount();
    }
    /**
     * Enqueues events into the buffer to be published to the Event Hub.
     * If there is no capacity in the buffer when this method is invoked,
     * it will wait for space to become available and ensure that the events
     * have been enqueued.
     *
     * When this call returns, the events have been accepted into the buffer,
     * but it may not have been published yet.
     * Publishing will take place at a nondeterministic point in the future as the buffer is processed.
     *
     * @param events - An array of {@link EventData} or `AmqpAnnotatedMessage`.
     * @param options - A set of options that can be specified to influence the way in which
     * events are sent to the associated Event Hub.
     * - `abortSignal`  : A signal used to cancel the enqueueEvents operation.
     * - `partitionId`  : The partition this set of events will be sent to. If set, `partitionKey` can not be set.
     * - `partitionKey` : A value that is hashed to produce a partition assignment. If set, `partitionId` can not be set.
     * @returns The total number of events that are currently buffered and waiting to be published, across all partitions.
     */
    async enqueueEvents(events, options = {}) {
        for (const event of events) {
            await this.enqueueEvent(event, options);
        }
        return this._getTotalBufferedEventsCount();
    }
    /**
     * Attempts to publish all events in the buffer immediately.
     * This may result in multiple batches being published,
     * the outcome of each of which will be individually reported by
     * the `onSendEventsSuccessHandler` and `onSendEventsErrorHandler` handlers.
     *
     * @param options - The set of options to apply to the operation call.
     */
    async flush(options = {}) {
        await Promise.all(Array.from(this._partitionChannels.values()).map((channel) => channel.flush(options)));
    }
    /**
     * Provides the Event Hub runtime information.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the Event Hub instance.
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getEventHubProperties(options = {}) {
        return this._producer.getEventHubProperties(options);
    }
    /**
     * Provides the id for each partition associated with the Event Hub.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with an Array of strings representing the id for
     * each partition associated with the Event Hub.
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionIds(options = {}) {
        return this._producer.getPartitionIds(options);
    }
    /**
     * Provides information about the state of the specified partition.
     * @param partitionId - The id of the partition for which information is required.
     * @param options - The set of options to apply to the operation call.
     * @returns A promise that resolves with information about the state of the partition .
     * @throws Error if the underlying connection has been closed, create a new EventHubBufferedProducerClient.
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    getPartitionProperties(partitionId, options = {}) {
        return this._producer.getPartitionProperties(partitionId, options);
    }
    /**
     * Gets the `BatchingPartitionChannel` associated with the partitionId.
     *
     * If one does not exist, it is created.
     */
    _getPartitionChannel(partitionId) {
        var _a;
        const partitionChannel = (_a = this._partitionChannels.get(partitionId)) !== null && _a !== void 0 ? _a : new BatchingPartitionChannel({
            loopAbortSignal: this._abortController.signal,
            maxBufferSize: this._clientOptions.maxEventBufferLengthPerPartition || 1500,
            maxWaitTimeInMs: this._clientOptions.maxWaitTimeInMs || 1000,
            onSendEventsErrorHandler: this._clientOptions.onSendEventsErrorHandler,
            onSendEventsSuccessHandler: this._clientOptions.onSendEventsSuccessHandler,
            partitionId,
            producer: this._producer,
        });
        this._partitionChannels.set(partitionId, partitionChannel);
        return partitionChannel;
    }
    /**
     * Returns the total number of buffered events across all partitions.
     */
    _getTotalBufferedEventsCount() {
        let total = 0;
        for (const [_, channel] of this._partitionChannels) {
            total += channel.getCurrentBufferedCount();
        }
        return total;
    }
    async _updatePartitionIds() {
        logger.verbose("Checking for partition Id updates...");
        const queriedPartitionIds = await this.getPartitionIds();
        if (this._partitionIds.length !== queriedPartitionIds.length) {
            logger.verbose("Applying partition Id updates");
            this._partitionIds = queriedPartitionIds;
            this._partitionAssigner.setPartitionIds(this._partitionIds);
        }
    }
    async _startPartitionIdsUpdateLoop() {
        logger.verbose("Starting a background loop to check and apply partition id updates...");
        while (!this._abortController.signal.aborted && !this._isClosed) {
            await coreAmqp.delay(this._backgroundManagementInterval);
            if (!this._isClosed) {
                await this._updatePartitionIds();
            }
        }
    }
}

// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
/**
 * A function that constructs an event data adapter. That adapter can be used
 * with `@azure/schema-registry-avro` to encode and decode body in event data.
 *
 * @param params - parameters to create the event data
 * @returns An event data adapter that can produce and consume event data
 */
function createEventDataAdapter(params = {}) {
    return {
        produce: ({ data: body, contentType }) => {
            return Object.assign(Object.assign({}, params), { body,
                contentType });
        },
        consume: (message) => {
            const { body, contentType } = message;
            if (body === undefined) {
                throw new Error("Expected the body field to be defined");
            }
            if (contentType === undefined) {
                throw new Error("Expected the contentType field to be defined");
            }
            return {
                /**
                 * If the raw response was parsed as JSON, we need to convert it to a Uint8Array,
                 * otherwise, leave the payload as is.
                 */
                data: typeof body === "object" ? Uint8Array.from(Object.values(body)) : body,
                contentType,
            };
        },
    };
}

Object.defineProperty(exports, 'MessagingError', {
    enumerable: true,
    get: function () { return coreAmqp.MessagingError; }
});
Object.defineProperty(exports, 'RetryMode', {
    enumerable: true,
    get: function () { return coreAmqp.RetryMode; }
});
exports.EventHubBufferedProducerClient = EventHubBufferedProducerClient;
exports.EventHubConsumerClient = EventHubConsumerClient;
exports.EventHubProducerClient = EventHubProducerClient;
exports.createEventDataAdapter = createEventDataAdapter;
exports.earliestEventPosition = earliestEventPosition;
exports.latestEventPosition = latestEventPosition;
exports.logger = logger;
exports.parseEventHubConnectionString = parseEventHubConnectionString;
//# sourceMappingURL=index.js.map
