/// <reference path="../../@types/bonjour-hap.d.ts" />
/// <reference types="node" />
import { MDNSServerOptions, ServiceTxt } from "@homebridge/ciao";
import { InterfaceName, IPAddress } from "@homebridge/ciao/lib/NetworkManager";
import { InvokeError } from "@homebridge/dbus-native";
import { MulticastOptions } from "bonjour-hap";
import { EventEmitter } from "events";
import { AccessoryInfo } from "./model/AccessoryInfo";
/**
 * This enum lists all bitmasks for all known status flags.
 * When the bit for the given bitmask is set, it represents the state described by the name.
 *
 * @group Advertiser
 */
export declare const enum StatusFlag {
    NOT_PAIRED = 1,
    NOT_JOINED_WIFI = 2,
    PROBLEM_DETECTED = 4
}
/**
 * This enum lists all bitmasks for all known pairing feature flags.
 * When the bit for the given bitmask is set, it represents the state described by the name.
 *
 * @group Advertiser
 */
export declare const enum PairingFeatureFlag {
    SUPPORTS_HARDWARE_AUTHENTICATION = 1,
    SUPPORTS_SOFTWARE_AUTHENTICATION = 2
}
/**
 * @group Advertiser
 */
export declare const enum AdvertiserEvent {
    /**
     * Emitted if the underlying mDNS advertisers signals, that the service name
     * was automatically changed due to some naming conflicts on the network.
     */
    UPDATED_NAME = "updated-name"
}
/**
 * @group Advertiser
 */
export declare interface Advertiser {
    on(event: "updated-name", listener: (name: string) => void): this;
    emit(event: "updated-name", name: string): boolean;
}
/**
 * @group Advertiser
 */
export interface ServiceNetworkOptions {
    /**
     * If defined it restricts the service to be advertised on the specified
     * ip addresses or interface names.
     *
     * If an interface name is specified, ANY address on that given interface will be advertised
     * (if an IP address of the given interface is also given in the array, it will be overridden).
     * If an IP address is specified, the service will only be advertised for the given addresses.
     *
     * Interface names and addresses can be mixed in the array.
     * If an ip address is given, the ip address must be valid at the time of service creation.
     *
     * If the service is set to advertise on a given interface, though the MDNSServer is
     * configured to ignore this interface, the service won't be advertised on the interface.
     */
    restrictedAddresses?: (InterfaceName | IPAddress)[];
    /**
     * The service won't advertise ipv6 address records.
     * This can be used to simulate binding on 0.0.0.0.
     * May be combined with {@link restrictedAddresses}.
     */
    disabledIpv6?: boolean;
}
/**
 * A generic Advertiser interface required for any MDNS Advertiser backend implementations.
 *
 * All implementations have to extend NodeJS' {@link EventEmitter} and emit the events defined in {@link AdvertiserEvent}.
 *
 * @group Advertiser
 */
export interface Advertiser {
    initPort(port: number): void;
    startAdvertising(): Promise<void>;
    updateAdvertisement(silent?: boolean): void;
    destroy(): void;
}
/**
 * Advertiser uses mdns to broadcast the presence of an Accessory to the local network.
 *
 * Note that as of iOS 9, an accessory can only pair with a single client. Instead of pairing your
 * accessories with multiple iOS devices in your home, Apple intends for you to use Home Sharing.
 * To support this requirement, we provide the ability to be "discoverable" or not (via a "service flag" on the
 * mdns payload).
 *
 * @group Advertiser
 */
export declare class CiaoAdvertiser extends EventEmitter implements Advertiser {
    static protocolVersion: string;
    static protocolVersionService: string;
    private readonly accessoryInfo;
    private readonly setupHash;
    private readonly responder;
    private readonly advertisedService;
    constructor(accessoryInfo: AccessoryInfo, responderOptions?: MDNSServerOptions, serviceOptions?: ServiceNetworkOptions);
    initPort(port: number): void;
    startAdvertising(): Promise<void>;
    updateAdvertisement(silent?: boolean): void;
    destroy(): Promise<void>;
    static createTxt(accessoryInfo: AccessoryInfo, setupHash: string): ServiceTxt;
    static computeSetupHash(accessoryInfo: AccessoryInfo): string;
    static ff(...flags: PairingFeatureFlag[]): number;
    static sf(...flags: StatusFlag[]): number;
}
/**
 * Advertiser base on the legacy "bonjour-hap" library.
 *
 * @group Advertiser
 */
export declare class BonjourHAPAdvertiser extends EventEmitter implements Advertiser {
    private readonly accessoryInfo;
    private readonly setupHash;
    private readonly serviceOptions?;
    private bonjour;
    private advertisement?;
    private port?;
    private destroyed;
    constructor(accessoryInfo: AccessoryInfo, responderOptions?: MulticastOptions, serviceOptions?: ServiceNetworkOptions);
    initPort(port: number): void;
    startAdvertising(): Promise<void>;
    updateAdvertisement(silent?: boolean): void;
    destroy(): void;
}
/**
 * @group Advertiser
 */
export declare class DBusInvokeError extends Error {
    readonly errorName: string;
    constructor(errorObject: InvokeError);
}
/**
 * Advertiser based on the Avahi D-Bus library.
 * For (very crappy) docs on the interface, see the XML files at: https://github.com/lathiat/avahi/tree/master/avahi-daemon.
 *
 * Refer to https://github.com/lathiat/avahi/blob/fd482a74625b8db8547b8cfca3ee3d3c6c721423/avahi-common/defs.h#L120-L155 for a
 * rough API usage guide of Avahi.
 *
 * @group Advertiser
 */
export declare class AvahiAdvertiser extends EventEmitter implements Advertiser {
    private readonly accessoryInfo;
    private readonly setupHash;
    private port?;
    private bus?;
    private avahiServerInterface?;
    private path?;
    private readonly stateChangeHandler;
    constructor(accessoryInfo: AccessoryInfo);
    private createTxt;
    initPort(port: number): void;
    startAdvertising(): Promise<void>;
    /**
     * Event handler for the `StateChanged` event of the `org.freedesktop.Avahi.Server` DBus interface.
     *
     * This is called once the state of the running avahi-daemon changes its running state.
     * @param state - The state the server changed into {@see AvahiServerState}.
     */
    private handleStateChangedEvent;
    updateAdvertisement(silent?: boolean): Promise<void>;
    destroy(): Promise<void>;
    static isAvailable(): Promise<boolean>;
    private static avahiInvoke;
    private static avahiInterface;
}
/**
 * Advertiser based on the systemd-resolved D-Bus library.
 * For docs on the interface, see: https://www.freedesktop.org/software/systemd/man/org.freedesktop.resolve1.html
 *
 * @group Advertiser
 */
export declare class ResolvedAdvertiser extends EventEmitter implements Advertiser {
    private readonly accessoryInfo;
    private readonly setupHash;
    private port?;
    private bus?;
    private path?;
    constructor(accessoryInfo: AccessoryInfo);
    private createTxt;
    initPort(port: number): void;
    startAdvertising(): Promise<void>;
    updateAdvertisement(silent?: boolean): Promise<void>;
    private stopAdvertising;
    destroy(): Promise<void>;
    static isAvailable(): Promise<boolean>;
    private static resolvedInvoke;
    private static managerInvoke;
}
//# sourceMappingURL=Advertiser.d.ts.map