/// <reference types="node" />
import { EventEmitter } from "events";
import { CharacteristicJsonObject, CharacteristicValue, Nullable, PartialAllowingNull, VoidCallback } from "../types";
import { CharacteristicWarningType } from "./Accessory";
import type { AccessCodeControlPoint, AccessCodeSupportedConfiguration, AccessControlLevel, AccessoryFlags, AccessoryIdentifier, Active, ActiveIdentifier, ActivityInterval, AdministratorOnlyAccess, AirParticulateDensity, AirParticulateSize, AirPlayEnable, AirQuality, AppMatchingIdentifier, AssetUpdateReadiness, AudioFeedback, BatteryLevel, Brightness, ButtonEvent, CameraOperatingModeIndicator, CarbonDioxideDetected, CarbonDioxideLevel, CarbonDioxidePeakLevel, CarbonMonoxideDetected, CarbonMonoxideLevel, CarbonMonoxidePeakLevel, Category, CCAEnergyDetectThreshold, CCASignalDetectThreshold, CharacteristicValueActiveTransitionCount, CharacteristicValueTransitionControl, ChargingState, ClosedCaptions, ColorTemperature, ConfigurationState, ConfigureBridgedAccessory, ConfigureBridgedAccessoryStatus, ConfiguredName, ContactSensorState, CoolingThresholdTemperature, CryptoHash, CurrentAirPurifierState, CurrentAmbientLightLevel, CurrentDoorState, CurrentFanState, CurrentHeaterCoolerState, CurrentHeatingCoolingState, CurrentHorizontalTiltAngle, CurrentHumidifierDehumidifierState, CurrentMediaState, CurrentPosition, CurrentRelativeHumidity, CurrentSlatState, CurrentTemperature, CurrentTiltAngle, CurrentTime, CurrentTransport, CurrentVerticalTiltAngle, CurrentVisibilityState, DataStreamHAPTransport, DataStreamHAPTransportInterrupt, DayoftheWeek, DiagonalFieldOfView, DigitalZoom, DiscoverBridgedAccessories, DiscoveredBridgedAccessories, DisplayOrder, EventRetransmissionMaximum, EventSnapshotsActive, EventTransmissionCounters, FilterChangeIndication, FilterLifeLevel, FirmwareRevision, FirmwareUpdateReadiness, FirmwareUpdateStatus, HardwareFinish, HardwareRevision, HeartBeat, HeatingThresholdTemperature, HoldPosition, HomeKitCameraActive, Hue, Identifier, Identify, ImageMirroring, ImageRotation, InputDeviceType, InputSourceType, InUse, IsConfigured, LeakDetected, LinkQuality, ListPairings, LockControlPoint, LockCurrentState, LockLastKnownAction, LockManagementAutoSecurityTimeout, LockPhysicalControls, LockTargetState, Logs, MACRetransmissionMaximum, MACTransmissionCounters, ManagedNetworkEnable, ManuallyDisabled, Manufacturer, MaximumTransmitPower, MetricsBufferFullState, Model, MotionDetected, MultifunctionButton, Mute, Name, NetworkAccessViolationControl, NetworkClientProfileControl, NetworkClientStatusControl, NFCAccessControlPoint, NFCAccessSupportedConfiguration, NightVision, NitrogenDioxideDensity, ObstructionDetected, OccupancyDetected, On, OperatingStateResponse, OpticalZoom, OutletInUse, OzoneDensity, PairingFeatures, PairSetup, PairVerify, PasswordSetting, PeriodicSnapshotsActive, PictureMode, Ping, PM10Density, PM2_5Density, PositionState, PowerModeSelection, ProductData, ProgrammableSwitchEvent, ProgrammableSwitchOutputState, ProgramMode, Reachable, ReceivedSignalStrengthIndication, ReceiverSensitivity, RecordingAudioActive, RelativeHumidityDehumidifierThreshold, RelativeHumidityHumidifierThreshold, RelayControlPoint, RelayEnabled, RelayState, RemainingDuration, RemoteKey, ResetFilterIndication, RotationDirection, RotationSpeed, RouterStatus, Saturation, SecuritySystemAlarmType, SecuritySystemCurrentState, SecuritySystemTargetState, SelectedAudioStreamConfiguration, SelectedCameraRecordingConfiguration, SelectedDiagnosticsModes, SelectedRTPStreamConfiguration, SelectedSleepConfiguration, SerialNumber, ServiceLabelIndex, ServiceLabelNamespace, SetDuration, SetupDataStreamTransport, SetupEndpoints, SetupTransferTransport, SignalToNoiseRatio, SiriEnable, SiriEndpointSessionStatus, SiriEngineVersion, SiriInputType, SiriLightOnUse, SiriListening, SiriTouchToUse, SlatType, SleepDiscoveryMode, SleepInterval, SmokeDetected, SoftwareRevision, StagedFirmwareVersion, StatusActive, StatusFault, StatusJammed, StatusLowBattery, StatusTampered, StreamingStatus, SulphurDioxideDensity, SupportedAssetTypes, SupportedAudioRecordingConfiguration, SupportedAudioStreamConfiguration, SupportedCameraRecordingConfiguration, SupportedCharacteristicValueTransitionConfiguration, SupportedDataStreamTransportConfiguration, SupportedDiagnosticsModes, SupportedDiagnosticsSnapshot, SupportedFirmwareUpdateConfiguration, SupportedMetrics, SupportedRouterConfiguration, SupportedRTPConfiguration, SupportedSleepConfiguration, SupportedTransferTransportConfiguration, SupportedVideoRecordingConfiguration, SupportedVideoStreamConfiguration, SwingMode, TapType, TargetAirPurifierState, TargetAirQuality, TargetControlList, TargetControlSupportedConfiguration, TargetDoorState, TargetFanState, TargetHeaterCoolerState, TargetHeatingCoolingState, TargetHorizontalTiltAngle, TargetHumidifierDehumidifierState, TargetMediaState, TargetPosition, TargetRelativeHumidity, TargetSlatState, TargetTemperature, TargetTiltAngle, TargetVerticalTiltAngle, TargetVisibilityState, TemperatureDisplayUnits, ThirdPartyCameraActive, ThreadControlPoint, ThreadNodeCapabilities, ThreadOpenThreadVersion, ThreadStatus, TimeUpdate, Token, TransmitPower, TunnelConnectionTimeout, TunneledAccessoryAdvertising, TunneledAccessoryConnected, TunneledAccessoryStateNumber, ValveType, Version, VideoAnalysisActive, VOCDensity, Volume, VolumeControlType, VolumeSelector, WakeConfiguration, WANConfigurationList, WANStatusList, WaterLevel, WiFiCapabilities, WiFiConfigurationControl, WiFiSatelliteStatus } from "./definitions";
import { HAPStatus } from "./HAPServer";
import { IdentifierCache } from "./model/IdentifierCache";
import { HAPConnection } from "./util/eventedhttp";
import { HapStatusError } from "./util/hapStatusError";
/**
 * @group Characteristic
 */
export declare const enum Formats {
    BOOL = "bool",
    /**
     * Signed 32-bit integer
     */
    INT = "int",
    /**
     * Signed 64-bit floating point
     */
    FLOAT = "float",
    /**
     * String encoded in utf8
     */
    STRING = "string",
    /**
     * Unsigned 8-bit integer.
     */
    UINT8 = "uint8",
    /**
     * Unsigned 16-bit integer.
     */
    UINT16 = "uint16",
    /**
     * Unsigned 32-bit integer.
     */
    UINT32 = "uint32",
    /**
     * Unsigned 64-bit integer.
     */
    UINT64 = "uint64",
    /**
     * Data is base64 encoded string.
     */
    DATA = "data",
    /**
     * Base64 encoded tlv8 string.
     */
    TLV8 = "tlv8",
    /**
     * @deprecated Not contained in the HAP spec
     */
    ARRAY = "array",
    /**
     * @deprecated Not contained in the HAP spec
     */
    DICTIONARY = "dict"
}
/**
 * @group Characteristic
 */
export declare const enum Units {
    /**
     * Celsius is the only temperature unit in the HomeKit Accessory Protocol.
     * Unit conversion is always done on the client side e.g. on the iPhone in the Home App depending on
     * the configured unit on the device itself.
     */
    CELSIUS = "celsius",
    PERCENTAGE = "percentage",
    ARC_DEGREE = "arcdegrees",
    LUX = "lux",
    SECONDS = "seconds"
}
/**
 * @group Characteristic
 */
export declare const enum Perms {
    /**
     * @deprecated replaced by {@link PAIRED_READ}. Kept for backwards compatibility.
     */
    READ = "pr",
    /**
     * @deprecated replaced by {@link PAIRED_WRITE}. Kept for backwards compatibility.
     */
    WRITE = "pw",
    PAIRED_READ = "pr",
    PAIRED_WRITE = "pw",
    NOTIFY = "ev",
    EVENTS = "ev",
    ADDITIONAL_AUTHORIZATION = "aa",
    TIMED_WRITE = "tw",
    HIDDEN = "hd",
    WRITE_RESPONSE = "wr"
}
/**
 * @group Characteristic
 */
export interface CharacteristicProps {
    format: Formats | string;
    perms: Perms[];
    unit?: Units | string;
    description?: string;
    /**
     * Defines the minimum value for a numeric characteristic
     */
    minValue?: number;
    /**
     * Defines the maximum value for a numeric characteristic
     */
    maxValue?: number;
    minStep?: number;
    /**
     * Maximum number of characters when format is {@link Formats.STRING}.
     * Default is 64 characters. Maximum allowed is 256 characters.
     */
    maxLen?: number;
    /**
     * Maximum number of characters when format is {@link Formats.DATA}.
     * Default is 2097152 characters.
     */
    maxDataLen?: number;
    /**
     * Defines an array of valid values to be used for the characteristic.
     */
    validValues?: number[];
    /**
     * Two element array where the first value specifies the lowest valid value and
     * the second element specifies the highest valid value.
     */
    validValueRanges?: [min: number, max: number];
    adminOnlyAccess?: Access[];
}
/**
 * Describes the abstract access to a {@link Characteristic}.
 * It abstracts the more granular access described by {@link Perms}.
 *
 * It is used in {@link CharacteristicProps.adminOnlyAccess}.
 *
 * @group Characteristic
 */
export declare const enum Access {
    READ = 0,
    WRITE = 1,
    NOTIFY = 2
}
/**
 * @group Characteristic
 */
export type CharacteristicChange = {
    originator?: HAPConnection;
    newValue: Nullable<CharacteristicValue>;
    oldValue: Nullable<CharacteristicValue>;
    reason: ChangeReason;
    context?: CharacteristicContext;
};
/**
 * @group Characteristic
 */
export declare const enum ChangeReason {
    /**
     * Reason used when HomeKit writes a value or the API user calls {@link Characteristic.setValue}.
     */
    WRITE = "write",
    /**
     * Reason used when the API user calls the method {@link Characteristic.updateValue}.
     */
    UPDATE = "update",
    /**
     * Used when HomeKit reads a value or the API user calls the deprecated method `Characteristic.getValue`.
     */
    READ = "read",
    /**
     * Used when call to {@link Characteristic.sendEventNotification} was made.
     */
    EVENT = "event"
}
/**
 * This format for a context object can be used to pass to any characteristic write operation.
 * It can contain additional information used by the internal event handlers of hap-nodejs.
 * The context object can be combined with any custom data for own use.
 *
 * @group Characteristic
 */
export interface CharacteristicOperationContext {
    /**
     * If set to true for any characteristic write operation
     * the Accessory won't send any event notifications to HomeKit controllers
     * for that particular change.
     */
    omitEventUpdate?: boolean;
}
/**
 * @group Characteristic
 */
export interface SerializedCharacteristic {
    displayName: string;
    UUID: string;
    eventOnlyCharacteristic: boolean;
    constructorName?: string;
    value: Nullable<CharacteristicValue>;
    props: CharacteristicProps;
}
/**
 * @group Characteristic
 */
export declare const enum CharacteristicEventTypes {
    /**
     * This event is thrown when a HomeKit controller wants to read the current value of the characteristic.
     * The event handler should call the supplied callback as fast as possible.
     *
     * HAP-NodeJS will complain about slow running get handlers after 3 seconds and terminate the request after 10 seconds.
     */
    GET = "get",
    /**
     * This event is thrown when a HomeKit controller wants to write a new value to the characteristic.
     * The event handler should call the supplied callback as fast as possible.
     *
     * HAP-NodeJS will complain about slow running set handlers after 3 seconds and terminate the request after 10 seconds.
     */
    SET = "set",
    /**
     * Emitted after a new value is set for the characteristic.
     * The new value can be set via a request by a HomeKit controller or via an API call.
     */
    CHANGE = "change",
    /**
     * @private
     */
    SUBSCRIBE = "subscribe",
    /**
     * @private
     */
    UNSUBSCRIBE = "unsubscribe",
    /**
     * @private
     */
    CHARACTERISTIC_WARNING = "characteristic-warning"
}
/**
 * @group Characteristic
 */
export type CharacteristicContext = any;
/**
 * @group Characteristic
 */
export type CharacteristicGetCallback = (status?: HAPStatus | null | Error, value?: Nullable<CharacteristicValue>) => void;
/**
 * @group Characteristic
 */
export type CharacteristicSetCallback = (error?: HAPStatus | null | Error, writeResponse?: Nullable<CharacteristicValue>) => void;
/**
 * @group Characteristic
 */
export type CharacteristicGetHandler = (context: CharacteristicContext, connection?: HAPConnection) => Promise<Nullable<CharacteristicValue>> | Nullable<CharacteristicValue>;
/**
 * @group Characteristic
 */
export type CharacteristicSetHandler = (value: CharacteristicValue, context: CharacteristicContext, connection?: HAPConnection) => Promise<Nullable<CharacteristicValue> | void> | Nullable<CharacteristicValue> | void;
/**
 * @group Characteristic
 */
export type AdditionalAuthorizationHandler = (additionalAuthorizationData: string | undefined) => boolean;
/**
 * @group Characteristic
 */
export declare interface Characteristic {
    on(event: "get", listener: (callback: CharacteristicGetCallback, context: CharacteristicContext, connection?: HAPConnection) => void): this;
    on(event: "set", listener: (value: CharacteristicValue, callback: CharacteristicSetCallback, context: CharacteristicContext, connection?: HAPConnection) => void): this;
    on(event: "change", listener: (change: CharacteristicChange) => void): this;
    /**
     * @private
     */
    on(event: "subscribe", listener: VoidCallback): this;
    /**
     * @private
     */
    on(event: "unsubscribe", listener: VoidCallback): this;
    /**
     * @private
     */
    on(event: "characteristic-warning", listener: (type: CharacteristicWarningType, message: string, stack?: string) => void): this;
    /**
     * @private
     */
    emit(event: "get", callback: CharacteristicGetCallback, context: CharacteristicContext, connection?: HAPConnection): boolean;
    /**
     * @private
     */
    emit(event: "set", value: CharacteristicValue, callback: CharacteristicSetCallback, context: CharacteristicContext, connection?: HAPConnection): boolean;
    /**
     * @private
     */
    emit(event: "change", change: CharacteristicChange): boolean;
    /**
     * @private
     */
    emit(event: "subscribe"): boolean;
    /**
     * @private
     */
    emit(event: "unsubscribe"): boolean;
    /**
     * @private
     */
    emit(event: "characteristic-warning", type: CharacteristicWarningType, message: string, stack?: string): boolean;
}
/**
 * Characteristic represents a particular typed variable that can be assigned to a Service. For instance, a
 * "Hue" Characteristic might store a 'float' value of type 'arcdegrees'. You could add the Hue Characteristic
 * to a {@link Service} in order to store that value. A particular Characteristic is distinguished from others by its
 * UUID. HomeKit provides a set of known Characteristic UUIDs defined in HomeKit.ts along with a
 * corresponding concrete subclass.
 *
 * You can also define custom Characteristics by providing your own UUID. Custom Characteristics can be added
 * to any native or custom Services, but Siri will likely not be able to work with these.
 *
 * @group Characteristic
 */
export declare class Characteristic extends EventEmitter {
    /**
     * @deprecated Please use the Formats const enum above.
     */
    static Formats: typeof Formats;
    /**
     * @deprecated Please use the Units const enum above.
     */
    static Units: typeof Units;
    /**
     * @deprecated Please use the Perms const enum above.
     */
    static Perms: typeof Perms;
    /**
     * @group Characteristic Definitions
     */
    static AccessCodeControlPoint: typeof AccessCodeControlPoint;
    /**
     * @group Characteristic Definitions
     */
    static AccessCodeSupportedConfiguration: typeof AccessCodeSupportedConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static AccessControlLevel: typeof AccessControlLevel;
    /**
     * @group Characteristic Definitions
     */
    static AccessoryFlags: typeof AccessoryFlags;
    /**
     * @group Characteristic Definitions
     */
    static AccessoryIdentifier: typeof AccessoryIdentifier;
    /**
     * @group Characteristic Definitions
     */
    static Active: typeof Active;
    /**
     * @group Characteristic Definitions
     */
    static ActiveIdentifier: typeof ActiveIdentifier;
    /**
     * @group Characteristic Definitions
     */
    static ActivityInterval: typeof ActivityInterval;
    /**
     * @group Characteristic Definitions
     */
    static AdministratorOnlyAccess: typeof AdministratorOnlyAccess;
    /**
     * @group Characteristic Definitions
     */
    static AirParticulateDensity: typeof AirParticulateDensity;
    /**
     * @group Characteristic Definitions
     */
    static AirParticulateSize: typeof AirParticulateSize;
    /**
     * @group Characteristic Definitions
     */
    static AirPlayEnable: typeof AirPlayEnable;
    /**
     * @group Characteristic Definitions
     */
    static AirQuality: typeof AirQuality;
    /**
     * @group Characteristic Definitions
     */
    static AppMatchingIdentifier: typeof AppMatchingIdentifier;
    /**
     * @group Characteristic Definitions
     */
    static AssetUpdateReadiness: typeof AssetUpdateReadiness;
    /**
     * @group Characteristic Definitions
     */
    static AudioFeedback: typeof AudioFeedback;
    /**
     * @group Characteristic Definitions
     */
    static BatteryLevel: typeof BatteryLevel;
    /**
     * @group Characteristic Definitions
     */
    static Brightness: typeof Brightness;
    /**
     * @group Characteristic Definitions
     */
    static ButtonEvent: typeof ButtonEvent;
    /**
     * @group Characteristic Definitions
     */
    static CameraOperatingModeIndicator: typeof CameraOperatingModeIndicator;
    /**
     * @group Characteristic Definitions
     */
    static CarbonDioxideDetected: typeof CarbonDioxideDetected;
    /**
     * @group Characteristic Definitions
     */
    static CarbonDioxideLevel: typeof CarbonDioxideLevel;
    /**
     * @group Characteristic Definitions
     */
    static CarbonDioxidePeakLevel: typeof CarbonDioxidePeakLevel;
    /**
     * @group Characteristic Definitions
     */
    static CarbonMonoxideDetected: typeof CarbonMonoxideDetected;
    /**
     * @group Characteristic Definitions
     */
    static CarbonMonoxideLevel: typeof CarbonMonoxideLevel;
    /**
     * @group Characteristic Definitions
     */
    static CarbonMonoxidePeakLevel: typeof CarbonMonoxidePeakLevel;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static Category: typeof Category;
    /**
     * @group Characteristic Definitions
     */
    static CCAEnergyDetectThreshold: typeof CCAEnergyDetectThreshold;
    /**
     * @group Characteristic Definitions
     */
    static CCASignalDetectThreshold: typeof CCASignalDetectThreshold;
    /**
     * @group Characteristic Definitions
     */
    static CharacteristicValueActiveTransitionCount: typeof CharacteristicValueActiveTransitionCount;
    /**
     * @group Characteristic Definitions
     */
    static CharacteristicValueTransitionControl: typeof CharacteristicValueTransitionControl;
    /**
     * @group Characteristic Definitions
     */
    static ChargingState: typeof ChargingState;
    /**
     * @group Characteristic Definitions
     */
    static ClosedCaptions: typeof ClosedCaptions;
    /**
     * @group Characteristic Definitions
     */
    static ColorTemperature: typeof ColorTemperature;
    /**
     * @group Characteristic Definitions
     */
    static ConfigurationState: typeof ConfigurationState;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static ConfigureBridgedAccessory: typeof ConfigureBridgedAccessory;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static ConfigureBridgedAccessoryStatus: typeof ConfigureBridgedAccessoryStatus;
    /**
     * @group Characteristic Definitions
     */
    static ConfiguredName: typeof ConfiguredName;
    /**
     * @group Characteristic Definitions
     */
    static ContactSensorState: typeof ContactSensorState;
    /**
     * @group Characteristic Definitions
     */
    static CoolingThresholdTemperature: typeof CoolingThresholdTemperature;
    /**
     * @group Characteristic Definitions
     */
    static CryptoHash: typeof CryptoHash;
    /**
     * @group Characteristic Definitions
     */
    static CurrentAirPurifierState: typeof CurrentAirPurifierState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentAmbientLightLevel: typeof CurrentAmbientLightLevel;
    /**
     * @group Characteristic Definitions
     */
    static CurrentDoorState: typeof CurrentDoorState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentFanState: typeof CurrentFanState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentHeaterCoolerState: typeof CurrentHeaterCoolerState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentHeatingCoolingState: typeof CurrentHeatingCoolingState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentHorizontalTiltAngle: typeof CurrentHorizontalTiltAngle;
    /**
     * @group Characteristic Definitions
     */
    static CurrentHumidifierDehumidifierState: typeof CurrentHumidifierDehumidifierState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentMediaState: typeof CurrentMediaState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentPosition: typeof CurrentPosition;
    /**
     * @group Characteristic Definitions
     */
    static CurrentRelativeHumidity: typeof CurrentRelativeHumidity;
    /**
     * @group Characteristic Definitions
     */
    static CurrentSlatState: typeof CurrentSlatState;
    /**
     * @group Characteristic Definitions
     */
    static CurrentTemperature: typeof CurrentTemperature;
    /**
     * @group Characteristic Definitions
     */
    static CurrentTiltAngle: typeof CurrentTiltAngle;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static CurrentTime: typeof CurrentTime;
    /**
     * @group Characteristic Definitions
     */
    static CurrentTransport: typeof CurrentTransport;
    /**
     * @group Characteristic Definitions
     */
    static CurrentVerticalTiltAngle: typeof CurrentVerticalTiltAngle;
    /**
     * @group Characteristic Definitions
     */
    static CurrentVisibilityState: typeof CurrentVisibilityState;
    /**
     * @group Characteristic Definitions
     */
    static DataStreamHAPTransport: typeof DataStreamHAPTransport;
    /**
     * @group Characteristic Definitions
     */
    static DataStreamHAPTransportInterrupt: typeof DataStreamHAPTransportInterrupt;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static DayoftheWeek: typeof DayoftheWeek;
    /**
     * @group Characteristic Definitions
     */
    static DiagonalFieldOfView: typeof DiagonalFieldOfView;
    /**
     * @group Characteristic Definitions
     */
    static DigitalZoom: typeof DigitalZoom;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static DiscoverBridgedAccessories: typeof DiscoverBridgedAccessories;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static DiscoveredBridgedAccessories: typeof DiscoveredBridgedAccessories;
    /**
     * @group Characteristic Definitions
     */
    static DisplayOrder: typeof DisplayOrder;
    /**
     * @group Characteristic Definitions
     */
    static EventRetransmissionMaximum: typeof EventRetransmissionMaximum;
    /**
     * @group Characteristic Definitions
     */
    static EventSnapshotsActive: typeof EventSnapshotsActive;
    /**
     * @group Characteristic Definitions
     */
    static EventTransmissionCounters: typeof EventTransmissionCounters;
    /**
     * @group Characteristic Definitions
     */
    static FilterChangeIndication: typeof FilterChangeIndication;
    /**
     * @group Characteristic Definitions
     */
    static FilterLifeLevel: typeof FilterLifeLevel;
    /**
     * @group Characteristic Definitions
     */
    static FirmwareRevision: typeof FirmwareRevision;
    /**
     * @group Characteristic Definitions
     */
    static FirmwareUpdateReadiness: typeof FirmwareUpdateReadiness;
    /**
     * @group Characteristic Definitions
     */
    static FirmwareUpdateStatus: typeof FirmwareUpdateStatus;
    /**
     * @group Characteristic Definitions
     */
    static HardwareFinish: typeof HardwareFinish;
    /**
     * @group Characteristic Definitions
     */
    static HardwareRevision: typeof HardwareRevision;
    /**
     * @group Characteristic Definitions
     */
    static HeartBeat: typeof HeartBeat;
    /**
     * @group Characteristic Definitions
     */
    static HeatingThresholdTemperature: typeof HeatingThresholdTemperature;
    /**
     * @group Characteristic Definitions
     */
    static HoldPosition: typeof HoldPosition;
    /**
     * @group Characteristic Definitions
     */
    static HomeKitCameraActive: typeof HomeKitCameraActive;
    /**
     * @group Characteristic Definitions
     */
    static Hue: typeof Hue;
    /**
     * @group Characteristic Definitions
     */
    static Identifier: typeof Identifier;
    /**
     * @group Characteristic Definitions
     */
    static Identify: typeof Identify;
    /**
     * @group Characteristic Definitions
     */
    static ImageMirroring: typeof ImageMirroring;
    /**
     * @group Characteristic Definitions
     */
    static ImageRotation: typeof ImageRotation;
    /**
     * @group Characteristic Definitions
     */
    static InputDeviceType: typeof InputDeviceType;
    /**
     * @group Characteristic Definitions
     */
    static InputSourceType: typeof InputSourceType;
    /**
     * @group Characteristic Definitions
     */
    static InUse: typeof InUse;
    /**
     * @group Characteristic Definitions
     */
    static IsConfigured: typeof IsConfigured;
    /**
     * @group Characteristic Definitions
     */
    static LeakDetected: typeof LeakDetected;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static LinkQuality: typeof LinkQuality;
    /**
     * @group Characteristic Definitions
     */
    static ListPairings: typeof ListPairings;
    /**
     * @group Characteristic Definitions
     */
    static LockControlPoint: typeof LockControlPoint;
    /**
     * @group Characteristic Definitions
     */
    static LockCurrentState: typeof LockCurrentState;
    /**
     * @group Characteristic Definitions
     */
    static LockLastKnownAction: typeof LockLastKnownAction;
    /**
     * @group Characteristic Definitions
     */
    static LockManagementAutoSecurityTimeout: typeof LockManagementAutoSecurityTimeout;
    /**
     * @group Characteristic Definitions
     */
    static LockPhysicalControls: typeof LockPhysicalControls;
    /**
     * @group Characteristic Definitions
     */
    static LockTargetState: typeof LockTargetState;
    /**
     * @group Characteristic Definitions
     */
    static Logs: typeof Logs;
    /**
     * @group Characteristic Definitions
     */
    static MACRetransmissionMaximum: typeof MACRetransmissionMaximum;
    /**
     * @group Characteristic Definitions
     */
    static MACTransmissionCounters: typeof MACTransmissionCounters;
    /**
     * @group Characteristic Definitions
     */
    static ManagedNetworkEnable: typeof ManagedNetworkEnable;
    /**
     * @group Characteristic Definitions
     */
    static ManuallyDisabled: typeof ManuallyDisabled;
    /**
     * @group Characteristic Definitions
     */
    static Manufacturer: typeof Manufacturer;
    /**
     * @group Characteristic Definitions
     */
    static MaximumTransmitPower: typeof MaximumTransmitPower;
    /**
     * @group Characteristic Definitions
     */
    static MetricsBufferFullState: typeof MetricsBufferFullState;
    /**
     * @group Characteristic Definitions
     */
    static Model: typeof Model;
    /**
     * @group Characteristic Definitions
     */
    static MotionDetected: typeof MotionDetected;
    /**
     * @group Characteristic Definitions
     */
    static MultifunctionButton: typeof MultifunctionButton;
    /**
     * @group Characteristic Definitions
     */
    static Mute: typeof Mute;
    /**
     * @group Characteristic Definitions
     */
    static Name: typeof Name;
    /**
     * @group Characteristic Definitions
     */
    static NetworkAccessViolationControl: typeof NetworkAccessViolationControl;
    /**
     * @group Characteristic Definitions
     */
    static NetworkClientProfileControl: typeof NetworkClientProfileControl;
    /**
     * @group Characteristic Definitions
     */
    static NetworkClientStatusControl: typeof NetworkClientStatusControl;
    /**
     * @group Characteristic Definitions
     */
    static NFCAccessControlPoint: typeof NFCAccessControlPoint;
    /**
     * @group Characteristic Definitions
     */
    static NFCAccessSupportedConfiguration: typeof NFCAccessSupportedConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static NightVision: typeof NightVision;
    /**
     * @group Characteristic Definitions
     */
    static NitrogenDioxideDensity: typeof NitrogenDioxideDensity;
    /**
     * @group Characteristic Definitions
     */
    static ObstructionDetected: typeof ObstructionDetected;
    /**
     * @group Characteristic Definitions
     */
    static OccupancyDetected: typeof OccupancyDetected;
    /**
     * @group Characteristic Definitions
     */
    static On: typeof On;
    /**
     * @group Characteristic Definitions
     */
    static OperatingStateResponse: typeof OperatingStateResponse;
    /**
     * @group Characteristic Definitions
     */
    static OpticalZoom: typeof OpticalZoom;
    /**
     * @group Characteristic Definitions
     */
    static OutletInUse: typeof OutletInUse;
    /**
     * @group Characteristic Definitions
     */
    static OzoneDensity: typeof OzoneDensity;
    /**
     * @group Characteristic Definitions
     */
    static PairingFeatures: typeof PairingFeatures;
    /**
     * @group Characteristic Definitions
     */
    static PairSetup: typeof PairSetup;
    /**
     * @group Characteristic Definitions
     */
    static PairVerify: typeof PairVerify;
    /**
     * @group Characteristic Definitions
     */
    static PasswordSetting: typeof PasswordSetting;
    /**
     * @group Characteristic Definitions
     */
    static PeriodicSnapshotsActive: typeof PeriodicSnapshotsActive;
    /**
     * @group Characteristic Definitions
     */
    static PictureMode: typeof PictureMode;
    /**
     * @group Characteristic Definitions
     */
    static Ping: typeof Ping;
    /**
     * @group Characteristic Definitions
     */
    static PM10Density: typeof PM10Density;
    /**
     * @group Characteristic Definitions
     */
    static PM2_5Density: typeof PM2_5Density;
    /**
     * @group Characteristic Definitions
     */
    static PositionState: typeof PositionState;
    /**
     * @group Characteristic Definitions
     */
    static PowerModeSelection: typeof PowerModeSelection;
    /**
     * @group Characteristic Definitions
     */
    static ProductData: typeof ProductData;
    /**
     * @group Characteristic Definitions
     */
    static ProgrammableSwitchEvent: typeof ProgrammableSwitchEvent;
    /**
     * @group Characteristic Definitions
     */
    static ProgrammableSwitchOutputState: typeof ProgrammableSwitchOutputState;
    /**
     * @group Characteristic Definitions
     */
    static ProgramMode: typeof ProgramMode;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static Reachable: typeof Reachable;
    /**
     * @group Characteristic Definitions
     */
    static ReceivedSignalStrengthIndication: typeof ReceivedSignalStrengthIndication;
    /**
     * @group Characteristic Definitions
     */
    static ReceiverSensitivity: typeof ReceiverSensitivity;
    /**
     * @group Characteristic Definitions
     */
    static RecordingAudioActive: typeof RecordingAudioActive;
    /**
     * @group Characteristic Definitions
     */
    static RelativeHumidityDehumidifierThreshold: typeof RelativeHumidityDehumidifierThreshold;
    /**
     * @group Characteristic Definitions
     */
    static RelativeHumidityHumidifierThreshold: typeof RelativeHumidityHumidifierThreshold;
    /**
     * @group Characteristic Definitions
     */
    static RelayControlPoint: typeof RelayControlPoint;
    /**
     * @group Characteristic Definitions
     */
    static RelayEnabled: typeof RelayEnabled;
    /**
     * @group Characteristic Definitions
     */
    static RelayState: typeof RelayState;
    /**
     * @group Characteristic Definitions
     */
    static RemainingDuration: typeof RemainingDuration;
    /**
     * @group Characteristic Definitions
     */
    static RemoteKey: typeof RemoteKey;
    /**
     * @group Characteristic Definitions
     */
    static ResetFilterIndication: typeof ResetFilterIndication;
    /**
     * @group Characteristic Definitions
     */
    static RotationDirection: typeof RotationDirection;
    /**
     * @group Characteristic Definitions
     */
    static RotationSpeed: typeof RotationSpeed;
    /**
     * @group Characteristic Definitions
     */
    static RouterStatus: typeof RouterStatus;
    /**
     * @group Characteristic Definitions
     */
    static Saturation: typeof Saturation;
    /**
     * @group Characteristic Definitions
     */
    static SecuritySystemAlarmType: typeof SecuritySystemAlarmType;
    /**
     * @group Characteristic Definitions
     */
    static SecuritySystemCurrentState: typeof SecuritySystemCurrentState;
    /**
     * @group Characteristic Definitions
     */
    static SecuritySystemTargetState: typeof SecuritySystemTargetState;
    /**
     * @group Characteristic Definitions
     */
    static SelectedAudioStreamConfiguration: typeof SelectedAudioStreamConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SelectedCameraRecordingConfiguration: typeof SelectedCameraRecordingConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SelectedDiagnosticsModes: typeof SelectedDiagnosticsModes;
    /**
     * @group Characteristic Definitions
     */
    static SelectedRTPStreamConfiguration: typeof SelectedRTPStreamConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SelectedSleepConfiguration: typeof SelectedSleepConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SerialNumber: typeof SerialNumber;
    /**
     * @group Characteristic Definitions
     */
    static ServiceLabelIndex: typeof ServiceLabelIndex;
    /**
     * @group Characteristic Definitions
     */
    static ServiceLabelNamespace: typeof ServiceLabelNamespace;
    /**
     * @group Characteristic Definitions
     */
    static SetDuration: typeof SetDuration;
    /**
     * @group Characteristic Definitions
     */
    static SetupDataStreamTransport: typeof SetupDataStreamTransport;
    /**
     * @group Characteristic Definitions
     */
    static SetupEndpoints: typeof SetupEndpoints;
    /**
     * @group Characteristic Definitions
     */
    static SetupTransferTransport: typeof SetupTransferTransport;
    /**
     * @group Characteristic Definitions
     */
    static SignalToNoiseRatio: typeof SignalToNoiseRatio;
    /**
     * @group Characteristic Definitions
     */
    static SiriEnable: typeof SiriEnable;
    /**
     * @group Characteristic Definitions
     */
    static SiriEndpointSessionStatus: typeof SiriEndpointSessionStatus;
    /**
     * @group Characteristic Definitions
     */
    static SiriEngineVersion: typeof SiriEngineVersion;
    /**
     * @group Characteristic Definitions
     */
    static SiriInputType: typeof SiriInputType;
    /**
     * @group Characteristic Definitions
     */
    static SiriLightOnUse: typeof SiriLightOnUse;
    /**
     * @group Characteristic Definitions
     */
    static SiriListening: typeof SiriListening;
    /**
     * @group Characteristic Definitions
     */
    static SiriTouchToUse: typeof SiriTouchToUse;
    /**
     * @group Characteristic Definitions
     */
    static SlatType: typeof SlatType;
    /**
     * @group Characteristic Definitions
     */
    static SleepDiscoveryMode: typeof SleepDiscoveryMode;
    /**
     * @group Characteristic Definitions
     */
    static SleepInterval: typeof SleepInterval;
    /**
     * @group Characteristic Definitions
     */
    static SmokeDetected: typeof SmokeDetected;
    /**
     * @group Characteristic Definitions
     */
    static SoftwareRevision: typeof SoftwareRevision;
    /**
     * @group Characteristic Definitions
     */
    static StagedFirmwareVersion: typeof StagedFirmwareVersion;
    /**
     * @group Characteristic Definitions
     */
    static StatusActive: typeof StatusActive;
    /**
     * @group Characteristic Definitions
     */
    static StatusFault: typeof StatusFault;
    /**
     * @group Characteristic Definitions
     */
    static StatusJammed: typeof StatusJammed;
    /**
     * @group Characteristic Definitions
     */
    static StatusLowBattery: typeof StatusLowBattery;
    /**
     * @group Characteristic Definitions
     */
    static StatusTampered: typeof StatusTampered;
    /**
     * @group Characteristic Definitions
     */
    static StreamingStatus: typeof StreamingStatus;
    /**
     * @group Characteristic Definitions
     */
    static SulphurDioxideDensity: typeof SulphurDioxideDensity;
    /**
     * @group Characteristic Definitions
     */
    static SupportedAssetTypes: typeof SupportedAssetTypes;
    /**
     * @group Characteristic Definitions
     */
    static SupportedAudioRecordingConfiguration: typeof SupportedAudioRecordingConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedAudioStreamConfiguration: typeof SupportedAudioStreamConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedCameraRecordingConfiguration: typeof SupportedCameraRecordingConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedCharacteristicValueTransitionConfiguration: typeof SupportedCharacteristicValueTransitionConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedDataStreamTransportConfiguration: typeof SupportedDataStreamTransportConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedDiagnosticsModes: typeof SupportedDiagnosticsModes;
    /**
     * @group Characteristic Definitions
     */
    static SupportedDiagnosticsSnapshot: typeof SupportedDiagnosticsSnapshot;
    /**
     * @group Characteristic Definitions
     */
    static SupportedFirmwareUpdateConfiguration: typeof SupportedFirmwareUpdateConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedMetrics: typeof SupportedMetrics;
    /**
     * @group Characteristic Definitions
     */
    static SupportedRouterConfiguration: typeof SupportedRouterConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedRTPConfiguration: typeof SupportedRTPConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedSleepConfiguration: typeof SupportedSleepConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedTransferTransportConfiguration: typeof SupportedTransferTransportConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedVideoRecordingConfiguration: typeof SupportedVideoRecordingConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SupportedVideoStreamConfiguration: typeof SupportedVideoStreamConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static SwingMode: typeof SwingMode;
    /**
     * @group Characteristic Definitions
     */
    static TapType: typeof TapType;
    /**
     * @group Characteristic Definitions
     */
    static TargetAirPurifierState: typeof TargetAirPurifierState;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static TargetAirQuality: typeof TargetAirQuality;
    /**
     * @group Characteristic Definitions
     */
    static TargetControlList: typeof TargetControlList;
    /**
     * @group Characteristic Definitions
     */
    static TargetControlSupportedConfiguration: typeof TargetControlSupportedConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static TargetDoorState: typeof TargetDoorState;
    /**
     * @group Characteristic Definitions
     */
    static TargetFanState: typeof TargetFanState;
    /**
     * @group Characteristic Definitions
     */
    static TargetHeaterCoolerState: typeof TargetHeaterCoolerState;
    /**
     * @group Characteristic Definitions
     */
    static TargetHeatingCoolingState: typeof TargetHeatingCoolingState;
    /**
     * @group Characteristic Definitions
     */
    static TargetHorizontalTiltAngle: typeof TargetHorizontalTiltAngle;
    /**
     * @group Characteristic Definitions
     */
    static TargetHumidifierDehumidifierState: typeof TargetHumidifierDehumidifierState;
    /**
     * @group Characteristic Definitions
     */
    static TargetMediaState: typeof TargetMediaState;
    /**
     * @group Characteristic Definitions
     */
    static TargetPosition: typeof TargetPosition;
    /**
     * @group Characteristic Definitions
     */
    static TargetRelativeHumidity: typeof TargetRelativeHumidity;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static TargetSlatState: typeof TargetSlatState;
    /**
     * @group Characteristic Definitions
     */
    static TargetTemperature: typeof TargetTemperature;
    /**
     * @group Characteristic Definitions
     */
    static TargetTiltAngle: typeof TargetTiltAngle;
    /**
     * @group Characteristic Definitions
     */
    static TargetVerticalTiltAngle: typeof TargetVerticalTiltAngle;
    /**
     * @group Characteristic Definitions
     */
    static TargetVisibilityState: typeof TargetVisibilityState;
    /**
     * @group Characteristic Definitions
     */
    static TemperatureDisplayUnits: typeof TemperatureDisplayUnits;
    /**
     * @group Characteristic Definitions
     */
    static ThirdPartyCameraActive: typeof ThirdPartyCameraActive;
    /**
     * @group Characteristic Definitions
     */
    static ThreadControlPoint: typeof ThreadControlPoint;
    /**
     * @group Characteristic Definitions
     */
    static ThreadNodeCapabilities: typeof ThreadNodeCapabilities;
    /**
     * @group Characteristic Definitions
     */
    static ThreadOpenThreadVersion: typeof ThreadOpenThreadVersion;
    /**
     * @group Characteristic Definitions
     */
    static ThreadStatus: typeof ThreadStatus;
    /**
     * @group Characteristic Definitions
     * @deprecated Removed and not used anymore
     */
    static TimeUpdate: typeof TimeUpdate;
    /**
     * @group Characteristic Definitions
     */
    static Token: typeof Token;
    /**
     * @group Characteristic Definitions
     */
    static TransmitPower: typeof TransmitPower;
    /**
     * @group Characteristic Definitions
     */
    static TunnelConnectionTimeout: typeof TunnelConnectionTimeout;
    /**
     * @group Characteristic Definitions
     */
    static TunneledAccessoryAdvertising: typeof TunneledAccessoryAdvertising;
    /**
     * @group Characteristic Definitions
     */
    static TunneledAccessoryConnected: typeof TunneledAccessoryConnected;
    /**
     * @group Characteristic Definitions
     */
    static TunneledAccessoryStateNumber: typeof TunneledAccessoryStateNumber;
    /**
     * @group Characteristic Definitions
     */
    static ValveType: typeof ValveType;
    /**
     * @group Characteristic Definitions
     */
    static Version: typeof Version;
    /**
     * @group Characteristic Definitions
     */
    static VideoAnalysisActive: typeof VideoAnalysisActive;
    /**
     * @group Characteristic Definitions
     */
    static VOCDensity: typeof VOCDensity;
    /**
     * @group Characteristic Definitions
     */
    static Volume: typeof Volume;
    /**
     * @group Characteristic Definitions
     */
    static VolumeControlType: typeof VolumeControlType;
    /**
     * @group Characteristic Definitions
     */
    static VolumeSelector: typeof VolumeSelector;
    /**
     * @group Characteristic Definitions
     */
    static WakeConfiguration: typeof WakeConfiguration;
    /**
     * @group Characteristic Definitions
     */
    static WANConfigurationList: typeof WANConfigurationList;
    /**
     * @group Characteristic Definitions
     */
    static WANStatusList: typeof WANStatusList;
    /**
     * @group Characteristic Definitions
     */
    static WaterLevel: typeof WaterLevel;
    /**
     * @group Characteristic Definitions
     */
    static WiFiCapabilities: typeof WiFiCapabilities;
    /**
     * @group Characteristic Definitions
     */
    static WiFiConfigurationControl: typeof WiFiConfigurationControl;
    /**
     * @group Characteristic Definitions
     */
    static WiFiSatelliteStatus: typeof WiFiSatelliteStatus;
    displayName: string;
    UUID: string;
    iid: Nullable<number>;
    value: Nullable<CharacteristicValue>;
    /**
     * @deprecated replaced by {@link statusCode}
     * @private
     */
    status: Nullable<Error>;
    /**
     * @private
     */
    statusCode: HAPStatus;
    props: CharacteristicProps;
    /**
     * The {@link Characteristic.onGet} handler
     */
    private getHandler?;
    /**
     * The {@link Characteristic.onSet} handler
     */
    private setHandler?;
    private subscriptions;
    /**
     * @private
     */
    additionalAuthorizationHandler?: AdditionalAuthorizationHandler;
    constructor(displayName: string, UUID: string, props: CharacteristicProps);
    /**
     * Accepts a function that will be called to retrieve the current value of a Characteristic.
     * The function must return a valid Characteristic value for the Characteristic type.
     * May optionally return a promise.
     *
     * @example
     * ```ts
     * Characteristic.onGet(async () => {
     *   return true;
     * });
     * ```
     * @param handler
     */
    onGet(handler: CharacteristicGetHandler): Characteristic;
    /**
     * Removes the {@link CharacteristicGetHandler} handler which was configured using {@link onGet}.
     */
    removeOnGet(): Characteristic;
    /**
     * Accepts a function that will be called when setting the value of a Characteristic.
     * If the characteristic supports {@link Perms.WRITE_RESPONSE} and the request requests a write-response value,
     * the returned value will be used.
     * May optionally return a promise.
     *
     * @example
     * ```ts
     * Characteristic.onSet(async (value: CharacteristicValue) => {
     *   console.log(value);
     * });
     * ```
     * @param handler
     */
    onSet(handler: CharacteristicSetHandler): Characteristic;
    /**
     * Removes the {@link CharacteristicSetHandler} which was configured using {@link onSet}.
     */
    removeOnSet(): Characteristic;
    /**
     * Updates the properties of this characteristic.
     * Properties passed via the parameter will be set. Any parameter set to null will be deleted.
     * See {@link CharacteristicProps}.
     *
     * @param props - Partial properties object with the desired updates.
     */
    setProps(props: PartialAllowingNull<CharacteristicProps>): Characteristic;
    /**
     * This method can be used to gain an Iterator to loop over all valid values defined for this characteristic.
     *
     * The range of valid values can be defined using three different ways via the {@link CharacteristicProps} object
     * (set via the {@link setProps} method):
     *  * First method is to specifically list every valid value inside {@link CharacteristicProps.validValues}
     *  * Second you can specify a range via {@link CharacteristicProps.minValue} and {@link CharacteristicProps.maxValue} (with optionally defining
     *    {@link CharacteristicProps.minStep})
     *  * And lastly you can specify a range via {@link CharacteristicProps.validValueRanges}
     *  * Implicitly a valid value range is predefined for characteristics with Format {@link Formats.UINT8}, {@link Formats.UINT16},
     *    {@link Formats.UINT32} and {@link Formats.UINT64}: starting by zero to their respective maximum number
     *
     * The method will automatically detect which type of valid values definition is used and provide
     * the correct Iterator for that case.
     *
     * Note: This method is (obviously) only valid for numeric characteristics.
     *
     * @example
     * ```ts
     * // use the iterator to loop over every valid value...
     * for (const value of characteristic.validValuesIterator()) {
     *   // Insert logic to run for every
     * }
     *
     * // ... or collect them in an array for storage or manipulation
     * const validValues = Array.from(characteristic.validValuesIterator());
     * ```
     */
    validValuesIterator(): Iterable<number>;
    /**
     * This method can be used to set up additional authorization for a characteristic.
     * For one, it adds the {@link Perms.ADDITIONAL_AUTHORIZATION} permission to the characteristic
     * (if it wasn't already) to signal support for additional authorization to HomeKit.
     * Additionally, an {@link AdditionalAuthorizationHandler} is set up which is called
     * before a write request is performed.
     *
     * Additional Authorization Data can be added to SET request via a custom iOS App.
     * Before hap-nodejs executes a write request it will call the {@link AdditionalAuthorizationHandler}
     * with 'authData' supplied in the write request. The 'authData' is a base64 encoded string
     * (or undefined if no authData was supplied).
     * The {@link AdditionalAuthorizationHandler} must then return true or false to indicate if the write request
     * is authorized and should be accepted.
     *
     * @param handler - Handler called to check additional authorization data.
     */
    setupAdditionalAuthorization(handler: AdditionalAuthorizationHandler): void;
    /**
     * Updates the current value of the characteristic.
     *
     * @param callback
     * @param context
     * @private use to return the current value on HAP requests
     *
     * @deprecated
     */
    getValue(callback?: CharacteristicGetCallback, context?: CharacteristicContext): void;
    /**
     * This updates the value by calling the {@link CharacteristicEventTypes.SET} event handler associated with the characteristic.
     * This acts the same way as when a HomeKit controller sends a `/characteristics` request to update the characteristic.
     * An event notification will be sent to all connected HomeKit controllers which are registered
     * to receive event notifications for this characteristic.
     *
     * This method behaves like a {@link updateValue} call with the addition that the own {@link CharacteristicEventTypes.SET}
     * event handler is called.
     *
     * @param value - The new value.
     *
     * Note: If you don't want the {@link CharacteristicEventTypes.SET} to be called, refer to {@link updateValue}.
     */
    setValue(value: CharacteristicValue): Characteristic;
    /**
     * Sets the state of the characteristic to an errored state.
     *
     * If a {@link onGet} or {@link CharacteristicEventTypes.GET} handler is set up,
     * the errored state will be ignored and the characteristic will always query the latest state by calling the provided handler.
     *
     * If a generic error object is supplied, the characteristic tries to extract a {@link HAPStatus} code
     * from the error message string. If not possible a generic {@link HAPStatus.SERVICE_COMMUNICATION_FAILURE} will be used.
     * If the supplied error object is an instance of {@link HapStatusError} the corresponding status will be used.
     *
     * This doesn't call any registered {@link onSet} or {@link CharacteristicEventTypes.SET} handlers.
     *
     * Have a look at the
     * {@link https://github.com/homebridge/HAP-NodeJS/wiki/Presenting-Erroneous-Accessory-State-to-the-User Presenting Erroneous Accessory State to the User}
     * guide for more information on how to present erroneous state to the user.
     *
     * @param error - The error object
     *
     * Note: Erroneous state is never **pushed** to the client side. Only, if the HomeKit client requests the current
     *  state of the Characteristic, the corresponding {@link HapStatusError} is returned. As described above,
     *  any {@link onGet} or {@link CharacteristicEventTypes.GET} handlers have precedence.
     */
    setValue(error: HapStatusError | Error): Characteristic;
    /**
     * This updates the value by calling the {@link CharacteristicEventTypes.SET} event handler associated with the characteristic.
     * This acts the same way as when a HomeKit controller sends a `/characteristics` request to update the characteristic.
     * An event notification will be sent to all connected HomeKit controllers which are registered
     * to receive event notifications for this characteristic.
     *
     * This method behaves like a {@link updateValue} call with the addition that the own {@link CharacteristicEventTypes.SET}
     * event handler is called.
     *
     * @param value - The new value.
     * @param callback - Deprecated parameter there to provide backwards compatibility. Called once the
     *   {@link CharacteristicEventTypes.SET} event handler returns.
     * @param context - Passed to the {@link CharacteristicEventTypes.SET} and {@link CharacteristicEventTypes.CHANGE} event handler.
     *
     * @deprecated Parameter `callback` is deprecated.
     */
    setValue(value: CharacteristicValue, callback?: CharacteristicSetCallback, context?: CharacteristicContext): Characteristic;
    /**
     * This updates the value by calling the {@link CharacteristicEventTypes.SET} event handler associated with the characteristic.
     * This acts the same way as when a HomeKit controller sends a `/characteristics` request to update the characteristic.
     * An event notification will be sent to all connected HomeKit controllers which are registered
     * to receive event notifications for this characteristic.
     *
     * This method behaves like a {@link updateValue} call with the addition that the own {@link CharacteristicEventTypes.SET}
     * event handler is called.
     *
     * @param value - The new value.
     * @param context - Passed to the {@link CharacteristicEventTypes.SET} and {@link CharacteristicEventTypes.CHANGE} event handler.
     *
     * Note: If you don't want the {@link CharacteristicEventTypes.SET} to be called, refer to {@link updateValue}.
     */
    setValue(value: CharacteristicValue, context?: CharacteristicContext): Characteristic;
    /**
     * This updates the value of the characteristic. If the value changed, an event notification will be sent to all connected
     * HomeKit controllers which are registered to receive event notifications for this characteristic.
     *
     * @param value - The new value or a `Error` or {@link HapStatusError}.
     *
     * Note: Refer to the respective overloads for {@link CharacteristicValue} or {@link HapStatusError} for respective documentation.
     */
    updateValue(value: Nullable<CharacteristicValue> | Error | HapStatusError): Characteristic;
    /**
     * This updates the value of the characteristic. If the value changed, an event notification will be sent to all connected
     * HomeKit controllers which are registered to receive event notifications for this characteristic.
     *
     * @param value - The new value.
     */
    updateValue(value: Nullable<CharacteristicValue>): Characteristic;
    /**
     * Sets the state of the characteristic to an errored state.
     * If a {@link onGet} or {@link CharacteristicEventTypes.GET} handler is set up,
     * the errored state will be ignored and the characteristic will always query the latest state by calling the provided handler.
     *
     * If a generic error object is supplied, the characteristic tries to extract a {@link HAPStatus} code
     * from the error message string. If not possible a generic {@link HAPStatus.SERVICE_COMMUNICATION_FAILURE} will be used.
     * If the supplied error object is an instance of {@link HapStatusError} the corresponding status will be used.
     *
     * Have a look at the
     * {@link https://github.com/homebridge/HAP-NodeJS/wiki/Presenting-Erroneous-Accessory-State-to-the-User Presenting Erroneous Accessory State to the User}
     * guide for more information on how to present erroneous state to the user.
     *
     * @param error - The error object
     *
     * Note: Erroneous state is never **pushed** to the client side. Only, if the HomeKit client requests the current
     *  state of the Characteristic, the corresponding {@link HapStatusError} is returned. As described above,
     *  any {@link onGet} or {@link CharacteristicEventTypes.GET} handlers have precedence.
     */
    updateValue(error: Error | HapStatusError): Characteristic;
    /**
     * This updates the value of the characteristic. If the value changed, an event notification will be sent to all connected
     * HomeKit controllers which are registered to receive event notifications for this characteristic.
     *
     * @param value - The new value.
     * @param callback - Deprecated parameter there to provide backwards compatibility. Callback is called instantly.
     * @param context - Passed to the {@link CharacteristicEventTypes.CHANGE} event handler.
     *
     * @deprecated Parameter `callback` is deprecated.
     */
    updateValue(value: Nullable<CharacteristicValue>, callback?: () => void, context?: CharacteristicContext): Characteristic;
    /**
     * This updates the value of the characteristic. If the value changed, an event notification will be sent to all connected
     * HomeKit controllers which are registered to receive event notifications for this characteristic.
     *
     * @param value - The new value.
     * @param context - Passed to the {@link CharacteristicEventTypes.CHANGE} event handler.
     */
    updateValue(value: Nullable<CharacteristicValue>, context?: CharacteristicContext): Characteristic;
    /**
     * This method acts similarly to {@link updateValue} by setting the current value of the characteristic
     * without calling any {@link CharacteristicEventTypes.SET} or {@link onSet} handlers.
     * The difference is that this method forces an event notification sent (updateValue only sends one if the value changed).
     * This is especially useful for characteristics like {@link Characteristic.ButtonEvent} or {@link Characteristic.ProgrammableSwitchEvent}.
     *
     * @param value - The new value.
     * @param context - Passed to the {@link CharacteristicEventTypes.CHANGE} event handler.
     */
    sendEventNotification(value: CharacteristicValue, context?: CharacteristicContext): Characteristic;
    /**
     * Called when a HAP requests wants to know the current value of the characteristic.
     *
     * @param connection - The HAP connection from which the request originated from.
     * @param context - Deprecated parameter. There for backwards compatibility.
     * @private Used by the Accessory to load the characteristic value
     */
    handleGetRequest(connection?: HAPConnection, context?: CharacteristicContext): Promise<Nullable<CharacteristicValue>>;
    /**
     * Called when a HAP requests update the current value of the characteristic.
     *
     * @param value - The updated value
     * @param connection - The connection from which the request originated from
     * @param context - Deprecated parameter. There for backwards compatibility.
     * @returns Promise resolve to void in normal operation. When characteristic supports write-response, HAP
     *  requests a write-response and the set handler returns a write-response value, the respective
     *  write response value is resolved.
     * @private
     */
    handleSetRequest(value: CharacteristicValue, connection?: HAPConnection, context?: CharacteristicContext): Promise<CharacteristicValue | void>;
    /**
     * Called once a HomeKit controller subscribes to events of this characteristic.
     * @private
     */
    subscribe(): void;
    /**
     * Called once a HomeKit controller unsubscribe to events of this characteristic or a HomeKit controller
     * which was subscribed to this characteristic disconnects.
     * @private
     */
    unsubscribe(): void;
    protected getDefaultValue(): Nullable<CharacteristicValue>;
    /**
     * Checks if the value received from the HAP request is valid.
     * If returned false the received value is not valid and {@link HAPStatus.INVALID_VALUE_IN_REQUEST}
     * must be returned.
     * @param value - Value supplied by the HomeKit controller
     */
    private validateClientSuppliedValue;
    /**
     * Checks if the value received from the API call is valid.
     * It adjusts the value where it makes sense, prints a warning where values may be rejected with an error
     * in the future and throws an error which can't be converted to a valid value.
     *
     * @param value - The value received from the API call
     * @param warningType - Optionally defines the warning type to use when raising a {@link CharacteristicEventTypes.CHARACTERISTIC_WARNING}.
     */
    private validateUserInput;
    /**
     * @private used to assign iid to characteristic
     */
    _assignID(identifierCache: IdentifierCache, accessoryName: string, serviceUUID: string, serviceSubtype?: string): void;
    private characteristicWarning;
    /**
     * @param event
     * @private
     */
    removeAllListeners(event?: string | symbol): this;
    /**
     * @param characteristic
     * @private
     */
    replaceBy(characteristic: Characteristic): void;
    /**
     * Returns a JSON representation of this characteristic suitable for delivering to HAP clients.
     * @private used to generate response to /accessories query
     */
    toHAP(connection: HAPConnection, contactGetHandlers?: boolean): Promise<CharacteristicJsonObject>;
    /**
     * Returns a JSON representation of this characteristic without the value.
     * @private used to generate the config hash
     */
    internalHAPRepresentation(): CharacteristicJsonObject;
    /**
     * Serialize characteristic into json string.
     *
     * @param characteristic - Characteristic object.
     * @private used to store characteristic on disk
     */
    static serialize(characteristic: Characteristic): SerializedCharacteristic;
    /**
     * Deserialize characteristic from json string.
     *
     * @param json - Json string representing a characteristic.
     * @private used to recreate characteristic from disk
     */
    static deserialize(json: SerializedCharacteristic): Characteristic;
}
import "./definitions/CharacteristicDefinitions";
//# sourceMappingURL=Characteristic.d.ts.map