"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SwitchCreator = exports.isSwitchConfig = void 0;
const z2mModels_1 = require("../z2mModels");
const hap_1 = require("../hap");
const helpers_1 = require("../helpers");
const monitor_1 = require("./monitor");
// eslint-disable-next-line @typescript-eslint/no-explicit-any
const isSwitchConfig = (x) => (x !== undefined && (x.type === undefined
    || (typeof x.type === 'string' && x.type.length > 0 &&
        [SwitchCreator.CONFIG_TYPE_SWITCH, SwitchCreator.CONFIG_TYPE_OUTLET].includes(x.type.toLowerCase()))));
exports.isSwitchConfig = isSwitchConfig;
class SwitchCreator {
    constructor(converterConfigRegistry) {
        converterConfigRegistry.registerConverterConfiguration(SwitchCreator.CONFIG_TAG, SwitchCreator.isValidConverterConfiguration);
    }
    static isValidConverterConfiguration(config) {
        return (0, exports.isSwitchConfig)(config);
    }
    createServicesFromExposes(accessory, exposes) {
        let exposeAsOutlet = false;
        const converterConfig = accessory.getConverterConfiguration(SwitchCreator.CONFIG_TAG);
        if ((0, exports.isSwitchConfig)(converterConfig) && converterConfig.type === SwitchCreator.CONFIG_TYPE_OUTLET) {
            exposeAsOutlet = true;
        }
        exposes.filter(e => e.type === z2mModels_1.ExposesKnownTypes.SWITCH && (0, z2mModels_1.exposesHasFeatures)(e)
            && (0, z2mModels_1.exposesHasAllRequiredFeatures)(e, [SwitchHandler.PREDICATE_STATE], accessory.isPropertyExcluded.bind(accessory))
            && !accessory.isServiceHandlerIdKnown(SwitchHandler.generateIdentifier(exposeAsOutlet, e.endpoint)))
            .forEach(e => this.createService(e, accessory, exposeAsOutlet));
    }
    createService(expose, accessory, exposeAsOutlet) {
        try {
            const handler = new SwitchHandler(expose, accessory, exposeAsOutlet);
            accessory.registerServiceHandler(handler);
        }
        catch (error) {
            accessory.log.warn(`Failed to setup switch for accessory ${accessory.displayName} from expose "${JSON.stringify(expose)}": ${error}`);
        }
    }
}
exports.SwitchCreator = SwitchCreator;
SwitchCreator.CONFIG_TAG = 'switch';
SwitchCreator.CONFIG_TYPE_SWITCH = 'switch';
SwitchCreator.CONFIG_TYPE_OUTLET = 'outlet';
class SwitchHandler {
    constructor(expose, accessory, exposeAsOutlet) {
        this.accessory = accessory;
        const endpoint = expose.endpoint;
        const serviceTypeName = exposeAsOutlet ? 'Outlet' : 'Switch';
        this.identifier = SwitchHandler.generateIdentifier(exposeAsOutlet, endpoint);
        const potentialStateExpose = expose.features.find(e => SwitchHandler.PREDICATE_STATE(e)
            && !accessory.isPropertyExcluded(e.property));
        if (potentialStateExpose === undefined) {
            throw new Error(`Required "state" property not found for ${serviceTypeName}.`);
        }
        this.stateExpose = potentialStateExpose;
        const serviceName = accessory.getDefaultServiceDisplayName(endpoint);
        accessory.log.debug(`Configuring ${serviceTypeName} for ${serviceName}`);
        const service = accessory.getOrAddService(exposeAsOutlet ?
            new hap_1.hap.Service.Outlet(serviceName, endpoint) :
            new hap_1.hap.Service.Switch(serviceName, endpoint));
        (0, helpers_1.getOrAddCharacteristic)(service, hap_1.hap.Characteristic.On).on('set', this.handleSetOn.bind(this));
        const onOffValues = new Map();
        onOffValues.set(this.stateExpose.value_on, true);
        onOffValues.set(this.stateExpose.value_off, false);
        this.monitor = new monitor_1.MappingCharacteristicMonitor(this.stateExpose.property, service, hap_1.hap.Characteristic.On, onOffValues);
    }
    get getableKeys() {
        const keys = [];
        if ((0, z2mModels_1.exposesCanBeGet)(this.stateExpose)) {
            keys.push(this.stateExpose.property);
        }
        return keys;
    }
    updateState(state) {
        this.monitor.callback(state);
    }
    handleSetOn(value, callback) {
        const data = {};
        data[this.stateExpose.property] = value ? this.stateExpose.value_on : this.stateExpose.value_off;
        this.accessory.queueDataForSetAction(data);
        callback(null);
    }
    static generateIdentifier(exposeAsOutlet, endpoint) {
        let identifier = exposeAsOutlet ? hap_1.hap.Service.Outlet.UUID : hap_1.hap.Service.Switch.UUID;
        if (endpoint !== undefined) {
            identifier += '_' + endpoint.trim();
        }
        return identifier;
    }
}
SwitchHandler.PREDICATE_STATE = (e) => (0, z2mModels_1.exposesHasBinaryProperty)(e)
    && e.name === 'state' && (0, z2mModels_1.exposesCanBeSet)(e) && (0, z2mModels_1.exposesIsPublished)(e);
//# sourceMappingURL=switch.js.map