/*
 * shell.h
 */
 /**
\ingroup clish
\defgroup clish_shell shell
@{

\brief This class represents the top level container for a CLI session.

*/
#ifndef _clish_shell_h
#define _clish_shell_h

#include <stdio.h>
#include <sys/types.h>
#include <pwd.h>

#include "lub/c_decl.h"
#include "lub/types.h"
#include "lub/argv.h"
#include "tinyrl/tinyrl.h"
#include "clish/macros.h"
#include "clish/view.h"
#include "clish/ptype.h"
#include "clish/var.h"
#include "clish/plugin.h"
#include "konf/net.h"

#define CLISH_LOCK_PATH "/tmp/clish.lock"
#define CLISH_LOCK_WAIT 20

#define CLISH_STDOUT_CHUNK 1024
#define CLISH_STDOUT_MAXBUF (CLISH_STDOUT_CHUNK * 1024)

#define CLISH_XML_ERROR_STR "Error parsing XML: "
#define CLISH_XML_ERROR_ATTR(attr) CLISH_XML_ERROR_STR"The \""attr"\" attribute is required.\n"

typedef struct clish_shell_s clish_shell_t;
typedef struct clish_context_s clish_context_t;

/* Context functions */

_BEGIN_C_DECL
clish_context_t *clish_context_new(clish_shell_t *shell);
int clish_context_init(clish_context_t *instance, clish_shell_t *shell);
void clish_context_free(clish_context_t *instance);
int clish_context_dup(clish_context_t *dst, const clish_context_t *src);

clish_shell_t *clish_context__get_shell(const void *instance);
void clish_context__set_cmd(void *instance, const clish_command_t *cmd);
const clish_command_t *clish_context__get_cmd(const void *instance);
void clish_context__set_pargv(void *instance, clish_pargv_t *pargv);
clish_pargv_t *clish_context__get_pargv(const void *instance);
void clish_context__set_action(void *instance, const clish_action_t *action);
const clish_action_t *clish_context__get_action(const void *instance);
bool_t clish_context__get_expand(const clish_context_t *context);
_END_C_DECL

/* Shell */

typedef enum {
	SHELL_STATE_OK = 0,
	SHELL_STATE_UNKNOWN = 1,
	SHELL_STATE_IO_ERROR = 2,
	SHELL_STATE_SCRIPT_ERROR = 3,/* Script execution error */
	SHELL_STATE_SYNTAX_ERROR = 4, /* Illegal line entered */
	SHELL_STATE_SYSTEM_ERROR = 5, /* Some internal system error */
	SHELL_STATE_INITIALISING = 6,
	SHELL_STATE_HELPING = 7,
	SHELL_STATE_EOF = 8, /* EOF of input stream */
	SHELL_STATE_CLOSING = 9
} clish_shell_state_e;

typedef enum {
	SHELL_VAR_NONE, /* Nothing to escape */
	SHELL_VAR_ACTION, /* Variable expanding for ACTION script */
	SHELL_VAR_REGEX /* Variable expanding for regex usage */
} clish_shell_var_e;

typedef enum {
	SHELL_EXPAND_PARAM = 1,
	SHELL_EXPAND_VIEW = 2,
	SHELL_EXPAND_CONTEXT = 4,
	SHELL_EXPAND_VAR = 8,
	SHELL_EXPAND_ENV = 16,
	SHELL_EXPAND_ALL = 255
} clish_shell_expand_e;

_BEGIN_C_DECL

clish_shell_t *clish_shell_new(FILE *istream, FILE *ostream, bool_t stop_on_error);
int clish_shell_startup(clish_shell_t *instance);
void clish_shell_delete(clish_shell_t *instance);
clish_view_t *clish_shell_find_create_view(clish_shell_t *instance, const char *name);
clish_ptype_t *clish_shell_find_create_ptype(clish_shell_t *instance,
	const char *name,
	const char *text,
	const char *pattern,
	clish_ptype_method_e method,
	clish_ptype_preprocess_e preprocess);
clish_view_t *clish_shell_find_view(clish_shell_t * instance,
	const char *name);
clish_ptype_t *clish_shell_find_ptype(clish_shell_t *instance,
	const char *name);
void clish_shell_help(clish_shell_t * instance, const char *line);
int clish_shell_exec_action(clish_context_t *context, char **out);
int clish_shell_execute(clish_context_t *context, char **out);
int clish_shell_forceline(clish_shell_t *instance, const char *line, char ** out);
int clish_shell_readline(clish_shell_t *instance, char ** out);
void clish_shell_dump(clish_shell_t * instance);
/**
 * Push the specified file handle on to the stack of file handles
 * for this shell. The specified file will become the source of 
 * commands, until it is exhausted.
 *
 * \return
 * BOOL_TRUE - the file was successfully associated with the shell.
 * BOOL_FALSE - there was insufficient resource to associate this file.
 */
int clish_shell_push_file(clish_shell_t * instance, const char * fname,
	bool_t stop_on_error);
int clish_shell_push_fd(clish_shell_t * instance, FILE * file,
	bool_t stop_on_error);
clish_var_t *clish_shell_find_var(clish_shell_t *instance, const char *name);
char *clish_shell_expand_var(const char *name, clish_context_t *context);
char *clish_shell_expand_var_ex(const char *name, clish_context_t *context, clish_shell_expand_e flags);
char *clish_shell_expand(const char *str, clish_shell_var_e vtype, clish_context_t *context);
char * clish_shell_mkfifo(clish_shell_t * instance, char *name, size_t n);
int clish_shell_rmfifo(clish_shell_t * instance, const char *name);

/*-----------------
 * attributes
 *----------------- */
_CLISH_GET_STR(shell, overview);
_CLISH_SET_STR(shell, lockfile);
_CLISH_GET_STR(shell, lockfile);
_CLISH_SET_STR(shell, default_shebang);
_CLISH_GET_STR(shell, default_shebang);
_CLISH_SET(shell, bool_t, default_expand);
_CLISH_GET(shell, bool_t, default_expand);
_CLISH_SET(shell, unsigned int, idle_timeout);
_CLISH_SET(shell, unsigned int, wdog_timeout);
_CLISH_GET(shell, unsigned int, wdog_timeout);
_CLISH_GET(shell, unsigned int, depth);
_CLISH_SET(shell, int, log_facility);
_CLISH_GET(shell, int, log_facility);
_CLISH_GET(shell, konf_client_t *, client);
_CLISH_GET(shell, struct passwd *, user);
_CLISH_SET(shell, clish_shell_state_e, state);
_CLISH_GET(shell, clish_shell_state_e, state);
_CLISH_SET(shell, bool_t, interactive);
_CLISH_GET(shell, bool_t, interactive);
_CLISH_SET(shell, bool_t, log);
_CLISH_GET(shell, bool_t, log);
_CLISH_SET(shell, bool_t, dryrun);
_CLISH_GET(shell, bool_t, dryrun);
_CLISH_SET(shell, bool_t, canon_out);
_CLISH_GET(shell, bool_t, canon_out);

clish_view_t *clish_shell__get_view(const clish_shell_t * instance);
clish_view_t *clish_shell__set_depth(clish_shell_t *instance, unsigned int depth);
const char *clish_shell__get_viewid(const clish_shell_t * instance);
tinyrl_t *clish_shell__get_tinyrl(const clish_shell_t * instance);
void clish_shell__set_pwd(clish_shell_t *instance, const char * line,
	clish_view_t * view, const char * viewid, clish_context_t *context);
char *clish_shell__get_pwd_line(const clish_shell_t * instance,
	 unsigned int index);
clish_pargv_t *clish_shell__get_pwd_pargv(const clish_shell_t *instance,
	unsigned int index);
char *clish_shell__get_pwd_cmd(const clish_shell_t *instance,
	unsigned int index);
char *clish_shell__get_pwd_prefix(const clish_shell_t *instance,
	unsigned int index);
char *clish_shell__get_pwd_full(const clish_shell_t * instance,
	unsigned int depth);
clish_view_t *clish_shell__get_pwd_view(const clish_shell_t * instance,
	unsigned int index);
FILE *clish_shell__get_istream(const clish_shell_t * instance);
FILE *clish_shell__get_ostream(const clish_shell_t * instance);
int clish_shell__set_socket(clish_shell_t * instance, const char * path);
int clish_shell_load_scheme(clish_shell_t * instance, const char * xml_path, const char *xslt_path);
int clish_shell_loop(clish_shell_t * instance);
void clish_shell__set_startup_view(clish_shell_t * instance, const char * viewname);
void clish_shell__set_startup_viewid(clish_shell_t * instance, const char * viewid);
bool_t clish_shell__get_utf8(const clish_shell_t * instance);
void clish_shell__set_utf8(clish_shell_t * instance, bool_t utf8);
char *clish_shell__get_line(clish_context_t *context);
char *clish_shell__get_full_line(clish_context_t *context);
char *clish_shell__get_params(clish_context_t *context);
int clish_shell_wdog(clish_shell_t *instance);
int clish_shell__save_history(const clish_shell_t *instance, const char *fname);
int clish_shell__restore_history(clish_shell_t *instance, const char *fname);
void clish_shell__stifle_history(clish_shell_t *instance, unsigned int stifle);
char *clish_shell_format_username(const clish_shell_t *instance);

/* Plugin functions */
clish_plugin_t * clish_shell_create_plugin(clish_shell_t *instance,
	const char *name);
clish_plugin_t * clish_shell_find_plugin(clish_shell_t *instance,
	const char *name);
clish_plugin_t * clish_shell_find_create_plugin(clish_shell_t *instance,
	const char *name);
int clish_shell_load_plugins(clish_shell_t *instance);
int clish_shell_link_plugins(clish_shell_t *instance);

/* Unresolved symbols functions */
clish_sym_t *clish_shell_find_sym(clish_shell_t *instance,
	const char *name, int type);
clish_sym_t *clish_shell_add_sym(clish_shell_t *instance,
	void *func, const char *name, int type);
clish_sym_t *clish_shell_add_unresolved_sym(clish_shell_t *instance,
	const char *name, int type);
clish_sym_t *clish_shell_get_hook(const clish_shell_t *instance, int type);

/* Hook wrappers */
const void *clish_shell_check_hook(const clish_context_t *clish_context, int type);
CLISH_HOOK_CONFIG(clish_shell_exec_config);
CLISH_HOOK_LOG(clish_shell_exec_log);

/* User data functions */
void *clish_shell__get_udata(const clish_shell_t *instance, const char *name);
void *clish_shell__del_udata(clish_shell_t *instance, const char *name);
int clish_shell__set_udata(clish_shell_t *instance,
	const char *name, void *data);

/* Access functions */
int clish_shell_prepare(clish_shell_t *instance);

/* Machine/Human oriented protocol */
bool_t clish_shell_is_machine_interface(const clish_shell_t *shell);
void clish_shell_set_machine_interface(clish_shell_t *shell);
void clish_shell_set_human_interface(clish_shell_t *shell);
void clish_shell_machine_retval(clish_shell_t *shell, int retval);

/*
 * Non shell specific functions.
 * Start and Stop XML parser engine.
 */
int clish_xmldoc_start(void);
int clish_xmldoc_stop(void);

_END_C_DECL

#endif				/* _clish_shell_h */
/** @} clish_shell */
