/*   FILE: io.vala -- Handle PiFace CAD or keyboard I/O.
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 22 February 2018
 *
 * Copyright (c) 2018 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* The PiFace CAD provides the following inputs:
 *   From left to right:
 *     Button 0 = 0
 *     Button 1 = 1
 *     Button 2 = 2
 *     Button 3 = 3
 *     Button 4 = 4
 *   Rocker switch:
 *     Select     = 5
 *     Rock left  = 6
 *     Rock right = 7
 */

using Gee;

const uint LIGHTS_OUT_SECONDS       = 10;
const uint SWITCH_POLL_MILLISECONDS = 20;  // Sized to avoid double-press.
const uint OUTPUT_REFRESH_FREQUENCY = 750;

enum SwitchNumber {
	STOP        = 0,
	SEEKBACK    = 1,
	PAUSEPLAY   = 2,
	SEEKFORWARD = 3,
	ESCAPE      = 4,
	SELECT      = 5,
	LEFT        = 6,
	RIGHT       = 7
}

enum KeyNumber {
	STOP        = 'z',
	SEEKBACK    = 'q',
	PAUSEPLAY   = ' ',
	SEEKFORWARD = 'w',
	ESCAPE      =  0x1b,
	SELECT      = '\n',
	LEFT        = 'a',
	RIGHT       = 's',
}

class InputOutput : GLib.Object {
	static uint output_shift = 0;
	static string output_str = "";

	private bool pifacecad;

	uint lights_out_id = 0;
	private uint8[] lastSwitchState = { 1, 1, 1, 1, 1, 1, 1, 1 };

	public signal void stop();
	public signal void seekback();
	public signal void pauseplay();
	public signal void seekforward();
	public signal void escape();
	public signal void left();
	public signal void right();
	public signal void select();

	public InputOutput(bool pifacecad) {
		this.pifacecad = pifacecad;

		if (pifacecad) {
			Pifacecad.open ();
			GLib.Timeout.add (750, render_output);
		}

		register_handlers();
		lights();
	}

	private bool render_output () {
		Pifacecad.lcd_clear ();

		if (output_str.length <= 15) {
			Pifacecad.lcd_write (output_str);
		} else {
			var builder = new StringBuilder ();
			uint rest = output_str.length > 30 ? 15 : output_str.length - 15;
			uint cutoff = output_str.length > 30 ? output_str.length - 30 : 0;

			// FIXME: Should be able to replace 15 with 16,
			// but libpifacecad seems to require a "cell" for '\n'.
			// See https://github.com/piface/libpifacecad/issues/2.
			builder.append (output_str.substring (output_shift, 15));
			builder.append ("\n ");
			builder.append (output_str.substring (15 + output_shift, rest));

			Pifacecad.lcd_write (builder.str);

			output_shift = output_shift < cutoff ? output_shift + 1 : 0;
		}

		return true;
	}

	public void output (string str) {
		GLib.stdout.printf ("%s\n", str);
		output_shift = 0;
		output_str = str;
		if (! no_pifacecad) {
			render_output(); /* Also registered as GLib.Timeout function for long-string animation. */
		}
	}

	protected void register_handlers () {
		try {
			IOChannel channel = new IOChannel.unix_new (Posix.STDIN_FILENO);
			channel.set_encoding (null);
			channel.add_watch (IOCondition.IN, check_keyboard);

			if (pifacecad) {
				Timeout.add (SWITCH_POLL_MILLISECONDS, check_switches);
			}
		} catch (IOChannelError e) {
			GLib.error ("Channel error initializing state");
		}
	}

	private void lights() {
		if (pifacecad) {
			Pifacecad.lcd_backlight_on ();	

			if (0 != lights_out_id) {
				Source.remove(lights_out_id);
			}
			lights_out_id = Timeout.add_seconds (LIGHTS_OUT_SECONDS, () => {
				Pifacecad.lcd_backlight_off ();
				return false;
			});
		}
	}

	protected bool check_switches () {
		GLib.assert(pifacecad);

		for (uint8 switch = 0; switch < 8; switch++) {
			uint8 current = Pifacecad.read_switch (switch);

			if (0 == current && 1 == lastSwitchState[switch]) {
				GLib.debug("Switch %d input\n", switch);

				switch(switch) {
				case SwitchNumber.STOP:
					stop();
					break;
				case SwitchNumber.SEEKBACK:
					seekback();
					break;
				case SwitchNumber.PAUSEPLAY:
					pauseplay();
					break;
				case SwitchNumber.SEEKFORWARD:
					seekforward();
					break;
				case SwitchNumber.ESCAPE:
					escape();
					break;
				case SwitchNumber.SELECT:
					select();
					break;
				case SwitchNumber.LEFT:
					left();
					break;
				case SwitchNumber.RIGHT:
					right();
					break;
				default:
					GLib.warning ("Invalid key pressed");
					break;
				}

				lights();
			}

			lastSwitchState[switch] = current;
		}

		return true;
	}

	public bool check_keyboard (IOChannel channel, IOCondition condition) {
		char []c = new char[1];

		if (condition == IOCondition.HUP) {
			GLib.stderr.printf ("Standard in closed?\n");
			return false;
		}

		try {
			size_t length = -1;
			IOStatus status = channel.read_chars (c, out length);
			if (status == IOStatus.EOF || length != 1) {
				GLib.stderr.printf ("Standard in closed?\n");
				return false;
			}
		} catch (IOChannelError e) {
			GLib.stderr.printf ("IOChannelError: %s\n", e.message);
			return false;
		} catch (ConvertError e) {
			GLib.stderr.printf ("ConvertError: %s\n", e.message);
			return false;
		}

		switch (c[0]) {
		case KeyNumber.STOP:
			stop();
			break;
		case KeyNumber.SEEKBACK:
			seekback();
			break;
		case KeyNumber.SEEKFORWARD:
			seekforward();
			break;
		case KeyNumber.PAUSEPLAY:
			pauseplay();
			break;
		case KeyNumber.LEFT:
			left();
			break;
		case KeyNumber.RIGHT:
			right();
			break;
		case KeyNumber.SELECT:
			select();
			break;
		case KeyNumber.ESCAPE:
			escape();
			break;
		default:
			GLib.stderr.printf ("Valid keys are:\n");
			GLib.stderr.printf ("  'z'     (stop)\n");
			GLib.stderr.printf ("  'q'     (seek backward)\n");
			GLib.stderr.printf ("  'w'     (seek forward)\n");
			GLib.stderr.printf ("  ' '     (pause/play)\n");
			GLib.stderr.printf ("  'a'     (left)\n");
			GLib.stderr.printf ("  's'     (right)\n");
			GLib.stderr.printf ("  'enter' (select)\n");
			GLib.stderr.printf ("  'esc'   (escape)\n");
			break;
		}

		return true;
	}
}
