/*   FILE: state-choose-category.vala -- A state machine implementing lcdgrilo's menu
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 December 2013 
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stop:        stop currently playing and clear playlist.
 * pauseplay:   pause if playing else transition from paused to playing.
 * seekback:    no operation.
 * seekforward: no operation.
 * next:        select next menu item.
 * previous:    select previous menu item.
 * transition:  enter folder, display more, play all or play; state to
 *              StateConnecting (StateChooseCategory), StateConnecting (self) or
 *              StatePlay.
 */

const int PAGE_SIZE = 100;

private enum CategoryType {
	BOX,
	MORE,
	TRACK,
	PLAYALL,
}

private struct Category {
	public CategoryType type;
	public Grl.Media?   media;

	public Category (CategoryType type, Grl.Media? media) {
		this.type  = type;
		this.media = media;
	}
}

private int media_compare(Grl.Media m1, Grl.Media m2) {
	if (m1.is_audio() && m2.is_audio()) {
		var disc1 = ((Grl.Media)m1).get_album_disc_number ();
		var disc2 = ((Grl.Media)m2).get_album_disc_number ();

		if (disc1 != disc2) {
			return disc1 - disc2;
		} else {
			var track1 = ((Grl.Media)m1).get_track_number ();
			var track2 = ((Grl.Media)m2).get_track_number ();

				return track1 - track2;
		}
	} else {
		return strcmp (m1.get_title (), m2.get_title ());
	}
}

class StateChooseCategory : State {
	private int page = 0;
	private Grl.Media parent_selection;
	private MainLoop loop;
	private Grl.Source source;
	private Gee.ArrayList<Category?> categories;
	private int current = 0;
	private Grl.Media canary = null;
	private Gee.ArrayList<Grl.Media> media = new Gee.ArrayList<Grl.Media> ();
	private ChildMediaListBuilder child_media_list_builder; // FIXME: This is here so it won't get freed when it goes out of scope.
	private delegate string GetFunc (Grl.Media media);

	// FIXME: Handle connections that require authentication.
	public void connected_cb (Grl.Source source) {
		var state = player.stateStack.pop (); // Pop dummy "Connecting to service" state.
		GLib.assert (state is StateConnecting);

		var next_state = state.transition ();
		GLib.assert (next_state is StateChooseCategory);
		((StateChooseCategory) next_state).build_menu ();

		// Do not push if already on top of stack after popping connection (e.g., user selected "more").
		if (next_state != player.stateStack.peek()) {
			player.stateStack.push (next_state);
		}
	}

	private class ChildMediaListBuilder {
		private StateChooseCategory state;
		private uint count = 0;

		public ChildMediaListBuilder (StateChooseCategory state) {
			this.state = state;
		}

		// Possibly called PAGE_SIZE + 1 times.
		// PAGE_SIZE to fill menu and + 1 for a "canary" that indicates more values will follow.
		// If we don't get to the + 1, then there are no more values for another page.
		// If there is a previous canary, then add it to the menu.
		public void browse_cb (Grl.Source source, uint operation_id, owned Grl.Media? media, uint remaining, GLib.Error? error) {
			GLib.assert (count <= PAGE_SIZE);

			// FIXME: Nearly the same implementation exists elsewhere.
			if (null != error) {
				return;
			}

			if (null == media) {
				return;
			}

			count++;

			if (null != state.canary) {
				state.media.add (state.canary);
				state.canary = null;
			}

			if (count == PAGE_SIZE) {
				state.canary = media;
			} else {
				state.media.add (media);
			}

			if (0 == remaining) {
				// FIXME: Replace ArrayList with a data structure that does a sorted insert?
				state.media.sort (media_compare);

				state.connected_cb (source);
			}
		}
	}

	public void add_media (Grl.Media media) {
		this.media.add (media);
	}

	public StateChooseCategory (MainLoop loop, LCDPlayer player, Grl.Source source, Grl.Media? parent_selection) {
		this.loop             = loop;
		this.player           = player;
		this.source           = source;
		this.parent_selection = parent_selection;
	}

	// FIXME: When to use this vs. setting with _cb?
	public void build_menu () {
		bool have_media = false;

		this.categories = new Gee.ArrayList<Category?> ();

		foreach (var m in media) {
			if (m.is_container()) {
				categories.add (Category (CategoryType.BOX, m));
			} else {
				have_media = true;
				categories.add (Category (CategoryType.TRACK, m));
			}
		}

		if (true == have_media) {
			categories.insert (0, Category (CategoryType.PLAYALL, null));
		}

		// TODO: When using Jamando, directly list media for user?

		// FIXME: This canary breaks when we get to the point of selecting category when using Jamendo.
		// This is because we next show a list of categories not media.
		// There are two consequences of this:
		// (1) We get a "more" when we don't need one (e.g., "Title", "More...")
		// (2) The user never gets a  chance to load a second page of media, since we move on to, e.g., StateChooseTitle.
		if (null != canary) {
			categories.add (Category (CategoryType.MORE, null));
		}

		print_selected ();
	}

	// Print selected item.
	public override void print_selected () {
		switch (categories[current].type) {
		case CategoryType.BOX:
			player.io.output (categories[current].media.get_title ());
			break;
		case CategoryType.TRACK:
			var builder = new StringBuilder ();

			if (categories[current].media is Grl.Media) {
				var disc = ((Grl.Media) categories[current].media).get_album_disc_number ();
				var track = ((Grl.Media) categories[current].media).get_track_number ();
				if (track != 0) {
					builder.append (disc.to_string ());
					builder.append ("-");
					builder.append (track.to_string ());
					builder.append (". ");
				}
			}
			builder.append (categories[current].media.get_title ());

			player.io.output (builder.str);
			break;
		case CategoryType.MORE:
			player.io.output ("More...");
			break;
		case CategoryType.PLAYALL:
			player.io.output ("Play all");
			break;
		default:
			GLib.assert_not_reached ();
		}
	}

	// Seek to previous track.
	public override void seekback () {
	}

	// Seek to next track.
	public override void seekforward () {
	}

	// Select next menu item.
	public override void next () {
		current = (current + 1) % categories.size;
		print_selected ();
	}

	// Select previous menu item.
	public override void previous () {
		current = (current + categories.size - 1) % categories.size;
		print_selected ();
	}

	// Transition to next state, based on current selection.
	public override State transition () {
		switch (categories[current].type) {
		case CategoryType.BOX:
			// Nearly the same exists elsewhere.
			GLib.List keys = Grl.MetadataKey.list_new (Grl.MetadataKey.ALBUM,
			                                           Grl.MetadataKey.ARTIST,
			                                           Grl.MetadataKey.TITLE,
			                                           Grl.MetadataKey.TRACK_NUMBER,
			                                           Grl.MetadataKey.URL);

			var caps               = source.get_caps (Grl.SupportedOps.BROWSE);
			var options            = new Grl.OperationOptions (caps);

			StateChooseCategory next_state = new StateChooseCategory (loop,
			                                                          this.player,
			                                                          this.source,
			                                                          categories[current].media);

			page++;
			// Note: first round will actually have PAGE_SIZE - 1 displayed,
			// as the last item will become the first canary.
			options.set_count  (PAGE_SIZE);

			child_media_list_builder = new ChildMediaListBuilder (next_state);
			source.do_browse (categories[current].media, keys, options, child_media_list_builder.browse_cb);

			return new StateConnecting (loop,
			                            player,
			                            categories[current].media.get_title (),
			                            next_state);
		case CategoryType.MORE:
			// Remove this "more", as we will add another to the end later.
			categories.remove (categories[current]);

			// Nearly the same exists elsewhere.
			GLib.List keys = Grl.MetadataKey.list_new (Grl.MetadataKey.ALBUM,
			                                           Grl.MetadataKey.ARTIST,
			                                           Grl.MetadataKey.TITLE,
			                                           Grl.MetadataKey.TRACK_NUMBER,
			                                           Grl.MetadataKey.URL);

			var caps               = source.get_caps (Grl.SupportedOps.BROWSE);
			var options            = new Grl.OperationOptions (caps);

			page++;
			options.set_skip  (PAGE_SIZE * page);
			options.set_count (PAGE_SIZE);

			child_media_list_builder = new ChildMediaListBuilder (this);
			source.do_browse (parent_selection, keys, options, child_media_list_builder.browse_cb);

			return new StateConnecting (loop,
			                            player,
			                            parent_selection.get_title (),
			                            this);
		case CategoryType.PLAYALL:
			foreach (var m in media) {
				if (m is Grl.Media) {
					player.playlist.add ((Grl.Media) m);
				}
			}

			player.playlist.sort (media_compare);

			return new StatePlay(player);
		case CategoryType.TRACK:
			GLib.assert (categories[current].media is Grl.Media);
			var media = categories[current].media;
			if (media in player.playlist) {
				player.io.output (media.get_title () + " already in queue");
			} else {
				player.playlist.add ((Grl.Media) media);
				player.io.output ("Added '" + media.get_title () + "' to queue");
			}
			return new StatePlay(player);
		default:
			GLib.assert_not_reached ();
		}
	}
}
