/*   FILE: state-choose-service.vala -- A state machine implementing lcdgrilo's menu
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 December 2013 
 *
 * Copyright (c) 2013 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * stop:        stop currently playing and clear playlist.
 * pauseplay:   pause if playing else transition from paused to playing.
 * seekback:    no operation.
 * seekforward: no operation.
 * next:        select next source in menu.
 * previous:    select previous source in menu.
 * transition:  connect to the selected source; state to
 *              StateConnecting (StateChooseCategory).
 */

class StateChooseSource : State {
	private MainLoop loop;
	private Gee.ArrayList<unowned Grl.Source> sources;
	private int current = 0;
	private MediaListBuilder media_list_builder; // FIXME: This is here so it won't get freed when it goes out of scope.

	private class MediaListBuilder {
                private StateChooseCategory state;

                public MediaListBuilder (StateChooseCategory state) {
                        this.state = state;
                }

		public void browse_cb (Grl.Source source, uint operation_id, owned Grl.Media? media, uint remaining, GLib.Error? error) {
			if (null != error) {
				var builder = new StringBuilder ();
				builder.printf ("Error browsing: %s\n", error.message);
				state.player.io.output (builder.str);
				return;
			}

			if (null != media) {
				state.add_media (media);
			}
			if (0 == remaining) {
				state.connected_cb (source);
			}

			if (null == media && remaining != 0) {
				/* media is only null on error, or when finishing a list
				 * of unknown length */
				state.player.io.output ("Error browsing");
				return;
			}
		}
	}

	public StateChooseSource (MainLoop loop, LCDPlayer player, Gee.ArrayList<unowned Grl.Source> sources) {
		this.loop     = loop;
		this.sources  = sources;
		this.player   = player;

		print_selected ();
	}

	// Print selected item.
	public override void print_selected () {
		if (0 == sources.size) {
			player.io.output ("No media source found");
		} else {
			player.io.output (sources[current].get_name ());
		}
	}

	// Seek to previous track.
	public override void seekback () {
	}

	// Seek to next track.
	public override void seekforward () {
	}

	// Select next menu item.
	public override void next () {
		current = sources.size > 0 ? (current + 1) % sources.size : 0;
		print_selected ();
	}

	// Select previous menu item.
	public override void previous () {
		current = sources.size > 0 ? (current + sources.size - 1) % sources.size : 0;
		print_selected ();
	}

	// Transition to next state, based on current selection.
	public override State transition () {
		if (0 == sources.size) {
			return this;
		}

		GLib.List keys = Grl.MetadataKey.list_new (Grl.MetadataKey.ALBUM,
		                                           Grl.MetadataKey.ARTIST,
		                                           Grl.MetadataKey.TITLE,
		                                           Grl.MetadataKey.TRACK_NUMBER,
		                                           Grl.MetadataKey.URL,
		                                           Grl.MetadataKey.AUDIO_TRACK);
		var caps               = sources[current].get_caps (Grl.SupportedOps.BROWSE);
		var options            = new Grl.OperationOptions (caps); 

		StateChooseCategory next_state = new StateChooseCategory (loop, player, sources[current], null);

		media_list_builder = new MediaListBuilder (next_state);
                sources[current].do_browse (null, keys, options, media_list_builder.browse_cb);

		return new StateConnecting (loop,
		                            player,
		                            sources[current].get_name (),
		                            next_state);
	}
}
