"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MockServer = void 0;
const mock_1 = require("@zwave-js/serial/mock");
const testing_1 = require("@zwave-js/testing");
const deferred_promise_1 = require("alcalzone-shared/deferred-promise");
const net_1 = require("net");
const Utils_1 = require("./Utils");
class MockServer {
    constructor(options = {}) {
        this.options = options;
    }
    async start() {
        const { port: serialport, binding } = await (0, mock_1.createAndOpenMockedZWaveSerialPort)("/tty/FAKE");
        this.serialport = serialport;
        this.binding = binding;
        console.log("Mock serial port opened");
        // Hook up a fake controller and nodes
        prepareMocks(binding, this.options.config?.controller, this.options.config?.nodes);
        // Start a TCP server, listen for connections, and forward them to the serial port
        this.server = (0, net_1.createServer)((socket) => {
            if (!this.serialport) {
                console.error("Serial port not initialized");
                socket.destroy();
                return;
            }
            console.log("Client connected");
            socket.pipe(this.serialport);
            this.serialport.on("data", (chunk) => {
                if (typeof chunk === "number") {
                    socket.write(Buffer.from([chunk]));
                }
                else {
                    socket.write(chunk);
                }
            });
            // when the connection is closed, unpipe the streams
            socket.on("close", () => {
                console.log("Client disconnected");
                socket.unpipe(this.serialport);
                this.serialport?.removeAllListeners("data");
            });
        });
        // Do not allow more than one client to connect
        this.server.maxConnections = 1;
        const promise = (0, deferred_promise_1.createDeferredPromise)();
        this.server.on("error", (err) => {
            if (err.code === "EADDRINUSE") {
                promise.reject(err);
            }
        });
        this.server.listen({
            host: this.options.interface,
            port: this.options.port ?? 5555,
        }, () => {
            const address = this.server.address();
            console.log(`Server listening on tcp://${address.address}:${address.port}`);
            promise.resolve();
        });
    }
    async stop() {
        console.log("Shutting down mock server...");
        this.server?.close();
        await this.serialport?.close();
        if (this.binding?.isOpen)
            await this.binding?.close();
        console.log("Mock server shut down");
    }
}
exports.MockServer = MockServer;
function prepareMocks(mockPort, controller = {}, nodes = []) {
    const mockController = new testing_1.MockController({
        homeId: 0x7e570001,
        ownNodeId: 1,
        ...controller,
        serial: mockPort,
    });
    // Apply default behaviors that are required for interacting with the driver correctly
    mockController.defineBehavior(...(0, Utils_1.createDefaultMockControllerBehaviors)());
    for (const node of nodes) {
        const mockNode = new testing_1.MockNode({
            ...node,
            controller: mockController,
        });
        mockController.addNode(mockNode);
        // Apply default behaviors that are required for interacting with the driver correctly
        mockNode.defineBehavior(...(0, Utils_1.createDefaultMockNodeBehaviors)());
    }
}
//# sourceMappingURL=mockServer.js.map