"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.behaviors = void 0;
const ConfigurationCC_1 = require("@zwave-js/cc/ConfigurationCC");
const safe_1 = require("@zwave-js/core/safe");
const testing_1 = require("@zwave-js/testing");
const defaultCapabilities = {
    bulkSupport: false,
    parameters: [],
};
const STATE_KEY_PREFIX = "Configuration_";
const StateKeys = {
    value: (param) => `${STATE_KEY_PREFIX}value_${param}`,
};
const respondToConfigurationGet = {
    async onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCGet) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            const parameter = frame.payload.parameter;
            const paramInfo = capabilities.parameters.find((p) => p["#"] === parameter);
            // Do not respond if the parameter is not supported
            if (!paramInfo)
                return true;
            const value = self.state.get(StateKeys.value(parameter)) ??
                paramInfo.defaultValue ??
                0;
            const cc = new ConfigurationCC_1.ConfigurationCCReport(self.host, {
                nodeId: controller.host.ownNodeId,
                parameter,
                value,
                valueSize: paramInfo.valueSize,
                valueFormat: paramInfo.format,
            });
            await self.sendToController((0, testing_1.createMockZWaveRequestFrame)(cc, {
                ackRequested: false,
            }));
            return true;
        }
        return false;
    },
};
const respondToConfigurationSet = {
    onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCSet) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            const parameter = frame.payload.parameter;
            const paramInfo = capabilities.parameters.find((p) => p["#"] === parameter);
            // Do nothing if the parameter is not supported
            if (!paramInfo)
                return true;
            if (frame.payload.resetToDefault) {
                self.state.delete(StateKeys.value(parameter));
                return true;
            }
            const value = frame.payload.value;
            // Do nothing if the value is out of range
            if (paramInfo.minValue != undefined && value < paramInfo.minValue) {
                return true;
            }
            else if (paramInfo.maxValue != undefined &&
                value > paramInfo.maxValue) {
                return true;
            }
            self.state.set(StateKeys.value(parameter), value);
            return true;
        }
        return false;
    },
};
const respondToConfigurationDefaultReset = {
    onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCDefaultReset) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            for (const paramInfo of capabilities.parameters) {
                self.state.delete(StateKeys.value(paramInfo["#"]));
            }
            return true;
        }
        return false;
    },
};
const respondToConfigurationNameGet = {
    async onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCNameGet) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            const parameter = frame.payload.parameter;
            const paramInfo = capabilities.parameters.find((p) => p["#"] === parameter);
            // Do nothing if the parameter is not supported
            if (!paramInfo)
                return true;
            const cc = new ConfigurationCC_1.ConfigurationCCNameReport(self.host, {
                nodeId: controller.host.ownNodeId,
                parameter,
                name: paramInfo.name ?? "",
                reportsToFollow: 0,
            });
            await self.sendToController((0, testing_1.createMockZWaveRequestFrame)(cc, {
                ackRequested: false,
            }));
            return true;
        }
        return false;
    },
};
const respondToConfigurationInfoGet = {
    async onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCInfoGet) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            const parameter = frame.payload.parameter;
            const paramInfo = capabilities.parameters.find((p) => p["#"] === parameter);
            // Do nothing if the parameter is not supported
            if (!paramInfo)
                return true;
            const cc = new ConfigurationCC_1.ConfigurationCCInfoReport(self.host, {
                nodeId: controller.host.ownNodeId,
                parameter,
                info: paramInfo.info ?? "",
                reportsToFollow: 0,
            });
            await self.sendToController((0, testing_1.createMockZWaveRequestFrame)(cc, {
                ackRequested: false,
            }));
            return true;
        }
        return false;
    },
};
const respondToConfigurationPropertiesGet = {
    async onControllerFrame(controller, self, frame) {
        if (frame.type === testing_1.MockZWaveFrameType.Request &&
            frame.payload instanceof ConfigurationCC_1.ConfigurationCCPropertiesGet) {
            const capabilities = {
                ...defaultCapabilities,
                ...self.getCCCapabilities(safe_1.CommandClasses.Configuration, frame.payload.endpointIndex),
            };
            const parameter = frame.payload.parameter;
            const paramIndex = capabilities.parameters.findIndex((p) => p["#"] === parameter);
            const paramInfo = capabilities.parameters[paramIndex];
            const nextParameter = capabilities.parameters[paramIndex + 1];
            let cc;
            // If the parameter is not supported, respond with the first supported parameter
            if (!paramInfo) {
                cc = new ConfigurationCC_1.ConfigurationCCPropertiesReport(self.host, {
                    nodeId: controller.host.ownNodeId,
                    parameter,
                    valueFormat: 0,
                    valueSize: 0,
                    nextParameter: nextParameter?.["#"] ?? 0,
                });
            }
            else {
                cc = new ConfigurationCC_1.ConfigurationCCPropertiesReport(self.host, {
                    nodeId: controller.host.ownNodeId,
                    parameter,
                    valueSize: paramInfo.valueSize,
                    valueFormat: paramInfo.format ?? safe_1.ConfigValueFormat.SignedInteger,
                    minValue: paramInfo.minValue,
                    maxValue: paramInfo.maxValue,
                    defaultValue: paramInfo.defaultValue,
                    isAdvanced: paramInfo.isAdvanced ?? false,
                    altersCapabilities: paramInfo.altersCapabilities ?? false,
                    isReadonly: paramInfo.readonly ?? false,
                    noBulkSupport: !(capabilities.bulkSupport ?? false),
                    nextParameter: nextParameter?.["#"] ?? 0,
                });
            }
            await self.sendToController((0, testing_1.createMockZWaveRequestFrame)(cc, {
                ackRequested: false,
            }));
            return true;
        }
        return false;
    },
};
exports.behaviors = [
    respondToConfigurationGet,
    respondToConfigurationSet,
    respondToConfigurationNameGet,
    respondToConfigurationInfoGet,
    respondToConfigurationPropertiesGet,
    respondToConfigurationDefaultReset,
];
//# sourceMappingURL=Configuration.js.map