/*
* Copyright (c) 2003-2008 Rony Shapiro <ronys@users.sourceforge.net>.
* All rights reserved. Use of the code is allowed under the
* Artistic License 2.0 terms, as specified in the LICENSE file
* distributed with this code, or available from
* http://www.opensource.org/licenses/artistic-license-2.0.php
*/
// pws_platform.h
//------------------------------------------------------------------------------
//
// This header file exists to determine, at compile time, which target we're
// compiling to and to make the appropriate #defines and #includes.
//
// The following macros are defined:
//
//    PWS_BIG_ENDIAN    - Defined only if the target CPU is big-endian.
//    PWS_LITTLE_ENDIAN - Defined only if the target CPU is little-endian.
//    PWS_PLATFORM      - A string, the target platform, e.g. "Pocket PC".
//    PWS_PLATFORM_EX   - A string, the target platform, e.g. "Pocket PC 2000".
//    POCKET_PC         - Defined only if target is Pocket PC 2000 or later.
//    POCKET_PC_VER     - Defined only if target is Pocket PC 2000 or later.
//
// Notes:
//
// 1. PWS_BIG_ENDIAN and PWS_LITTLE_ENDIAN are mutually exclusive.
// 2. PWS_BIG_ENDIAN and PWS_LITTLE_ENDIAN may be defined on the complier
//    command line but checks are made to ensure that one and only one is
//    defined.
//
// Supported Configurations:
// -------------------------
//
// Pocket PC:
//
//    2000 ARM
//    2000 MIPS
//    2000 SH3
//    2000 x86 Emulator
//    2002 ARM
//    2002 x86 Emulator
//    2003 ARM (untested)
//    2003 x86 Emulator (untested)
//
// Windows
//
//    Win32 X86
//
// Linux - work in progress
// Cygwin - work in progress

#ifndef PwsPlatform_h
#define PwsPlatform_h

#if defined(_WIN32)
#include "../stdafx.h" // ONLY place in corelib which refers to parent. Ugh.
#else
// some globally useful includes for non-Windows
#include <cassert>
#endif

// Following seems needed on Linux/cygwin
#if defined(__linux__) || defined(__CYGWIN__)
#define LTC_NO_ROLC
#endif

// stop MS VC++ 6 and earlier being quite so pedantic
#if defined(_MSC_VER) && (_MSC_VER <= 1200) // i.e. only MSVC++ 6 and earlier
#pragma warning ( disable : 4097 )
#endif

#undef PWS_PLATFORM
#undef POCKET_PC

#if defined(_WIN32)
#ifdef BIG_ENDIAN
#define PWD_BIG_ENDIAN
#endif
#ifdef LITTLE_ENDIAN
#define PWS_LITTLE_ENDIAN
#endif
#endif

// PWS_BIG_ENDIAN and PWS_LITTLE_ENDIAN can be specified on the 
#if defined(PWS_BIG_ENDIAN)
#undef PWS_BIG_ENDIAN
#define PWS_BIG_ENDIAN
#endif

#if defined(PWS_LITTLE_ENDIAN)
#undef PWS_LITTLE_ENDIAN
#define PWS_LITTLE_ENDIAN
#endif

#if defined(_WIN32_WCE_PSPC)
// **********************************************
// * Pocket PC 2000                             *
// **********************************************
#if (WIN32_PLATFORM_PSPC == 1)
#define PWS_PLATFORM "Pocket PC"
#define PWS_PLATFORM_EX "Pocket PC 2000"
#define POCKET_PC_VER 2000
#define POCKET_PC
#if !defined(PWS_LITTLE_ENDIAN) && !defined(PWS_BIG_ENDIAN)
#if defined(ARM) || defined(_ARM)
#define PWS_LITTLE_ENDIAN
#elif defined(MIPS) || defined(_MIPS)
#define PWS_LITTLE_ENDIAN
#elif defined(SH3) || defined(_SH3)
#define PWS_LITTLE_ENDIAN
#elif defined(x86) || defined(_x86) || defined(_X86) || defined(_X86_)
#define PWS_LITTLE_ENDIAN
#endif
#endif
// **********************************************
// * Pocket PC 2002 and later                   *
// **********************************************
#elif (WIN32_PLATFORM_PSPC >= 310)
#define PWS_PLATFORM "Pocket PC"
#define PWS_PLATFORM_EX "Pocket PC 2002"
#define POCKET_PC_VER 2002
#define POCKET_PC
#if !defined(PWS_LITTLE_ENDIAN) && !defined(PWS_BIG_ENDIAN)
#if defined(ARM) || defined(_ARM)
#define PWS_LITTLE_ENDIAN
#elif defined(x86) || defined(_x86) || defined(_X86) || defined(_X86_)
#define PWS_LITTLE_ENDIAN
#endif
#endif
#else
#error Only Pocket PC 2000 and later are supported
#endif
// **********************************************
// * Windows 32                                 *
// **********************************************
#elif defined(_WIN32)
#if defined(x86) || defined(_x86) || defined(_X86) || defined(_X86_) || defined(_M_IX86) || defined(_M_X64)
#define PWS_PLATFORM "Windows"
#define PWS_LITTLE_ENDIAN
#endif
// **********************************************
// * Linux on Intel                             *
// **********************************************
#elif defined(__linux)
#define PWS_PLATFORM "Linux"
#if defined(__i386__) || defined(__i486__)
#define PWS_LITTLE_ENDIAN
#endif
// **********************************************
// * cygwin on Intel                             *
// **********************************************
#elif defined(__CYGWIN__)
#define PWS_PLATFORM "Cygwin"
#if defined(__i386__) || defined(__i486__)
#define PWS_LITTLE_ENDIAN
 #endif
// **********************************************
// * Add other platforms here...                *
// **********************************************
#endif

// 
#if !defined(PWS_PLATFORM)
#error Unable to determine the target platform - please fix PwsPlatform.h
#endif

#if !defined(PWS_LITTLE_ENDIAN) && !defined(PWS_BIG_ENDIAN)
#error Cannot determine whether the target CPU is big or little endian - please fix PwsPlatform.h
#endif

#if defined(PWS_BIG_ENDIAN) && defined(PWS_LITTLE_ENDIAN)
#error Both PWS_BIG_ENDIAN and PWS_LITTLE_ENDIAN are defined, only one should be defined.
#endif

// Following from libtomcrypt, for twofish & SHA256
/* Controls endianess and size of registers.  Leave uncommented to get platform neutral [slower] code 
* 
* Note: in order to use the optimized macros your platform must support unaligned 32 and 64 bit read/writes.
* The x86 platforms allow this but some others [ARM for instance] do not.  On those platforms you **MUST**
* use the portable [slower] macros.
*/

/* detect x86-32 machines somewhat */
#if defined(INTEL_CC) || (defined(_MSC_VER) && defined(WIN32)) || (defined(__GNUC__) && (defined(__DJGPP__) || defined(__CYGWIN__) || defined(__MINGW32__) || defined(__i386__)))
#define ENDIAN_LITTLE
#define ENDIAN_32BITWORD
#endif

/* detects MIPS R5900 processors (PS2) */
#if (defined(__R5900) || defined(R5900) || defined(__R5900__)) && (defined(_mips) || defined(__mips__) || defined(mips))
#define ENDIAN_LITTLE
#define ENDIAN_64BITWORD
#endif

/* detect amd64 */
#if defined(__x86_64__)
#define ENDIAN_LITTLE
#define ENDIAN_64BITWORD
#endif

/* #define ENDIAN_LITTLE */
/* #define ENDIAN_BIG */

/* #define ENDIAN_32BITWORD */
/* #define ENDIAN_64BITWORD */

#if (defined(ENDIAN_BIG) || defined(ENDIAN_LITTLE)) && !(defined(ENDIAN_32BITWORD) || defined(ENDIAN_64BITWORD))
#error You must specify a word size as well as endianess in mycrypt_cfg.h
#endif

#if !(defined(ENDIAN_BIG) || defined(ENDIAN_LITTLE))
#define ENDIAN_NEUTRAL
#endif

/* fix for MSVC ...evil! */
#ifdef _MSC_VER
#define CONST64(n) n ## ui64
typedef unsigned __int64 ulong64;
#else
#define CONST64(n) n ## ULL
typedef unsigned long long ulong64;
#endif

/* this is the "32-bit at least" data type 
* Re-define it to suit your platform but it must be at least 32-bits 
*/
#if defined(__x86_64__)
typedef unsigned ulong32;
#else
typedef unsigned long ulong32;
#endif

/* ---- HELPER MACROS ---- */
#ifdef ENDIAN_NEUTRAL

#define STORE32L(x, y)                                                                     \
{ (y)[3] = (unsigned char)(((x)>>24)&255); (y)[2] = (unsigned char)(((x)>>16)&255);   \
  (y)[1] = (unsigned char)(((x)>>8)&255); (y)[0] = (unsigned char)((x)&255); }

#define LOAD32L(x, y)                            \
{ x = ((unsigned long)((y)[3] & 255)<<24) | \
  ((unsigned long)((y)[2] & 255)<<16) | \
  ((unsigned long)((y)[1] & 255)<<8)  | \
  ((unsigned long)((y)[0] & 255)); }

#define STORE64L(x, y)                                                                     \
{ (y)[7] = (unsigned char)(((x)>>56)&255); (y)[6] = (unsigned char)(((x)>>48)&255);   \
  (y)[5] = (unsigned char)(((x)>>40)&255); (y)[4] = (unsigned char)(((x)>>32)&255);   \
  (y)[3] = (unsigned char)(((x)>>24)&255); (y)[2] = (unsigned char)(((x)>>16)&255);   \
  (y)[1] = (unsigned char)(((x)>>8)&255); (y)[0] = (unsigned char)((x)&255); }

#define LOAD64L(x, y)                                                       \
{ x = (((ulong64)((y)[7] & 255))<<56)|(((ulong64)((y)[6] & 255))<<48)| \
  (((ulong64)((y)[5] & 255))<<40)|(((ulong64)((y)[4] & 255))<<32)| \
  (((ulong64)((y)[3] & 255))<<24)|(((ulong64)((y)[2] & 255))<<16)| \
  (((ulong64)((y)[1] & 255))<<8)|(((ulong64)((y)[0] & 255))); }

#define STORE32H(x, y)                                                                     \
{ (y)[0] = (unsigned char)(((x)>>24)&255); (y)[1] = (unsigned char)(((x)>>16)&255);   \
  (y)[2] = (unsigned char)(((x)>>8)&255); (y)[3] = (unsigned char)((x)&255); }

#define LOAD32H(x, y)                            \
{ x = ((unsigned long)((y)[0] & 255)<<24) | \
  ((unsigned long)((y)[1] & 255)<<16) | \
  ((unsigned long)((y)[2] & 255)<<8)  | \
  ((unsigned long)((y)[3] & 255)); }

#define STORE64H(x, y)                                                                     \
{ (y)[0] = (unsigned char)(((x)>>56)&255); (y)[1] = (unsigned char)(((x)>>48)&255);     \
  (y)[2] = (unsigned char)(((x)>>40)&255); (y)[3] = (unsigned char)(((x)>>32)&255);     \
  (y)[4] = (unsigned char)(((x)>>24)&255); (y)[5] = (unsigned char)(((x)>>16)&255);     \
  (y)[6] = (unsigned char)(((x)>>8)&255); (y)[7] = (unsigned char)((x)&255); }

#define LOAD64H(x, y)                                                      \
{ x = (((ulong64)((y)[0] & 255))<<56)|(((ulong64)((y)[1] & 255))<<48) | \
  (((ulong64)((y)[2] & 255))<<40)|(((ulong64)((y)[3] & 255))<<32) | \
  (((ulong64)((y)[4] & 255))<<24)|(((ulong64)((y)[5] & 255))<<16) | \
  (((ulong64)((y)[6] & 255))<<8)|(((ulong64)((y)[7] & 255))); }

#endif /* ENDIAN_NEUTRAL */

#ifdef ENDIAN_LITTLE

#define STORE32H(x, y)                                                                     \
{ (y)[0] = (unsigned char)(((x)>>24)&255); (y)[1] = (unsigned char)(((x)>>16)&255);   \
  (y)[2] = (unsigned char)(((x)>>8)&255); (y)[3] = (unsigned char)((x)&255); }

#define LOAD32H(x, y)                            \
{ x = ((unsigned long)((y)[0] & 255)<<24) | \
  ((unsigned long)((y)[1] & 255)<<16) | \
  ((unsigned long)((y)[2] & 255)<<8)  | \
  ((unsigned long)((y)[3] & 255)); }

#define STORE64H(x, y)                                                                     \
{ (y)[0] = (unsigned char)(((x)>>56)&255); (y)[1] = (unsigned char)(((x)>>48)&255);     \
  (y)[2] = (unsigned char)(((x)>>40)&255); (y)[3] = (unsigned char)(((x)>>32)&255);     \
  (y)[4] = (unsigned char)(((x)>>24)&255); (y)[5] = (unsigned char)(((x)>>16)&255);     \
  (y)[6] = (unsigned char)(((x)>>8)&255); (y)[7] = (unsigned char)((x)&255); }

#define LOAD64H(x, y)                                                      \
{ x = (((ulong64)((y)[0] & 255))<<56)|(((ulong64)((y)[1] & 255))<<48) | \
  (((ulong64)((y)[2] & 255))<<40)|(((ulong64)((y)[3] & 255))<<32) | \
  (((ulong64)((y)[4] & 255))<<24)|(((ulong64)((y)[5] & 255))<<16) | \
  (((ulong64)((y)[6] & 255))<<8)|(((ulong64)((y)[7] & 255))); }

#ifdef ENDIAN_32BITWORD 

#define STORE32L(x, y)        \
{ unsigned long __t = (x); memcpy(y, &__t, 4); }

#define LOAD32L(x, y)         \
  memcpy(&(x), y, 4);

#define STORE64L(x, y)                                                                     \
{ (y)[7] = (unsigned char)(((x)>>56)&255); (y)[6] = (unsigned char)(((x)>>48)&255);   \
  (y)[5] = (unsigned char)(((x)>>40)&255); (y)[4] = (unsigned char)(((x)>>32)&255);   \
  (y)[3] = (unsigned char)(((x)>>24)&255); (y)[2] = (unsigned char)(((x)>>16)&255);   \
  (y)[1] = (unsigned char)(((x)>>8)&255); (y)[0] = (unsigned char)((x)&255); }

#define LOAD64L(x, y)                                                       \
{ x = (((ulong64)((y)[7] & 255))<<56)|(((ulong64)((y)[6] & 255))<<48)| \
  (((ulong64)((y)[5] & 255))<<40)|(((ulong64)((y)[4] & 255))<<32)| \
  (((ulong64)((y)[3] & 255))<<24)|(((ulong64)((y)[2] & 255))<<16)| \
  (((ulong64)((y)[1] & 255))<<8)|(((ulong64)((y)[0] & 255))); }

#else /* 64-bit words then  */

#define STORE32L(x, y)        \
{ unsigned long __t = (x); memcpy(y, &__t, 4); }

#define LOAD32L(x, y)         \
{ memcpy(&(x), y, 4); x &= 0xFFFFFFFF; }

#define STORE64L(x, y)        \
{ ulong64 __t = (x); memcpy(y, &__t, 8); }

#define LOAD64L(x, y)         \
{ memcpy(&(x), y, 8); }

#endif /* ENDIAN_64BITWORD */

#endif /* ENDIAN_LITTLE */

#ifdef ENDIAN_BIG
#define STORE32L(x, y)                                                                     \
{ (y)[3] = (unsigned char)(((x)>>24)&255); (y)[2] = (unsigned char)(((x)>>16)&255);   \
  (y)[1] = (unsigned char)(((x)>>8)&255); (y)[0] = (unsigned char)((x)&255); }

#define LOAD32L(x, y)                            \
{ x = ((unsigned long)((y)[3] & 255)<<24) | \
  ((unsigned long)((y)[2] & 255)<<16) | \
  ((unsigned long)((y)[1] & 255)<<8)  | \
  ((unsigned long)((y)[0] & 255)); }

#define STORE64L(x, y)                                                                     \
{ (y)[7] = (unsigned char)(((x)>>56)&255); (y)[6] = (unsigned char)(((x)>>48)&255);     \
  (y)[5] = (unsigned char)(((x)>>40)&255); (y)[4] = (unsigned char)(((x)>>32)&255);     \
  (y)[3] = (unsigned char)(((x)>>24)&255); (y)[2] = (unsigned char)(((x)>>16)&255);     \
  (y)[1] = (unsigned char)(((x)>>8)&255); (y)[0] = (unsigned char)((x)&255); }

#define LOAD64L(x, y)                                                      \
{ x = (((ulong64)((y)[7] & 255))<<56)|(((ulong64)((y)[6] & 255))<<48) | \
  (((ulong64)((y)[5] & 255))<<40)|(((ulong64)((y)[4] & 255))<<32) | \
  (((ulong64)((y)[3] & 255))<<24)|(((ulong64)((y)[2] & 255))<<16) | \
  (((ulong64)((y)[1] & 255))<<8)|(((ulong64)((y)[0] & 255))); }

#ifdef ENDIAN_32BITWORD 

#define STORE32H(x, y)        \
{ unsigned long __t = (x); memcpy(y, &__t, 4); }

#define LOAD32H(x, y)         \
  memcpy(&(x), y, 4);

#define STORE64H(x, y)                                                                     \
{ (y)[0] = (unsigned char)(((x)>>56)&255); (y)[1] = (unsigned char)(((x)>>48)&255);   \
  (y)[2] = (unsigned char)(((x)>>40)&255); (y)[3] = (unsigned char)(((x)>>32)&255);   \
  (y)[4] = (unsigned char)(((x)>>24)&255); (y)[5] = (unsigned char)(((x)>>16)&255);   \
  (y)[6] = (unsigned char)(((x)>>8)&255);  (y)[7] = (unsigned char)((x)&255); }

#define LOAD64H(x, y)                                                       \
{ x = (((ulong64)((y)[0] & 255))<<56)|(((ulong64)((y)[1] & 255))<<48)| \
  (((ulong64)((y)[2] & 255))<<40)|(((ulong64)((y)[3] & 255))<<32)| \
  (((ulong64)((y)[4] & 255))<<24)|(((ulong64)((y)[5] & 255))<<16)| \
  (((ulong64)((y)[6] & 255))<<8)| (((ulong64)((y)[7] & 255))); }

#else /* 64-bit words then  */

#define STORE32H(x, y)        \
{ unsigned long __t = (x); memcpy(y, &__t, 4); }

#define LOAD32H(x, y)         \
{ memcpy(&(x), y, 4); x &= 0xFFFFFFFF; }

#define STORE64H(x, y)        \
{ ulong64 __t = (x); memcpy(y, &__t, 8); }

#define LOAD64H(x, y)         \
{ memcpy(&(x), y, 8); }

#endif /* ENDIAN_64BITWORD */
#endif /* ENDIAN_BIG */

#define BSWAP(x)  ( ((x>>24)&0x000000FFUL) | ((x<<24)&0xFF000000UL)  | \
  ((x>>8)&0x0000FF00UL)  | ((x<<8)&0x00FF0000UL) )


/* 32-bit Rotates */
#if defined(_MSC_VER)

/* instrinsic rotate */
#include <stdlib.h>
#pragma intrinsic(_lrotr,_lrotl)
#define ROR(x,n) _lrotr(x,n)
#define ROL(x,n) _lrotl(x,n)
#define RORc(x,n) _lrotr(x,n)
#define ROLc(x,n) _lrotl(x,n)

#elif defined(__GNUC__) && (defined(__i386__) || defined(__x86_64__)) && !defined(INTEL_CC) && !defined(LTC_NO_ASM)

static inline unsigned ROL(unsigned word, int i)
{
  asm ("roll %%cl,%0"
    :"=r" (word)
    :"0" (word),"c" (i));
  return word;
}

static inline unsigned ROR(unsigned word, int i)
{
  asm ("rorl %%cl,%0"
    :"=r" (word)
    :"0" (word),"c" (i));
  return word;
}

#ifndef LTC_NO_ROLC

static inline unsigned ROLc(unsigned word, const int i)
{
  asm ("roll %2,%0"
    :"=r" (word)
    :"0" (word),"I" (i));
  return word;
}

static inline unsigned RORc(unsigned word, const int i)
{
  asm ("rorl %2,%0"
    :"=r" (word)
    :"0" (word),"I" (i));
  return word;
}

#else

#define ROLc ROL
#define RORc ROR

#endif

#else

/* rotates the hard way */
#define ROL(x, y) ( (((unsigned long)(x)<<(unsigned long)((y)&31)) | (((unsigned long)(x)&0xFFFFFFFFUL)>>(unsigned long)(32-((y)&31)))) & 0xFFFFFFFFUL)
#define ROR(x, y) ( ((((unsigned long)(x)&0xFFFFFFFFUL)>>(unsigned long)((y)&31)) | ((unsigned long)(x)<<(unsigned long)(32-((y)&31)))) & 0xFFFFFFFFUL)
#define ROLc(x, y) ( (((unsigned long)(x)<<(unsigned long)((y)&31)) | (((unsigned long)(x)&0xFFFFFFFFUL)>>(unsigned long)(32-((y)&31)))) & 0xFFFFFFFFUL)
#define RORc(x, y) ( ((((unsigned long)(x)&0xFFFFFFFFUL)>>(unsigned long)((y)&31)) | ((unsigned long)(x)<<(unsigned long)(32-((y)&31)))) & 0xFFFFFFFFUL)

#endif


/* 64-bit Rotates */
#if defined(__GNUC__) && defined(__x86_64__) && !defined(LTC_NO_ASM)

static inline unsigned long ROL64(unsigned long word, int i)
{
  asm("rolq %%cl,%0"
    :"=r" (word)
    :"0" (word),"c" (i));
  return word;
}

static inline unsigned long ROR64(unsigned long word, int i)
{
  asm("rorq %%cl,%0"
    :"=r" (word)
    :"0" (word),"c" (i));
  return word;
}

#ifndef LTC_NO_ROLC

static inline unsigned long ROL64c(unsigned long word, const int i)
{
  asm("rolq %2,%0"
    :"=r" (word)
    :"0" (word),"J" (i));
  return word;
}

static inline unsigned long ROR64c(unsigned long word, const int i)
{
  asm("rorq %2,%0"
    :"=r" (word)
    :"0" (word),"J" (i));
  return word;
}

#else /* LTC_NO_ROLC */

#define ROL64c ROL
#define ROR64c ROR

#endif

#else /* Not x86_64  */

#define ROL64(x, y) \
  ( (((x)<<((ulong64)(y)&63)) | \
  (((x)&CONST64(0xFFFFFFFFFFFFFFFF))>>((ulong64)64-((y)&63)))) & CONST64(0xFFFFFFFFFFFFFFFF))

#define ROR64(x, y) \
  ( ((((x)&CONST64(0xFFFFFFFFFFFFFFFF))>>((ulong64)(y)&CONST64(63))) | \
  ((x)<<((ulong64)(64-((y)&CONST64(63)))))) & CONST64(0xFFFFFFFFFFFFFFFF))

#define ROL64c(x, y) \
  ( (((x)<<((ulong64)(y)&63)) | \
  (((x)&CONST64(0xFFFFFFFFFFFFFFFF))>>((ulong64)64-((y)&63)))) & CONST64(0xFFFFFFFFFFFFFFFF))

#define ROR64c(x, y) \
  ( ((((x)&CONST64(0xFFFFFFFFFFFFFFFF))>>((ulong64)(y)&CONST64(63))) | \
  ((x)<<((ulong64)(64-((y)&CONST64(63)))))) & CONST64(0xFFFFFFFFFFFFFFFF))

#endif

#undef MAX
#undef MIN
#define MAX(x, y) ( ((x)>(y))?(x):(y) )
#define MIN(x, y) ( ((x)<(y))?(x):(y) )

/* extract a byte portably */
#ifdef _MSC_VER
#define byte(x, n) ((unsigned char)((x) >> (8 * (n))))
#else
#define byte(x, n) (((x) >> (8 * (n))) & 255)
#endif   


#endif
