#!/usr/bin/env python3
"""
Change your nautilus directories icons easily

Author : Bilal Elmoussaoui (bil.elmoussaoui@gmail.com)
Version : 3.0.1
Website : https://github.com/bilelmoussaoui/nautilus-folder-icons
Licence : GPL-3.0
nautilus-folder-icons is free software: you can redistribute it and/or
modify it under the terms of the GNU General Public License as published
by the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.
nautilus-folder-icons is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
You should have received a copy of the GNU General Public License
along with nautilus-folder-icons. If not, see <http://www.gnu.org/licenses/>.
"""
from gettext import gettext as _
from gettext import textdomain
from os import path
from sys import path as sys_path

from gi import require_version
require_version("Gtk", "3.0")
from gi.repository import GObject, Gtk

if "nemo" == "nautilus":
    require_version("Nautilus", "3.0")
    from gi.repository import Nautilus as FileManager
else:
    require_version("Nemo", "3.0")
    from gi.repository import Nemo as FileManager

textdomain('com.github.bilelmoussaoui.nautilus-folder-icons')
sys_path.insert(0, "/usr/share/com.github.bilelmoussaoui.nautilus-folder-icons")

from utils import change_folder_icon, restore_default_icon, uriparse


class OpenFolderIconProvider(GObject.GObject,
                             FileManager.LocationWidgetProvider):

    def __init__(self):
        self._window = None
        self._folder = None
        self._accel_group = None

    def _create_accel_group(self):
        self._accel_group = Gtk.AccelGroup()
        key, mod = Gtk.accelerator_parse("<Shift><Ctrl>S")
        self._accel_group.connect(key, mod, Gtk.AccelFlags.VISIBLE,
                                  self._open_folder_icon)

    def _open_folder_icon(self, *args):
        change_folder_icon([self._folder], self._window)

    def get_widget(self, uri, window):
        self._folder = uriparse(uri)
        if self._window:
            self._window.remove_accel_group(self._accel_group)
        if path.isdir(self._folder):
            self._create_accel_group()
            window.add_accel_group(self._accel_group)
        self._window = window
        return None


class FileManagerFolderIcons(GObject.GObject, FileManager.MenuProvider):

    def get_file_items(self, window, files):
        # Force use to select only directories
        folders = []
        for file_ in files:
            if not file_.is_directory() or file_.get_uri_scheme() != "file":
                pass
            folder = uriparse(file_)
            folders.append(folder)
        if not folders:
            return

        top_item = FileManager.MenuItem(name='nemoPython::folder-icons',
                                    label=_('Folder Icon'),
                                    tip=_('Change folder icon'))

        submenu = FileManager.Menu()
        top_item.set_submenu(submenu)

        item = FileManager.MenuItem(name='nemoPython::change_folder_icon',
                                    label=_("Select a new icon"))
        item.connect('activate', self._chagne_folder_icon, folders, window)
        submenu.append_item(item)

        if len(folders) == 1:
            item = FileManager.MenuItem(name='nemoPython::restore_default_icon',
                                    label=_("Restore default"), tip=_("Restore default icon"))
            item.connect('activate', self._restore_default_icon, folders[0], window)
            submenu.append_item(item)
        elif len(folders) > 1:
            item = FileManager.MenuItem(name='nemoPython::restore_all_default_icon',
                                    label=_("Restore all to default"), tip=_("Restore all folders to default icon"))
            item.connect('activate', self._restore_default_icon, folders, window)
            submenu.append_item(item)


        return [top_item]

    def _chagne_folder_icon(self, menu_item, folders, nautilus_window):
        change_folder_icon(folders, nautilus_window)

    def _restore_default_icon(self, menu_item, folders, nautilus_window):
        if isinstance(folders, str):
            restore_default_icon(folders)
        else:
            for folder in folders:
                restore_default_icon(folder)
        if nautilus_window.has_action("reload"):
            action = nautilus_window.lookup_action("reload")
            action.emit("activate", None)
