-- scatter.e - Hiding/retrieving bytes in/from an image - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file contains routines for hiding bytes in a sequence representing
-- the pixels of an image, and for retrieving bytes from such a sequence.
-- To hide/retrieve a byte, it computes the position of each bit in the
-- image (with help of a pseudo-random number generator). After writing
-- or reading the bit, it marks that bit in the sequence as already used,
-- so that no position is read/written twice.

include errors.e
include rand.e
include encrypt.e

constant include_name = "scatter.e"

constant bit_pos = #10000   -- If you change this, do not forget to change
			    -- the constant in wr_bmp as well.

integer bits_left, current_bit, data_len, bit_val, table_size
sequence data, table1, table2


procedure init_get_pos()
    -- Create empty table counters and initialize other variables

    integer rem

    -- Initialize variables
    data_len = length(data)
    bits_left = data_len

    -- How many positions will each table entry represent?
    -- This is an arbitrary formula which seems to give good results
    table_size = -floor(-power(data_len, 1/3)/2)

    -- Create the table representation for data
    -- Each table entry contains table_size unused positions now ...
    table1 = repeat( table_size, -floor(-data_len/table_size))
    -- ... except for the last entry, which represents less positions
    rem = remainder(data_len, table_size)
    if rem then  table1[length(table1)] = rem  end if

    -- Create the table representation for table1
    -- Each entry contains table_size^2 unused positions now ...
    table2 = repeat( table_size * table_size, -floor(-length(table1)/table_size))
    -- ... except for the last entry, which represents less positions
    rem = remainder(data_len, table_size * table_size)
    if rem then  table2[length(table2)] = rem  end if

end procedure

function find_unused_pos()
    -- Finds an unused position
    integer data_pos, table1_pos, table2_pos, end_pos, end_pos1

    -- Get the initial position to test
    data_pos = random()

    if data[data_pos] < current_bit then     -- Not used yet, so that's it
	return data_pos
    end if

    -- That position is already used, so find the next unused position
    -- Try one more position; if that one is also used, use the tables to find an unused one

    data_pos += 1    -- Move to next position
    if data_pos > data_len then data_pos = 1 end if -- If past the last position, go back to the first

    -- Test position
    if data[data_pos] < current_bit then     -- Not used yet
	return data_pos
    end if

    data_pos += 1    -- Move to next pos, etc.
    if data_pos > data_len then data_pos = 1 end if


    -- Calculate table entry which contains the position
    table1_pos = floor( (data_pos-1)/ table_size) + 1

    -- Verify table1 - is there any free position in current entry?
    if table1[table1_pos] != 0 then     -- There is, but it might be before the current position

	-- Calculate end of table entry position
	if table1_pos = length(table1) then
	    end_pos = data_len
	else
	    end_pos = table1_pos * table_size
	end if

	-- Find the empty position
	for pos = data_pos to end_pos do
	    if data[pos] < current_bit then     -- Not used yet
		return pos                          -- Found
	    end if
	end for

	-- Not found... the free position was before the initial position
    end if


    -- Move to next table1 entry
    table1_pos += 1
    if table1_pos > length(table1) then table1_pos = 1 end if

    -- Calculate table2 entry which contains the current table1 one
    table2_pos = floor( (table1_pos-1) / table_size) + 1

    -- Verify table2 - is there any free position in current entry?
    if table2[table2_pos] != 0 then     -- There might be

	-- Calculate end of table entry position
	if table2_pos = length(table2) then
	    end_pos1 = length(table1)
	else
	    end_pos1 = table2_pos * table_size
	end if

	-- Find the entry containing the empty position
	for pos1 = table1_pos to end_pos1 do


	    if table1[pos1] != 0 then     -- Seems we've found it

		-- Calculate first position
		data_pos = (pos1-1)*table_size + 1

		-- Calculate last position
		if pos1 = length(table1) then
		    end_pos = data_len
		else
		    end_pos = pos1 * table_size
		end if

		-- Search elements in table entry
		for pos = data_pos to end_pos do
		    if data[pos] < current_bit then     -- Not used yet
			return pos                      -- Found
		    end if
		end for

		-- The code should never get to this point
		abort_error (include_name, "find_unused_pos", "No unused position found (1)")

	    end if

	end for

    end if

    while 1 do

	-- Move to next element
	table2_pos += 1
	if table2_pos > length(table2) then table2_pos = 1 end if


	if table2[table2_pos] != 0 then -- Not all in current entry have been used

	    table1_pos = (table2_pos - 1) * table_size + 1

	    if table2_pos = length(table2) then -- Last table entry
		end_pos1 = length(table1)
	    else
		end_pos1 = table2_pos * table_size
	    end if

	    -- Search table1
	    for pos1 = table1_pos to end_pos1 do

		if table1[pos1] != 0 then   -- Not all in table1 entry have been used

		    -- The empty position is in this entry

		    -- Calculate first position
		    data_pos = (pos1 - 1) * table_size + 1

		    -- Calculate last position
		    if pos1 = length(table1) then
			end_pos = data_len
		    else
			end_pos = pos1 * table_size
		    end if

		    -- Search elements in table entry
		    for pos = data_pos to end_pos do
			if data[pos] < current_bit then     -- Not used yet
			    return pos                      -- Found
			end if
		    end for

		    -- The code should never get to this point
		    abort_error (include_name, "find_unused_pos", "No unused position found (2)")


		end if
	    end for

	    -- The code should never get to this point
	    abort_error (include_name, "find_unused_pos", "No unused position found (3)")

	end if

    end while



end function


function get_pos()
    integer pos, table1_pos, table2_pos

    if bits_left = 0 then
	current_bit += bit_pos
	bit_val *= 2
	init_get_pos()
    end if

    -- Get unused position
    pos = find_unused_pos()

    -- The position is now used, update the tables
    table1_pos = floor( (pos-1)/ table_size) + 1
    table1[table1_pos] -= 1

    table2_pos = floor( (table1_pos-1) / table_size) + 1
    table2[table2_pos] -= 1

    -- One more bit is used
    bits_left -= 1

    return pos

end function


global procedure write_byte(atom byte)
    integer pos

    -- Write a byte (least-significant bits first)
    for bit = 1 to 8 do
	pos = get_pos()

	if and_bits (byte, 1) then
	    data[pos] += bit_pos + bit_val
	else
	    data[pos] += bit_pos
	end if

	byte = floor(byte / 2)

    end for

end procedure

global procedure write_string(sequence bytes)
    for x = 1 to length(bytes) do
	write_byte (bytes[x])
    end for
end procedure

global function get_scattered_data()
    -- Returns the written data
    return data
end function

global function read_byte()
    -- Reads a byte written with write_byte()

    integer pos, byte, bit_value
    byte = 0
    bit_value = 1

    for bit = 1 to 8 do

	pos = get_pos()

	-- Get bit and add it to the byte
	if and_bits(data[pos], bit_val) then
	    byte += bit_value
	end if

	data[pos] += bit_pos

	bit_value *= 2
    end for

    return byte
end function

global function read_string (atom n)
    sequence bytes
    bytes = repeat(-1, n)
    for x = 1 to n do
	bytes [x] = read_byte()
    end for

    return bytes
end function


global procedure init_scatter_write(atom data_length)
    data_len = data_length
    data = repeat(0, data_len)

    bit_val = 1
    current_bit = bit_pos

    init_get_pos()
end procedure

global procedure init_scatter_read(sequence scattered)
    data = scattered

    bit_val = 1
    current_bit = bit_pos

    init_get_pos()
end procedure

global procedure set_scatter_key (sequence key)
    sequence temp_key
    temp_key = expand_key (0 & key, 256)
    init_random (temp_key, data_len)
end procedure

global function get_current_bit ()
    return current_bit / bit_pos
end function

global procedure end_scatter ()
    -- Free memory
    data = {}
    table1 = {}
    table2 = {}
    init_random ( repeat(0, 256), 10 )  -- Erase random state
end procedure



include truerand.e

global function make_erase_data (atom data_length)
    -- Write random bits in the data, destroying any information present

    -- Since these will be written in sequential order, we must be sure
    -- it is not possible to detect they were written

    -- To do this, we get a random key, initialize our random number
    -- generator and add some further random data to its numbers;
    -- we also choose a random starting point for the data


    atom initial_pos
    sequence data, rand_bits

    init_random (get_random_data (64), 2)

    rand_bits = rand_bytes()   -- get rand_bits length and discard first random bits

    -- set data size to smallest multiple of length(rand_bits) larger than data_length
    data = repeat (0, data_length + length (rand_bits) - remainder (data_length, length(rand_bits)))

    for n = 1 to data_length by length (rand_bits) do

	rand_bits = and_bits (rand_bytes(), 1)
	data [n..n+length(rand_bits)-1] = rand_bits + bit_pos

    end for

    initial_pos = bytes_to_int (get_random_data (4))
    if initial_pos < 0 then initial_pos += #100000000 end if

    -- Choose initial position (don't pick always the first one, this is safer)
    initial_pos = remainder (initial_pos, length(data)-data_length+1) + 1

    init_random ( repeat(0, 256), 10 )  -- Erase random state

    return data [initial_pos..initial_pos+data_length-1]
end function

