-- truerand.e - Random number generator - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This include file is used to get high-quality random numbers for use
-- with HIP. The standard Euphoria RNG has a seed of only 32 bits, and it
-- is necessary to generate 128 bits of random data for each file to be
-- hidden. In order to increase security, HIP gets random data from
-- several sources (key presses, mouse movement, system time...) .
-- Other files add random data to the buffer by calling add_random_data.

include machine.e
include sha1.e
include fileutil.e


sequence rand_buffer

-- Get some pseudo-random bytes
rand_buffer = rand (repeat(256, 16))-1 & date()


constant max_buffer_size = 500,     -- When rand_buffer is larger than
				    -- this, hash it
	 hashs_to_compute = 3       -- Number of times buffer will be
				    -- hashed when it exceeds max_buffer_size


global procedure add_random_data (sequence data)
    -- Add random data to buffer
    sequence new_rand_buffer

    rand_buffer = rand_buffer & data & date() & int_to_bytes (floor(time()*10000)) & 0

    if length (rand_buffer) >= max_buffer_size then
	-- Buffer is too big, compute 200-byte hash of it

	new_rand_buffer = {}

	for n = 1 to hashs_to_compute do
	    new_rand_buffer &= sha_1 (n & rand_buffer)
	end for

	rand_buffer = new_rand_buffer

    end if

end procedure

global procedure linux_add_dev_random(atom bytes)
    -- If in Linux, get random bytes from /dev/random

    atom fn
    if platform() = LINUX then

	fn = open ( "/dev/random", "rb")
	if fn = -1 then
	    fn = open ("/dev/urandom", "rb")
	end if

	if fn != -1 then
	    add_random_data (read_bytes (fn, bytes))
	    close (fn)
	end if
    end if
end procedure

linux_add_dev_random (16)

global function get_random_data (atom bytes)
    -- Compute hash of buffer
    sequence random_data

    linux_add_dev_random (bytes)

    random_data = {}

    -- Generate the necessary amount of random bytes
    while length(random_data) < bytes do
	random_data = random_data & sha_1 (random_data & rand_buffer)
    end while

    -- Add some random data just to make sure the buffer will be different
    -- the next time get_random_data() is called
    add_random_data (date() & rand({256,256,256,256})-1)

    return random_data [1..bytes]
end function

