/*
 * four-in-a-row game - terminal/textmode functions
 * Copyright (c) 2017 Andreas K. Foerster <info@akfoerster.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "row4.h"

static void cputv P ((unsigned int));

/* board offsets */
static short unsigned int xoffs, yoffs, score[2];
static const char playerchip[4] = { ' ', 'X', 'O', '#' };

static void (*border) P ((char *));

#if !defined(ASCII) && !defined(CP437) && !defined(VT100)
#if defined(__MSDOS__) || defined(_WIN32)
#define CP437
#elif defined(__unix__)
#define VT100
#endif
#endif

#define SCREENWIDTH 80
#define HALFSCREENWIDTH (SCREENWIDTH/2)

#ifdef __MSDOS__
#ifdef __DJGPP__
#undef __STRICT_ANSI__
#include <conio.h>
#elif defined(__BCC__) && defined(__AS386_16__)
#include <conio.h>
/* *INDENT-OFF* */

static void
akf_clrscr ()
{
#asm
  xor cx, cx  ; upper left
  mov dl, #79 ; width
  mov dh, #24 ; height
  mov bh, #7  ; attribute
  xor al, al  ; whole screen
  mov ah, #6  ; scroll up
  int $10
#endasm
}


/* without range check! */
static void
akf_gotoxy (x, y)
{
#asm
  mov bx, sp
  mov ax, [bx+2]
  mov dl, al ; x
  mov ax, [bx+4]
  mov dh, al ; y
  dec dl ; 0-based
  dec dh ; 0-based
  mov bh, #0 ; page number
  mov ah, #2
  int $10
#endasm
}
/* *INDENT-ON* */

#define clrscr() akf_clrscr()
#define gotoxy(x,y) akf_gotoxy((x), (y))

#endif /* __BCC__ */

/* like cputs but inserts \r before each \n */
static void
ctext (s)
     char *s;
{
  while (*s)
    {
      if (*s == '\n')
	putch ('\r');
      putch (*s++);
    }
}

#else /* not __MSDOS__ */

/* vt100 terminal */
#define ESC  "\033"
#define CSI  ESC "["
#define cputs(s)  write (STDOUT_FILENO, (s), strlen (s))
#define ctext(s)  write (STDOUT_FILENO, (s), strlen (s))

#define clrscr()  cputs (CSI "2J")
#define clreol()  cputs (CSI "K")

static void
putch (c)
     int c;
{
  char ch = (char) c;
  write (STDOUT_FILENO, &ch, 1);
}

/* starts with 1, 1 in upper left */
static void
gotoxy (x, y)
     int x, y;
{
  cputs (CSI);
  cputv (y);
  putch (';');
  cputv (x);
  putch ('H');
}

/* press enter to confirm  - not needed in DOS */
static int
getche ()
{
  char in[8];

  read (STDIN_FILENO, in, sizeof (in));
  return (int) *in;
}

#define getch()  getche()
#endif /* not __MSDOS__ */


#pragma GCC poison  printf fprintf sprintf snprintf

static void
cputv (v)
     unsigned int v;
{
  char buffer[40], *p;
  register unsigned int i;

  p = &buffer[sizeof (buffer) - 1];
  *p = '\0';

  i = v;
  do
    {
      *--p = (i % 10) | 0x30;
      i /= 10;
    }
  while (i);

  cputs (p);
}


#ifdef clreol

static void
clear_line (y)
     int y;
{
  gotoxy (1, y);
  clreol ();
}

#else

static void
clear_line (y)
     int y;
{
  int i;

  gotoxy (1, y);

  i = SCREENWIDTH;
  while (i--)
    putch (' ');
}

#endif


static void
centered (y, s)
     int y;
     char *s;
{
  gotoxy ((int) (HALFSCREENWIDTH - (strlen (s) / 2)), y);
  cputs (s);
}


static void
ascii_border (s)
     char *s;
{
  for (; *s; ++s)
    {
      int c;

      if ('x' == *s)
	c = '|';
      else if ('o' <= *s && *s <= 's')
	c = '-';
      else if (('j' <= *s && *s <= 'n') || ('t' <= *s && *s <= 'w'))
	c = '+';
      else
	c = *s;

      putch (c);
    }
}


#if defined(VT100)
static void
vt100_border (s)
     char *s;
{
  putch (0x0E);			/* SO (shift out) */
  cputs (s);
  putch (0x0F);			/* SI (shift in) */
}

#define native_border  vt100_border
#elif defined(CP437)

/* VT100 graphics to CP437 translation, incomplete */
static int
vt100_cp437 (c)
     int c;
{
  switch (c)
    {
      /* missing: graphical representations of control codes */

    case '`':
      return 0x04;		/* diamond */

    case 'a':
      return 0xB1;		/* checkerboard (error indicator) */

    case 'f':
      return 0xF8;		/* degree symbol */

    case 'g':
      return 0xF1;		/* plus/minus */

    case 'j':
      return 0xD9;		/* lower-right corner */

    case 'k':
      return 0xBF;		/* upper-right corner */

    case 'l':
      return 0xDA;		/* upper-left corner */

    case 'm':
      return 0xC0;		/* lower-left corner */

    case 'n':
      return 0xC5;		/* crossing lines */

      /* horizontal lines of various heights */
      /* 'q' for box drawing */
    case 'o':
    case 'p':
    case 'q':
    case 'r':
    case 's':
      return 0xC4;

    case 't':
      return 0xC3;		/* left "T" */

    case 'u':
      return 0xB4;		/* right "T" */

    case 'v':
      return 0xC1;		/* bottom "T" */

    case 'w':
      return 0xC2;		/* top "T" */

    case 'x':
      return 0xB3;		/* vertical bar */

    case 'y':
      return 0xF3;		/* less than or equal */

    case 'z':
      return 0xF2;		/* greater than or equal */

    case '{':
      return 0xE3;		/* Pi */

      /* missigng: '|': not equal to */

    case '}':
      return 0x9C;		/* UK pound */

    case '~':
      return 0xFA;		/* centered dot */
    }

  return c;
}


static void
cp437_border (s)
     char *s;
{
  while (*s)
    putch (vt100_cp437 (*s++));
}

#define native_border cp437_border
#else
#define native_border  ascii_border
#endif


extern void
initialize (mode)
     int mode;
{
  border = (MOD_ASCII == mode) ? ascii_border : native_border;

#ifdef VT100
  /* load vt100 special graphics character set into G1 */
  cputs (ESC ")0");
#endif
}


/* put chip into specified position */
extern void
chip (x, y, player)
     int x, y, player;
{
  gotoxy (xoffs + (4 * x) + 2, yoffs + (2 * (5 - y)) + 1);
  putch (playerchip[player & 3]);
}


static void
numbers ()
{
  cputs ("  1   2   3   4   5   6   7");
}


extern void
draw_board ()
{
  int y;

  clrscr ();
  centered (1, TITLE);
  centered (2, AUTHOR);

  yoffs = 4;
  xoffs = HALFSCREENWIDTH - 15;

  gotoxy (xoffs, yoffs++);
  numbers ();

  for (y = 0; y < 2 * 6; y += 2)
    {
      gotoxy (xoffs, yoffs + y);
      border (y ? "tqqqnqqqnqqqnqqqnqqqnqqqnqqqu"
	      : "lqqqwqqqwqqqwqqqwqqqwqqqwqqqk");
      gotoxy (xoffs, yoffs + y + 1);
      border ("x   x   x   x   x   x   x   x");
    }

  gotoxy (xoffs, yoffs + y);
  border ("mqqqvqqqvqqqvqqqvqqqvqqqvqqqj");

  gotoxy (xoffs, yoffs + y + 1);
  numbers ();

  centered (22, COMMENT);
}


extern void
clear_board ()
{
  int x, y;

  for (x = 0; x <= 6; ++x)
    for (y = 0; y <= 5; ++y)
      chip (x, y, NONE);

  clear_line (21);
}


static void
show_license ()
{
  clrscr ();
  gotoxy (1, 1);

  ctext (TITLE);
  ctext ("\n");
  ctext (copyright);
  ctext ("\n\n");
  ctext (license);
  ctext ("\n\n" ENTER);
  getche ();
}


/* returns slot number, or ERROR, or NEW */
extern int
ask_slot (player)
     int player;
{
  int r, ch;

  ch = playerchip[player & 3];

  r = -1;
  while (r < 0)
    {
      char c;

      clear_line (20);
      gotoxy ((int) (HALFSCREENWIDTH - 2), 20);
      putch (ch);
      cputs (": ");

      c = getche ();
      if ('n' == c || 'N' == c)
	return NEW;
      else if ('e' == c || 'E' == c)
	return ERROR;
      else if ('1' <= c && c <= '7')
	r = c - '1';
    }

  return r;
}



static void
finish ()
{
  clrscr ();
  gotoxy (1, 1);
}


extern int
ask_players ()
{
  int r;

  atexit (finish);

  r = -1;
  while (r < 0)
    {
      char c;

      clrscr ();
      gotoxy (1, 1);
      cputs (TITLE);
      cputs ("\r\n");
      cputs (copyright);
      cputs ("\r\n\n");
      cputs (PLAYERS);

      c = getche ();

      switch (c)
	{
	case '1':
	  r = 1;
	  break;

	case '2':
	  r = 2;
	  break;

	case 'l':
	case 'L':
	  show_license ();
	  break;

	case 'e':
	case 'E':
	  r = 0;
	  break;
	}
    }

  return r;
}


static void
show_score (player, x)
     int player, x;
{
  gotoxy (x, 24);
  putch (playerchip[player]);
  cputs (": ");
  cputv (score[player - 1]);
}


extern void
win (player)
     int player;
{
  char t[81];

  if (1 > player || player > 2)
    return;

  score[player - 1] += 1;

  show_score (1, HALFSCREENWIDTH - 7);
  show_score (2, HALFSCREENWIDTH + 2);

  strcpy (t, WINMSG);
  t[0] = playerchip[player];

  clear_line (21);
  centered (21, t);
}
