/* sshdos.c       Copyright (c) 2000-2002 Nagy Daniel
 *
 * $Date: 2002/06/19 14:09:35 $
 * $Revision: 1.16 $
 *
 * This module is the main part:
 *  - command line parsing
 *  - client loop
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <conio.h>
#include <string.h>
#include <time.h>
#include <stdarg.h> 

#if defined (__DJGPP__)
 #include <unistd.h>
 #include <go32.h>
 #include <dpmi.h>
 #include "tcp_djgp.h"
#elif defined (__TURBOC__)
 #include <io.h>
 #include <dos.h> 
 #include "tcp.h"
 #include "zlib.h"
#endif

#include "ssh.h"
#include "protocol.h"
#include "cipher.h"
#include "version.h"
#include "vt100.h"
#include "xmalloc.h"
#include "config.h"

/* external functions */
SendFuncPtr SendPacket;
extern SessKeyInitPtr SessKeyInit;
extern CryptPtr EncryptPacket;
extern CryptPtr DecryptPacket;

/* external structures, variables */
extern struct Packet pktin;	/* incoming SSH packet */
extern struct Packet pktout;	/* outgoing SSH packet */
extern char *RemoteClosed;
extern char *ConnectionClosed;

/* global variables */
Config GlobalConfig;		/* global configuration structure */
unsigned short Configuration = 0;	/* Configuration bits */

/* local variables */
static char *terminals[]={"xterm","vt100","linux","xterm-color"}; /* terminal types */
static char *command = NULL;
static unsigned char tty = 0;

#if defined (__DJGPP__)
/*
 * we don't want DJGPP to extend wildcards, so include
 * this dummy function
 */
char **__crt0_glob_function (char *arg)
{
   return 0;
}

/*
 * Some keepalive support tricks. In DJGPP, we must ensure,
 * that interrupt handlers remain always in memory, not
 * swapped out to disk. That's why we need end_keepalive
 * dummy function. Also note, that we don't need to call
 * the old handler under DJGPP.
 */
unsigned long timer = 0;		/* increased by timer interrupt */
_go32_dpmi_seginfo oldhandler, my_handler;	/* for old and new timer interrupt */

/* Timer handler */
void keepalive(void)
{
   timer++;
}
void end_keepalive(void) {}	/* dummy, not to swap out keepalive func. */

#elif defined (__TURBOC__)

volatile unsigned long timer = 0;	/* increased by timer interrupt */

void interrupt (*oldhandler)(void); /* for old timer interrupt */
void interrupt keepalive(void)
{
   timer++;
   oldhandler();
}

#endif

/*
 * fatal error handler
 */
void fatal(const char *fmt, ...)
{
va_list ap;
char buf[256];

   va_start(ap, fmt);
   vsprintf(buf, fmt, ap);
   va_end(ap);
   printf("%s\n", buf);
   if(GlobalConfig.brailab)
	fclose(GlobalConfig.brailab);
   if(GlobalConfig.debugfile)
	fclose(GlobalConfig.debugfile);
   if(GlobalConfig.logfile)
	fclose(GlobalConfig.logfile);
   sock_close(&GlobalConfig.s);
   exit(255);
}

/*
 * Initialize global variables
 */
static void Config_Init(void)
{
   GlobalConfig.username = NULL;
   GlobalConfig.password = NULL;
   GlobalConfig.remotehost = NULL;
   GlobalConfig.RemotePort = SSH_PORT;
   GlobalConfig.term = terminals[0]; /* default is "xterm" */
   GlobalConfig.KeyMapFile = NULL;
   GlobalConfig.identity = NULL;
   GlobalConfig.keepalives = 0;
   GlobalConfig.statusline = 1;
   GlobalConfig.CipherType = SSH_CIPHER_BLOWFISH;
   GlobalConfig.debugfile = NULL;
   GlobalConfig.logfile = NULL;
   GlobalConfig.brailab = NULL;

   SendPacket = SendSSHPacket;
   SessKeyInit = blowfish_sesskey;
   EncryptPacket = blowfish_encrypt_blk;
   DecryptPacket = blowfish_decrypt_blk;
}

/*
 * Allocate a pseudo terminal
 */
static void request_pty(char *termtype)
{
unsigned short len;

   s_wrpkt_start(SSH_CMSG_REQUEST_PTY, (len = strlen(termtype)) + 21);
   pktout.body[0] = pktout.body[1] = pktout.body[2] = 0;
   pktout.body[3] = len;
   memcpy(pktout.body + 4, termtype, len);
   memset(pktout.body + 4 + len, 0, 17);
   pktout.body[7+len] = 25 - GlobalConfig.statusline;
   pktout.body[11+len] = 80;
   s_wrpkt();

   packet_read_expect(SSH_SMSG_SUCCESS);
   xfree(pktin.whole);
}

/*
 * Start interactive shell or run command
 */
static void Start_Shell_Or_Command(void)
{
unsigned short len;

   if(command != NULL && *command != '\0') {
	s_wrpkt_start(SSH_CMSG_EXEC_CMD, (len = strlen(command)) + 4);
	pktout.body[0] = pktout.body[1] = 0;
	pktout.body[2] = (len >> 8) & 0xFF;
	pktout.body[3] = len & 0xFF;		/* up to 32k of string */
	memcpy(pktout.body + 4, command, len);
	xfree(command);
	s_wrpkt();
   }
   else{
	s_wrpkt_start(SSH_CMSG_EXEC_SHELL, 0);
	s_wrpkt();
   }
}

/*
 * Client loop. This runs when the user successfully logged in,
 * until SSH connection is terminated
 */
static short dosession(void)
{
unsigned short i;
int status;		/* status of communication */

   /* send keepalive SSH_MSG_IGNORE packet if configured */
   if(timer > GlobalConfig.keepalives){
	s_wrpkt_start(SSH_MSG_IGNORE, 4);
	pktout.body[0] = pktout.body[1] = pktout.body[2] = pktout.body[3] = 0;
	s_wrpkt();
	timer = 0;
   }

   while(ConChk())	/* examine STDIN */
	DoKey();

   sock_tick(&GlobalConfig.s, &status); /* TCP wait */
   if(!sock_dataready(&GlobalConfig.s))
	return(0);	/* Begin loop again if none */

   ssh_gotdata(); /* uncrypt and get valuable data */

   switch(pktin.type){
	case SSH_SMSG_STDOUT_DATA:
		for(i = 0; i < pktin.length - 5; i++){
		   if(tty)
			ConOut(pktin.body[i+4]);
		   else
			putchar(pktin.body[i+4]);
		}
		if(GlobalConfig.logfile)
		   fwrite(pktin.body + 4, 1, pktin.length - 5, GlobalConfig.logfile);
		break;

	case SSH_SMSG_STDERR_DATA:
		pktin.body[pktin.length - 1] = '\0';
		cputs(pktin.body + 4);
		break;

	case SSH_SMSG_EXITSTATUS:
		s_wrpkt_start(SSH_CMSG_EXIT_CONFIRMATION, 0);
		s_wrpkt();
                return(EXIT_SSH);

        case SSH_MSG_IGNORE:
		break;

        case SSH_MSG_DEBUG:
		if(Configuration & VERBOSE_MODE){
		   pktin.body[pktin.length - 1] = '\0';
		   cputs(pktin.body + 4);
		}
		break;

        case SSH_MSG_DISCONNECT:
		pktin.body[pktin.length - 1] = '\0';
		cputs(pktin.body + 4);
		return(EXIT_SSH);

	default:
		cprintf("Unsupported packet received. Type: %d\n\r",pktin.type);
		break;
   } /* switch */
   xfree(pktin.whole);
   return(0);

sock_err:
   switch(status){
	case 1:
	   cputs(ConnectionClosed);
	   break;

	case -1:
	   cputs(RemoteClosed);
	   break;
   } /* switch */
   return(EXIT_SSH);
}

/*
 * Get command line arguments
 */
static void getargs(int argc, char *argv[])
{
unsigned short i, j, len;
char *s;
char *keymaperror="Specified keymap not found!";
char *usage="Usage: sshdos [options] username remotehost [command [args]]\n"
	    "Options:\n"
	    "-c <3des|blowfish>                 - cipher type (default: blowfish)\n"
	    "-i <identity file>                 - key file for public key authentication\n"
	    "-t <xterm|vt100|linux|xterm-color> - terminal type (default: xterm)\n"
	    "-p <port number>                   - remote port\n"
	    "-k <keymap file>                   - path to keymap file\n"
	    "-s <password>                      - remote password\n"
	    "-l <log file>                      - log session to file\n"
	    "-a <minutes>                       - time between keepalive packets\n"
	    "-b <COM[1234]>                     - Brailab PC adapter on COM[1234] port\n"
	    "-P                                 - don't allocate a privileged port\n"
	    "-C                                 - enable compression\n"
	    "-S                                 - disable status line\n"
	    "-n                                 - add CR if server sends only LF\n"
	    "-d                                 - save SSH packets to debug.pkt\n"
	    "-v                                 - verbose output";

   for(i = 1; i < argc; ++i){
	s = argv[i];
	if(*s != '-')
           break;
	switch(*++s){
	   case '\0':
		fatal(usage);
		return;

	   case 'c':
		if(*++s){
		   if(!strcmp(s,"blowfish"))
			break;
		   else if(!strcmp(s,"3des")){
			GlobalConfig.CipherType = SSH_CIPHER_3DES;
			SessKeyInit = des3_sesskey;
			EncryptPacket = des3_encrypt_blk;
			DecryptPacket = des3_decrypt_blk;
		   }
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i],"blowfish"))
			break;
		   else if(!strcmp(argv[i],"3des")){
			GlobalConfig.CipherType = SSH_CIPHER_3DES;
			SessKeyInit = des3_sesskey;
			EncryptPacket = des3_encrypt_blk;
			DecryptPacket = des3_decrypt_blk;
		   }
		   else
			fatal(usage);
		}
		else
		   fatal(usage);
		continue;

	   case 'i':
		if(*++s)
		   GlobalConfig.identity = s;
		else if(++i < argc)
		   GlobalConfig.identity = argv[i];
		else
		   fatal(usage);
		continue;

	   case 's':
		if(*++s)
		   GlobalConfig.password = strdup(s);
		else if(++i < argc)
		   GlobalConfig.password = strdup(argv[i]);
		else
		   fatal(usage);
		GlobalConfig.password[MAX_PASSWORD_LENGTH] = '\0';
		continue;

	   case 'l':
		if(*++s){
		   if((GlobalConfig.logfile = fopen(s,"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else if(++i < argc){
		   if((GlobalConfig.logfile = fopen(argv[i],"w+b")) == NULL)
			fatal("Cannot create log file");
		}
		else
		   fatal(usage);
		continue;

	   case 't':
		if(*++s){
		   if(!strcmp(s,terminals[0]) ||
		      !strcmp(s,terminals[1]) ||
		      !strcmp(s,terminals[2]) ||
		      !strcmp(s,terminals[3]) )
			GlobalConfig.term = s;
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i],terminals[0]) ||
		      !strcmp(argv[i],terminals[1]) ||
		      !strcmp(argv[i],terminals[2]) ||
		      !strcmp(argv[i],terminals[3]) )
			GlobalConfig.term = argv[i];
		   else
			fatal(usage);
		}
	    	else
		   fatal(usage);
		continue;

	   case 'p':
		if(*++s)
		   GlobalConfig.RemotePort = atoi(s);
		else if(++i < argc)
		   GlobalConfig.RemotePort = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'a':
		if(*++s)
		   GlobalConfig.keepalives = atoi(s);
		else if(++i < argc)
		   GlobalConfig.keepalives = atoi(argv[i]);
		else
		   fatal(usage);
		continue;

	   case 'b':
		if(*++s){
		   if(!strcmp(s, "COM1") ||
		      !strcmp(s, "COM2") ||
		      !strcmp(s, "COM3") ||
		      !strcmp(s, "COM4")){
			if((GlobalConfig.brailab = fopen(s,"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else if(++i < argc){
		   if(!strcmp(argv[i], "COM1") ||
		      !strcmp(argv[i], "COM2") ||
		      !strcmp(argv[i], "COM3") ||
		      !strcmp(argv[i], "COM4")){
			if((GlobalConfig.brailab = fopen(argv[i],"w+b")) == NULL){
			   fatal("Cannot open COM port");
			}
		   }
		   else
			fatal(usage);
		}
		else
		   fatal(usage);
		continue;

	   case 'k':
		if(*++s){
		   if(keymap_init(s))
			fatal(keymaperror);
		}
		else if(++i < argc){
		   if(keymap_init(argv[i]))
			fatal(keymaperror);
		}
		else
		   fatal(usage);
		continue;

	   case 'P':
		Configuration |= NONPRIVILEGED_PORT;
		continue;

	   case 'S':
		GlobalConfig.statusline = 0;
		continue;

	   case 'C':
		Configuration |= COMPRESSION_REQUESTED;
		continue;

	   case 'n':
		Configuration |= NEWLINE;
		continue;

	   case 'd':
		if((GlobalConfig.debugfile = fopen("debug.pkt","w+")) == NULL)
		   fatal("Cannot create debug file");
		else
		   fputs("\n-------------------\n",GlobalConfig.debugfile);
		continue;

	   case 'v':
		Configuration |= VERBOSE_MODE;
		continue;

	   default:
		fatal(usage);
	} /* end switch */

    } /* end for */

   /* no_more_options */
   if(i + 2 > argc)
	fatal(usage);
   GlobalConfig.username = argv[i++];
   GlobalConfig.remotehost = argv[i++];
   if(i >= argc)			/* command args? */
	return;
   /* collect remaining arguments and make a command line of them */
   for(len = 0, j = i; j < argc; j++)
	len += strlen(argv[j]) + 1;	/* 1 for the separating space */
   command = (char *)xmalloc(len);
   for(*command = '\0', j = i; j < argc; j++){
	strcat(command, argv[j]);
	if(j < argc - 1)		/* add ' 'if not last argument */
            strcat(command, " ");
   }
}

/*
 * Main program starts here
 */
int main(int argc, char **argv)
{
   printf("SSHDOS v%s\n", SSH_VERSION);

   Config_Init();	/* Initialize global variables */
   srand (time(NULL));	/* Initialize random number generator */

   getargs(argc, argv); /* Process command line */

   if(TCPConnect(GlobalConfig.remotehost))	/* Connect to server */
	fatal("TCP connection error");
   if(SSHConnect())				/* begin SSH negotiation */
	fatal("SSH connection error");

   /* Request a pseudo terminal if stdout is the terminal*/
   if(isatty(fileno(stdout))){
	tty = 1;
        request_pty(GlobalConfig.term);
   }

   /* Start an interactive shell or specified command */
   Start_Shell_Or_Command();

#if defined (__DJGPP__)
   tcp_cbreak(1);	/* No Break checking under DJGPP */
#endif

   KeyInit();
   VidInit(GlobalConfig.remotehost);
   VTInit();

   if(GlobalConfig.keepalives ){	/* install keepalive timer */
	GlobalConfig.keepalives = GlobalConfig.keepalives * 18 * 60; /* correct keepalives value */
#if defined (__DJGPP__)
	_go32_dpmi_lock_data(&timer, sizeof(timer));
	_go32_dpmi_lock_code(keepalive, (long)end_keepalive - (long) keepalive);
	_go32_dpmi_get_protected_mode_interrupt_vector(0x1C, &oldhandler);
	my_handler.pm_offset = (int) keepalive;
	my_handler.pm_selector = _go32_my_cs();
	_go32_dpmi_chain_protected_mode_interrupt_vector(0x1C, &my_handler);
#elif defined (__TURBOC__)
	oldhandler = getvect(0x1C);
	setvect(0x1C, keepalive);
#endif
   } /* if */

   while(EXIT_SSH != dosession());	/* Loop until session end */
   xfree(pktin.whole);

#ifdef __TURBOC__
   if(Configuration & COMPRESSION_ENABLED)
	Disable_Compression();
#endif

   if(GlobalConfig.keepalives)
#if defined (__DJGPP__)
	_go32_dpmi_set_protected_mode_interrupt_vector(0x1C, &oldhandler);
#elif defined (__TURBOC__)
	setvect(0x1C, oldhandler);
#endif


   sock_close(&GlobalConfig.s);	/* Close TCP socket */

   /* Close open files */
   if(GlobalConfig.brailab)
	fclose(GlobalConfig.brailab);
   if(GlobalConfig.debugfile)
	fclose(GlobalConfig.debugfile);
   if(GlobalConfig.logfile)
	fclose(GlobalConfig.logfile);

   return(0);
}
