/*
 *    TTTTTTTTTTTTTT  EEEEEEEEEEEEEE  OOOOOOOOOOOOOO
 *    TTTTTTTTTTTTTT  EEEEEEEEEEEEEE  OOOOOOOOOOOOOO
 *          TT        EE              OO          OO
 *          TT        EE              OO          OO
 *          TT        EE              OO          OO
 *          TT        EEEEEEEEEE      OO          OO
 *          TT        EEEEEEEEEE      OO          OO
 *          TT        EE              OO          OO
 *          TT        EE              OO          OO
 *          TT        EE              OO          OO
 *          TT        EEEEEEEEEEEEEE  OOOOOOOOOOOOOO
 *          TT        EEEEEEEEEEEEEE  OOOOOOOOOOOOOO
 *
 *                  L'mulateur Thomson TO8
 *
 *  Copyright (C) 1997-2013 Gilles Ftis, Eric Botcazou, Alexandre Pukall,
 *                          Jrmie Guillaume, Franois Mouret
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *  Module     : hardware.c
 *  Version    : 1.8.2
 *  Cr par   : Gilles Ftis
 *  Modifi par: Eric Botcazou 24/10/2003
 *               Franois Mouret 18/09/2006 02/02/2012 29/09/2012
 *
 *  Emulation de l'environnement matriel du MC6809E:
 *	- carte mmoire
 *	- circuits d'entre/sortie du systme
 *	- circuits d'entre/sortie des priphriques
 */


#ifndef SCAN_DEPEND
   #include <stdio.h>
   #include <string.h>
   #include <time.h>
#endif

#include "defs.h"
#include "teo.h"
#include "errors.h"
#include "hardware.h"
#include "mc68xx/mc6809.h"
#include "mc68xx/mc6821.h"
#include "mc68xx/mc6846.h"
#include "media/disk/controlr.h"
#include "media/disk.h"
#include "media/keyboard.h"
#include "media/cass.h"
#include "media/mouse.h"
#include "media/printer.h"


/* les composants mtriels de l'mulateur */
struct MC6846_PIA mc6846;      /* PIA 6846 systme         */
struct MC6821_PIA pia_int;     /* PIA 6821 systme         */
struct MC6821_PIA pia_ext;     /* PIA 6821 musique et jeux */
struct GATE_ARRAY mode_page;   /* Gate Array mode page     */
struct EF9369 pal_chip;        /* circuit de palette vido */
struct MEMORY_PAGER mempager;  /* carte mmoire logique    */
struct MEMORY mem;             /* carte mmoire physique   */
struct MOTHERBOARD mb;         /* (pseudo) carte mre      */

mc6809_clock_t screen_clock;


/* Fonctions de commutation de l'espace mmoire:
 */
static void update_cart(void)
{
    static uint8 cart_garbage[32];

    if (mode_page.cart&0x20)
    {   /* l'espace cartouche est mapp sur la RAM */
        mempager.segment[0x0]=mem.ram.bank[mempager.cart.ram_page];
        mempager.segment[0x1]=mem.ram.bank[mempager.cart.ram_page]+0x1000;
        mempager.segment[0x2]=mem.ram.bank[mempager.cart.ram_page]+0x2000;
        mempager.segment[0x3]=mem.ram.bank[mempager.cart.ram_page]+0x3000;

        /* ncessaire pour Bob Morane SF !! */
        if (mode_page.system1&8)
            mc6846.prc|=4;
    }
    else if (mc6846.prc&4)
    {   /* l'espace cartouche est mapp sur la ROM interne */
        mempager.segment[0x0]=mem.rom.bank[mempager.cart.rom_page];
        mempager.segment[0x1]=mem.rom.bank[mempager.cart.rom_page]+0x1000;
        mempager.segment[0x2]=mem.rom.bank[mempager.cart.rom_page]+0x2000;
        mempager.segment[0x3]=mem.rom.bank[mempager.cart.rom_page]+0x3000;
    }
    else
    {   /* l'espace cartouche est mapp sur la cartouche Mmo7 */
        if (mem.cart.nbank==0)
            mempager.segment[0x0]=cart_garbage;
        else
        {
            mempager.segment[0x0]=mem.cart.bank[mempager.cart.page];
            mempager.segment[0x1]=mem.cart.bank[mempager.cart.page]+0x1000;
            mempager.segment[0x2]=mem.cart.bank[mempager.cart.page]+0x2000;
            mempager.segment[0x3]=mem.cart.bank[mempager.cart.page]+0x3000;
        }
    }
}


static void update_screen(void)
{
    mempager.segment[0x4]=mem.ram.bank[0]+(1-mempager.screen.page)*0x2000;
    mempager.segment[0x5]=mem.ram.bank[0]+(1-mempager.screen.page)*0x2000+0x1000;
}


static void update_system(void)
{
    mempager.segment[0x6]=mem.ram.bank[mempager.system.page];
    mempager.segment[0x7]=mem.ram.bank[mempager.system.page]+0x1000;
    mempager.segment[0x8]=mem.ram.bank[mempager.system.page]+0x2000;
    mempager.segment[0x9]=mem.ram.bank[mempager.system.page]+0x3000;
}


static void update_data(void)
{
    if (mode_page.system1&0x10)
        mempager.data.page = mempager.data.reg_page;
    else
        mempager.data.page = mempager.data.pia_page;

    mempager.segment[0xA]=mem.ram.bank[mempager.data.page]+0x2000;
    mempager.segment[0xB]=mem.ram.bank[mempager.data.page]+0x3000;
    mempager.segment[0xC]=mem.ram.bank[mempager.data.page];
    mempager.segment[0xD]=mem.ram.bank[mempager.data.page]+0x1000;
}


static void update_mon(void)
{
    mempager.segment[0xE]=mem.mon.bank[mempager.mon.page];
    mempager.segment[0xF]=mem.mon.bank[mempager.mon.page]+0x1000;
}



/* update_color:
 *  Met  jour la couleur correspond  l'index spcifi.
 */
static void update_color(int index)
{
    static int gamma[16]={0,100,127,147,163,179,191,203,215,223,231,239,243,247,251,255};

    teo_SetColor(index, gamma[ pal_chip.color[index].gr&0xF],
                        gamma[(pal_chip.color[index].gr&0xF0)>>4],
                        gamma[ pal_chip.color[index].b&0xF]);
}



/* SetDeviceRegister:
 *  Dpose un octet dans le registre du priphrique et
 *  modifie en consquence son tat.
 */
static void SetDeviceRegister(int addr, int val)
{
    static int new_lsb;
    int index;

    switch (addr)
    {
        /* PIA 6846 systme */
        case 0xE7C1:
            mc6846_WriteCommand(&mc6846, val);

            if ((mc6846.crc&0x30) == 0x30)
                teo_PutSoundByte(mc6809_clock(), mc6846.crc&8 ? 0x00 : (mc6821_ReadPort(&pia_ext.portb)&0x3F)<<2);
            break;

        case 0xE7C2:
            mc6846.ddrc=val;
            break;

        case 0xE7C3:
            mc6846_WriteData(&mc6846, val);

            /* bit 0: slection demi-page VRAM */
            mempager.screen.page = (mc6846.prc&1 ? 1 : 0);
            mempager.screen.update();

            /* bit 2: commutation page cartouche */
            mempager.cart.update();

            /* bit 4: slection page moniteur */
            mempager.mon.page = (mc6846.prc&0x10 ? 1 : 0);
            mempager.mon.update();

            /* bit 5: ACK clavier */
            keyboard_SetACK(mc6846.prc&0x20);
            break;

        case 0xE7C5:
            mc6846_SetTcr(&mc6846, val);
            break;

        case 0xE7C6:
            mc6846_SetTmsb(&mc6846, val);
            break;

        case 0xE7C7:
            mc6846_SetTlsb(&mc6846, val);
            break;

        /* PIA 6821 systme */
        case 0xE7C8:
            mc6821_WriteData(&pia_int.porta, val);

            /* criture sur le port donne de l'imprimante */
            printer_WriteData(0xFE, mc6821_ReadPort(&pia_int.porta));
            break;

        case 0xE7C9:
            mc6821_WriteData(&pia_int.portb, val);

            /* criture sur le port donne de l'imprimante */
            printer_WriteData(0x01, mc6821_ReadPort(&pia_int.portb));

            /* criture sur le STROBE de l'imprimante */
            printer_SetStrobe(mc6821_ReadPort(&pia_int.portb)&0x02);

            switch (mc6821_ReadPort(&pia_int.portb)&0xF8)
            {
                case 0xF0:  /* DDRB 0x0F */
                    mempager.data.pia_page=2;
                    break;

                case 0xE8:  /* DDRB 0x17 */
                    mempager.data.pia_page=3;
                    break;

                case 0x18:  /* DDRB 0xE7 */
                    mempager.data.pia_page=4;
                    break;

                case 0x58:  /* DDRB 0xA7 */
                    mempager.data.pia_page=5;
                    break;

                case 0x98:  /* DDRB 0x67 */
                    mempager.data.pia_page=6;
                    break;

                case 0xD8:  /* DDRB 0x27 */
                    mempager.data.pia_page=7;
                    break;

/* le TO7-70 et le TO9 ont une correspondance subtilement diffrente: 
   les cas 0x58 et 0x98 sont interchangs !!!
   (voir Chip: le Crpuscule du Naja routines 8029 et 8085 et le manuel
    technique TO8/9/9+ page 45) */
            }

            mempager.data.update();
            break;

        case 0xE7CA:
            mc6821_WriteCommand(&pia_int.porta, val);
#ifdef LEP_Motor
            if ((val&0x30) == 0x30)
                LEP_Motor(pia_int.porta.cr&8 ? OFF : ON);
#endif
            break;

        case 0xE7CB:
            mc6821_WriteCommand(&pia_int.portb, val);
            break;

        /* PIA 6821 musique et jeux */
        case 0xE7CC:
            mc6821_WriteData(&pia_ext.porta, val);
            break;

        case 0xE7CD:
            mc6821_WriteData(&pia_ext.portb, val);

            if (!(mc6846.crc&8))  /* MUTE son inactif */
                teo_PutSoundByte(mc6809_clock(), (mc6821_ReadPort(&pia_ext.portb)&0x3F)<<2);
            break;

        case 0xE7CE:
            mc6821_WriteCommand(&pia_ext.porta, val);
            break;

        case 0xE7CF:
            mc6821_WriteCommand(&pia_ext.portb, val);
            break;

        /* Gate Array lecteur de disquettes */
        case 0xE7D0:
            dkc->SetReg0(val);
            break;

        case 0xE7D1:
            dkc->SetReg1(val);
            break;

        case 0xE7D2:
            dkc->SetReg2(val);
            break;

        case 0xE7D3:
            dkc->SetReg3(val);
            break;

        case 0xE7D4:
            dkc->SetReg4(val);
            break;

        case 0xE7D5:
            dkc->SetReg5(val);
            break;

        case 0xE7D6:
            dkc->SetReg6(val);
            break;

        case 0xE7D7:
            dkc->SetReg7(val);
            break;

        case 0xE7D8:
            dkc->SetReg8(val);
            break;

        case 0xE7D9:
            dkc->SetReg9(val);
            break;

        /* Gate Array mode page */
        case 0xE7DA:
            index = (mode_page.p_addr>>1);

            if (mode_page.p_addr&1)
            {
                val = 0xE0|(val&0xF);

                if ((pal_chip.color[index].b != val) || new_lsb)
                {
                    pal_chip.color[index].b = val;
                    pal_chip.update(index);
                    new_lsb = FALSE;
                    teo_new_video_params = TRUE;
                }
            }
            else if (pal_chip.color[index].gr != val)
            {
                pal_chip.color[index].gr = val;
                new_lsb = TRUE;
            }

            mode_page.p_addr=(mode_page.p_addr+1)&0x1F;
            break;

        case 0xE7DB:
            mode_page.p_addr=val&0x1F;
            break;

        case 0xE7DC:
            if (val != mode_page.lgamod)
            {
                mode_page.lgamod=val;
                teo_new_video_params=TRUE;
            }
            break;

        case 0xE7DD:
            if (mempager.screen.vram_page != (val>>6)) /* commutation de la VRAM */
            {
               mempager.screen.vram_page=val>>6;
               teo_new_video_params=TRUE;
            }
            if ( ((mode_page.system2&0xF) != (val&0xF)) && teo_SetBorderColor)
                teo_SetBorderColor(mode_page.lgamod, val&0xF);

            mode_page.system2=val;
            break;

        case 0xE7E5:
            mode_page.ram_data=val;
 
            /* bit 0-4: slection page de l'espace donnes */
            mempager.data.reg_page=val&0x1F;
            mempager.data.update();
            break;

        case 0xE7E6:  /* commutation de l'espace cartouche */
            mode_page.cart=val;
            mempager.cart.ram_page=val&0x1F;
            mempager.cart.update();
            break;

        case 0xE7E7:
            mode_page.system1=(val&0x5F)+(mode_page.system1&0xA0);

            /* bit 4: mode de commutation de l'espace donnes */
            mempager.data.update();
            break;

    } /* end of switch */
}



/* GetRealValue_lp4:
 *  Rcupre la valeur ajuste de mode_page.lp4
 */
static inline int GetRealValue_lp4(void)
{
    mc6809_clock_t spot_pos, spot_point;
    int lp4;

    /* Positionne d'office b7 et b5  1 */
    lp4=mode_page.lp4|0xA0;

    /* Position du spot dans l'cran */
    spot_pos=mc6809_clock()-screen_clock;

    /* Ajoute 1 cycle si ligne impaire:
        Pour assurer la "fbrilit" du spot ?
        Chinese Stack ne fonctionne bien qu'avec un ajout
        de 1 cycle pour les lignes impaires, et le raster de
        couleurs de HCL en est parfaitement centr */
    if ((spot_pos/FULL_LINE_CYCLES)&1)
        spot_pos+=1;
    
    /* Positionne b7  0 si le spot est au dessus ou au dessous de l'cran affichable */
    spot_point=spot_pos/FULL_LINE_CYCLES;    /* Numro de ligne */
    if ((spot_point<TOP_SHADOW_LINES+TOP_BORDER_LINES)
        || (spot_point>=TOP_SHADOW_LINES+TOP_BORDER_LINES+WINDOW_LINES))
        lp4&=0x7f;

    /* Positionne b5  0 si le spot est  droite ou  gauche de l'cran affichable */
    spot_point=spot_pos%FULL_LINE_CYCLES;    /* Numro de colonne */
    if ((spot_point<LEFT_SHADOW_CYCLES+LEFT_BORDER_CYCLES)
        || (spot_point>=LEFT_SHADOW_CYCLES+LEFT_BORDER_CYCLES+WINDOW_LINE_CYCLES))
        lp4&=0xDF;

    return lp4;
}


/* StoreWord:
 *  Ecrit deux octets en mmoire.
 */
static void StoreWord(int addr, int val)
{
    hardware_StoreByte(addr, (val>>8));
    hardware_StoreByte(addr+1, val&0xFF);
}



/* LoadByte:
 *  Lit un octet en mmoire.
 */
static int LoadByte(int addr)
{
    int index;

    if ((0xE7C0<=addr) && (addr<=0xE7FF))
        switch (addr)
        {
            /* PIA 6846 systme */
            case 0xE7C0:
            case 0xE7C4:
                return mc6846.csr;

            case 0xE7C1:
                return mc6846.crc;

            case 0xE7C2:
                return mc6846.ddrc;

            case 0xE7C3:
                return mc6846.prc;

            case 0xE7C5:
                return mc6846_tcr(&mc6846);

            case 0xE7C6:
                return mc6846_tmsb(&mc6846);

            case 0xE7C7:
                return mc6846_tlsb(&mc6846);

            /* PIA 6821 systme */
            case 0xE7C8:
                return mc6821_ReadData(&pia_int.porta);

            case 0xE7C9:
                return mc6821_ReadData(&pia_int.portb);

            case 0xE7CA:
		return mc6821_ReadCommand(&pia_int.porta);

            case 0xE7CB:
		return mc6821_ReadCommand(&pia_int.portb);

            /* PIA 6821 musique et jeux */
            case 0xE7CC:
                return mc6821_ReadData(&pia_ext.porta);

            case 0xE7CD:
                return mc6821_ReadData(&pia_ext.portb);

            case 0xE7CE:
		return mc6821_ReadCommand(&pia_ext.porta);

            case 0xE7CF:
		return mc6821_ReadCommand(&pia_ext.portb);

            /* Gate Array lecteur de disquettes */
            case 0xE7D0:
                return dkc->GetReg0();

            case 0xE7D1:
                return dkc->GetReg1();

            case 0xE7D2:
                return dkc->GetReg2();

            case 0xE7D3:
                return dkc->GetReg3();

            case 0xE7D4:
                return dkc->GetReg4();

            case 0xE7D5:
                return dkc->GetReg5();

            case 0xE7D6:
                return dkc->GetReg6();

            case 0xE7D7:
                return dkc->GetReg7();

            case 0xE7D8:
                return dkc->GetReg8();

            case 0xE7D9:
                return dkc->GetReg9();

            /* Gate Array mode page */
            case 0xE7DA:
                index = (mode_page.p_addr>>1);

                if (mode_page.p_addr&1)
                    mode_page.p_data = pal_chip.color[index].b;
                else
                    mode_page.p_data = pal_chip.color[index].gr;

                mode_page.p_addr = (mode_page.p_addr+1)&0x1F;
                return mode_page.p_data;

            case 0xE7DB:
                return mode_page.p_addr;

            case 0xE7DC:
            case 0xE7DD:
            case 0xE7DE:
            case 0xE7DF:
            case 0xE7E0:
            case 0xE7E1:
            case 0xE7E2:
            case 0xE7E3: 
                return 0xCC;

            case 0xE7E4:
                return (mode_page.commut&1 ? mode_page.lp1 : mode_page.system2&0xC0);

            case 0xE7E5:
                /* Retourne le numro de banque courante mme en mode PIA */
                return (mode_page.commut&1 ? mode_page.lp2 : mempager.data.page&0x1F);

            case 0xE7E6:
                return (mode_page.commut&1 ? mode_page.lp3 : mode_page.cart);

            case 0xE7E7:
                return (/*mode_page.lp4*/ GetRealValue_lp4()&0xFE) | (mode_page.commut&1);
        }

    return LOAD_BYTE(addr);
}



/* LoadWord:
 *  Lit deux octets en mmoire.
 */
static int LoadWord(int addr)
{
    if ((0xE7C0-1<=addr) && (addr<=0xE7FF))
        return (LoadByte(addr)<<8)+LoadByte(addr+1);
    else
        return LOAD_WORD(addr);
}



/* FetchInstr:
 *  Remplit le buffer de fetch du CPU.
 */
static void FetchInstr(int addr, unsigned char fetch_buffer[])
{
    register int i;

    for (i=0; i<MC6809_FETCH_BUFFER_SIZE; i++)
        fetch_buffer[i]=LOAD_BYTE(addr+i);
}



/* BiosCall:
 *  Appel aux routines du BIOS de gestion des priphriques.
 */
static int BiosCall(struct MC6809_REGS *regs)
{
    time_t x;
    struct tm *t;

    switch (regs->pc)
    {
        case 0x25D4:  /* routine d'affichage de la date */
            time(&x);
            t = gmtime(&x);
            STORE_BYTE(0x607C,t->tm_mday);
            STORE_BYTE(0x607D,t->tm_mon+1);
            STORE_BYTE(0x607E,t->tm_year%100);
            return 0x10; /* LDY immdiat */

        case 0x315A:  /* routine de slection souris/crayon optique */
            teo_SetPointer( LOAD_BYTE(0x6074)&0x80 ? TEO_STATUS_MOUSE: TEO_STATUS_LIGHTPEN);
            mouse_Reset();
            return 0x8E; /* LDX immdiat */

        case 0x357A:  /* page de modification palette: fin */
            mode_page.lgamod=TEO_COL40;
            return 0x7D; /* TST tendu */

        case 0x3686:  /* page de modification palette: dbut */
            mode_page.lgamod=TEO_PALETTE;
            return 0x86; /* LDA immdiat */

        case 0x337E:  /* routine GETL crayon optique   */
        case 0x3F97:  /* routine GETL crayon optique 2 */
	    mouse_GetLightpen(&regs->xr, &regs->yr, &regs->cc);
            break;

        case 0xFA5A:  /* routine CASS */
            cass_Event(&regs->br,&regs->cc);
            break;

 	/* Imprimante */
        case 0xFB66:
            printer_Close();
            return 0x8a; /* ORA immdiat */

    } /* end of switch */
    return 0x12;  /* NOP */
}


/* ------------------------------------------------------------------------- */


#ifdef DEBUG  /* la fonction n'est pas inline */

void DrawGPL(int addr)
{ 
    int pt,col;

    if (addr>=0x1F40)
        return;

    pt =mem.ram.bank[mempager.screen.vram_page][addr];
    col=mem.ram.bank[mempager.screen.vram_page][addr+0x2000];

    teo_DrawGPL(mode_page.lgamod, addr, pt, col);
}

#endif



/* hardware_StoreByte:
 *  Ecrit un octet en mmoire.
 */
void hardware_StoreByte(int addr, int val)
{
    /* masque de commutation des pages de la cartouche */
    static const int page_mask[]={0, 0x0, 0x1, 0x2, 0x2};
    
    int msq=addr>>12;

    switch (msq)
    {
        case 0x0: /* espace cartouche */
        case 0x1:
        case 0x2:
        case 0x3:
            if (mode_page.cart&0x20)  /* l'espace est mapp sur la RAM */
            {
                if (mode_page.cart&0x40) /* l'criture est autorise */
                {
                    STORE_BYTE(addr, val);

                    if ((mempager.screen.vram_page==mempager.cart.ram_page) && mb.direct_screen_mode)
                        DrawGPL(addr&0x1FFF);
                }
            }
            else if (addr<=0x1FFF)  /* commutation par latchage */
            {
                if (mc6846.prc&4)
                    mempager.cart.rom_page = addr&3;
                else
                    mempager.cart.page = addr&page_mask[mem.cart.nbank];
                    
                mempager.cart.update();
            }
            break;

        case 0x4: /* espace cran */
        case 0x5:
            STORE_BYTE(addr, val);

            if ((mempager.screen.vram_page==0) && mb.direct_screen_mode)
                DrawGPL(addr&0x1FFF);

            break;

        case 0x6: /* espace systme non commutable */
        case 0x7:
        case 0x8:
        case 0x9:
            STORE_BYTE(addr, val);

            if ((mempager.screen.vram_page==1) && mb.direct_screen_mode)
                DrawGPL(addr&0x1FFF);

            break;

        case 0xA: /* espace donnes */
        case 0xB:
        case 0xC:
        case 0xD:
            STORE_BYTE(addr, val);

            if ((mempager.screen.vram_page==mempager.data.page) && mb.direct_screen_mode)
                DrawGPL(addr&0x1FFF);

            break;

        case 0xE:
        case 0xF:
            if ((0xE7C0<=addr) && (addr<=0xE7FF))
                SetDeviceRegister(addr,val&0xff);

            break;
    }
}



/* hardware_Init:
 *  Initialise la carte mmoire de l'mulateur.
 */
void hardware_Init(void)
{
    register int i;

    struct MC6809_INTERFACE interface={ FetchInstr,
                                        LoadByte,
                                        LoadWord,
                                        hardware_StoreByte,
                                        StoreWord,
                                        BiosCall };
    mc6809_Init(&interface);

    /* circuit de palette vido */
    pal_chip.update = update_color;

    /* dfinition de la carte mmoire physique */
    /* pas de cartouche */
    mem.cart.nbank = 0;
    mem.cart.size = 0x4000;

    /* 64ko de ROM */
    mem.rom.nbank = 4;
    mem.rom.size = 0x4000;
    for (i=0; i<mem.rom.nbank; i++)
        mem.rom.bank[i] = NULL;

#ifdef DEBIAN_BUILD
    strcpy(mem.rom.filename[0], "/usr/share/teo/system/rom/basic512.rom");
    strcpy(mem.rom.filename[1], "/usr/share/teo/system/rom/extramon.rom");
    strcpy(mem.rom.filename[2], "/usr/share/teo/system/rom/basic1.rom");
    strcpy(mem.rom.filename[3], "/usr/share/teo/system/rom/expl.rom");
#else
    strcpy(mem.rom.filename[0], "system/rom/basic512.rom");
    strcpy(mem.rom.filename[1], "system/rom/extramon.rom");
    strcpy(mem.rom.filename[2], "system/rom/basic1.rom");
    strcpy(mem.rom.filename[3], "system/rom/expl.rom");
#endif

    /* 512ko de RAM */
    mem.ram.nbank = 32;            
    mem.ram.size = 0x4000;
    for (i=0; i<mem.ram.nbank; i++)
        mem.ram.bank[i] = NULL;

    /* 16ko de ROM moniteur */
    mem.mon.nbank = 2;
    mem.mon.size = 0x2000;
    for (i=0; i<mem.mon.nbank; i++)
        mem.mon.bank[i] = NULL;

#ifdef DEBIAN_BUILD
    strcpy(mem.mon.filename[0], "/usr/share/teo/system/rom/monitor1.rom");
    strcpy(mem.mon.filename[1], "/usr/share/teo/system/rom/monitor2.rom");
#else
    strcpy(mem.mon.filename[0], "system/rom/monitor1.rom");
    strcpy(mem.mon.filename[1], "system/rom/monitor2.rom");
#endif

    /* dfinition de la carte mmoire logique */
    mempager.cart.update = update_cart;
    mempager.screen.update = update_screen;
    mempager.system.update = update_system;
    mempager.data.update = update_data;
    mempager.mon.update = update_mon;

    /* carte mre */
    mb.exact_clock = 0;
    mb.direct_screen_mode = TRUE;

    LOCK_VARIABLE(mc6809_irq);
    LOCK_VARIABLE(mc6846);
    LOCK_VARIABLE(pia_int);
    LOCK_VARIABLE(pia_ext);
    LOCK_VARIABLE(mode_page);
    LOCK_VARIABLE(pal_chip);
    LOCK_VARIABLE(mempager);
    LOCK_VARIABLE(mem);
    LOCK_VARIABLE(screen_clock);
}

