package jp.sourceforge.sxdbutils;

import java.math.BigDecimal;
import java.sql.Clob;
import java.sql.Types;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import jp.sourceforge.sxdbutils.types.BigDecimalType;
import jp.sourceforge.sxdbutils.types.BooleanCharType;
import jp.sourceforge.sxdbutils.types.BooleanNumericType;
import jp.sourceforge.sxdbutils.types.BooleanType;
import jp.sourceforge.sxdbutils.types.ByteArrayType;
import jp.sourceforge.sxdbutils.types.ByteType;
import jp.sourceforge.sxdbutils.types.CharNumericType;
import jp.sourceforge.sxdbutils.types.CharType;
import jp.sourceforge.sxdbutils.types.DateTimestampType;
import jp.sourceforge.sxdbutils.types.DoubleType;
import jp.sourceforge.sxdbutils.types.FloatType;
import jp.sourceforge.sxdbutils.types.IntegerType;
import jp.sourceforge.sxdbutils.types.LongTimestampType;
import jp.sourceforge.sxdbutils.types.LongType;
import jp.sourceforge.sxdbutils.types.ObjectType;
import jp.sourceforge.sxdbutils.types.ShortType;
import jp.sourceforge.sxdbutils.types.SqlDateType;
import jp.sourceforge.sxdbutils.types.StringClobType;
import jp.sourceforge.sxdbutils.types.StringNumericType;
import jp.sourceforge.sxdbutils.types.StringType;

public class TypeMappings {
	private TypeMappings() {
	}

	/**
	 * {@link BigDecimal}で受け取る{@link ValueType}
	 */
	public static final ValueType BIGDECIMAL_TYPE = new BigDecimalType();
	/**
	 * booleanで受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_PRIMITIVE_TYPE = new BooleanType(
			true);
	/**
	 * {@link Boolean}で受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_OBJECT_TYPE = new BooleanType(
			false);

	/**
	 * CHARフィールドをbooleanで受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_CHAR_PRIMITIVE_TYPE = new BooleanCharType(
			true);
	/**
	 * CHARフィールドを{@link Boolean}で受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_CHAR_OBJECT_TYPE = new BooleanCharType(
			false);

	/**
	 * 数値型フィールドをbooleanで受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_NUMERIC_PRIMITIVE_TYPE = new BooleanNumericType(
			true);
	/**
	 * 数値型フィールドを{@link Boolean}で受け取る{@link ValueType}
	 */
	public static final ValueType BOOLEAN_NUMERIC_OBJECT_TYPE = new BooleanNumericType(
			false);

	/**
	 * byte[]で受け取る{@link ValueType}
	 */
	public static final ValueType BYTEARRAY_TYPE = new ByteArrayType();
	/**
	 * byteで受け取る{@link ValueType}
	 */
	public static final ValueType BYTE_PRIMITIVE_TYPE = new ByteType(true);
	/**
	 * {@link Byte}で受け取る{@link ValueType}
	 */
	public static final ValueType BYTE_OBJECT_TYPE = new ByteType(false);
	/**
	 * charで受け取る{@link ValueType}
	 */
	public static final ValueType CHAR_PRIMITIVE_TYPE = new CharType(true);
	/**
	 * {@link Character}で受け取る{@link ValueType}
	 */
	public static final ValueType CHAR_OBJECT_TYPE = new CharType(false);
	/**
	 * DBのDateTime型を {@link Date} で受け取る{@link ValueType}
	 */
	public static final ValueType DATE_TIMESTAMP_TYPE = new DateTimestampType();
	/**
	 * DBのDateTime型を long で受け取る{@link ValueType}
	 */
	public static final ValueType LONG_TIMESTAMP_PRIMITIVE_TYPE = new LongTimestampType(true);
	
	/**
	 * DBのDateTime型を long で受け取る{@link ValueType}
	 */
	public static final ValueType LONG_TIMESTAMP_OBJECT_TYPE = new LongTimestampType(false);
	
	
	
	/**
	 * doubleで受け取る{@link ValueType}
	 */
	public static final ValueType DOUBLE_PRIMITIVE_TYPE = new DoubleType(
			true);
	/**
	 * {@link Double}で受け取る{@link ValueType}
	 */
	public static final ValueType DOUBLE_OBJECT_TYPE = new DoubleType(false);

	/**
	 * floatで受け取る{@link ValueType}
	 */
	public static final ValueType FLOAT_PRIMITIVE_TYPE = new FloatType(true);
	/**
	 * {@link Float}で受け取る{@link ValueType}
	 */
	public static final ValueType FLOAT_OBJECT_TYPE = new FloatType(false);

	/**
	 * intで受け取る{@link ValueType}
	 */
	public static final ValueType INT_PRIMITIVE_TYPE = new IntegerType(true);
	/**
	 * {@link Integer}で受け取る{@link ValueType}
	 */
	public static final ValueType INT_OBJECT_TYPE = new IntegerType(false);

	/**
	 * longで受け取る{@link ValueType}
	 */
	public static final ValueType LONG_PRIMITIVE_TYPE = new LongType(true);
	/**
	 * {@link Long}で受け取る{@link ValueType}
	 */
	public static final ValueType LONG_OBJECT_TYPE = new LongType(false);

	/**
	 * {@link Object}で受け取る{@link ValueType}
	 */
	public static final ValueType OBJECT_TYPE = new ObjectType();

	/**
	 * shortで受け取る{@link ValueType}
	 */
	public static final ValueType SHORT_PRIMITIVE_TYPE = new ShortType(true);
	/**
	 * {@link Short}で受け取るValueType
	 */
	public static final ValueType SHORT_OBJECT_TYPE = new ShortType(false);

	/**
	 * {@link java.sql.Date}で受け取る{@link ValueType}
	 */
	public static final ValueType SQLDATE_OBJECT_TYPE = new SqlDateType();

	/**
	 * {@link Clob} を {@link String}で受け取る {@link ValueType}
	 */
	public static final ValueType STRING_CLOB_TYPE = new StringClobType();

	/**
	 * {@link String}で受け取る {@link ValueType}
	 */
	public static final ValueType STRING_TYPE = new StringType();

	public static final ValueType CHAR_NUMERIC_PRIMITIVE_TYPE = new CharNumericType(
			true);
	public static final ValueType CHAR_NUMERIC_OBJECT_TYPE = new CharNumericType(
			false);
	public static final ValueType STRING_NUMERIC_OBJECT_TYPE = new StringNumericType();

	public static Map valueTypes = new HashMap();
	public static Map defaultValueTypes = new HashMap();

	// staticイニシャライザで登録する。
	static {
		registerAllNumericValueType(BigDecimal.class, BIGDECIMAL_TYPE);

		registerValueType(boolean.class, Types.BOOLEAN, BOOLEAN_PRIMITIVE_TYPE);
		registerValueType(Boolean.class, Types.BOOLEAN, BOOLEAN_OBJECT_TYPE);

		registerValueType(boolean.class, Types.CHAR,
				BOOLEAN_CHAR_PRIMITIVE_TYPE);
		registerValueType(Boolean.class, Types.CHAR, BOOLEAN_CHAR_OBJECT_TYPE);

		registerAllNumericValueType(boolean.class,
				BOOLEAN_NUMERIC_PRIMITIVE_TYPE);
		registerAllNumericValueType(Boolean.class,
				BOOLEAN_NUMERIC_PRIMITIVE_TYPE);

		// バイナリ
		registerAllBinaryValueType(byte[].class, BYTEARRAY_TYPE);
		// byte
		registerAllNumericValueType(byte.class, BYTE_PRIMITIVE_TYPE);
		registerAllNumericValueType(Byte.class, BYTE_OBJECT_TYPE);
		// char
		registerAllStringValueType(char.class, CHAR_PRIMITIVE_TYPE);
		registerAllNumericValueType(char.class, CHAR_PRIMITIVE_TYPE);
		registerAllStringValueType(Character.class, CHAR_PRIMITIVE_TYPE);
		registerAllNumericValueType(Character.class, CHAR_PRIMITIVE_TYPE);
		registerAllNumericValueType(char.class, CHAR_NUMERIC_PRIMITIVE_TYPE);
		registerAllNumericValueType(Character.class, CHAR_NUMERIC_OBJECT_TYPE);

		// Date
		registerAllDateValueType(Date.class, DATE_TIMESTAMP_TYPE);
		registerAllDateValueType(long.class, LONG_TIMESTAMP_PRIMITIVE_TYPE);
		registerAllDateValueType(Long.class, LONG_TIMESTAMP_OBJECT_TYPE);
		
		
		// double
		registerAllNumericValueType(double.class, DOUBLE_PRIMITIVE_TYPE);
		registerAllNumericValueType(Double.class, DOUBLE_OBJECT_TYPE);
		// float
		registerAllNumericValueType(float.class, FLOAT_PRIMITIVE_TYPE);
		registerAllNumericValueType(Float.class, FLOAT_OBJECT_TYPE);
		// int
		registerAllNumericValueType(int.class, INT_PRIMITIVE_TYPE);
		registerAllNumericValueType(Integer.class, INT_OBJECT_TYPE);
		// long
		registerAllNumericValueType(long.class, LONG_PRIMITIVE_TYPE);
		registerAllNumericValueType(Long.class, LONG_OBJECT_TYPE);
		// short
		registerAllNumericValueType(short.class, SHORT_PRIMITIVE_TYPE);
		registerAllNumericValueType(Short.class, SHORT_OBJECT_TYPE);
		// java.sql.Date
		registerValueType(java.sql.Date.class, Types.DATE, SQLDATE_OBJECT_TYPE);
		// String
		registerAllStringValueType(String.class, STRING_TYPE);
		registerAllNumericValueType(String.class, STRING_TYPE);
		registerValueType(String.class, Types.CLOB, STRING_CLOB_TYPE);
		registerAllNumericValueType(String.class, STRING_NUMERIC_OBJECT_TYPE);
		registerAllDateValueType(String.class, STRING_TYPE);

		
		// Object
		registerAllNumericValueType(Object.class, OBJECT_TYPE);
		registerAllStringValueType(Object.class, OBJECT_TYPE);
		registerAllBinaryValueType(Object.class, OBJECT_TYPE);
		registerAllDateValueType(Object.class, OBJECT_TYPE);
		registerValueType(Object.class, Types.CLOB, STRING_CLOB_TYPE);
		registerValueType(Object.class, Types.BOOLEAN, STRING_CLOB_TYPE);
		// DEFAULT
		registerDefaultValueType(String.class, STRING_TYPE);
		registerDefaultValueType(char.class, CHAR_PRIMITIVE_TYPE);
		registerDefaultValueType(Character.class, CHAR_OBJECT_TYPE);

		registerDefaultValueType(boolean.class, BOOLEAN_PRIMITIVE_TYPE);
		registerDefaultValueType(Boolean.class, BOOLEAN_OBJECT_TYPE);

		registerDefaultValueType(int.class, INT_PRIMITIVE_TYPE);
		registerDefaultValueType(Integer.class, INT_OBJECT_TYPE);

		registerDefaultValueType(long.class, LONG_PRIMITIVE_TYPE);
		registerDefaultValueType(Long.class, LONG_OBJECT_TYPE);

		registerDefaultValueType(short.class, SHORT_PRIMITIVE_TYPE);
		registerDefaultValueType(Short.class, SHORT_OBJECT_TYPE);

		registerDefaultValueType(float.class, FLOAT_PRIMITIVE_TYPE);
		registerDefaultValueType(Float.class, FLOAT_OBJECT_TYPE);

		registerDefaultValueType(byte.class, BYTE_PRIMITIVE_TYPE);
		registerDefaultValueType(Byte.class, BYTE_OBJECT_TYPE);

		registerDefaultValueType(double.class, DOUBLE_PRIMITIVE_TYPE);
		registerDefaultValueType(Double.class, DOUBLE_OBJECT_TYPE);

		registerDefaultValueType(BigDecimal.class, BIGDECIMAL_TYPE);

		registerDefaultValueType(Date.class, DATE_TIMESTAMP_TYPE);
	}

	static class Key {
		final Class clazz;
		final int sqlType;

		public Key(Class clazz, int sqlType) {
			this.clazz = clazz;
			this.sqlType = sqlType;
		}

		public int hashCode() {
			int result = 17;
			result = result * 31 + clazz.hashCode();
			result = result * 31 + sqlType;
			return result;
		}

		public boolean equals(Object obj) {
			if (obj instanceof Key) {
				Key other = (Key) obj;
				return clazz.equals(other.clazz) && sqlType == other.sqlType;
			}
			return false;
		}
	}

	static void registerValueType(Class clazz, int sqlType, ValueType type) {
		registerValueType(new Key(clazz, sqlType), type);
	}

	/*
	 * 数値型のタイプをセットする
	 */
	static void registerAllNumericValueType(Class clazz, ValueType type) {
		registerValueType(clazz, Types.DECIMAL, type);
		registerValueType(clazz, Types.NUMERIC, type);
		registerValueType(clazz, Types.INTEGER, type);
		registerValueType(clazz, Types.BIGINT, type);
		registerValueType(clazz, Types.REAL, type);
		registerValueType(clazz, Types.DOUBLE, type);
		registerValueType(clazz, Types.FLOAT, type);
		registerValueType(clazz, Types.SMALLINT, type);
		registerValueType(clazz, Types.TINYINT, type);
	}

	/*
	 * 文字型のタイプをセットする
	 */
	static void registerAllStringValueType(Class clazz, ValueType type) {
		registerValueType(clazz, Types.VARCHAR, type);
		registerValueType(clazz, Types.LONGVARCHAR, type);
		registerValueType(clazz, Types.CHAR, type);
	}

	/*
	 * バイナリ型のタイプをセットする
	 */
	static void registerAllBinaryValueType(Class clazz, ValueType type) {
		registerValueType(clazz, Types.BINARY, type);
		registerValueType(clazz, Types.BLOB, type);
		registerValueType(clazz, Types.VARBINARY, type);
		registerValueType(clazz, Types.LONGVARBINARY, type);
	}

	/*
	 * 日付型のタイプをセットする
	 */
	static void registerAllDateValueType(Class clazz, ValueType type) {
		registerValueType(clazz, Types.TIMESTAMP, type);
		registerValueType(clazz, Types.DATE, type);
		registerValueType(clazz, Types.TIME, type);
	}

	static void registerValueType(Key key, ValueType type) {
		valueTypes.put(key, type);

	}

	/**
	 * デフォルトのValueTypeを登録する。
	 * 
	 * @param clazz
	 * @param type
	 */
	static void registerDefaultValueType(Class clazz, ValueType type) {
		defaultValueTypes.put(clazz, type);

	}

	/**
	 * ValueTypeを取得します。
	 * 
	 * @param clazz
	 * @param sqlType
	 */
	public static ValueType getValueType(Class clazz, int sqlType) {
		ValueType result = (ValueType) valueTypes.get(new Key(clazz, sqlType));
		return result == null ? OBJECT_TYPE : result;
	}

	public static ValueType getValueType(int sqlType) {
		switch (sqlType) {
		case Types.CHAR:
		case Types.VARCHAR:
		case Types.LONGVARCHAR:
			return STRING_TYPE;
		case Types.BIGINT:
		case Types.DECIMAL:
		case Types.DOUBLE:
		case Types.NUMERIC:
		case Types.INTEGER:
		case Types.FLOAT:
		case Types.SMALLINT:
		case Types.TINYINT:
		case Types.REAL:
			return BIGDECIMAL_TYPE;
		case Types.CLOB:
			return STRING_CLOB_TYPE;
		case Types.BLOB:
		case Types.VARBINARY:
		case Types.LONGVARBINARY:
		case Types.BINARY:
			return BYTEARRAY_TYPE;
		case Types.DATE:
		case Types.TIMESTAMP:
		case Types.TIME:
			return DATE_TIMESTAMP_TYPE;
		default:
			return OBJECT_TYPE;
		}
	}

	public static ValueType getValueType(Class type) {
		ValueType result = (ValueType)defaultValueTypes.get(type);
		return (result == null) ? OBJECT_TYPE : result;

	}
}
