<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Controller.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

/**
 * バッファリング設定
 */
ob_start();
ob_implicit_flush(0);

/**
 * magic_quotes_runtime -> off
 */
set_magic_quotes_runtime(0);

/**
 * 設定ファイルクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_Config.php';
/**
 * コンテナクラス
 */
require_once SYL_FRAMEWORK_DIR . '/core/SyL_Container.php';

/**
 * コントローラクラス
 *
 * 全体の処理の流れを管理する
 * 主な内容は
 *   ・DIコンテナの保持
 *   ・処理の流れに伴う遷移イベント処理
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_Controller.php,v 1.1 2009/01/11 05:34:30 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_Controller
{
    /**
     * エラーハンドラメソッド名
     * ''の場合はエラーハンドラを設定しない
     *
     * ※SyL_Errorで使用している
     *
     * @access public
     * @var string
     */
    var $error_handler_method = 'errorStream';
    /**
     * 例外ハンドラメソッド名
     * ''の場合は例外ハンドラを設定しない
     *
     * ※SyL_Errorで使用している
     *
     * @access public
     * @var string
     */
    var $exception_handler_method = 'exceptionStream';
    /**
     * 処理後破棄メソッド名
     * ''の場合は処理後破棄メソッドを設定しない
     *
     * @access private
     * @var string
     */
    var $destory_handler_method = 'finalStream';
    /**
     * エラー配列
     *
     * @access private
     * @var array
     */
    var $error = array();
    /**
     * コンテナオブジェクト
     *
     * @access private
     * @var object
     */
    var $container = null;

    /**
     * コンストラクタ
     *
     * @access protected
     * @param array 環境設定配列
     */
    function SyL_Controller($config)
    {
        // アプリケーション名定義
        if (!isset($config['app_name']) || !$config['app_name']) {
            trigger_error("[SyL error] App_name not found", E_USER_ERROR);
        }

        // アクションキー定義
        define('SYL_ACTION_KEY', (isset($config['action_key']) && $config['action_key']) ? $config['action_key'] : 'action');
        // プロジェクトディレクトリ定義
        define('SYL_PROJECT_DIR', $config['project_dir']);
        // プロジェクト設定ファイルディレクトリ定義
        define('SYL_PROJECT_CONFIG_DIR',  SYL_PROJECT_DIR . '/config');
        // プロジェクトライブラリディレクトリ定義
        define('SYL_PROJECT_LIB_DIR',  SYL_PROJECT_DIR . '/lib');
        // アプリケーション名
        define('SYL_APP_NAME', $config['app_name']);
        // アプリケーションディレクトリ定義
        define('SYL_APP_DIR', SYL_PROJECT_DIR . '/apps/' . SYL_APP_NAME);
        // アプリケーション設定ファイルディレクトリ定義
        define('SYL_APP_CONFIG_DIR', SYL_APP_DIR . '/config');
        // アプリケーションライブラリディレクトリ定義
        define('SYL_APP_LIB_DIR', SYL_APP_DIR . '/lib');
        // アプリケーションキャッシュディレクトリ定義
        define('SYL_APP_CACHE_DIR', SYL_PROJECT_DIR . '/var/cache/' . SYL_APP_NAME . '/');
        // アプリケーションキャッシュ使用の有無
        define('SYL_CACHE', (isset($config['cache']) && $config['cache']));

        // PHPシステムエラーログの保存
        // カレントでログを取得していない場合、ログを保存する
        if (!isset($config['syslog']) || $config['syslog']) {
            if (!ini_get('log_errors')) {
                ini_set('log_errors', true);
                ini_set('error_log', SYL_PROJECT_DIR . '/var/syslogs/' . SYL_APP_NAME . '/phperror_' . date('Ymd') . '.log');
            }
        }

        // 設定定数読み込み
        SyL_Config::get();

        // コンテナ生成
        $this->container =& SyL_Container::singleton();
        $this->container->setComponent('controller', $this);

        // shutdownハンドラ定義
        if ($this->destory_handler_method != '') {
            register_shutdown_function(array(&$this, $this->destory_handler_method));
        }
    }

    /**
     * エントリポイント
     *
     * @static
     * @access public
     * @param array 環境設定配列
     */
    function streaming($config)
    {
        if (isset($config['type']) && $config['type']) {
            define('SYL_ENV_TYPE', strtolower($config['type']));
        } else {
            if ((PHP_SAPI == 'cgi') && !isset($_SERVER['REQUEST_METHOD'])) {
                trigger_error("[SyL error] Command line SAPI is CLI only", E_USER_ERROR);
            } else if (PHP_SAPI == 'cli') {
                define('SYL_ENV_TYPE', 'cmd');
            } else {
                define('SYL_ENV_TYPE', 'web');
            }
        }
        $class_name = 'SyL_Controller' . ucfirst(SYL_ENV_TYPE);
        include_once SYL_FRAMEWORK_DIR . "/core/Controller/{$class_name}.php";
        $singleton = new $class_name($config);
        $singleton->stream();
    }

    /**
     * 初期化処理
     *
     * @access protected
     */
    function initStream()
    {
        // イベント実行
        $this->raiseEvent('initStream', 'components');

        // リクエスト初期ロギング
        if (SYL_ENV_TYPE == 'cmd') {
            $script_file = isset($_SERVER['SCRIPT_FILENAME']) ? $_SERVER['SCRIPT_FILENAME'] : '';
            SyL_Loggers::info("cmd start: {$script_file}");
        } else {
            $method = isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : '';
            $addr   = isset($_SERVER['REMOTE_ADDR'])    ? $_SERVER['REMOTE_ADDR']    : '';
            $uri    = isset($_SERVER['REQUEST_URI'])    ? $_SERVER['REQUEST_URI']    : '';
            SyL_Loggers::info("request start: \"{$method}: {$uri}\" - {$addr}");
        }
    }

    /**
     * アクション前処理実行
     *
     * @access protected
     */
    function loadStream()
    {
        // イベント実行
        $this->raiseEvent('loadStream', 'filters');
    }

    /**
     * アクション処理実行
     *
     * @access protected
     */
    function executeStream()
    {
        $context =& $this->container->getComponent('context');
        if ($context->isExecute()) {
            // イベント実行
            $this->raiseEvent('executeStream', 'actions');
        }
    }

    /**
     * ビュー表示実行前処理
     *
     * @access protected
     */
    function middleStream()
    {
        // イベント実行
        $this->raiseEvent('middleStream');
    }

    /**
     * ビュー表示処理
     *
     * @access protected
     */
    function renderStream()
    {
        // イベント実行
        $this->raiseEvent('renderStream');
    }

    /**
     * ビュー表示実行後処理
     *
     * @access protected
     */
    function unloadStream()
    {
        // イベント実行
        $this->raiseEvent('unloadStream');
    }

    /**
     * 最終処理
     * ※register_shutdown_function使用
     *
     * @access protected
     */
    function finalStream()
    {
        // メモリ量デバック
        if (function_exists('memory_get_usage') && class_exists('SyL_Loggers')) {
            $debug = 'Memory Status  current: ' . number_format(floor(memory_get_usage()/1024)) . 'KB';
            if (function_exists('memory_get_peak_usage')) {
                $debug .= ' max: ' . number_format(floor(memory_get_peak_usage()/1024)) . 'KB';
            }
            SyL_Loggers::debug($debug);
        }

        // イベント実行
        $this->raiseEvent('finalStream');
    }

    /**
     * 例外ハンドラ実行処理
     *
     * @access protected
     * @param object 例外オブジェクト
     */
    function exceptionStream($ex)
    {
        $this->errorStream(E_USER_ERROR, $ex->getMessage(), $ex->getFile(), $ex->getLine(), array(), $ex->getTrace());
    }

    /**
     * エラーハンドラ実行処理
     *
     * @access protected
     * @param int エラーのレベル
     * @param string エラーメッセージ
     * @param string ファイル
     * @param int 行番号
     * @param array コンテキスト
     * @param array バックトレース
     */
    function errorStream()
    {
        static $num = 1;
        // 現在のエラーレベルからエラー画面を表示するか判定
        if (func_get_arg(0) & error_reporting()) {
            if ($num++ > 1) {
                $error_message = "[SyL error] Error handler trigger 2";
                SyL_Loggers::warn($error_message);
                echo $error_message;
                exit;
            }

            for ($i=0; $i<5; $i++) {
                $this->error[$i] = func_get_arg($i);
            }
            if (func_num_args() == 6) {
                $this->error[5] = func_get_arg(5);
            } else {
                $this->error[5] = debug_backtrace();
                // errorStream()メソッドは削除
                array_shift($this->error[5]);
            }

            if (count($this->error[5]) > 0) {
                if (!isset($this->error[5][0]['file'])) {
                    $this->error[5][0]['file'] = $this->error[2];
                }
                if (!isset($this->error[5][0]['line'])) {
                    $this->error[5][0]['line'] = $this->error[3];
                }
            }

            // イベント実行
            $this->raiseEvent('errorStream');
            exit;
        } else {
            $error_message = "[SyL_error] Background logging out of error_reporting. level: " . func_get_arg(0) . " message: " . func_get_arg(1) . " file: " . func_get_arg(2) . " line: " . func_get_arg(3);
            SyL_Loggers::notice($error_message);
        }
    }

    /**
     * 実行環境ごとのフロー制御
     *
     * @access abstract
     */
    function stream()
    {
    }

    /**
     * イベント実行
     *
     * @access protected
     * @param string イベントメソッド名
     * @param string 設定ファイル名
     */
    function raiseEvent($event, $config_type='')
    {
        $this->container->raiseEvent($event, $config_type);
    }

    /**
     * エラーNo取得
     * 
     * @access public
     * @return string エラーNo
     */
    function getErrorNo()
    {
        return isset($this->error[0]) ? $this->error[0] : '';
    }

    /**
     * エラーファイルを取得
     * 
     * @access public
     * @return string エラーファイル
     */
    function getErrorFile()
    {
        return isset($this->error[2]) ? $this->error[2] : '';
    }

    /**
     * エラー行数を取得
     * 
     * @access public
     * @return string エラー行数
     */
    function getErrorLine()
    {
        return isset($this->error[3]) ? $this->error[3] : '';
    }

    /**
     * エラーメッセージ取得
     * 
     * @access public
     * @return string エラーメッセージ
     */
    function getErrorMessage()
    {
        $error_message = isset($this->error[1]) ? $this->error[1] : '';
        if ($error_message) {
            $error_message .= ' (' . $this->getErrorFile() . ' on line ' . $this->getErrorLine()  . ')';
        }
        return $error_message;
    }

    /**
     * エラーが発生したスコープ内でのすべての変数の内容を取得
     * 
     * @access public
     * @return array エラーコンテキスト
     */
    function getErrorContext()
    {
        return isset($this->error[4]) ? $this->error[4] : array();
    }

    /**
     * エラーが発生したデバックトレース
     * 
     * @access public
     * @return array デバックトレース
     */
    function getErrorTrace()
    {
        return isset($this->error[5]) ? $this->error[5] : array();
    }
}
