<?php
/**
 * -----------------------------------------------------------------------------
 *
 * SyL - Web Application Framework for PHP
 *
 * PHP version 4 (>= 4.3.x) or 5
 *
 * Copyright (C) 2006-2009 k.watanabe
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * -----------------------------------------------------------------------------
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheMemcache.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 * -----------------------------------------------------------------------------
 */

if (!defined('SYL_CACHE_MEMCACHE_HOST')) {
    define('SYL_CACHE_MEMCACHE_HOST', 'localhost:11211');
}

/**
 * Memcacheキャッシュクラス
 *
 * @package   SyL
 * @author    Koki Watanabe <k.watanabe@syl.jp>
 * @copyright 2006-2009 k.watanabe
 * @license   http://www.opensource.org/licenses/lgpl-license.php
 * @version   CVS: $Id: SyL_CacheMemcache.php,v 1.1 2009/01/11 05:34:31 seasonstream Exp $
 * @link      http://syl.jp/
 */
class SyL_CacheMemcache extends SyL_Cache
{
    /**
     * memcacheオブジェクト
     *
     * @access private
     * @var string
     */
    var $memcache = null;
    /**
     * キャッシュ更新時間の接尾辞
     *
     * @access private
     * @var string
     */
    var $update_suffix = '_update';
    /**
     * Memcache圧縮フラグ
     *
     * @access private
     * @var int
     */
    var $compress = 0;

    /**
     * Memcachedと接続
     *
     * @access public
     * @param string ホスト名
     * @param int ポート番号
     * @return bool true: 成功、false: 失敗
     */
    function SyL_CacheMemcache($host='')
    {
        list($host, $port) = ($host) ? explode(':', $host, 2) : explode(':', SYL_CACHE_MEMCACHE_HOST, 2);

        static $singleton = array();
        if (!isset($singleton[$host.$port])) {
            $singleton[$host.$port] = new Memcache();
            if (!$singleton[$host.$port]->connect($host, $port)) {
                trigger_error("[SyL error] Unable to connect memcahed server ({$host}:{$port})", E_USER_ERROR);
            }
        }

        $this->memcache =& $singleton[$host.$port];
    }

    /**
     * Memcachedとの接続断
     *
     * @access public
     */
    function close()
    {
        if ($this->memcache != null) {
            $this->memcache->close();
            $this->memcache = null;
        }
    }

    /**
     * キャッシュの更新時間を更新する
     *
     * @access public
     * @param int 更新時間(UNIX Time)
     * @return bool true: 更新OK、false: 更新エラー
     */
    function setModifyTime($mtime=0)
    {
        $key  = $this->getKey();
        $data = $this->memcache->get($key);
        if ($data) {
            $limit = 0;
            if ($this->life_time > 0) {
                $limit = ($mtime) ? $mtime - time() + $this->life_time : $this->life_time;
            }
            if ($this->memcache->replace($key, $data, $this->compress, $limit)) {
                if (!$mtime) {
                    $mtime = time();
                }
                $key_updaete = $key . $this->update_suffix;
                return $this->memcache->replace($key_updaete, $mtime, $this->compress, $limit);
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * キャッシュファイルの更新時間を取得する
     *
     * @access public
     * @return int 更新時間(UNIX Time)
     */
    function getModifyTime()
    {
        $key_updaete = $this->getKey() . $this->update_suffix;
        return $this->memcache->get($key_updaete);
    }

    /**
     * キャッシュを読み込む
     *
     * @access public
     * @param bool キャッシュ有効期間を更新
     * @return mixed キャッシュデータ
     */
    function read($modify_time_update=false)
    {
        $key = $this->getKey();
        $data = $this->memcache->get($key);
        if ($data !== false) {
            $key_updaete = $key . $this->update_suffix;
            $mtime = $this->memcache->get($key_updaete);
            // 更新時間の最小値判定
            if ($mtime && ($this->min_mtime > 0)) {
                if ($mtime < $this->min_mtime) {
                    $this->delete();
                    return false;
                }
            }
            // 更新時間の最大値判定
            if ($mtime && ($this->max_mtime > 0)) {
                if ($mtime > $this->max_mtime) {
                    $this->delete();
                    return false;
                }
            }

            if ($modify_time_update) {
                $this->setModifyTime();
            }
            return $data;
        } else {
            // キャッシュファイル無し or 期限切れ
            return false;
        }
    }

    /**
     * キャッシュを保存する
     *
     * @access public
     * @param mixed キャッシュデータ
     * @return bool true: 保存成功、false: 保存エラー
     */
    function write($data)
    {
        $key = $this->getKey();
        if ($this->memcache->set($key, $data, $this->compress, $this->life_time)) {
            $key_updaete = $key . $this->update_suffix;
            $this->memcache->set($key_updaete, time(), $this->compress, $this->life_time);
            return true;
        } else {
            return false;
        }
    }

    /**
     * キャッシュを削除する
     * 
     * @access public
     * @return bool true: 削除成功、false: 削除エラー
     */
    function delete()
    {
        $key         = $this->getKey();
        $key_updaete = $key . $this->update_suffix;
        $this->memcache->delete($key_updaete);
        return $this->memcache->delete($key);
    }

    /**
     * キャッシュ中の期限切れキャッシュを削除する
     *
     * @access public
     */
    function clean()
    {
    }

    /**
     * キャッシュを全て削除する
     *
     * @access public
     */
    function cleanAll()
    {
        $this->memcache->flush();
    }

    /**
     * Memcache圧縮フラグをセットする
     *
     * @access public
     * @param int Memcache圧縮フラグ
     */
    function setCompress($compress)
    {
        $this->compress = $compress;
    }
}
